/*  Inti: Integrated Foundation Classes
 *  Copyright (C) 2002-2003 The Inti Development Team.
 *
 *  text.cc - AtkText C++ wrapper implementation
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 
#include "text.h"
#include "private/text_p.h"
#include "../glib/object.h"
#include "../glib/unicode.h"
#include "../glib/object.h"

using namespace Inti;

/*  Atk::text_attribute_register
 */

Atk::TextAttribute
Atk::text_attribute_register(const String& name)
{
	return (TextAttribute)atk_text_attribute_register(name.c_str());
}

/*  Atk::text_attribute_get_name
 */

String 
Atk::text_attribute_get_name(TextAttribute attr)
{
	return atk_text_attribute_get_name((AtkTextAttribute)attr);
}

/*  Atk::text_attribute_for_name
 */

Atk::TextAttribute
Atk::text_attribute_for_name(const String& name)
{
	return (TextAttribute)atk_text_attribute_for_name(name.c_str());
}

/*  Atk::text_attribute_get_value
 */

String 
Atk::text_attribute_get_value(TextAttribute attr, int index)
{
	return atk_text_attribute_get_value((AtkTextAttribute)attr, index);
}

namespace { // copy and free AtkAttribute

void copy_attribute(AtkAttribute *dest, const AtkAttribute *src)
{
	dest->name = g_strdup(src->name);
	dest->value = g_strdup(src->value);
}

void free_attribute(AtkAttribute *attrib)
{
	if (attrib->name)
	{
		g_free(attrib->name);
		attrib->name = 0;
	}

	if (attrib->value)
	{
		g_free(attrib->value);
		attrib->value = 0;
	}
}

} // copy and free AtkAttribute

/*  Atk::Attribute
 */

Atk::Attribute::Attribute()
{
	attrib_.name = 0;
	attrib_.value = 0;
}

Atk::Attribute::Attribute(TextAttribute attr, int index)
{
	set(text_attribute_get_name(attr), text_attribute_get_value(attr, index));
}

Atk::Attribute::Attribute(TextAttribute attr, const String& value)
{
	set(text_attribute_get_name(attr), value);
}

Atk::Attribute::Attribute(const String& name, const String& value)
{
	set(name, value);
}

Atk::Attribute::Attribute(const AtkAttribute& attrib)
{
	copy_attribute(&attrib_, &attrib);
}

Atk::Attribute::Attribute(const Attribute& src)
{
	copy_attribute(&attrib_, &src.attrib_);
}

Atk::Attribute::~Attribute()
{
	free_attribute(&attrib_);
}

Atk::Attribute& 
Atk::Attribute::operator=(const Attribute& src)
{
	if (this != &src)
	{
		free_attribute(&attrib_);
		copy_attribute(&attrib_, &src.attrib_);
	}
	return *this;
}

Atk::TextAttribute
Atk::Attribute::attr() const
{
	return (TextAttribute)atk_text_attribute_for_name(attrib_.name);
}

String
Atk::Attribute::name() const
{
	return attrib_.name;
}

String 
Atk::Attribute::value() const
{
	return attrib_.value;
}
	
void 
Atk::Attribute::set(const String& name, const String& value)
{
	free_attribute(&attrib_);
	attrib_.name = g_strndup(name.c_str(), name.size());
	attrib_.value = g_strndup(value.c_str(), value.size());
}

/*  Atk::Text
 */

Atk::Text::~Text()
{
}

AtkTextIface*
Atk::Text::atk_text_iface() const 
{
	return peek<AtkTextIface>(ATK_TYPE_TEXT); 
}

Atk::Text::operator AtkText * () const
{
	return this ? atk_text() : 0;
}

String 
Atk::Text::get_text(int start_offset, int end_offset) const
{
	char *text = atk_text_get_text(atk_text(), start_offset, end_offset);
	String s(text);
	g_free(text);
	return s;
}

G::Unichar 
Atk::Text::get_character_at_offset(int offset) const
{
	return atk_text_get_character_at_offset(atk_text(), offset);
}

String 
Atk::Text::get_text_after_offset(int offset, TextBoundary boundary_type, int *start_offset, int *end_offset) const
{
	char *text = atk_text_get_text_after_offset(atk_text(), offset, (AtkTextBoundary)boundary_type, start_offset, end_offset);
	String s(text);
	g_free(text);
	return s;
}

String
Atk::Text::get_text_at_offset(int offset, TextBoundary boundary_type, int *start_offset, int *end_offset) const
{
	char *text = atk_text_get_text_at_offset(atk_text(), offset, (AtkTextBoundary)boundary_type, start_offset, end_offset);
	String s(text);
	g_free(text);
	return s;
}

String 
Atk::Text::get_text_before_offset(int offset, TextBoundary boundary_type, int *start_offset, int *end_offset) const
{
	char *text = atk_text_get_text_before_offset(atk_text(), offset, (AtkTextBoundary)boundary_type, start_offset, end_offset);
	String s(text);
	g_free(text);
	return s;
}

int 
Atk::Text::get_caret_offset() const
{
	return atk_text_get_caret_offset(atk_text());
}

void 
Atk::Text::get_character_extents(int offset, int *x, int *y, int *width, int *height, CoordType coords) const
{
	atk_text_get_character_extents(atk_text(), offset, x, y, width, height, (AtkCoordType)coords);
}

std::vector<Atk::Attribute>
Atk::Text::get_run_attributes(int offset, int *start_offset, int *end_offset) const
{
	std::vector<Attribute> attribs;
	GSList *first = atk_text_get_run_attributes(atk_text(), offset, start_offset, end_offset);
	GSList *next = first;

	while (next != 0)
	{
		Attribute attrib(*((AtkAttribute*)next->data));
		attribs.push_back(attrib);
		next = g_slist_next(next);
	}

	atk_attribute_set_free(first);
	return attribs;
}

std::vector<Atk::Attribute>
Atk::Text::get_default_attributes() const
{
	std::vector<Attribute> attribs;
	GSList *first = atk_text_get_default_attributes(atk_text());
	GSList *next = first;

	while (next != 0)
	{
		Attribute attrib(*((AtkAttribute*)next->data));
		attribs.push_back(attrib);
		next = g_slist_next(next);
	}
	
	atk_attribute_set_free(first);
	return attribs;
}
	
int 
Atk::Text::get_character_count() const
{
	return atk_text_get_character_count(atk_text());
}

int
Atk::Text::get_offset_at_point(int x, int y, CoordType coords) const
{
	return atk_text_get_offset_at_point(atk_text(), x, y, (AtkCoordType)coords);
}

int 
Atk::Text::get_n_selections() const
{
	return atk_text_get_n_selections(atk_text());
}

String 
Atk::Text::get_selection(int selection_num, int *start_offset, int *end_offset) const
{
	return atk_text_get_selection(atk_text(), selection_num, start_offset, end_offset);
}

bool 
Atk::Text::add_selection(int start_offset, int end_offset)
{
	return atk_text_add_selection(atk_text(), start_offset, end_offset);
}

bool 
Atk::Text::remove_selection(int selection_num)
{
	return atk_text_remove_selection(atk_text(), selection_num);
}

bool 
Atk::Text::set_selection(int selection_num, int start_offset, int end_offset)
{
	return atk_text_set_selection(atk_text(), selection_num, start_offset, end_offset);
}

bool 
Atk::Text::set_caret_offset(int offset)
{
	return atk_text_set_caret_offset(atk_text(), offset);
}

/*  Atk::TextIface
 */

void
Atk::TextIface::init(AtkTextIface *g_iface)
{
	g_iface->get_text = &get_text_proxy;
	g_iface->get_text_after_offset = &get_text_after_offset_proxy;
	g_iface->get_text_at_offset = &get_text_at_offset_proxy;
	g_iface->get_character_at_offset = &get_character_at_offset_proxy;
	g_iface->get_text_before_offset = &get_text_before_offset_proxy;
	g_iface->get_caret_offset = &get_caret_offset_proxy;
	g_iface->get_run_attributes = &get_run_attributes_proxy;
	g_iface->get_default_attributes = &get_default_attributes_proxy;
	g_iface->get_character_extents = &get_character_extents_proxy;
	g_iface->get_character_count = &get_character_count_proxy;
	g_iface->get_offset_at_point = &get_offset_at_point_proxy;
	g_iface->get_n_selections = &get_n_selections_proxy;
	g_iface->get_selection = &get_selection_proxy;
	g_iface->add_selection = &add_selection_proxy;
	g_iface->remove_selection = &remove_selection_proxy;
	g_iface->set_selection = &set_selection_proxy;
	g_iface->set_caret_offset = &set_caret_offset_proxy;
	g_iface->text_changed = &text_changed_proxy;
	g_iface->text_caret_moved = &text_caret_moved_proxy;
	g_iface->text_selection_changed = &text_selection_changed_proxy;
	g_iface->text_attributes_changed = text_attributes_changed_proxy;
}

gchar*
Atk::TextIface::get_text_proxy(AtkText *text, gint start_offset, gint end_offset)
{
	gchar *result = 0;
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(text));
	Atk::Text *tmp_text = dynamic_cast<Atk::Text*>(object);
	if (tmp_text)
		result = tmp_text->do_get_text(start_offset, end_offset);
	else
	{
		AtkTextIface *g_iface = static_cast<AtkTextIface*>(g_type_interface_peek_parent(ATK_TEXT_GET_IFACE(text)));
		if (g_iface->get_text)
			result = g_iface->get_text(text, start_offset, end_offset);
	}
	return result;
}

gchar*
Atk::TextIface::get_text_after_offset_proxy(AtkText *text, gint offset, AtkTextBoundary boundary_type, gint *start_offset, gint *end_offset)
{
	gchar *result = 0;
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(text));
	Atk::Text *tmp_text = dynamic_cast<Atk::Text*>(object);
	if (tmp_text)
		result = tmp_text->do_get_text_after_offset(offset, boundary_type, start_offset, end_offset);
	else
	{
		AtkTextIface *g_iface = static_cast<AtkTextIface*>(g_type_interface_peek_parent(ATK_TEXT_GET_IFACE(text)));
		if (g_iface->get_text_after_offset)
			result = g_iface->get_text_after_offset(text, offset, boundary_type, start_offset, end_offset);
	}
	return result;
}

gchar*
Atk::TextIface::get_text_at_offset_proxy(AtkText *text, gint offset, AtkTextBoundary boundary_type, gint *start_offset, gint *end_offset)
{
	gchar *result = 0;
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(text));
	Atk::Text *tmp_text = dynamic_cast<Atk::Text*>(object);
	if (tmp_text)
		result = tmp_text->do_get_text_at_offset(offset, boundary_type, start_offset, end_offset);
	else
	{
		AtkTextIface *g_iface = static_cast<AtkTextIface*>(g_type_interface_peek_parent(ATK_TEXT_GET_IFACE(text)));
		if (g_iface->get_text_at_offset)
			result = g_iface->get_text_at_offset(text, offset, boundary_type, start_offset, end_offset);
	}
	return result;
}

gunichar
Atk::TextIface::get_character_at_offset_proxy(AtkText *text, gint offset)
{
	gunichar result = 0;
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(text));
	Atk::Text *tmp_text = dynamic_cast<Atk::Text*>(object);
	if (tmp_text)
		result = tmp_text->do_get_character_at_offset(offset);
	else
	{
		AtkTextIface *g_iface = static_cast<AtkTextIface*>(g_type_interface_peek_parent(ATK_TEXT_GET_IFACE(text)));
		if (g_iface->get_character_at_offset)
			result = g_iface->get_character_at_offset(text, offset);
	}
	return result;
}

gchar*
Atk::TextIface::get_text_before_offset_proxy(AtkText *text, gint offset, AtkTextBoundary boundary_type, gint *start_offset, gint *end_offset)
{
	gchar *result = 0;
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(text));
	Atk::Text *tmp_text = dynamic_cast<Atk::Text*>(object);
	if (tmp_text)
		result = tmp_text->do_get_text_before_offset(offset, boundary_type, start_offset, end_offset);
	else
	{
		AtkTextIface *g_iface = static_cast<AtkTextIface*>(g_type_interface_peek_parent(ATK_TEXT_GET_IFACE(text)));
		if (g_iface->get_text_before_offset)
			result = g_iface->get_text_before_offset(text, offset, boundary_type, start_offset, end_offset);
	}
	return result;
}

gint
Atk::TextIface::get_caret_offset_proxy(AtkText *text)
{
	gint result = 0;
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(text));
	Atk::Text *tmp_text = dynamic_cast<Atk::Text*>(object);
	if (tmp_text)
		result = tmp_text->do_get_caret_offset();
	else
	{
		AtkTextIface *g_iface = static_cast<AtkTextIface*>(g_type_interface_peek_parent(ATK_TEXT_GET_IFACE(text)));
		if (g_iface->get_caret_offset)
			result = g_iface->get_caret_offset(text);
	}
	return result;
}

AtkAttributeSet*
Atk::TextIface::get_run_attributes_proxy(AtkText *text, gint offset, gint *start_offset, gint *end_offset)
{
	AtkAttributeSet *result = 0;
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(text));
	Atk::Text *tmp_text = dynamic_cast<Atk::Text*>(object);
	if (tmp_text)
		result = tmp_text->do_get_run_attributes(offset, start_offset, end_offset);
	else
	{
		AtkTextIface *g_iface = static_cast<AtkTextIface*>(g_type_interface_peek_parent(ATK_TEXT_GET_IFACE(text)));
		if (g_iface->get_run_attributes)
			result = g_iface->get_run_attributes(text, offset, start_offset, end_offset);
	}
	return result;
}

AtkAttributeSet*
Atk::TextIface::get_default_attributes_proxy(AtkText *text)
{
	AtkAttributeSet *result = 0;
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(text));
	Atk::Text *tmp_text = dynamic_cast<Atk::Text*>(object);
	if (tmp_text)
		result = tmp_text->do_get_default_attributes();
	else
	{
		AtkTextIface *g_iface = static_cast<AtkTextIface*>(g_type_interface_peek_parent(ATK_TEXT_GET_IFACE(text)));
		if (g_iface->get_default_attributes)
			result = g_iface->get_default_attributes(text);
	}
	return result;
}

void
Atk::TextIface::get_character_extents_proxy(AtkText *text, gint offset, gint *x, gint *y, gint *width, gint *height, AtkCoordType coords)
{
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(text));
	Atk::Text *tmp_text = dynamic_cast<Atk::Text*>(object);
	if (tmp_text)
		tmp_text->do_get_character_extents(offset, x, y, width, height, coords);
	else
	{
		AtkTextIface *g_iface = static_cast<AtkTextIface*>(g_type_interface_peek_parent(ATK_TEXT_GET_IFACE(text)));
		if (g_iface->get_character_extents)
			g_iface->get_character_extents(text, offset, x, y, width, height, coords);
	}
}

gint
Atk::TextIface::get_character_count_proxy(AtkText *text)
{
	gint result = 0;
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(text));
	Atk::Text *tmp_text = dynamic_cast<Atk::Text*>(object);
	if (tmp_text)
		result = tmp_text->do_get_character_count();
	else
	{
		AtkTextIface *g_iface = static_cast<AtkTextIface*>(g_type_interface_peek_parent(ATK_TEXT_GET_IFACE(text)));
		if (g_iface->get_character_count)
			result = g_iface->get_character_count(text);
	}
	return result;
}

gint
Atk::TextIface::get_offset_at_point_proxy(AtkText *text, gint  x, gint y, AtkCoordType coords)
{
	gint result = 0;
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(text));
	Atk::Text *tmp_text = dynamic_cast<Atk::Text*>(object);
	if (tmp_text)
		result = tmp_text->do_get_offset_at_point(x, y, coords);
	else
	{
		AtkTextIface *g_iface = static_cast<AtkTextIface*>(g_type_interface_peek_parent(ATK_TEXT_GET_IFACE(text)));
		if (g_iface->get_offset_at_point)
			result = g_iface->get_offset_at_point(text, x, y, coords);
	}
	return result;
}

gint
Atk::TextIface::get_n_selections_proxy(AtkText *text)
{
	gint result = 0;
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(text));
	Atk::Text *tmp_text = dynamic_cast<Atk::Text*>(object);
	if (tmp_text)
		result = tmp_text->do_get_n_selections();
	else
	{
		AtkTextIface *g_iface = static_cast<AtkTextIface*>(g_type_interface_peek_parent(ATK_TEXT_GET_IFACE(text)));
		if (g_iface->get_n_selections)
			result = g_iface->get_n_selections(text);
	}
	return result;
}

gchar*
Atk::TextIface::get_selection_proxy(AtkText *text, gint selection_num, gint *start_offset, gint *end_offset)
{
	gchar *result = 0;
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(text));
	Atk::Text *tmp_text = dynamic_cast<Atk::Text*>(object);
	if (tmp_text)
		result = tmp_text->do_get_selection(selection_num, start_offset, end_offset);
	else
	{
		AtkTextIface *g_iface = static_cast<AtkTextIface*>(g_type_interface_peek_parent(ATK_TEXT_GET_IFACE(text)));
		if (g_iface->get_selection)
			result = g_iface->get_selection(text, selection_num, start_offset, end_offset);
	}
	return result;
}

gboolean
Atk::TextIface::add_selection_proxy(AtkText *text, gint start_offset, gint end_offset)
{
	gboolean result = false;
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(text));
	Atk::Text *tmp_text = dynamic_cast<Atk::Text*>(object);
	if (tmp_text)
		result = tmp_text->do_add_selection(start_offset, end_offset);
	else
	{
		AtkTextIface *g_iface = static_cast<AtkTextIface*>(g_type_interface_peek_parent(ATK_TEXT_GET_IFACE(text)));
		if (g_iface->add_selection)
			result = g_iface->add_selection(text, start_offset, end_offset);
	}
	return result;
}

gboolean
Atk::TextIface::remove_selection_proxy(AtkText *text, gint selection_num)
{
	gboolean result = false;
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(text));
	Atk::Text *tmp_text = dynamic_cast<Atk::Text*>(object);
	if (tmp_text)
		result = tmp_text->do_remove_selection(selection_num);
	else
	{
		AtkTextIface *g_iface = static_cast<AtkTextIface*>(g_type_interface_peek_parent(ATK_TEXT_GET_IFACE(text)));
		if (g_iface->remove_selection)
			result = g_iface->remove_selection(text, selection_num);
	}
	return result;
}

gboolean
Atk::TextIface::set_selection_proxy(AtkText *text, gint selection_num, gint	start_offset, gint end_offset)
{
	gboolean result = false;
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(text));
	Atk::Text *tmp_text = dynamic_cast<Atk::Text*>(object);
	if (tmp_text)
		result = tmp_text->do_set_selection(selection_num, start_offset, end_offset);
	else
	{
		AtkTextIface *g_iface = static_cast<AtkTextIface*>(g_type_interface_peek_parent(ATK_TEXT_GET_IFACE(text)));
		if (g_iface->set_selection)
			result = g_iface->set_selection(text, selection_num, start_offset, end_offset);
	}
	return result;
}

gboolean
Atk::TextIface::set_caret_offset_proxy(AtkText *text, gint offset)
{
	gboolean result = false;
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(text));
	Atk::Text *tmp_text = dynamic_cast<Atk::Text*>(object);
	if (tmp_text)
		result = tmp_text->do_set_caret_offset(offset);
	else
	{
		AtkTextIface *g_iface = static_cast<AtkTextIface*>(g_type_interface_peek_parent(ATK_TEXT_GET_IFACE(text)));
		if (g_iface->set_caret_offset)
			result = g_iface->set_caret_offset(text, offset);
	}
	return result;
}

void
Atk::TextIface::text_changed_proxy(AtkText *text, gint position, gint length)
{
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(text));
	Atk::Text *tmp_text = dynamic_cast<Atk::Text*>(object);
	if (tmp_text)
		tmp_text->on_text_changed(position, length);
	else
	{
		AtkTextIface *g_iface = static_cast<AtkTextIface*>(g_type_interface_peek_parent(ATK_TEXT_GET_IFACE(text)));
		if (g_iface->text_changed)
			g_iface->text_changed(text, position, length);
	}
}

void
Atk::TextIface::text_caret_moved_proxy(AtkText *text, gint location)
{
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(text));
	Atk::Text *tmp_text = dynamic_cast<Atk::Text*>(object);
	if (tmp_text)
		tmp_text->on_text_caret_moved(location);
	else
	{
		AtkTextIface *g_iface = static_cast<AtkTextIface*>(g_type_interface_peek_parent(ATK_TEXT_GET_IFACE(text)));
		if (g_iface->text_caret_moved)
			g_iface->text_caret_moved(text, location);
	}
}

void
Atk::TextIface::text_selection_changed_proxy(AtkText *text)
{
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(text));
	Atk::Text *tmp_text = dynamic_cast<Atk::Text*>(object);
	if (tmp_text)
		tmp_text->on_text_selection_changed();
	else
	{
		AtkTextIface *g_iface = static_cast<AtkTextIface*>(g_type_interface_peek_parent(ATK_TEXT_GET_IFACE(text)));
		if (g_iface->text_selection_changed)
			g_iface->text_selection_changed(text);
	}
}

void
Atk::TextIface::text_attributes_changed_proxy(AtkText *text)
{
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(text));
	Atk::Text *tmp_text = dynamic_cast<Atk::Text*>(object);
	if (tmp_text)
		tmp_text->on_text_attributes_changed();
	else
	{
		AtkTextIface *g_iface = static_cast<AtkTextIface*>(g_type_interface_peek_parent(ATK_TEXT_GET_IFACE(text)));
		if (g_iface->text_attributes_changed)
			g_iface->text_attributes_changed(text);
	}
}

/*  Overridable methods
 */

char*
Atk::Text::do_get_text(int start_offset, int end_offset)
{
	char *result = 0;
	AtkTextIface *g_iface = peek_parent<AtkTextIface>(atk_text_iface());
	if (g_iface->get_text)
		result = g_iface->get_text(atk_text(), start_offset, end_offset);
	return result;
}

char*
Atk::Text::do_get_text_after_offset(int offset, AtkTextBoundary boundary_type, int *start_offset, int *end_offset)
{
	char *result = 0;
	AtkTextIface *g_iface = peek_parent<AtkTextIface>(atk_text_iface());
	if (g_iface->get_text_after_offset)
		result = g_iface->get_text_after_offset(atk_text(), offset, boundary_type, start_offset, end_offset);
	return result;
}

char*
Atk::Text::do_get_text_at_offset(int offset, AtkTextBoundary boundary_type, int *start_offset, int *end_offset)
{
	char *result = 0;
	AtkTextIface *g_iface = peek_parent<AtkTextIface>(atk_text_iface());
	if (g_iface->get_text_at_offset)
		result = g_iface->get_text_at_offset(atk_text(), offset, boundary_type, start_offset, end_offset);
	return result;
}

gunichar
Atk::Text::do_get_character_at_offset(int offset)
{
	gunichar result = 0;
	AtkTextIface *g_iface = peek_parent<AtkTextIface>(atk_text_iface());
	if (g_iface->get_character_at_offset)
		result = g_iface->get_character_at_offset(atk_text(), offset);
	return result;
}

char*
Atk::Text::do_get_text_before_offset(int offset, AtkTextBoundary boundary_type, int *start_offset, int *end_offset)
{
	char *result = 0;
	AtkTextIface *g_iface = peek_parent<AtkTextIface>(atk_text_iface());
	if (g_iface->get_text_before_offset)
		result = g_iface->get_text_before_offset(atk_text(), offset, boundary_type, start_offset, end_offset);
	return result;
}

int
Atk::Text::do_get_caret_offset()
{
	int result = 0;
	AtkTextIface *g_iface = peek_parent<AtkTextIface>(atk_text_iface());
	if (g_iface->get_caret_offset)
		result = g_iface->get_caret_offset(atk_text());
	return result;
}

AtkAttributeSet*
Atk::Text::do_get_run_attributes(int offset, int *start_offset, int *end_offset)
{
	AtkAttributeSet *result = 0;
	AtkTextIface *g_iface = peek_parent<AtkTextIface>(atk_text_iface());
	if (g_iface->get_run_attributes)
		result = g_iface->get_run_attributes(atk_text(), offset, start_offset, end_offset);
	return result;
}

AtkAttributeSet*
Atk::Text::do_get_default_attributes()
{
	AtkAttributeSet *result = 0;
	AtkTextIface *g_iface = peek_parent<AtkTextIface>(atk_text_iface());
	if (g_iface->get_default_attributes)
		result = g_iface->get_default_attributes(atk_text());
	return result;
}

void
Atk::Text::do_get_character_extents(int offset, int *x, int *y, int *width, int *height, AtkCoordType coords)
{
	AtkTextIface *g_iface = peek_parent<AtkTextIface>(atk_text_iface());
	if (g_iface->get_character_extents)
		g_iface->get_character_extents(atk_text(), offset, x, y, width, height, coords);
}

int
Atk::Text::do_get_character_count()
{
	int result = 0;
	AtkTextIface *g_iface = peek_parent<AtkTextIface>(atk_text_iface());
	if (g_iface->get_character_count)
		result = g_iface->get_character_count(atk_text());
	return result;
}

int
Atk::Text::do_get_offset_at_point(int  x, int y, AtkCoordType coords)
{
	int result = 0;
	AtkTextIface *g_iface = peek_parent<AtkTextIface>(atk_text_iface());
	if (g_iface->get_offset_at_point)
		result = g_iface->get_offset_at_point(atk_text(), x, y, coords);
	return result;
}

int
Atk::Text::do_get_n_selections()
{
	int result = 0;
	AtkTextIface *g_iface = peek_parent<AtkTextIface>(atk_text_iface());
	if (g_iface->get_n_selections)
		result = g_iface->get_n_selections(atk_text());
	return result;
}

char*
Atk::Text::do_get_selection(int selection_num, int *start_offset, int *end_offset)
{
	char *result = 0;
	AtkTextIface *g_iface = peek_parent<AtkTextIface>(atk_text_iface());
	if (g_iface->get_selection)
		result = g_iface->get_selection(atk_text(), selection_num, start_offset, end_offset);
	return result;
}

bool
Atk::Text::do_add_selection(int start_offset, int end_offset)
{
	bool result = false;
	AtkTextIface *g_iface = peek_parent<AtkTextIface>(atk_text_iface());
	if (g_iface->add_selection)
		result = g_iface->add_selection(atk_text(), start_offset, end_offset);
	return result;
}

bool
Atk::Text::do_remove_selection(int selection_num)
{
	bool result = false;
	AtkTextIface *g_iface = peek_parent<AtkTextIface>(atk_text_iface());
	if (g_iface->remove_selection)
		result = g_iface->remove_selection(atk_text(), selection_num);
	return result;
}

bool
Atk::Text::do_set_selection(int selection_num, int start_offset, int end_offset)
{
	bool result = false;
	AtkTextIface *g_iface = peek_parent<AtkTextIface>(atk_text_iface());
	if (g_iface->set_selection)
		result = g_iface->set_selection(atk_text(), selection_num, start_offset, end_offset);
	return result;
}

bool
Atk::Text::do_set_caret_offset(int offset)
{
	bool result = false;
	AtkTextIface *g_iface = peek_parent<AtkTextIface>(atk_text_iface());
	if (g_iface->set_caret_offset)
		result = g_iface->set_caret_offset(atk_text(), offset);
	return result;
}

/*  Signal handlers
 */

void
Atk::Text::on_text_changed(int position, int length)
{
	AtkTextIface *g_iface = peek_parent<AtkTextIface>(atk_text_iface());
	if (g_iface->text_changed)
		g_iface->text_changed(atk_text(), position, length);
}

void
Atk::Text::on_text_caret_moved(int location)
{
	AtkTextIface *g_iface = peek_parent<AtkTextIface>(atk_text_iface());
	if (g_iface->text_caret_moved)
		g_iface->text_caret_moved(atk_text(), location);
}

void
Atk::Text::on_text_selection_changed()
{
	AtkTextIface *g_iface = peek_parent<AtkTextIface>(atk_text_iface());
	if (g_iface->text_selection_changed)
		g_iface->text_selection_changed(atk_text());
}

void
Atk::Text::on_text_attributes_changed()
{
	AtkTextIface *g_iface = peek_parent<AtkTextIface>(atk_text_iface());
	if (g_iface->text_attributes_changed)
		g_iface->text_attributes_changed(atk_text());
}

/*  Signals
 */

const Atk::Text::TextChangedSignalType Atk::Text::text_changed_signal("text_changed");

const Atk::Text::TextCaretMovedSignalType Atk::Text::text_caret_moved_signal("text_caret_moved");

const Atk::Text::TextSelectionChangedSignalType Atk::Text::text_selection_changed_signal("text_selection_changed");

const Atk::Text::TextAttributesChangedSignalType Atk::Text::text_attributes_changed_signal("text_attributes_changed");

