/*  Inti: Integrated Foundation Classes
 *  Copyright (C) 2002 The Inti Development Team.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 
//! @file inti/gdk/bitmap.h
//! @brief A GdkBitmap C++ wrapper interface.
//!
//! Provides Bitmap, an object that represents an offscreen monochrome drawable.

#ifndef INTI_GDK_BITMAP_H
#define INTI_GDK_BITMAP_H

#ifndef INTI_GDK_PIXMAP_H
#include <inti/gdk/pixmap.h>
#endif

namespace Inti {

namespace Gdk {

//! @class Bitmap bitmap.h inti/gdk/bitmap.h
//! @brief A GdkBitmap C++ wrapper class.
//!
//! Bitmap, like Pixmaps are offscreen drawables. They can be drawn upon with the
//! standard drawing primitives, then copied to another drawable (such as a 
//! Gdk::Window) with Gdk::Pixmap::draw(). The depth of a pixmap is the number of
//! bits per pixels. Bitmaps are simply pixmaps with a depth of 1. (That is, they
//! are monochrome bitmaps, each pixel can be either on or off).

class Bitmap : public Pixmap
{
	friend class G::Object;

	Bitmap(const Bitmap&);
	Bitmap& operator=(const Bitmap&);

protected:
//! @name Constructors
//! @{

	explicit Bitmap(GdkBitmap *bitmap, bool reference = true);
	//!< Construct a new Bitmap from an existing GdkBitmap.
	//!< @param bitmap A pointer to a GdkBitmap.
	//!< @param reference Set false if the initial reference count is floating, set true if it's not.
	//!<
	//!< <BR>The <EM>bitmap</EM> can be a newly created GdkBitmap or an existing
	//!< GdkBitmap. (see G::Object::Object).

//! @}

public:
//! @name Constructors
//! @{

	Bitmap(const char *data, int width, int height);
	//!< Create a new bitmap from data in XBM format using the root window
	//!< to determine the default values.
	//!< @param data A pointer to the XBM data.
	//!< @param width The width of the new bitmap in pixels.
	//!< @param height The height of the new bitmap in pixels.

	Bitmap(const Window& window, const char *data, int width, int height);
	//!< Create a new bitmap from data in XBM format using window
	//!< to determine the default values.
	//!< @param window The window to use.
	//!< @param data A pointer to the XBM data.
	//!< @param width The width of the new bitmap in pixels.
	//!< @param height The height of the new bitmap in pixels.

	virtual ~Bitmap();
	//!< Destructor.

//! @}
//! @name Accessors
//! @{

	GdkBitmap* gdk_bitmap() const { return (GdkBitmap*)instance; }
	//!< Get a pointer to the GdkBitmap structure.

//! @}
};

} // namespace Gdk

} // namespace Inti

#endif // INTI_GDK_BITMAP_H

