/*  Inti: Integrated Foundation Classes
 *  Copyright (C) 2002-2003 The Inti Development Team.
 *
 *  object.cc - GObject C++ wrapper implementation
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "object.h"
#include "private/object_p.h"
#include "value.h"

using namespace Inti;

const G::Quark G::Object::pointer_quark("inti_object");

/*  G::Object
 */

G::Object::Object(GObject *object, bool reference)
{
	g_return_if_fail(object != 0);
	
	if (!g_object_get_qdata(object, pointer_quark))
	{
		g_object_set_qdata_full(object, pointer_quark, this, &destroy_notify);
		instance = (GTypeInstance*)object;
		set_referenced(reference);
	}
	else
	{
		const char *name = g_type_name(((GTypeInstance*)object)->g_class->g_type);
		g_error("This %s object already has an Inti wrapper.", name);
	}
}

G::Object::~Object()
{
	if (instance)
	{
		g_object_steal_qdata(g_object(), pointer_quark);
		instance = 0;
	}
}

void 
G::Object::destroy_notify(void *data)
{
	G::Object *object = static_cast<G::Object*>(data);
	if (object)
	{
 		object->remove_data(G::Quark::try_string("inti_object"));
		object->instance = 0;
		if (object->is_dynamic())
			delete object;
	}
}

GObjectClass*
G::Object::g_object_class() const
{
	return get_class<GObjectClass>();
}

G::Object::operator GObject* () const
{
	return this ? g_object() : 0;
}

void
G::Object::get(const char* first_property_name, ...) const
{
	va_list args;
	va_start(args, first_property_name);
	g_object_get_valist(g_object(), first_property_name, args);
	va_end(args);
}

void
G::Object::get_property(const char* property_name, G::Value& value) const
{
	g_object_get_property(g_object(), property_name, value.g_value());
}

void*
G::Object::get_data(const Quark& quark) const
{
	return g_object_get_qdata(g_object(), quark);
}

void*
G::Object::get_data(const char *key) const
{
	return g_object_get_qdata(g_object(), g_quark_try_string(key));
}

void*
G::Object::get_data(const String& key) const
{
	return g_object_get_qdata(g_object(), g_quark_try_string(key.c_str()));
}

void
G::Object::ref()
{
	g_object_ref(g_object());
}

void
G::Object::unref()
{
	g_object_unref(g_object());
}

void 
G::Object::dispose()
{
	g_object_run_dispose(g_object());
}

void 
G::Object::set(const char* first_property_name, ...)
{
	va_list args;
	va_start(args, first_property_name);
	g_object_set_valist(g_object(), first_property_name, args);
	va_end(args);
}

void 
G::Object::set_property(const char* property_name, const G::Value& value)
{
	g_object_set_property(g_object(), property_name, value.g_value());
}

void
G::Object::set_data(const Quark& quark, void *data, GDestroyNotify destroy)
{
	g_object_set_qdata_full(g_object(), quark, data, destroy);
}

void 
G::Object::set_data(const char *key, void *data, GDestroyNotify destroy)
{
	g_object_set_qdata_full(g_object(), g_quark_from_string(key), data, destroy);
}

void 
G::Object::set_data(const String& key, void *data, GDestroyNotify destroy)
{
	g_object_set_qdata_full(g_object(), g_quark_from_string(key.c_str()), data, destroy);
}

void* 
G::Object::remove_data(const Quark& quark, bool notify)
{
	void *ptr = 0;
	if (notify)
		set_data(quark, 0);
	else
		ptr = g_object_steal_qdata(g_object(), quark);
	return ptr;
}

gpointer 
G::Object::remove_data(const char *key, bool notify)
{
	return remove_data(G::Quark::try_string(key), notify);
}

gpointer 
G::Object::remove_data(const String& key, bool notify)
{
	return remove_data(G::Quark::try_string(key.c_str()), notify);
}

unsigned long 
G::Object::connect(const char *signal_name, GCallback handler, void *data, GClosureNotify destroy_data)
{
	return g_signal_connect_data(g_object(), signal_name, handler, data, destroy_data, GConnectFlags(0));
}

unsigned long
G::Object::connect_after(const char *signal_name, GCallback handler, void *data, GClosureNotify destroy_data)
{
	return g_signal_connect_data(g_object(), signal_name, handler, data, destroy_data, G_CONNECT_AFTER);
}

unsigned long
G::Object::connect_swapped(const char *signal_name, GCallback handler, void *data, GClosureNotify destroy_data)
{
	return g_signal_connect_data(g_object(), signal_name, handler, data, destroy_data, G_CONNECT_SWAPPED);
}

void
G::Object::emit_by_name(const char *signal_name, ...)
{
	g_return_if_fail(signal_name != 0);
	
	GSignalQuery query;
	g_signal_query(g_signal_lookup(signal_name, type()), &query);
	g_return_if_fail(query.signal_id != 0);

	va_list args;
	va_start(args, signal_name);
	g_signal_emit_valist(g_object(), query.signal_id, 0, args);
	va_end (args);
}

void 
G::Object::stop_emission_by_name(const char *detailed_signal)
{
	g_signal_stop_emission_by_name(g_object(), detailed_signal);
}

bool
G::Object::disconnect_by_name(const char* signal_name)
{
	unsigned int id = g_signal_lookup(signal_name, type());
	if (id)
	{
		id = g_signal_handler_find(g_object(), G_SIGNAL_MATCH_ID, id, 0, 0, 0, 0);
		if (id)
			g_signal_handler_disconnect(g_object(), id);
	}
	return id != 0;
}

/*  G::ObjectClass
 */

void
G::ObjectClass::init(GObjectClass *g_class)
{
	g_class->notify = &notify_proxy;
	g_class->set_property = &set_property_proxy;
	g_class->get_property = &get_property_proxy;
	g_class->dispose = &dispose_proxy;
	g_class->finalize = &finalize_proxy;
}

GType
G::ObjectClass::get_type()
{
	static GType type = 0;
	if (!type)
	{
		type = G::TypeInstance::register_type(G_TYPE_OBJECT, (GClassInitFunc)init);
	}
	return type;
}

void*
G::ObjectClass::create()
{
	return g_object_new(get_type(), 0);
}

void
G::ObjectClass::set_property_proxy(GObject *object, guint property_id, const GValue *value, GParamSpec *pspec)
{
	Object *tmp_object = Object::pointer<Object>(object);
	if (tmp_object)
		tmp_object->do_set_property(property_id, value, pspec);
	else
	{
		GObjectClass *g_class = TypeInstance::class_peek_parent<GObjectClass>(G_OBJECT_GET_CLASS(object));
		if (g_class->set_property)
			g_class->set_property(object, property_id, value, pspec);
	}
}

void
G::ObjectClass::get_property_proxy(GObject *object, guint property_id, GValue *value, GParamSpec *pspec)
{
	Object *tmp_object = Object::pointer<Object>(object);
	if (tmp_object)
		tmp_object->do_get_property(property_id, value, pspec);
	else
	{
		GObjectClass *g_class = TypeInstance::class_peek_parent<GObjectClass>(G_OBJECT_GET_CLASS(object));
		if (g_class->get_property)
			g_class->get_property(object, property_id, value, pspec);
	}
}

void
G::ObjectClass::dispose_proxy(GObject *object)
{
	Object *tmp_object = Object::pointer<Object>(object);
	if (tmp_object)
		tmp_object->do_dispose();
	else
	{
		GObjectClass *g_class = TypeInstance::class_peek_parent<GObjectClass>(G_OBJECT_GET_CLASS(object));
		if (g_class->dispose)
			g_class->dispose(object);
	}
}

void
G::ObjectClass::finalize_proxy(GObject *object)
{
	Object *tmp_object = Object::pointer<Object>(object);
	if (tmp_object)
		tmp_object->do_finalize();
	else
	{
		GObjectClass *g_class = TypeInstance::class_peek_parent<GObjectClass>(G_OBJECT_GET_CLASS(object));
		if (g_class->finalize)
			g_class->finalize(object);
	}
}

void
G::ObjectClass::notify_proxy(GObject *object, GParamSpec *pspec)
{
	Object *tmp_object = Object::pointer<Object>(object);
	if (tmp_object)
		tmp_object->on_notify(pspec);
	else
	{
		GObjectClass *g_class = TypeInstance::class_peek_parent<GObjectClass>(G_OBJECT_GET_CLASS(object));
		if (g_class->notify)
			g_class->notify(object, pspec);
	}
}

/*  Overridable GObject methods
 */

void
G::Object::do_set_property(unsigned int property_id, const GValue *value, GParamSpec *pspec)
{
	GObjectClass *g_class = class_peek_parent<GObjectClass>(g_object_class());
	if (g_class->set_property)
		g_class->set_property(g_object(), property_id, value, pspec);
}


void
G::Object::do_get_property(unsigned int property_id, GValue *value, GParamSpec *pspec)
{
	GObjectClass *g_class = class_peek_parent<GObjectClass>(g_object_class());
	if (g_class->get_property)
		g_class->get_property(g_object(), property_id, value, pspec);
}


void
G::Object::do_dispose()
{
	GObjectClass *g_class = class_peek_parent<GObjectClass>(g_object_class());
	if (g_class->dispose)
		g_class->dispose(g_object());
}


void
G::Object::do_finalize()
{
	GObjectClass *g_class = class_peek_parent<GObjectClass>(g_object_class());
	if (g_class->finalize)
		g_class->finalize(g_object());
}

/*  Signal handlers
 */

void
G::Object::on_notify(GParamSpec *pspec)
{
	GObjectClass *g_class = class_peek_parent<GObjectClass>(g_object_class());
	if (g_class->notify)
		g_class->notify(g_object(), pspec);
}

/*  Signals
 */

const G::Object::NotifySignalType G::Object::notify_signal("notify");

