/*  Inti: Integrated Foundation Classes
 *  Copyright (C) 2003 The Inti Development Team.
 *
 *  spawn.cc - A C++ interface for the GLib spawn functions.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "spawn.h"
#include <inti/glib/error.h>
#include <glib/gmem.h>
#include <glib/gshell.h>
#include <glib/gstrfuncs.h>

namespace Inti {

namespace G {

/*  G::SpawnData
 */

struct SpawnData
{
	const char *working_directory;
	char **argv;
	char **envp;
	GSpawnFlags flags;
};

/*  G::SpawnAsyncData
 */

struct SpawnAsyncData
{
	int child_pid;
	int *standard_input;
	int *standard_output;
	int *standard_error;
};

/*  G::SpawnSyncData
 */

struct SpawnSyncData
{
	char *standard_output;
	char *standard_error;
	int exit_status;
};

/*  G::Spawn
 */

Spawn::Spawn()
{
	data_ = new SpawnData;
	memset(data_, 0, sizeof(SpawnData));
}

Spawn::~Spawn()
{
	g_strfreev(data_->argv);
	g_strfreev(data_->envp);
	delete data_;
}

const char*
Spawn::get_working_directory() const
{
	return data_->working_directory;
}

char**
Spawn::get_argv() const
{
	return data_->argv;
}

char**
Spawn::get_envp() const
{
	return data_->envp;
}

GSpawnFlags
Spawn::get_flags() const
{
	return data_->flags;
}

void
Spawn::set_arguments(const std::vector<std::string>& argv, SpawnFlagsField flags)
{
	int count = argv.size();
	data_->argv = g_new0(char*, count + 1);

	int i = 0;
	while (i < count)
	{
		data_->argv[i] = const_cast<char*>(argv[i].c_str());
		++i;
	}

	data_->flags = (GSpawnFlags)flags;
}

void
Spawn::set_arguments(const std::string& command_line, G::Error *error)
{
	g_shell_parse_argv(command_line.c_str(), 0, &data_->argv, *error);
	data_->flags = G_SPAWN_SEARCH_PATH;
}

void
Spawn::set_working_directory(const std::string& working_directory)
{
	data_->working_directory = working_directory.c_str();
}

void
Spawn::set_enviroment(std::vector<std::string>& envp)
{
	int count = envp.size();
	data_->envp = g_new0(char*, count + 1);

	int i = 0;
	while (i < count)
	{
		data_->envp[i] = const_cast<char*>(envp[i].c_str());
		++i;
	}
}

/*  G::SpawnAsync
 */

namespace { // spawn_async_init

void spawn_async_init(SpawnAsyncData **data)
{
	*data = new SpawnAsyncData;
	memset(data, 0, sizeof(SpawnAsyncData));
}

} // namespace

G::SpawnAsync::SpawnAsync(const std::vector<std::string>& argv, SpawnFlagsField flags)
{
	spawn_async_init(&data_);
	set_arguments(argv, flags);
}

G::SpawnAsync::SpawnAsync(const std::string& command_line, G::Error *error)
{
	spawn_async_init(&data_);
	set_arguments(command_line, error);
}

G::SpawnAsync::~SpawnAsync()
{
	delete data_;
}

int
G::SpawnAsync::process_id() const
{
	return data_->child_pid;
}

namespace { // spawn_child_setup_slot_callback

void spawn_child_setup_slot_callback(gpointer user_data)
{
	G::Spawn::ChildSetupSlot *slot = static_cast<G::Spawn::ChildSetupSlot*>(user_data);
	slot->call();
}

} // namespace

bool
G::SpawnAsync::execute(G::Error *error, const ChildSetupSlot *slot)
{
	return g_spawn_async_with_pipes(get_working_directory(), get_argv(), get_envp(), get_flags(),
	                                &spawn_child_setup_slot_callback, (void*)slot, &data_->child_pid,
	                                (get_flags() & G_SPAWN_CHILD_INHERITS_STDIN) ? 0 : data_->standard_output,
	                                (get_flags() & G_SPAWN_STDOUT_TO_DEV_NULL) ? 0 : data_->standard_input,
	                                (get_flags() & G_SPAWN_STDERR_TO_DEV_NULL) ? 0 : data_->standard_error, *error);
}

void
G::SpawnAsync::set_standard_input(int& standard_input)
{
	data_->standard_input = &standard_input;
}

void
G::SpawnAsync::set_standard_output(int& standard_output)
{
	data_->standard_output = &standard_output;
}

void
G::SpawnAsync::set_standard_error(int& standard_error)
{
	data_->standard_error = &standard_error;
}

/*  G::SpawnSync
 */

namespace { // spawn_sync_init

void spawn_sync_init(SpawnSyncData **data)
{
	*data = new SpawnSyncData;
	memset(data, 0, sizeof(SpawnSyncData));
}

} // namespace

G::SpawnSync::SpawnSync(const std::vector<std::string>& argv, SpawnFlagsField flags)
{
	spawn_sync_init(&data_);
	set_arguments(argv, flags);
}

G::SpawnSync::SpawnSync(const std::string& command_line, G::Error *error)
{
	spawn_sync_init(&data_);
	set_arguments(command_line, error);
}

G::SpawnSync::~SpawnSync()
{
	if (data_->standard_output)
		g_free(data_->standard_output);

	if (data_->standard_error)
		g_free(data_->standard_error);

	delete data_;
}

int
G::SpawnSync::exit_status() const
{
	return data_->exit_status;
}

std::string
G::SpawnSync::standard_output() const
{
	return std::string(data_->standard_output ? data_->standard_output : "");
}

std::string
G::SpawnSync::standard_error() const
{
	return std::string(data_->standard_error ? data_->standard_error : "");
}

bool
G::SpawnSync::execute(G::Error *error, const ChildSetupSlot *slot)
{
	return g_spawn_sync(get_working_directory(), get_argv(), get_envp(), get_flags(),
	                    &spawn_child_setup_slot_callback, (void*)slot,
	                    (get_flags() & G_SPAWN_CHILD_INHERITS_STDIN) ? 0 : &data_->standard_output,
	                    (get_flags() & G_SPAWN_STDERR_TO_DEV_NULL) ? 0 : &data_->standard_error,
			    &data_->exit_status, *error);
}

} // namespace G

} // namespace Inti

