/*  Inti: Integrated Foundation Classes
 *  Copyright (C) 2002 The Inti Development Team.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 
//! @file inti/gtk/alignment.h
//! @brief A GtkAlignment C++ wrapper interface.
//!
//! Provides Alignment, a widget that controls the alignment and size of its child widget.

#ifndef INTI_GTK_ALIGNMENT_H
#define INTI_GTK_ALIGNMENT_H

#ifndef INTI_GTK_BIN_H
#include <inti/gtk/bin.h>
#endif

#ifndef __GTK_ALIGNMENT_H__
#include <gtk/gtkalignment.h>
#endif

namespace Inti {

namespace Gtk {

class AlignmentClass;

//! @class Alignment alignment.h inti/gtk/alignment.h
//! @brief A GtkAlignment C++ wrapper class.
//!
//! The Alignment widget controls the alignment and size of its child widget.
//! It has four settings: xscale, yscale, xalign, and yalign.
//!
//! The scale settings are used to specify how much the child widget should
//! expand to fill the space allocated to the Alignment. The values can range
//! from 0 (meaning the child doesn't expand at all) to 1 (meaning the child
//! expands to fill all of the available space).
//!
//! The align settings are used to place the child widget within the available
//! area. The values range from 0 (top or left) to 1 (bottom or right). 
//! Of course, if the scale settings are both set to 1, the alignment settings
//! have no effect.

class Alignment : public Bin
{
	friend class G::Object;
	friend class AlignmentClass;

	Alignment(const Alignment&);
	Alignment& operator=(const Alignment&);
	
protected:
//! @name Constructors
//! @{

	explicit Alignment(GtkAlignment *alignment, bool reference = false);
	//!< Construct a new Alignment from an existing GtkAlignment.
	//!< @param alignment T pointer to a GtkAlignment.
	//!< @param reference Set false if the initial reference count is floating, set true if it's not.
	//!<
	//!< <BR>The <EM>alignment</EM> can be a newly created GtkAlignment or an existing
	//!< GtkAlignment. (see G::Object::Object).

//! @}
//  Properties

	typedef G::Property<float> XAlignPropertyType;
	typedef G::PropertyProxy<G::Object, XAlignPropertyType> XAlignPropertyProxy;
	static const XAlignPropertyType xalign_property;

	typedef G::Property<float> YAlignPropertyType;
	typedef G::PropertyProxy<G::Object, YAlignPropertyType> YAlignPropertyProxy;
	static const YAlignPropertyType yalign_property;

	typedef G::Property<float> XScalePropertyType;
	typedef G::PropertyProxy<G::Object, XScalePropertyType> XScalePropertyProxy;
	static const XScalePropertyType xscale_property;

	typedef G::Property<float> YScalePropertyType;
	typedef G::PropertyProxy<G::Object, YScalePropertyType> YScalePropertyProxy;
	static const YScalePropertyType yscale_property;

public:
//! @name Constructors
//! @{

	Alignment();
	//!< Construct an Alignment with the default values 0.5 for xalign and yalign and
	//!< 1.0 for xscale and yscale.

	Alignment(float xalign, float yalign, float xscale = 0.0, float yscale = 0.0);
	//!< Construct a new Alignment.
	//!< @param xalign The horizontal alignment of the child widget, from 0 (left) to 1 (right).
	//!< @param yalign The vertical alignment of the child widget, from 0 (top) to 1 (bottom).
	//!< @param xscale The amount that the child widget expands horizontally to fill up unused
	//!<               space, from 0 to 1. A value of 0 indicates that the child widget should
	//!<               never expand. A value of 1 indicates that the child widget will expand
	//!<               to fill all of the space allocated for the GtkAlignment.
	//!< @param yscale The amount that the child widget expands vertically to fill up unused
	//!                space, from 0 to 1. The values are similar to xscale.
	
	virtual ~Alignment();
	//!< Destructor.

//! @}
//! @name Accessors
//! @{

	GtkAlignment* gtk_alignment() const { return (GtkAlignment*)instance; }
	//!< Get a pointer to the GtkAlignment structure.

	GtkAlignmentClass* gtk_alignment_class() const;
	//!< Get a pointer to the GtkAlignmentClass structure.

	operator GtkAlignment* () const;
	//!< Conversion operator; safely converts an Alignment to a GtkAlignment pointer.

//! @}
//! @name Methods
//! @{

	void set(float xalign, float yalign, float xscale, float yscale);
	//!< Sets the Alignment values.
	//!< @param xalign The horizontal alignment of the child widget, from 0 (left) to 1 (right).
	//!< @param yalign The vertical alignment of the child widget, from 0 (top) to 1 (bottom).
	//!< @param xscale The amount that the child widget expands horizontally to fill up unused
	//!<               space, from 0 to 1. A value of 0 indicates that the child widget should
	//!<               never expand. A value of 1 indicates that the child widget will expand
	//!<               to fill all of the space allocated for the GtkAlignment.
	//!< @param yscale The amount that the child widget expands vertically to fill up unused
	//!                space, from 0 to 1. The values are similar to xscale.

//! @}
//! @name Property Proxies
//! @{

	const XAlignPropertyProxy prop_xalign()
	{
		return XAlignPropertyProxy(this, &xalign_property);
	}
	//!< Horizontal position of child in available space; 0.0 is left aligned, 1.0 is right aligned
	//!< (float : Read / Write)

	const YAlignPropertyProxy prop_yalign()
	{
		return YAlignPropertyProxy(this, &yalign_property);
	}
	//!< Vertical position of child in available space; 0.0 is top aligned, 1.0 is bottom aligned
	//!< (float : Read / Write).

	const XScalePropertyProxy prop_xscale()
	{
		return XScalePropertyProxy(this, &xscale_property);
	}
	//!< If available horizontal space is bigger than needed for the child, how much of it to use
	//!< for the child; 0.0 means none, 1.0 means all (float : Read / Write).

	const YScalePropertyProxy prop_yscale()
	{
		return YScalePropertyProxy(this, &yscale_property);
	}
	//!< If available vertical space is bigger than needed for the child, how much of it to use 
	//!< for the child; 0.0 means none, 1.0 means all (float : Read / Write).

//! @}
};

} // namespace Gtk

} // namespace Inti

#endif // INTI_GTK_ALIGNMENT_H

