/*  Inti: Integrated Foundation Classes
 *  Copyright (C) 2002 The Inti Development Team.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 
//! @file inti/gtk/aspectframe.h
//! @brief A GtkAspectFrame C++ wrapper interface.
//!
//! Provides AspectFrame, a widget that is useful when you want pack a widget so that it can
//! resize but always retains the same aspect ratio.

#ifndef INTI_GTK_ASPECT_FRAME_H
#define INTI_GTK_ASPECT_FRAME_H

#ifndef INTI_GTK_FRAME_H
#include <inti/gtk/frame.h>
#endif

#ifndef __GTK_ASPECT_FRAME_H__
#include <gtk/gtkaspectframe.h>
#endif

namespace Inti {

namespace Gtk {

class AspectFrameClass;

//! @class AspectFrame aspectframe.h inti/gtk/aspectframe.h
//! @brief A GtkAspectFrame C++ wrapper class.
//!
//! The AspectFrame is useful when you want pack a widget so that it can resize but always retains
//! the same aspect ratio. For instance, one might be drawing a small preview of a larger image.
//! AspectFrame derives from Frame, so it can draw a label and a frame around the child. The frame
//! will be "shrink-wrapped" to the size of the child. 

class AspectFrame : public Frame
{
	friend class G::Object;
	friend class AspectFrameClass;

	AspectFrame(const AspectFrame&);
	AspectFrame& operator=(const AspectFrame&);
	
protected:
//! @name Constructors
//! @{

	explicit AspectFrame(GtkAspectFrame *aspect_frame, bool reference = false);
	//!< Construct a new AspectFrame from an existing GtkAspectFrame.
	//!< @param aspect_frame A pointer to a GtkAspectFrame.
	//!< @param reference Set false if the initial reference count is floating, set true if it's not.
	//!<
	//!< <BR>The <EM>aspect_frame</EM> can be a newly created GtkAspectFrame or an existing
	//!< GtkAspectFrame. (see G::Object::Object).

//! @}
//  Properties

	typedef G::Property<float> XAlignPropertyType;
	typedef G::PropertyProxy<G::Object, XAlignPropertyType> XAlignPropertyProxy;
	static const XAlignPropertyType xalign_property;

	typedef G::Property<float> YAlignPropertyType;
	typedef G::PropertyProxy<G::Object, YAlignPropertyType> YAlignPropertyProxy;
	static const YAlignPropertyType yalign_property;

	typedef G::Property<float> RatioPropertyType;
	typedef G::PropertyProxy<G::Object, RatioPropertyType> RatioPropertyProxy;
	static const RatioPropertyType ratio_property;

	typedef G::Property<bool> ObeyChildPropertyType;
	typedef G::PropertyProxy<G::Object, ObeyChildPropertyType> ObeyChildPropertyProxy;
	static const ObeyChildPropertyType obey_child_property;

public:
//! @name Constructors
//! @{

	AspectFrame();
	//!< Construct an AspectFrame with xalign 0.5, yalign 0.5, ratio 1.0 and obey_child true.

	AspectFrame(float xalign, float yalign);
	//!< Construct an AspectFrame with the specified xalign and yalign values.
	//!< @param xalign The horizontal alignment of the child within the allocation of the
	//!<               AspectFrame; this ranges from 0.0 (left aligned) to 1.0 (right aligned).
	//!< @param yalign The vertical alignment of the child within the allocation of the
	//!<               AspectFrame. This ranges from 0.0 (left aligned) to 1.0 (right aligned).
	//!<
	//!< <BR>This constructor takes the aspect ratio from the requistion of the child. This is the
	//!< same as setting obey_child to true.

	AspectFrame(float xalign, float yalign, float ratio);
	//!< Construct an AspectFrame with the specified xalign, yalign and ratio values.
	//!< @param xalign The horizontal alignment of the child within the allocation of the
	//!<               AspectFrame; this ranges from 0.0 (left aligned) to 1.0 (right aligned).
	//!< @param yalign The vertical alignment of the child within the allocation of the
	//!<               AspectFrame. This ranges from 0.0 (left aligned) to 1.0 (right aligned).
	//!< @param ratio  The desired aspect ratio.
	//!<
	//!< <BR>This constructor takes the aspect ratio from <EM>ratio</EM> and sets obey_child to false.

	AspectFrame(const String& label, float xalign, float yalign, float ratio);
	//!< Construct a labelled AspectFrame with the specified xalign, yalign and ratio values.
	//!< @param label The label text.
	//!< @param xalign The horizontal alignment of the child within the allocation of the
	//!<               AspectFrame; this ranges from 0.0 (left aligned) to 1.0 (right aligned).
	//!< @param yalign The vertical alignment of the child within the allocation of the
	//!<               AspectFrame. This ranges from 0.0 (left aligned) to 1.0 (right aligned).
	//!< @param ratio  The desired aspect ratio.
	//!<
	//!< <BR>This constructor takes the aspect ratio from <EM>ratio</EM> and sets obey_child to false.

	virtual ~AspectFrame();
	//!< Destructor.

//! @}
//! @name Accessors
//! @{

	GtkAspectFrame* gtk_aspect_frame() const { return (GtkAspectFrame*)instance; }
	//!< Get a pointer to the GtkAspectFrame structure.

	GtkAspectFrameClass* gtk_aspect_frame_class() const;
	//!< Get a pointer to the GtkAspectFrameClass structure.

	operator GtkAspectFrame* () const;
	//!< Conversion operator; safely converts an AspectFrame to a GtkAspectFrame pointer.

//! @}
//! @name Methods
//! @{

	void set(float xalign, float yalign, float ratio, bool obey_child);
	//!< Sets the parameters for an existing AspectFrame.
	//!< @param xalign The horizontal alignment of the child within the allocation of the
	//!<               AspectFrame; this ranges from 0.0 (left aligned) to 1.0 (right aligned).
	//!< @param yalign The vertical alignment of the child within the allocation of the
	//!<               AspectFrame. This ranges from 0.0 (left aligned) to 1.0 (right aligned).
	//!< @param ratio  The desired aspect ratio.
	//!< @param obey_child If <EM>true</EM>, ratio is ignored, and the aspect ratio is taken
	//!<                   from the requistion of the child.

//! @}
//! @name Property Proxies
//! @{

	const XAlignPropertyProxy prop_xalign()
	{
		return XAlignPropertyProxy(this, &xalign_property);
	}
	//!< X alignment of the child (float : Read / Write).

	const YAlignPropertyProxy prop_yalign()
	{
		return YAlignPropertyProxy(this, &yalign_property);
	}
	//!< Y alignment of the child (float : Read / Write).

	const RatioPropertyProxy prop_ratio()
	{
		return RatioPropertyProxy(this, &ratio_property);
	}
	//!< Aspect ratio if obey_child is false (float : Read / Write).

	const ObeyChildPropertyProxy prop_obey_child()
	{
		return ObeyChildPropertyProxy(this, &obey_child_property);
	}
	//!< Force aspect ratio to match that of the frame's child (bool : Read / Write).

//! @}
};

} // namespace Gtk

} // namespace Inti

#endif // INTI_GTK_ASPECT_FRAME_H

