/*  Inti: Integrated Foundation Classes
 *  Copyright (C) 2002-2003 The Inti Development Team.
 *
 *  button.cc - GtkButton C++ wrapper implementation
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 
#include "button.h"
#include "private/button_p.h"
#include "alignment.h"
#include "box.h"
#include "image.h"
#include "label.h"

using namespace Inti;

Gtk::Button::Button(GtkButton *button, bool reference)
: Bin((GtkBin*)button, reference)
{
}

Gtk::Button::Button() 
: Bin((GtkBin*)ButtonClass::create())
{
}

Gtk::Button::Button(Image& image)
: Bin((GtkBin*)ButtonClass::create())
{
	add(image);
	image.show();
}

Gtk::Button::Button(const String& label, bool use_underline)
: Bin((GtkBin*)ButtonClass::create())
{
	set_label(label);
	set_use_underline(use_underline);
}

Gtk::Button::Button(Image& image, const String& label, bool use_underline, bool horizontal)
: Bin((GtkBin*)ButtonClass::create())
{
	set_image(image, label, use_underline, horizontal);
}

Gtk::Button::~Button()
{
}

GtkButtonClass*
Gtk::Button::gtk_button_class() const 
{ 
	return get_class<GtkButtonClass>(); 
}
	
Gtk::Button::operator GtkButton* () const 
{ 
	return this ? gtk_button() : 0; 
}
	
bool 
Gtk::Button::depresssed() const
{
	return gtk_button()->depressed;
}
	
Gtk::ReliefStyle
Gtk::Button::get_relief() const
{
	return (ReliefStyle)gtk_button_get_relief(gtk_button());
}

String 
Gtk::Button::get_label() const
{
	return gtk_button_get_label(gtk_button());
}

bool 
Gtk::Button::get_use_underline() const
{
	return gtk_button_get_use_underline(gtk_button());
}

bool 
Gtk::Button::get_use_stock() const
{
	return gtk_button_get_use_stock(gtk_button());
}

void
Gtk::Button::pressed()
{
	gtk_button_pressed(gtk_button());
}

void
Gtk::Button::released()
{
	gtk_button_released(gtk_button());
}

void
Gtk::Button::clicked()
{
	gtk_button_clicked(gtk_button());
}

void
Gtk::Button::enter()
{
	gtk_button_enter(gtk_button());
}

void
Gtk::Button::leave()
{
	gtk_button_leave(gtk_button());
}

void
Gtk::Button::set_depressed(bool depressed)
{
	_gtk_button_set_depressed(gtk_button(), depressed);
}
	
void 
Gtk::Button::set_relief(ReliefStyle newstyle)
{
	gtk_button_set_relief(gtk_button(), (GtkReliefStyle)newstyle);
}

void
Gtk::Button::set_label(const String& label)
{
	gtk_button_set_label(gtk_button(), label.c_str());
}

void
Gtk::Button::set_use_underline(bool value)
{
	gtk_button_set_use_underline(gtk_button(), value);
}

void 
Gtk::Button::set_use_stock(bool value)
{
	gtk_button_set_use_stock(gtk_button(), value);
}

void
Gtk::Button::set_image(Image& image, const String& label, bool use_underline, bool horizontal)
{
	if (gtk_button()->label_text)
		g_free(gtk_button()->label_text);

	Gtk::Widget *child = get_child();
	if (child)
		remove(*child);

	Gtk::Box *box;
	if (horizontal)
		box = new Gtk::HBox(false, 2);
	else
		box = new Gtk::VBox;

	box->pack_start(image, false, false);
	Gtk::Label *label_widget = new Gtk::Label(label, use_underline);
	if (use_underline)
		label_widget->set_mnemonic_widget(this);
	box->pack_end(*label_widget, false, false);

	Gtk::Alignment *align = new Gtk::Alignment(0.5, 0.5);
	add(*align);
	align->add(*box);
	align->show_all();
}

/*  Gtk::ButtonClass
 */

void
Gtk::ButtonClass::init(GtkButtonClass *g_class)
{
	BinClass::init((GtkBinClass*)g_class);
	g_class->pressed = &pressed_proxy;
	g_class->released = &released_proxy;
	g_class->clicked = &clicked_proxy;
	g_class->enter = &enter_proxy;
	g_class->leave = &leave_proxy;
	g_class->activate = &activate_proxy;
}

GType
Gtk::ButtonClass::get_type()
{
	static GType type = 0;
	if (!type)
	{
		type = G::TypeInstance::register_type(GTK_TYPE_BUTTON, (GClassInitFunc)init);
	}
	return type;
}

void*
Gtk::ButtonClass::create()
{
	return g_object_new(get_type(), 0);
}

void
Gtk::ButtonClass::pressed_proxy(GtkButton *button)
{
	Button *tmp_button = G::Object::pointer<Button>(button);
	if (tmp_button)
		tmp_button->on_pressed();
	else
	{
		GtkButtonClass *g_class = G::TypeInstance::class_peek_parent<GtkButtonClass>(GTK_BUTTON_GET_CLASS(button));
		if (g_class->pressed)
			g_class->pressed(button);
	}
}

void
Gtk::ButtonClass::released_proxy(GtkButton *button)
{
	Button *tmp_button = G::Object::pointer<Button>(button);
	if (tmp_button)
		tmp_button->on_released();
	else
	{
		GtkButtonClass *g_class = G::TypeInstance::class_peek_parent<GtkButtonClass>(GTK_BUTTON_GET_CLASS(button));
		if (g_class->released)
			g_class->released(button);
	}
}

void
Gtk::ButtonClass::clicked_proxy(GtkButton *button)
{
	Button *tmp_button = G::Object::pointer<Button>(button);
	if (tmp_button)
		tmp_button->on_clicked();
	else
	{
		GtkButtonClass *g_class = G::TypeInstance::class_peek_parent<GtkButtonClass>(GTK_BUTTON_GET_CLASS(button));
		if (g_class->clicked)
			g_class->clicked(button);
	}
}

void
Gtk::ButtonClass::enter_proxy(GtkButton *button)
{
	Button *tmp_button = G::Object::pointer<Button>(button);
	if (tmp_button)
		tmp_button->on_enter();
	else
	{
		GtkButtonClass *g_class = G::TypeInstance::class_peek_parent<GtkButtonClass>(GTK_BUTTON_GET_CLASS(button));
		if (g_class->enter)
			g_class->enter(button);
	}
}

void
Gtk::ButtonClass::leave_proxy(GtkButton *button)
{
	Button *tmp_button = G::Object::pointer<Button>(button);
	if (tmp_button)
		tmp_button->on_leave();
	else
	{
		GtkButtonClass *g_class = G::TypeInstance::class_peek_parent<GtkButtonClass>(GTK_BUTTON_GET_CLASS(button));
		if (g_class->leave)
			g_class->leave(button);
	}
}

void
Gtk::ButtonClass::activate_proxy(GtkButton *button)
{
	Button *tmp_button = G::Object::pointer<Button>(button);
	if (tmp_button)
		tmp_button->on_activate();
	else
	{
		GtkButtonClass *g_class = G::TypeInstance::class_peek_parent<GtkButtonClass>(GTK_BUTTON_GET_CLASS(button));
		if (g_class->activate)
			g_class->activate(button);
	}
}

/*  Gtk::Button signal handlers
 */

void
Gtk::Button::on_pressed()
{
	GtkButtonClass *g_class = class_peek_parent<GtkButtonClass>(gtk_button_class());
	if (g_class->pressed)
		g_class->pressed(gtk_button());
}

void
Gtk::Button::on_released()
{
	GtkButtonClass *g_class = class_peek_parent<GtkButtonClass>(gtk_button_class());
	if (g_class->released)
		g_class->released(gtk_button());
}

void
Gtk::Button::on_clicked()
{
	GtkButtonClass *g_class = class_peek_parent<GtkButtonClass>(gtk_button_class());
	if (g_class->clicked)
		g_class->clicked(gtk_button());
}

void
Gtk::Button::on_enter()
{
	GtkButtonClass *g_class = class_peek_parent<GtkButtonClass>(gtk_button_class());
	if (g_class->enter)
		g_class->enter(gtk_button());
}

void
Gtk::Button::on_leave()
{
	GtkButtonClass *g_class = class_peek_parent<GtkButtonClass>(gtk_button_class());
	if (g_class->leave)
		g_class->leave(gtk_button());
}

void
Gtk::Button::on_activate()
{
	GtkButtonClass *g_class = class_peek_parent<GtkButtonClass>(gtk_button_class());
	if (g_class->activate)
		g_class->activate(gtk_button());
}

/*  Gtk::Button properties
 */

const Gtk::Button::LabelPropertyType Gtk::Button::label_property("label");

const Gtk::Button::UseUnderlinePropertyType Gtk::Button::use_underline_property("use_underline");

const Gtk::Button::UseStockPropertyType Gtk::Button::use_stock_property("use_stock");

const Gtk::Button::ReliefPropertyType Gtk::Button::relief_property("relief");

/*  Gtk::Button signals
 */

const Gtk::Button::PressedSignalType Gtk::Button::pressed_signal("pressed");

const Gtk::Button::ReleasedSignalType Gtk::Button::released_signal("released");

const Gtk::Button::ClickedSignalType Gtk::Button::clicked_signal("clicked");

const Gtk::Button::EnterSignalType Gtk::Button::enter_signal("enter");

const Gtk::Button::LeaveSignalType Gtk::Button::leave_signal("leave");

const Gtk::Button::ActivateSignalType Gtk::Button::activate_signal("activate");

/*  Gtk::StockButton
 */

Gtk::StockButton::StockButton(const char *stock_id)
: Button((GtkButton*)ButtonClass::create())
{
	gtk_button_set_label(gtk_button(), stock_id);
	set_use_stock(true);
	set_use_underline(true);
}

