/*  Inti: Integrated Foundation Classes
 *  Copyright (C) 2002-2003 The Inti Development Team.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 
//! @file inti/gtk/liststore.h
//! @brief A GtkListStore C++ wrapper interface.
//! 
//! Provides ListStore, a list model object for use with a TreeView widget.

#ifndef INTI_GTK_LIST_STORE_H
#define INTI_GTK_LIST_STORE_H

#ifndef INTI_G_OBJECT_H
#include <inti/glib/object.h>
#endif

#ifndef INTI_G_VALUE_H
#include <inti/glib/value.h>
#endif

#ifndef INTI_GTK_TREE_MODEL_H
#include <inti/gtk/treemodel.h>
#endif

#ifndef INTI_GTK_TREE_SORTABLE_H
#include <inti/gtk/treesortable.h>
#endif

#ifndef __GTK_LIST_STORE_H__
#include <gtk/gtkliststore.h>
#endif

namespace Inti {

namespace Gtk {

class ListStoreClass;
class TreeIter;

//! @class ListStore liststore.h inti/gtk/liststore.h
//! @brief A GtkListStore C++ wrapper class.
//!
//! The ListStore object is a list model for use with a TreeView widget. It implements
//! the TreeModel interface, and consequentialy, can use all of the methods available
//! there. It also implements the TreeSortable interface so it can be sorted by the
//! view. Finally, it also implements the tree drag and drop interfaces.
//!
//! <B>Example:</B> Creating a simple list store.
//! @code
//! #include <inti/main.h>
//! #include <inti/gtk/window.h>
//! #include "inti/gtk/liststore.h"
//! #include <inti/gtk/treeview.h>
//! #include "inti/gtk/cellrenderer.h"
//!
//! using namespace Inti;
//!
//! class ListStoreWindow : public Gtk::Window
//! {
//! 	Pointer<Gtk::ListStore> list_store;
//! 	Gtk::TreeView *tree_view;
//! public:
//! 	ListStoreWindow();
//! };
//!
//! enum
//! {
//! 	COLUMN_STRING,
//! 	COLUMN_INT,
//! 	COLUMN_BOOLEAN,
//! 	N_COLUMNS
//! };
//!
//! ListStoreWindow::ListStoreWindow()
//! {
//! 	list_store = new Gtk::ListStore(N_COLUMNS, G_TYPE_STRING, G_TYPE_INT, G_TYPE_BOOLEAN);
//!
//! 	Gtk::TreeIter iter;
//! 	for (int i = 0; i < 10; i++)
//! 	{
//! 		// Add a new row to the model
//! 		iter = list_store->append();
//! 		list_store->set_value(iter, COLUMN_STRING, "Some Data");
//! 		list_store->set_value(iter, COLUMN_INT, i);
//! 		list_store->set_value(iter, COLUMN_BOOLEAN, false);
//! 	}
//!
//! 	// Modify a particular row
//! 	Pointer<Gtk::TreePath> path = new Gtk::TreePath("4");
//! 	list_store->get_iter(iter, *path);
//! 	list_store->set_value(iter, COLUMN_BOOLEAN, true);
//!
//! 	// Create a TreeView
//! 	tree_view = new Gtk::TreeView(*list_store);
//!
//! 	// Create first column, associating the "text" attribute of the cell_renderer
//! 	// with the first column of the list_store. Add column to TreeView.
//! 	Gtk::CellRendererText *renderer = new Gtk::CellRendererText;
//! 	renderer->prop_foreground().set("red");
//! 	Gtk::TreeViewColumn *column = new Gtk::TreeViewColumn("Text", renderer, "text", COLUMN_STRING, 0);
//! 	tree_view->append_column(*column);
//!
//! 	// Create second column, associating the "text" attribute of the cell_renderer
//! 	// with the second column of the list_store. Add column to TreeView.
//! 	renderer = new Gtk::CellRendererText;
//! 	renderer->prop_foreground().set("blue");
//! 	column = new Gtk::TreeViewColumn("Number", renderer, "text", COLUMN_INT, 0);
//! 	tree_view->append_column(*column);
//!
//! 	// Create third column, associating the "active" attribute of the toggle cell_renderer
//! 	// with the second column of the list_store. Add column to TreeView. To update checkbox
//! 	// when toggled you must connect to the "toggled_signal".
//! 	Gtk::CellRendererToggle *toggle_renderer = new Gtk::CellRendererToggle;
//! 	column = new Gtk::TreeViewColumn("Boolean", toggle_renderer, "active", COLUMN_BOOLEAN, 0);
//! 	tree_view->append_column(*column);
//!
//! 	add(*tree_view);
//! 	show_all();
//! }
//!
//! int main (int argc, char *argv[])
//! {
//! 	using namespace Main;
//!
//! 	init(&argc, &argv);
//!
//! 	ListStoreWindow window;
//! 	window.sig_destroy().connect(slot(&Inti::Main::quit));
//!
//! 	run();
//! 	return 0;
//! }
//! @endcode

class ListStore : public G::Object, public TreeModel, public TreeSortable
{
	friend class G::Object;
	friend class ListStoreClass;

	ListStore(const ListStore&);
	ListStore& operator=(const ListStore&);

protected:
//! @name Constructors
//! @{

	explicit ListStore(GtkListStore *list_store, bool reference = true);
	//!< Construct a new ListStore from an existing GtkListStore.
	//!< @param list_store A pointer to a GtkListStore.
	//!< @param reference Set false if the initial reference count is floating, set true if it's not.
	//!<
	//!< <BR>The <EM>list_store</EM> can be a newly created GtkListStore or an existing
	//!< GtkListStore (see G::Object::Object).

//! @}

public:
//! @name Constructors
//! @{

	ListStore(int n_columns, ...);
	//!< Construct a list store of n_columns columns of each of the types passed in the variable argument list.
	//!< @param n_columns The number of columns in the list store.
	//!< @param ... All the GType types for the columns, from first to last.
	//!<
	//!< <BR>As an example, ListStore(3, G_TYPE_INT, G_TYPE_STRING, GDK_TYPE_PIXBUF);
	//!< will create a new list store with three columns, of type int, string and GdkPixbuf respectively.

	ListStore(const std::vector<GType>& types);
	//!< Construct a list store with a column for each type passed in the vector of GType.
	//!< @param types A vector of GType containing the column types, from first to last.

	virtual ~ListStore();
	//!< Destructor.
	
//! @}
//! @name Accessors
//! @{

	GtkListStore* gtk_list_store() const { return (GtkListStore*)instance; }
	//!< Get a pointer to the GtkListStore structure.

	GtkListStoreClass* gtk_list_store_class() const;
	//!< Get a pointer to the GtkListStoreClass structure.

	operator GtkListStore* () const;
	//!< Conversion operator; safely converts a ListStore to a GtkListStore pointer.

	bool iter_is_valid(const TreeIter& iter) const;
	//!< Checks if the given <EM>iter</EM> is a valid iter for the list store.
	//!< @param iter A TreeIter.
	//!< @return <EM>true</EM> if the iter is valid, <EM>false</EM> if it's not.
	//!<
	//!<<BR><B>WARNING:</B> This function is slow. Only use it for debugging and/or
	//!< testing purposes.

//! @}
//! @name Methods
//! @{

	TreeIter prepend();
	//!< Prepends a new row to a ListStore.
	//!< @return An unset TreeIter set to the prepended row.
	//!<
	//!< <BR>The row will be empty after this method is called. To fill in values,
	//!< you need to call set_value(), set_object() or set_pointer().

	TreeIter append();
	//!< Appends a new row to a ListStore.
	//!< @return An unset TreeIter set to the appended row.
	//!<
	//!< <BR>The row will be empty after this method is called. To fill in values,
	//!< you need to call set_value(), set_object() or set_pointer().

	TreeIter insert(int position);
	//!< Inserts a new row at <EM>position</EM>.
	//!< @param position The position to insert the new row.
	//!< @return An unset TreeIter set to the inserted row.
	//!<
	//!< <BR>If position is larger than the number of rows on the list, then the new row
	//!< will be appended to the list.The row will be empty after this method is called.
	//!< To fill in values, you need to call set_value(), set_object() or set_pointer().

	TreeIter insert_before(TreeIter& sibling);
	//!< Inserts a new row before sibling.
	//!< @param sibling A valid TreeIter.
	//!< @return An unset TreeIter set to the inserted row.
	//!<
	//!< <BR>The row will be empty after this method is called. To fill in values,
	//!< you need to call set_value(), set_object() or set_pointer().

	TreeIter insert_after(TreeIter& sibling);
	//!< Inserts a new row after sibling.
	//!< @param sibling A valid TreeIter.
	//!< @return An unset TreeIter set to the inserted row.
	//!<
	//!< <BR>The row will be empty after this method is called. To fill in values,
	//!< you need to call set_value(), set_object() or set_pointer().

	bool remove(TreeIter& iter);
	//!< Removes the given row from the list store. 
	//!< @param iter A valid TreeIter.
	//!< @return <EM>true</EM> if <EM>iter</EM> is valid, <EM>false</EM> if it's not.
	//!<
	//!< <BR>After being removed, iter is set to be the next valid row, or invalidated
	//!< if it pointed to the last row in list_store.

	void clear();
	//!< Removes all rows from the list store.

	void reorder(int *new_order);
	//!< Reorders the list store to follow the order indicated by <EM>new_order</EM>.
	//!< @param new_order An integer array indicating the new order for the list.
	//!<
	//!< <BR><B>Note:</B> This method only works with unsorted stores.

	void swap(const TreeIter& a, const TreeIter& b);
	//!< Swaps <EM>a</EM> and <EM>b</EM> in the list store.
	//!< @param a A TreeIter.
	//!< @param b Another TreeIter.
	//!<
	//!< <BR><B>Note:</B> This method only works with unsorted stores.

	void move_after(const TreeIter& iter, const TreeIter *position);
	//!< Moves <EM>iter</EM> in the list store to the position after position.
	//!< @param iter A TreeIter.
	//!< @param position A TreeIter, or null.
	//!<
	//!< <BR><B>Note:</B> This function only works with unsorted stores. If position
	//!< is null, iter will be moved to the start of the list.

	void move_before(const TreeIter& iter, const TreeIter *position);
	//!< Moves <EM>iter</EM> in the list store to the position before position.
	//!< @param iter A TreeIter.
	//!< @param position A TreeIter, or null.
	//!<
	//!< <BR><B>Note:</B> This function only works with unsorted stores. If position
	//!< is null, iter will be moved to the end of the list.

	void set_value(const TreeIter& iter, int column, const G::Value& value);
	//!< Sets the data in the cell specified by iter and column.
	//!< @param iter A valid TreeIter for the row being modified.
	//!< @param column The column number to modify.
	//!< @param value  The new value for the cell.
	//!<
	//!< <BR>The type of value must be convertible to the type of the column.

	void set_value(const TreeIter& iter, int column, const char *str);
	//!< Sets a string value in the cell specified by iter and column.
	//!< @param iter A valid TreeIter for the row being modified.
	//!< @param column The column number to modify.
	//!< @param str The new string for the cell.

//! @}
//! @name Templates
//! @{

	template<typename DataType>
	void set_value(const TreeIter& iter, int column, const DataType& data)
	{
		G::Value value(gtk_list_store()->column_headers[column]);
		value.set(data);
		gtk_list_store_set_value(gtk_list_store(), iter, column, value.g_value());
	}
	//!< Sets the data in the cell specified by iter and column.
	//!< @param iter A valid TreeIter for the row being modified.
	//!< @param column The column number to modify.
	//!< @param data The data to set for the cell.
	//!<
	//!< <BR>This method is used to set values corresponding to the standard data types used
	//!< by G::Value, such as bool, int, double, String and unsigned int. There is a good
	//!< example of setting values in the inti-demo program <demos/inti-demo/liststore.cc>.

	template<typename DataType>
	void set_enum(const TreeIter& iter, int column, const DataType& data)
	{
		G::Value value(gtk_list_store()->column_headers[column]);
		value.set((int)data);
		gtk_list_store_set_value(gtk_list_store(), iter, column, value.g_value());
	}
	//!< Sets the enum data in the cell specified by iter and column.
	//!< @param iter A valid TreeIter for the row being modified.
	//!< @param column The column number to modify.
	//!< @param data The enum value to set.
	//!<
	//!< <BR>This method is used to set enumeration values. The DataType is the type of the
	//!< enumeration being set. There is a good example of setting values in the inti-demo
	//!< program <demos/inti-demo/liststore.cc>.

	template<typename DataType>
	void set_object(const TreeIter& iter, int column, const DataType& data)
	{
		G::Value value(gtk_list_store()->column_headers[column]);
		value.set((G::Object*)data);
		gtk_list_store_set_value(gtk_list_store(), iter, column, value.g_value());
	}
	//!< Sets the object pointer <EM>data</EM> in the cell specified by iter and column.
	//!< @param iter A valid TreeIter for the row being modified.
	//!< @param column The column number to modify.
	//!< @param data A pointer to an object derived from G::Object, passed by reference.
	//!<
	//!< <BR>There is a good example of setting values in the inti-demo program
	//!< <demos/inti-demo/liststore.cc>.

	template<typename DataType>
	void set_pointer(const TreeIter& iter, int column, const DataType& data)
	{
		G::Value value(gtk_list_store()->column_headers[column]);
		value.set((void*)data);
		gtk_list_store_set_value(gtk_list_store(), iter, column, value.g_value());
	}
	//!< Sets the pointer <EM>data</EM> in the cell specified by iter and column.
	//!< @param iter A valid TreeIter for the row being modified.
	//!< @param column The column number to modify.
	//!< @param data The pointer to set in the cell.
	//!<
	//!< <BR>The <EM>data</EM> argument can be a pointer to any object. The pointer is managed
	//!< internally as a generic (void*) pointer. Unlike set_object() which passes the G::Object
	//!< pointer internally as a GObject pointer, set_pointer() passes the pointer as is, without
	//!< interpretation. There is a good example of setting values in the inti-demo program
	//!< <demos/inti-demo/liststore.cc>.

//! @}
};

} // namespace Gtk

} // namespace Inti

#endif // INTI_GTK_LIST_STORE_H

