/*  Inti: Integrated Foundation Classes
 *  Copyright (C) 2002-2003 The Inti Development Team.
 *
 *  spinbutton.cc - GtkSpinButton C++ wrapper implementation
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 
#include "spinbutton.h"
#include "private/spinbutton_p.h"
#include <math.h>
#include <stdlib.h>

const int max_digits = 20;

using namespace Inti;

/*  Gtk::Button
 */

Gtk::SpinButton::SpinButton(GtkSpinButton *spin_button, bool reference) 
: Entry((GtkEntry*)spin_button, reference)
{
}

Gtk::SpinButton::SpinButton()
: Entry((GtkEntry*)SpinButtonClass::create())
{
}

Gtk::SpinButton::SpinButton(Adjustment *adjustment, double climb_rate, unsigned int digits)
: Entry((GtkEntry*)SpinButtonClass::create())
{
	configure(adjustment, climb_rate, digits);
}

Gtk::SpinButton::SpinButton(double min, double max, double step)
: Entry((GtkEntry*)SpinButtonClass::create())
{
	g_return_if_fail(min < max);
	g_return_if_fail(step != 0.0);

	GtkObject *adjustment = gtk_adjustment_new(min, min, max, step, 10 * step, step);
	int digits;
	if (fabs(step) >= 1.0 || step == 0.0)
    	digits = 0;
  	else
	{
	    digits = abs((int)floor(log10(fabs(step))));
		if (digits > max_digits)
			digits = max_digits;
	}

	gtk_spin_button_configure(gtk_spin_button(), (GtkAdjustment*)adjustment, step, digits);
	set_numeric(true);
}

Gtk::SpinButton::~SpinButton()
{
}

GtkSpinButtonClass*
Gtk::SpinButton::gtk_spin_button_class() const 
{
	return get_class<GtkSpinButtonClass>();
}
	
Gtk::SpinButton::operator GtkSpinButton* () const 
{ 
	return this ? gtk_spin_button() : 0;
}
	
Gtk::Adjustment*
Gtk::SpinButton::get_adjustment() const
{
	return G::Object::wrap<Adjustment>(gtk_spin_button_get_adjustment(gtk_spin_button()));
}

unsigned int
Gtk::SpinButton::get_digits() const
{ 
	return gtk_spin_button_get_digits(gtk_spin_button()); 
}

void 
Gtk::SpinButton::get_increments(double *step, double *page) const
{
	gtk_spin_button_get_increments(gtk_spin_button(), step, page);
}

void 
Gtk::SpinButton::get_range(double *min, double *max) const
{	
	gtk_spin_button_get_range(gtk_spin_button(), min, max);
}

double
Gtk::SpinButton::get_value() const
{
	return gtk_spin_button_get_value(gtk_spin_button());
}

int
Gtk::SpinButton::get_value_as_int() const
{
	return gtk_spin_button_get_value_as_int(gtk_spin_button());
}

Gtk::SpinButtonUpdatePolicy
Gtk::SpinButton::get_update_policy() const
{	
	return (SpinButtonUpdatePolicy)gtk_spin_button_get_update_policy(gtk_spin_button());
}

bool
Gtk::SpinButton::get_numeric() const
{
	return gtk_spin_button_get_numeric(gtk_spin_button());
}

bool 
Gtk::SpinButton::get_wrap() const
{ 
	return gtk_spin_button_get_wrap(gtk_spin_button());
}

bool 
Gtk::SpinButton::get_snap_to_ticks() const
{ 
	return gtk_spin_button_get_snap_to_ticks(gtk_spin_button());
}

void
Gtk::SpinButton::configure(Adjustment *adjustment, double climb_rate, unsigned int digits)
{
	gtk_spin_button_configure(gtk_spin_button(), *adjustment, climb_rate, digits);
}

void 
Gtk::SpinButton::set_adjustment(Adjustment *adjustment)
{
	gtk_spin_button_set_adjustment(gtk_spin_button(), *adjustment);
}

void 
Gtk::SpinButton::set_digits(unsigned int digits)
{
	gtk_spin_button_set_digits(gtk_spin_button(), digits);
}

void
Gtk::SpinButton::set_increments(double step, double page)
{
	gtk_spin_button_set_increments(gtk_spin_button(), step, page);
}

void 
Gtk::SpinButton::set_range(double min, double max)
{
	gtk_spin_button_set_range(gtk_spin_button(), min, max);
}
void 
Gtk::SpinButton::set_value(double value)
{
	gtk_spin_button_set_value(gtk_spin_button(), value);
}

void
Gtk::SpinButton::set_update_policy(SpinButtonUpdatePolicy policy)
{
	gtk_spin_button_set_update_policy(gtk_spin_button(), (GtkSpinButtonUpdatePolicy)policy);
}

void 
Gtk::SpinButton::set_numeric(bool numeric)
{
	gtk_spin_button_set_numeric(gtk_spin_button(), numeric);
}

void 
Gtk::SpinButton::spin(SpinType direction, double increment)
{
	gtk_spin_button_spin(gtk_spin_button(), (GtkSpinType)direction, increment);
}

void 
Gtk::SpinButton::set_wrap(bool wrap)
{
	gtk_spin_button_set_wrap(gtk_spin_button(), wrap);
}

void 
Gtk::SpinButton::set_snap_to_ticks(bool snap_to_ticks)
{
	gtk_spin_button_set_snap_to_ticks(gtk_spin_button(), snap_to_ticks);
}

void 
Gtk::SpinButton::update()
{
	gtk_spin_button_update(gtk_spin_button());
}

/*  Gtk::SpinButtonClass
 */

void
Gtk::SpinButtonClass::init(GtkSpinButtonClass *g_class)
{
	EntryClass::init((GtkEntryClass*)g_class);
	g_class->input = &input_proxy;
	g_class->output = &output_proxy;
	g_class->value_changed = &value_changed_proxy;
}

GType
Gtk::SpinButtonClass::get_type()
{
	static GType type = 0;
	if (!type)
	{
		type = G::TypeInstance::register_type(GTK_TYPE_SPIN_BUTTON, (GClassInitFunc)init);
	}
	return type;
}

void*
Gtk::SpinButtonClass::create()
{
	return g_object_new(get_type(), 0);
}

gint
Gtk::SpinButtonClass::input_proxy(GtkSpinButton *spin_button, gdouble *new_value)
{
	SpinButton *tmp_spin_button = G::Object::pointer<SpinButton>(spin_button);
	gint result = 0;
	if (tmp_spin_button)
		result = tmp_spin_button->on_input(new_value);
	else
	{
		GtkSpinButtonClass *tmp_class = GTK_SPIN_BUTTON_GET_CLASS(spin_button);
		GtkSpinButtonClass *g_class = G::TypeInstance::class_peek_parent<GtkSpinButtonClass>(tmp_class);
		if (g_class->input)
			result = g_class->input(spin_button, new_value);
	}
	return result;
}

gint
Gtk::SpinButtonClass::output_proxy(GtkSpinButton *spin_button)
{
	SpinButton *tmp_spin_button = G::Object::pointer<SpinButton>(spin_button);
	gint result = 0;
	if (tmp_spin_button)
		result = tmp_spin_button->on_output();
	else
	{
		GtkSpinButtonClass *tmp_class = GTK_SPIN_BUTTON_GET_CLASS(spin_button);
		GtkSpinButtonClass *g_class = G::TypeInstance::class_peek_parent<GtkSpinButtonClass>(tmp_class);
		if (g_class->output)
			result = g_class->output(spin_button);
	}
	return result;
}

void
Gtk::SpinButtonClass::value_changed_proxy(GtkSpinButton *spin_button)
{
	SpinButton *tmp_spin_button = G::Object::pointer<SpinButton>(spin_button);
	if (tmp_spin_button)
		tmp_spin_button->on_value_changed();
	else
	{
		GtkSpinButtonClass *tmp_class = GTK_SPIN_BUTTON_GET_CLASS(spin_button);
		GtkSpinButtonClass *g_class = G::TypeInstance::class_peek_parent<GtkSpinButtonClass>(tmp_class);
		if (g_class->value_changed)
			g_class->value_changed(spin_button);
	}
}

/*  Signal handlers
 */

int
Gtk::SpinButton::on_input(double *new_value)
{
	GtkSpinButtonClass *g_class = class_peek_parent<GtkSpinButtonClass>(gtk_spin_button_class());
	gint result = 0;
	if (g_class->input)
		result = g_class->input(gtk_spin_button(), new_value);
	return result;
}

int
Gtk::SpinButton::on_output()
{
	GtkSpinButtonClass *g_class = class_peek_parent<GtkSpinButtonClass>(gtk_spin_button_class());
	gint result = 0;
	if (g_class->output)
		result = g_class->output(gtk_spin_button());
	return result;
}

void
Gtk::SpinButton::on_value_changed()
{
	GtkSpinButtonClass *g_class = class_peek_parent<GtkSpinButtonClass>(gtk_spin_button_class());
	if (g_class->value_changed)
		g_class->value_changed(gtk_spin_button());
}

/*  Properties
 */

const Gtk::SpinButton::AdjustmentPropertyType Gtk::SpinButton::adjustment_property("adjustment");

const Gtk::SpinButton::ClimbRatePropertyType Gtk::SpinButton::climb_rate_property("climb_rate");

const Gtk::SpinButton::DigitsPropertyType Gtk::SpinButton::digits_property("digits");

const Gtk::SpinButton::SnapToTicksPropertyType Gtk::SpinButton::snap_to_ticks_property("snap_to_ticks");

const Gtk::SpinButton::NumericPropertyType Gtk::SpinButton::numeric_property("numeric");

const Gtk::SpinButton::WrapPropertyType Gtk::SpinButton::wrap_property("wrap");

const Gtk::SpinButton::UpdatePolicyPropertyType Gtk::SpinButton::update_policy_property("update_policy");

const Gtk::SpinButton::ValuePropertyType Gtk::SpinButton::value_property("value");

/*  Signals
 */

const Gtk::SpinButton::InputSignalType Gtk::SpinButton::input_signal("input");

const Gtk::SpinButton::OutputSignalType Gtk::SpinButton::output_signal("output");

const Gtk::SpinButton::ValueChangedSignalType Gtk::SpinButton::value_changed_signal("value_changed");

