/*  Inti: Integrated Foundation Classes
 *  Copyright (C) 2002 The Inti Development Team.
 *  Copyright (C) 2000 Red Hat, Inc.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 
//! @file inti/gtk/statusbar.h
//! @brief A GtkStatusbar C++ wrapper interface.
//!
//! Provides Statusbar, a widget that allows you to report messages of minor importance to the user.

#ifndef INTI_GTK_STATUSBAR_H
#define INTI_GTK_STATUSBAR_H

#ifndef INTI_GTK_HBOX_H
#include <inti/gtk/box.h>
#endif

#ifndef __GTK_STATUSBAR_H__
#include <gtk/gtkstatusbar.h>
#endif

namespace Inti {

namespace Gtk {

class StatusbarClass;

//! @class Statusbar statusbar.h inti/gtk/statusbar.h
//! @brief A GtkStatusbar C++ wrapper class.
//!
//! A Statusbar is usually placed along the bottom of an application's main Gtk::Window. It may
//! provide a regular commentary of the application's status (as is usually the case in a web
//! browser, for example), or may be used to simply output a message when the status changes,
//! (when an upload is complete in an FTP client, for example). It may also have a resize grip
//! (a triangular area in the lower right corner) which can be clicked on to resize the window
//! containing the statusbar.
//!
//! Statusbars maintain a stack of messages. The message at the top of the each bar's stack
//! is the one that will currently be displayed. Any messages added to a statusbar's stack 
//! must specify a context_id that is used to uniquely identify the source of a message. This
//! context_id can be generated by get_context_id(), given a message and the statusbar that
//! it will be added to. Note that messages are stored in a stack, and when choosing which
//! message to display, the stack structure is adhered to, regardless of the context 
//! identifier of a message. Messages are added to the bar's stack with push(). The message
//! at the top of the stack can be removed using pop(). A message can be removed from anywhere
//! in the stack if it's message_id was recorded at the time it was added. This is done using
//! remove().

class Statusbar : public HBox
{
	friend class G::Object;
	friend class StatusbarClass;

	Statusbar(const Statusbar&);
	Statusbar& operator=(const Statusbar&);
	
protected:
//! @name Constructors
//! @{

	explicit Statusbar(GtkStatusbar *statusbar, bool reference = false);
	//!< Construct a new Statusbar from an existing GtkStatusbar.
	//!< @param statusbar A pointer to a GtkStatusbar.
	//!< @param reference Set false if the initial reference count is floating, set true if it's not.
	//!<
	//!< <BR>The <EM>statusbar</EM> can be a newly created GtkStatusbar or an existing
	//!< GtkStatusbar(see G::Object::Object).
	
//! @}
//! @name Signal Handlers
//! @{

	virtual void on_text_pushed(unsigned int context_id, const String& text);
	//!< Called whenever a new message gets pushed onto a statusbar's stack.
	//!< @param context_id The context id of the relevant message.
	//!< @param text The message that was pushed.
	
	virtual void on_text_popped(unsigned int context_id, const String& text);
	//!< Called whenever a new message is popped off a statusbar's stack.
	//!< @param context_id The context id of the relevant message.
	//!< @param text The message that was just popped.

//! @}
//  Signals

	typedef G::Signal2<void, unsigned int, const char*> TextPushedSignalType;
	typedef G::SignalProxy<TypeInstance, TextPushedSignalType> TextPushedSignalProxy;
	static const TextPushedSignalType text_pushed_signal;

	typedef G::Signal2<void, unsigned int, const char*> TextPoppedSignalType;
	typedef G::SignalProxy<TypeInstance, TextPoppedSignalType> TextPoppedSignalProxy;
	static const TextPoppedSignalType text_popped_signal;

public:
//! @name Constructors
//! @{

	Statusbar();
	//!< Construct a new statusbar.

	virtual ~Statusbar();
	//!< Destructor.
	
//! @}
//! @name Accessors
//! @{

	GtkStatusbar* gtk_statusbar() const { return (GtkStatusbar*)instance; }
	//!< Get a pointer to the GtkStatusbar structure.

	GtkStatusbarClass* gtk_statusbar_class() const;
	//!< Get a pointer to the GtkStatusbarClass structure.

	operator GtkStatusbar* () const;
	//!< Conversion operator; safely converts a Statusbar to a GtkStatusbar pointer.

	unsigned int get_context_id(const String& context_description) const;
	//!< Returns a new context identifier, given a description of the actual context.
	//!< @param context_description A textual description of what context the new message is being used in.
	//!< @return An integer id.

	bool get_has_resize_grip() const;
	//!< Returns whether the statusbar has a resize grip.

//! @}
//! @name Methods
//! @{

	unsigned int push(const String& text, unsigned int context_id = 0);
	//!< Pushes a new message onto a statusbar's stack.
	//!< @param context_id The message's context id, as returned by get_context_id().
	//!< @param text The message to add to the statusbar.
	//!< @return The message's new message id for use with remove().

	void pop(unsigned int context_id = 0);
	//!< Removes the message at the top of the statusbar's stack.
	//!< @param context_id A context identifier.

	void remove(unsigned int message_id, unsigned int context_id = 0);
	//!< Forces the removal of a message from the statusbar's stack.
	//!< @param context_id A context identifier.
	//!< @param message_id A message identifier, as returned by push()
	//!<
	//!< <BR>The exact context_id and message_id must be specified.
	
	void set_has_resize_grip(bool setting);
	//!< Sets whether the statusbar has a resize grip; true by default.
	//!< @param setting <EM>true</EM> to have a resize grip.

//! @}
//! @name Signal Proxies
//! @{

	const TextPushedSignalProxy sig_text_pushed()
	{
		return TextPushedSignalProxy(this, &text_pushed_signal);
	}
	//!< Connect to the text_pushed_signal; emitted whenever a new message gets pushed onto a statusbar's stack.

	const TextPoppedSignalProxy sig_text_popped()
	{
		return TextPoppedSignalProxy(this, &text_popped_signal);
	}
	//!< Connect to the text_popped_signal; emitted whenever a new message is popped off a statusbar's stack.
	
//! @}
};

} // namespace Gtk

} // namespace Inti

#endif // INTI_GTK_STATUSBAR_H

