/*  Inti: Integrated Foundation Classes
 *  Copyright (C) 2002-2003 The Inti Development Team.
 *
 *  window.cc - GtkWindow C++ wrapper implementation
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "window.h"
#include "private/window_p.h"
#include "accelgroup.h"
#include "../gdk/events.h"
#include "../gdk/screen.h"
#include "../gdk/window.h"
#include "../gdk-pixbuf/pixbuf.h"
#include "../glib/error.h"

using namespace Inti;

/*  Gtk::Window
 */

Gtk::Window::Window(GtkWindow *window, bool reference)
: Bin((GtkBin*)window, reference)
{
}

Gtk::Window::Window(WindowType type)
: Bin((GtkBin*)WindowClass::create())
{
	gtk_window()->type = (GtkWindowType)type;
}

Gtk::Window::~Window()
{
}

GtkWindowClass*
Gtk::Window::gtk_window_class() const
{
	return get_class<GtkWindowClass>();
}

Gtk::Window::operator GtkWindow* () const 
{ 
	return this ? gtk_window() : 0; 
}

bool
Gtk::Window::is_toplevel() const
{
	return gtk_window()->type == GTK_WINDOW_TOPLEVEL;
}

bool
Gtk::Window::is_popup() const
{
	return gtk_window()->type == GTK_WINDOW_POPUP;
}

String
Gtk::Window::get_title() const
{
	return gtk_window_get_title(gtk_window());
}

String 
Gtk::Window::get_role() const
{
	return gtk_window_get_role(gtk_window());
}

Gtk::Widget*
Gtk::Window::get_focus() const
{
	return G::Object::wrap<Widget>(gtk_window_get_focus(gtk_window()));
}

Gtk::Window*
Gtk::Window::get_transient_for() const
{
	GtkWindow *window = gtk_window_get_transient_for(gtk_window());
	return window ? G::Object::wrap<Window>(window) : 0;
}

Gdk::WindowTypeHint
Gtk::Window::get_type_hint() const
{
	return (Gdk::WindowTypeHint)gtk_window_get_type_hint(gtk_window());
}

bool
Gtk::Window::get_skip_taskbar_hint() const
{
	return gtk_window_get_skip_taskbar_hint(gtk_window());
}

bool
Gtk::Window::get_skip_pager_hint() const
{
	return gtk_window_get_skip_pager_hint(gtk_window());
}

bool
Gtk::Window::get_destroy_with_parent() const
{
	return gtk_window_get_destroy_with_parent(gtk_window());
}

bool 
Gtk::Window::get_resizable() const
{
	return gtk_window_get_resizable(gtk_window());
}

Gdk::Gravity
Gtk::Window::get_gravity() const
{
	return (Gdk::Gravity)gtk_window_get_gravity(gtk_window());
}

Gdk::Screen* 
Gtk::Window::get_screen() const
{
	return G::Object::wrap<Gdk::Screen>(gtk_window_get_screen(gtk_window()));
}

bool
Gtk::Window::get_has_frame() const
{
	return gtk_window_get_has_frame(gtk_window());
}

void
Gtk::Window::get_frame_dimensions(int *left, int *top, int *right, int *bottom) const
{
	gtk_window_get_frame_dimensions(gtk_window(), left, top, right, bottom);
}

bool
Gtk::Window::get_decorated() const
{
	return gtk_window_get_decorated(gtk_window());
}

namespace { // real_get_icon_list

bool real_get_icon_list(GtkWindow *window, std::vector<Gdk::Pixbuf*>& icons)
{
	g_return_val_if_fail(icons.empty(), false);
	
	GList *first = window ? gtk_window_get_icon_list(window) : gtk_window_get_default_icon_list();
	GList *next = first;

	while (next)
	{
		icons.push_back(G::Object::wrap<Gdk::Pixbuf>((GdkPixbuf*)next->data));
		next = g_list_next(next);
	}
	g_list_free(first);
	return !icons.empty();
}

} // real_get_icon_list

bool 
Gtk::Window::get_icon_list(std::vector<Gdk::Pixbuf*>& icons) const
{
	return real_get_icon_list(gtk_window(), icons);
}

Gdk::Pixbuf*
Gtk::Window::get_icon() const
{
	GdkPixbuf *icon = gtk_window_get_icon(gtk_window());
	return icon ? G::Object::wrap<Gdk::Pixbuf>(icon) : 0;
}

bool 
Gtk::Window::get_default_icon_list(std::vector<Gdk::Pixbuf*>& icons)
{
	return real_get_icon_list(0, icons);
}

bool 
Gtk::Window::get_modal() const
{
	return gtk_window_get_modal(gtk_window());
}

Gdk::ModifierTypeField
Gtk::Window::get_mnemonic_modifier() const
{
	return (Gdk::ModifierTypeField)gtk_window_get_mnemonic_modifier(gtk_window());
}

void 
Gtk::Window::get_default_size(int *width, int *height) const
{
	gtk_window_get_default_size(gtk_window(), width, height);
}

void 
Gtk::Window::get_size(int *width, int *height) const
{
	gtk_window_get_size(gtk_window(), width, height);
}

void
Gtk::Window::get_position(int *root_x, int *root_y) const
{
	gtk_window_get_position(gtk_window(), root_x, root_y);
}

Gdk::Point
Gtk::Window::get_position() const
{
	int root_x, root_y;
	get_position(&root_x, &root_y);
	return Gdk::Point(root_x, root_y);
}

Gtk::WindowGroup*
Gtk::Window::get_group() const
{
	return G::Object::wrap<WindowGroup>(_gtk_window_get_group(gtk_window()));
}

void
Gtk::Window::set_title(const String& title)
{
	gtk_window_set_title(gtk_window(), title.c_str());
}

void
Gtk::Window::set_wmclass(const String& wmclass_name, const String& wmclass_class)
{
	gtk_window_set_wmclass(gtk_window(), wmclass_name.c_str(), wmclass_class.c_str());
}

void
Gtk::Window::set_role(const String& role)
{
	gtk_window_set_role(gtk_window(), role.c_str());
}

Gtk::AccelGroup*
Gtk::Window::add_accel_group(AccelGroup *accel_group)
{
	AccelGroup *tmp_accel_group = accel_group;
	
	if (!tmp_accel_group)
		tmp_accel_group = new Gtk::AccelGroup;

	gtk_window_add_accel_group(gtk_window(), *tmp_accel_group);
	
	if (!accel_group)
		tmp_accel_group->unref();
	
	return tmp_accel_group;
}

void
Gtk::Window::remove_accel_group(AccelGroup& accel_group)
{
	gtk_window_remove_accel_group(gtk_window(), accel_group.gtk_accel_group());
}

void
Gtk::Window::set_position(WindowPosition position)
{
	gtk_window_set_position(gtk_window(), (GtkWindowPosition)position);
}
	
bool
Gtk::Window::activate_focus()
{
	return gtk_window_activate_focus(gtk_window());
}

void
Gtk::Window::set_focus(Widget *widget)
{
	gtk_window_set_focus(gtk_window(), *widget);
}

void
Gtk::Window::set_default(Widget *default_widget)
{
	gtk_window_set_default(gtk_window(), *default_widget);
}

bool
Gtk::Window::activate_default()
{
	return gtk_window_activate_default(gtk_window());
}

void
Gtk::Window::set_transient_for(Window *parent)
{
	gtk_window_set_transient_for(gtk_window(), parent->gtk_window());
}

void 
Gtk::Window::set_type_hint(Gdk::WindowTypeHint hint)
{
	gtk_window_set_type_hint(gtk_window(), (GdkWindowTypeHint)hint);
}


void 
Gtk::Window::set_skip_taskbar_hint(bool setting)
{
	gtk_window_set_skip_taskbar_hint(gtk_window(), setting);
}

void 
Gtk::Window::set_skip_pager_hint(bool setting)
{
	gtk_window_set_skip_pager_hint(gtk_window(), setting);
}

void
Gtk::Window::set_destroy_with_parent(bool setting)
{
	gtk_window_set_destroy_with_parent(gtk_window(), setting);
}

void
Gtk::Window::set_resizable(bool resizable)
{
	gtk_window_set_resizable(gtk_window(), resizable);
}

void
Gtk::Window::set_gravity(Gdk::Gravity gravity)
{
	gtk_window_set_gravity(gtk_window(), (GdkGravity)gravity);
}

void
Gtk::Window::set_geometry_hints(Widget *geometry_widget, const Gdk::Geometry& geometry)
{
	gtk_window_set_geometry_hints(gtk_window(), *geometry_widget, geometry.gdk_geometry(), geometry.gdk_window_hints());
}

void 
Gtk::Window::set_screen(const Gdk::Screen& screen)
{
	gtk_window_set_screen(gtk_window(), screen.gdk_screen());
}

void
Gtk::Window::set_has_frame(bool setting)
{
	gtk_window_set_has_frame(gtk_window(), setting);
}

void
Gtk::Window::set_frame_dimensions(int left, int top, int right, int bottom)
{
	gtk_window_set_frame_dimensions(gtk_window(), left, top, right, bottom);
}

void
Gtk::Window::set_decorated(bool setting)
{
	gtk_window_set_decorated(gtk_window(), setting);
}

namespace { // real_set_icon_list

void real_set_icon_list(GtkWindow *window, std::vector<Gdk::Pixbuf*>& icons)
{
	g_return_if_fail(!icons.empty());
	GList *tmp_icons = 0;
	int count = icons.size();

	int i = 0;
	while (i < count)
	{
		tmp_icons = g_list_append(tmp_icons, icons[i]->gdk_pixbuf());
		++i;
	}
 	window ? gtk_window_set_icon_list(window, tmp_icons) : gtk_window_set_default_icon_list(tmp_icons);
  	g_list_free(tmp_icons);
}

} // real_set_icon_list

void
Gtk::Window::set_icon_list(std::vector<Gdk::Pixbuf*>& icons)
{
	real_set_icon_list(gtk_window(), icons);
}

void 
Gtk::Window::set_icon(Gdk::Pixbuf& icon)
{
	gtk_window_set_icon(gtk_window(), icon.gdk_pixbuf());
}
	
bool 
Gtk::Window::set_icon_from_file(const String& filename, G::Error *error)
{
	return gtk_window_set_icon_from_file(gtk_window(), filename.c_str(), *error);
}

void
Gtk::Window::set_default_icon_list(std::vector<Gdk::Pixbuf*>& icons)
{
	real_set_icon_list(0, icons);
}

bool 
Gtk::Window::set_default_icon_from_file(const String& filename, G::Error *error)
{
	return gtk_window_set_default_icon_from_file(filename.c_str(), *error);
}

void 
Gtk::Window::set_auto_startup_notification(bool setting)
{
	gtk_window_set_auto_startup_notification(setting);
}

void
Gtk::Window::set_modal(bool modal)
{
	gtk_window_set_modal(gtk_window(), modal);
}

bool
Gtk::Window::list_toplevels(std::vector<Widget*>& toplevels)
{
	g_return_val_if_fail(toplevels.empty(), false);
	GList *first = gtk_window_list_toplevels();
	GList *next = first;

	while (next)
	{
		toplevels.push_back(G::Object::wrap<Widget>((GtkWidget*)next->data));
		next = g_list_next(next);
	}

	g_list_free(first);
	return !toplevels.empty();
}

void
Gtk::Window::add_mnemonic(unsigned int keyval, Widget& target)
{
	gtk_window_add_mnemonic(gtk_window(), keyval, target.gtk_widget());	
}		

void 
Gtk::Window::remove_mnemonic(unsigned int keyval, Widget& target)
{
	gtk_window_remove_mnemonic(gtk_window(), keyval, target.gtk_widget());
}

bool
Gtk::Window::mnemonic_activate(unsigned int keyval, Gdk::ModifierTypeField modifier)
{
	return gtk_window_mnemonic_activate(gtk_window(), keyval, (GdkModifierType)modifier);
}

void 
Gtk::Window::set_mnemonic_modifier(Gdk::ModifierTypeField modifier)
{
	gtk_window_set_mnemonic_modifier(gtk_window(), (GdkModifierType)modifier);
}

void 
Gtk::Window::present()
{
	gtk_window_present(gtk_window());
}

void
Gtk::Window::iconify()
{
	gtk_window_iconify(gtk_window());
}

void 
Gtk::Window::deiconify()
{
	gtk_window_deiconify(gtk_window());
}

void 
Gtk::Window::stick()
{
	gtk_window_stick(gtk_window());
}

void 
Gtk::Window::unstick()
{
	gtk_window_unstick(gtk_window());
}

void 
Gtk::Window::maximize()
{
	gtk_window_maximize(gtk_window());
}

void 
Gtk::Window::unmaximize()
{
	gtk_window_unmaximize(gtk_window());
}

void 
Gtk::Window::fullscreen()
{
	gtk_window_fullscreen(gtk_window());
}

void 
Gtk::Window::unfullscreen()
{
	gtk_window_unfullscreen(gtk_window());
}

void
Gtk::Window::begin_resize_drag(Gdk::WindowEdge edge, int button, int root_x, int root_y, unsigned int timestamp)
{
	gtk_window_begin_resize_drag(gtk_window(), (GdkWindowEdge)edge, button, root_x, root_y, timestamp);
}

void
Gtk::Window::begin_move_drag(int button, int root_x, int root_y, unsigned int timestamp)
{
	gtk_window_begin_move_drag(gtk_window(), button, root_x, root_y, timestamp);
}

void 
Gtk::Window::set_default_size(int width, int height)
{
	gtk_window_set_default_size(gtk_window(), width, height);
}

void 
Gtk::Window::resize(int width, int height)
{
	gtk_window_resize(gtk_window(), width, height);
}

void
Gtk::Window::move(int x, int y)
{
	gtk_window_move(gtk_window(), x, y);
}

void
Gtk::Window::move(const Gdk::Point& point)
{
	move(point.x(), point.y());
}

bool
Gtk::Window::parse_geometry(const char *geometry)
{
	return gtk_window_parse_geometry(gtk_window(), geometry);
}

void
Gtk::Window::reshow_with_initial_size()
{
	gtk_window_reshow_with_initial_size(gtk_window());
}

/*  Gtk::WindowClass
 */

void
Gtk::WindowClass::init(GtkWindowClass *g_class)
{
	BinClass::init((GtkBinClass*)g_class);
	g_class->set_focus = &set_focus_proxy;
	g_class->frame_event = &frame_event_proxy;
	g_class->keys_changed = &keys_changed_proxy;
	((GtkWidgetClass*)g_class)->delete_event = &delete_event_proxy;
}

GType
Gtk::WindowClass::get_type()
{
	static GType type = 0;
	if (!type)
	{
		type = G::TypeInstance::register_type(GTK_TYPE_WINDOW, (GClassInitFunc)init);
	}
	return type;
}

void*
Gtk::WindowClass::create()
{
	return g_object_new(get_type(), 0);
}

void
Gtk::WindowClass::set_focus_proxy(GtkWindow *window, GtkWidget *focus)
{
	Window *tmp_window = G::Object::pointer<Window>(window);
	if (tmp_window)
		tmp_window->on_set_focus(G::Object::wrap<Gtk::Widget>(focus));
	else
	{
		GtkWindowClass *g_class = G::TypeInstance::class_peek_parent<GtkWindowClass>(GTK_WINDOW_GET_CLASS(window));
		if (g_class->set_focus)
			g_class->set_focus(window, focus);
	}
}

gboolean
Gtk::WindowClass::frame_event_proxy(GtkWindow *window, GdkEvent *event)
{
	gboolean result = FALSE;
	Window *tmp_window = G::Object::pointer<Window>(window);
	if (tmp_window)
	{
		Gdk::Event tmp_event(event);
		result = tmp_window->on_frame_event(tmp_event);
	}
	else
	{
		GtkWindowClass *g_class = G::TypeInstance::class_peek_parent<GtkWindowClass>(GTK_WINDOW_GET_CLASS(window));
		if (g_class->frame_event)
			result = g_class->frame_event(window, event);
	}
	return result;
}

void
Gtk::WindowClass::keys_changed_proxy(GtkWindow *window)
{
	Window *tmp_window = G::Object::pointer<Window>(window);
	if (tmp_window)
		tmp_window->on_keys_changed();
	else
	{
		GtkWindowClass *g_class = G::TypeInstance::class_peek_parent<GtkWindowClass>(GTK_WINDOW_GET_CLASS(window));
		if (g_class->keys_changed)
			g_class->keys_changed(window);
	}
}

gboolean
Gtk::WindowClass::delete_event_proxy(GtkWidget *widget, GdkEventAny *event)
{
	gboolean result = FALSE;
	Window *tmp_window = G::Object::pointer<Window>(widget);
	if (tmp_window)
	{
		Gdk::Event tmp_event((GdkEvent*)event);
		result = tmp_window->on_delete_event(*tmp_event.any());
	}
	else
	{
		GtkWidgetClass *g_class = G::TypeInstance::class_peek_parent<GtkWidgetClass>(GTK_WIDGET_GET_CLASS(widget));
		if (g_class->delete_event)
			result = g_class->delete_event(widget, event);
	}
	return result;
}

/*  Gtk::Window signal handlers
 */

void
Gtk::Window::on_set_focus(Widget *focus)
{
	GtkWindowClass *g_class = class_peek_parent<GtkWindowClass>(gtk_window_class());
	if (g_class->set_focus)
		g_class->set_focus(gtk_window(), *focus);
}

bool
Gtk::Window::on_frame_event(const Gdk::Event& event)
{
	bool result = false;
	GtkWindowClass *g_class = class_peek_parent<GtkWindowClass>(gtk_window_class());
	if (g_class->frame_event)
		result = g_class->frame_event(gtk_window(), event.gdk_event());
	return result;
}

void
Gtk::Window::on_keys_changed()
{
	GtkWindowClass *g_class = class_peek_parent<GtkWindowClass>(gtk_window_class());
	if (g_class->keys_changed)
		g_class->keys_changed(gtk_window());
}

bool
Gtk::Window::on_delete_event(const Gdk::EventAny& event)
{
	bool result = false;
	GtkWidgetClass *g_class = class_peek_parent<GtkWidgetClass>(gtk_widget_class());
	if (g_class->delete_event)
		result = g_class->delete_event(gtk_widget(), event.gdk_event_any());
	return result;
}

/*  Gtk::Window properties
 */

const Gtk::Window::TypePropertyType Gtk::Window::type_property("type");

const Gtk::Window::TitlePropertyType Gtk::Window::title_property("title");

const Gtk::Window::AllowShrinkPropertyType Gtk::Window::allow_shrink_property("allow_shrink");

const Gtk::Window::AllowGrowPropertyType Gtk::Window::allow_grow_property("allow_grow");

const Gtk::Window::ResizablePropertyType Gtk::Window::resizable_property("resizable");

const Gtk::Window::ModalPropertyType Gtk::Window::modal_property("modal");

const Gtk::Window::WindowPositionPropertyType Gtk::Window::window_position_property("window_position");

const Gtk::Window::DefaultWidthPropertyType Gtk::Window::default_width_property("default_width");

const Gtk::Window::DefaultHeightPropertyType Gtk::Window::default_height_property("default_height");

const Gtk::Window::DestroyWithParentPropertyType Gtk::Window::destroy_with_parent_property("destroy_with_parent");

const Gtk::Window::IconPropertyType Gtk::Window::icon_property("icon");

const Gtk::Window::ScreenPropertyType Gtk::Window::screen_property("screen");

const Gtk::Window::IsActivePropertyType Gtk::Window::is_active_property("is_active");
	
const Gtk::Window::HasToplevelFocusPropertyType Gtk::Window::has_toplevel_focus_property("has_toplevel_focus");
	
const Gtk::Window::TypeHintPropertyType Gtk::Window::type_hint_property("type_hint");

const Gtk::Window::SkipTaskbarHintPropertyType Gtk::Window::skip_taskbar_hint_property("skip_taskbar_hint");

const Gtk::Window::SkipPagerHintPropertyType Gtk::Window::skip_pager_hint_property("skip_pager_hint");
	
/*  Gtk::Window signals
 */

const Gtk::Window::SetFocusSignalType Gtk::Window::set_focus_signal("set_focus");

const Gtk::Window::FrameEventSignalType Gtk::Window::frame_event_signal("frame_event");

const Gtk::Window::KeysChangedSignalType Gtk::Window::keys_changed_signal("keys_changed");

const Gtk::Window::DeleteEventSignalType Gtk::Window::delete_event_signal("delete_event");

/*  Gtk::WindowGroup
 */
 
Gtk::WindowGroup::WindowGroup(GtkWindowGroup *group, bool reference)
: G::Object((GObject*)group, reference)
{
}
	
Gtk::WindowGroup::WindowGroup()
: G::Object((GObject*)WindowGroupClass::create())
{
}

Gtk::WindowGroup::~WindowGroup() 
{
}

GtkWindowGroupClass* 
Gtk::WindowGroup::gtk_window_group_class() const 
{ 
	return get_class<GtkWindowGroupClass>(); 
}
	
Gtk::WindowGroup::operator GtkWindowGroup* () const
{ 
	return this ? gtk_window_group() : 0; 
}
	
void
Gtk::WindowGroup::add_window(Window& window)
{
	gtk_window_group_add_window(gtk_window_group(), window.gtk_window());
}

void
Gtk::WindowGroup::remove_window(Window& window)
{
	gtk_window_group_remove_window(gtk_window_group(), window.gtk_window());
}

/*  Gtk::WindowGroupClass
 */

void
Gtk::WindowGroupClass::init(GtkWindowGroupClass *g_class)
{
	G::ObjectClass::init((GObjectClass*)g_class);
}

GType
Gtk::WindowGroupClass::get_type()
{
	static GType type = 0;
	if (!type)
	{
		type = G::TypeInstance::register_type(GTK_TYPE_WINDOW_GROUP, (GClassInitFunc)init);
	}
	return type;
}

void*
Gtk::WindowGroupClass::create()
{
	return g_object_new(get_type(), 0);
}

