/***************************************************************************
 $RCSfile: accountimpl.h,v $
 -------------------
 cvs         : $Id: accountimpl.h,v 1.12 2003/07/02 23:07:05 aquamaniac Exp $
 begin       : Thu Nov 29 2001
 copyright   : (C) 2001 by Martin Preuss
 email       : openhbci@aquamaniac.de


 ***************************************************************************
 *                                                                         *
 *   This library is free software; you can redistribute it and/or         *
 *   modify it under the terms of the GNU Lesser General Public            *
 *   License as published by the Free Software Foundation; either          *
 *   version 2.1 of the License, or (at your option) any later version.    *
 *                                                                         *
 *   This library is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   Lesser General Public License for more details.                       *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the Free Software   *
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston,                 *
 *   MA  02111-1307  USA                                                   *
 *                                                                         *
 ***************************************************************************/


#ifndef HBCIACCOUNTIMPL_H
#define HBCIACCOUNTIMPL_H

namespace HBCI {
  class AccountImpl;
}

#include <list>
#include <string>

#include <openhbci/pointer.h>
#include <openhbci/accountparams.h>
#include <openhbci/account.h>
#include <openhbci/bankimpl.h>
#include <openhbci/standingorder.h>


namespace HBCI {

/** @short Implementation of the abstract base class Account.
 *
 * This class should only be visible inside OpenHBCI -- all external
 * applications are only supposed to use the interface class
 * Acccount.
 *
 * @author Martin Preuss<openhbci@aquamaniac.de> */
class DLLIMPORT AccountImpl: public Account, public accountParams {
    friend class Loader;
private:
    //Pointer<Bank> _bank;
    Pointer<BankImpl> _bankImpl;

    bool _managed;
    AccountBalance _balance;
    list<Transaction> _transactions;
    list<StandingOrder> _standingOrders;
    list<Pointer<Customer> > _auth;
	Limit _limit;

public:
    /** Various Constructors */
    //AccountImpl(Pointer<Bank> b);
    AccountImpl(Pointer<BankImpl> b);
    AccountImpl(Pointer<Bank> b, const accountParams &p);
    AccountImpl(Pointer<BankImpl> b, const accountParams &p);
    AccountImpl(Pointer<Bank> b,
                const string &nr,
                const string &subid);
    AccountImpl(Pointer<BankImpl> b,
                const string &nr,
                const string &subid);
    ~AccountImpl();

    /** @short Copy constructor from accountParams */
    Account &operator=(const accountParams &p);

    Pointer<Bank> bank() const;
    /** @short Returns the pointer to the BankImpl object this
     account belongs to. */
    Pointer<BankImpl> bankImpl() const { return _bankImpl;};


    bool managed() const { return _managed;};
    void setManaged(bool m) { _managed=m;};

    const list<Pointer<Customer> > &authorizedCustomers() const;
    void addAuthorizedCustomer(Pointer<Customer> c);
    void removeAuthorizedCustomer(Pointer<Customer> c);
    bool isAuthorized(const Pointer<Customer> c) const;

    const string &accountId() const
    { return accountParams::accountNumber();};
    void setAccountId(const string &s)
    { accountParams::setAccountNumber(s);};

    /**
     * @short Returns the account ID suffix of this account.
     *
     * Returns the account ID suffix of this account. This new in HBCI
     * 2.2; this field is only set by a few institutes */
    const string &accountSuffix() const
	{ return accountParams::accountSuffix();};
    void setAccountSuffix(const string &s) 
	{ accountParams::setAccountSuffix(s);};

    int countryCode() const { return accountParams::countryCode();};
    void setCountryCode(int i) { accountParams::setCountryCode(i);};

    const string &instituteCode() const
	{ return accountParams::instituteCode();};
    void setInstituteCode(const string &s) 
	{ accountParams::setInstituteCode(s);};

    /**
     * @short Returns the name of the owner of this account. 
     * 
     * Unfortunately this has no standard relation to the <i>customers</i>
     * that are allowed to use this account. 
     */
    const string &name1() const { return accountParams::name1();};
    void setName1(const string &s) { accountParams::setName1(s);};

    /** @short Returns the second part of the name of the owner of
     * this account.  */
    const string &name2() const { return accountParams::name2();};
    void setName2(const string &s) { accountParams::setName2(s);};

    /** 
     * @short Returns the name of this account. 
     *
     * This is set by the bank.
     */
    const string &accountName() const 
	{ return accountParams::accountName();};
    void setAccountName(const string &s) 
	{ accountParams::setAccountName(s);};

    /** 
     * @short Returns the default currency of this account (DEM, EUR)
     */
    const string &currency() const { return accountParams::currency();};
    void setCurrency(const string &s) { accountParams::setCurrency(s);};

	/**
     * @short Returns the type of general limit this account has, if it has
     * one. */
	const Limit &limit();

 private:
    /** 
     * @short Returns the type of general limit this account has, if it has
     * one. 
     * <b>DEPRECATED</b> use limit() instead
	 *
     * Returns the type of general limit this account has, if it has
     * one. (see HBCI-Spec V.3 Chapter User Parameter Data / account
     * information)
     *
     * Possible values: E - Einzelauftragslimit, T - Tageslimit, W -
     * Wochenlimit, M - Monatslimit, Z - Zeitlimit.
     */
    char limitType() const { return accountParams::limitType();};
    void setLimitType(char c) { accountParams::setLimitType(c);};

    /** @short Returns the value of the general limit this account
     * has, if it has one. 
	 * <b>DEPRECATED</b> use limit() instead  */
    const Value &limitValue() const 
	{ return accountParams::limitValue();};
    void setLimitValue(const Value &v) 
	{ accountParams::setLimitValue(v);};

    /** @short Returns the number of days for rotating time limits if
     * it has one.
     * <b>DEPRECATED</b> use limit() instead
	 *
     * Returns the number of days for rotating time limits
     * (Zeitlimit), if this account has a time limit
     * (Zeitlimit). Otherwise this is zero.  */
    int limitDays() const { return accountParams::limitDays();};
    void setLimitDays(int i) { accountParams::setLimitDays(i);};
 public:

    /** @short A list of allowed jobs */
    const list<updJob> &allowedJobs() const
	{ return accountParams::allowedJobs();};
    /** @short Add a job to the list of allowed jobs. */
    void addJob(updJob u) { accountParams::addJob(u);};
	const updJob *updForJob(const string jobId) const;

    const AccountBalance &balance() const { return _balance;};
    /** @short Set the stored balance. */
    void setBalance(AccountBalance b) { _balance=b;};

    const list<Transaction> &transactions() const;
    void clearTransactions();

    /** @short Add a transaction to the stored list. */
    void addTransaction(const Transaction &t);

    /** @short Remove all entries equal to <var>Transaction</var> from
     *	the stored list. */
    void removeTransaction (const Transaction &t);

    /**
     * Searches for a transaction equal to the given one and returns it.
     */
    const Transaction *findTransaction (const Transaction &t) const;

    const list<StandingOrder> &standingOrders() const;
    void addStandingOrder(const StandingOrder &o);
    void clearStandingOrders();
};

} /* namespace HBCI */

#endif
