#ifndef cllite_CLLITE_H_INCLUDED
#define cllite_CLLITE_H_INCLUDED 1

////////////////////////////////////////////////////////////////////////
// cllite.h:
// Defines a simplified front-end for the class_loader interfaces,
// encapsulating the most-commonly-used classloading features.
// It only supports string-keyed classloaders, and does not
// support classloaders which serve shared instanced of objects.
//
// Author: stephan beal <stephan@s11n.net>
// License: Public Domain
////////////////////////////////////////////////////////////////////////

#include <string>

#if HAVE_CONFIG_H
#  include "config.h"
#endif

#include <PACKAGE_NAMESPACE/class_loader.h> // core classloader framework.
#include <PACKAGE_NAMESPACE/cl_debuggering_macros.h> // CERR
#include <PACKAGE_NAMESPACE/path_finder.h> // path_finder class

#define LITECERR if(PACKAGE_NAMESPACE::class_loader_debug_level() > 0) CERR
//#define LITECERR if(0) CERR

#if ! WIN32
#ifdef HAVE_LTDL // prefer libltdl, but if it's not available...
#  include <ltdl.h>
#else           // use libdl instead:
#  include <dlfcn.h>
#endif // HAVE_LTDL
#else // WIN32:
#  include <windows.h> // LoadModule(). All WIN32 code is 100% untested.
#endif // !WIN32


////////////////////////////////////////////////////////////////////////////////
// Shorthand forms of the usual class-to-loader registration macros,
// for use by client code.
// These are all documented at length in the lib manual and class_loader.h.
#define CLLITE_BASE(BaseType) \
	PACKAGE_NAMESPACE_CLASSLOADER_REGISTER1(BaseType)
#define CLLITE_SUBTYPE(BaseType,SubType) \
	PACKAGE_NAMESPACE_CLASSLOADER_REGISTER2(BaseType,SubType)
#define CLLITE_ABSTRACT(BaseType) \
	PACKAGE_NAMESPACE_CLASSLOADER_ABSTRACT_BASE(BaseType)
////////////////////////////////////////////////////////////////////////////////



/**
   cllite is an experimental 'lite' front-end to the class_loader framework.
   It's intention is to provide the mostly-used class_loader<> functionality
   under one roof, and simplify it's interface.

   As of the introduction of cllite, the former dll_loader class is
   deprecated in favour of the functions in this namespace. The brief
   reason is that class_loader and dll_loader are not 100%
   usage-compatible, often causing some additional work in client
   code. The functions in this namespace provide more flexible DLL
   support than dll_loader without having to implement a class_loader
   subtype to do it. This code also demonstrates how class_loader can
   transparently work with arbitrary client-side DLL-opening code (reminder:
   class_loader requires NO special symbols in a DLL, as conventional
   classloaders do).
*/
namespace cllite
{
        /**
           The lookup key type used by classloaders in cllite.  Note
           that non-string types do not generically make sense in this
           context because DLL-awareness only works for strings. Why? 
           Consider: how do we load a DLL using an arbitrary
           non-string lookup key without imposing some type of
           translation conventions on clients?
        */
        typedef std::string key_type;


        /**
           Internal marker class for use with phoenix<>.
         */
        struct sharing_context;


        /**
           Returns this library's class search-path. It is up to specific
           classloaders to use or ignore this path: it is a
           suggestion, and not a rule. Clients are free to modify
           the returned object.
        */
        PACKAGE_NAMESPACE::path_finder & class_path();

        /**
           A basic classname-to-DLL-name transformer.

           Intended to be used to transform namespaced/templated names
           into filename-friendly names.


        */
        struct classname_to_dllname
        {
                typedef std::string key_type;
                classname_to_dllname();

                /**
                   ns_xlate sets the namespace separator (::) replacement which
                   will happen when this functor is activated. 


                   e.g., passing "/" will cause
                   foo::bar::MyType to translate to foo/bar/MyType
                */
                explicit classname_to_dllname( const key_type & ns_xlate );

                /**
                   General conventions:

                   Should return a DLL-friendly name of the given key,
                   minus any extension: a set of platform-specific
                   extensions is sed by class_path(). See the
                   classname_to_dllname class for a sample
                   implementation.

                   This implementation transforms class names such
                   that:

                   - By default namespace separators (::)
                   are replaced by, "_", but this can be changed via the ctor

                   - Any template parameters are simply removed.

                   e.g.:

                   std::list<foo> = std_list

                   MyType = MyType

                   foo::bar::MyType = foo_bar_MyType


                */
                key_type operator()( const key_type & key ) const;

        private:
                std::string m_nssep; // namespace separator replacement
        };


        /**
           Tries to open a DLL named basename, which should be
           either an absolute path/filename or a partial filename,
           without path and extension parts.
           
           class_path() will be used to resolve the full file name.

           Returns the path to the DLL, if one was found and opened,
           or an empty string on error.

           Reminder: the classloader architecture is such that simply
           opening a DLL is enough to trigger any classloader
           registrations which might live in it.
        */
        std::string open_dll( const std::string & basename );

        /**
           Returns open_dll( trans(key) ).
        */
        template <typename NameTransformer>
        std::string open_dll( const std::string & key,
                              const NameTransformer & trans )
        {
                return open_dll( xlated, trans(key) );
        } // open_dll()


        /**
           Returns the same thing as the underlying dlerror()
           implementation. Just as dlerror() does, this
           function will return NULL if called twice in a
           row. That is, it's error code is only valid once,
           and then it will return NULL until another error
           happens.           
        */
        const char * dll_error();



        /**
           Returns the BaseType classloader, as selected by the
           classloader_selector&lt;BaseType&gt; class.

           Note that this object itself is not DLL-aware: use
           classload() to enable DLL lookups for missing classes.
        */
        template <typename BaseType>
        PACKAGE_NAMESPACE::class_loader<BaseType> &
        classloader()
        {
                return PACKAGE_NAMESPACE::phoenix<
                        PACKAGE_NAMESPACE::class_loader<BaseType>,
                        sharing_context >::instance();
        }

        /**
           Tries to load the class named 'key' from BaseType's classloader.
           If no such class is registered a DLL search is initiated to
           find the class, using a default classname-to-DLL-name translator.

           The caller owns the returned pointer, which may be 0
           (indicating the classloader could not find the DLL).
        */
        template <typename BaseType>
        BaseType * classload( const key_type & key )
        {
                PACKAGE_NAMESPACE::class_loader<BaseType> & cl =
                        classloader<BaseType>();
                BaseType * ret = cl.load( key );
                if( ret ) return ret;
                LITECERR << "cllite looking for a DLL for '"<<key<<"'\n";
                if( open_dll( classname_to_dllname()(key) ).empty() )
                {
                        return 0;
                }
                return cl.load( key ); // try again!
        }

        /**
           Similar to classloader(), but the object it returns serves
           shared instances of objects.
        */
        template <typename BaseType>
        PACKAGE_NAMESPACE::class_loader<BaseType,key_type,true> &
        classloader_shared()
        {
                return PACKAGE_NAMESPACE::phoenix<
                        PACKAGE_NAMESPACE::class_loader<BaseType,key_type,true>,
                        sharing_context >::instance();
        }


        /**
           Similar to classload(), but objects returned by this function
           are NOT owned by the caller: the classloader cleans them up
           when it is destroyed (post-main()).
        */
        template <typename BaseType>
        BaseType * classload_shared( const key_type & key )
        {
                typedef PACKAGE_NAMESPACE::class_loader<BaseType,key_type,true> SharedCL;
                SharedCL & cl = classloader_shared<BaseType>();
                BaseType * ret = cl.load( key );
                if( ret ) return ret;
                LITECERR << "cllite looking for a DLL for '"<<key<<"'\n";
                if( open_dll( classname_to_dllname()(key) ).empty() )
                {
                        return 0;
                }
                return cl.load( key ); // try again!
        }


        /**
           Registers key with a factory returning (BaseType *).
        */
        template <typename BaseType>
        void register_factory( const key_type & key, BaseType * (*factory)() )
        {
                classloader<BaseType>().register_factory( key, factory );
        }

        /**
           Registers a default factory for creating (SubType *) returned
           as (BaseType *).
         */
        template <typename BaseType, typename SubType>
        void register_factory( const key_type & key )
        {
                register_factory<BaseType>( key,
                                            PACKAGE_NAMESPACE::object_factory<BaseType,SubType>::new_instance
                                            );
        }
        
        /**
           Registers a default factory for BaseType.
        */
        template <typename BaseType>
        void register_base( const key_type & key )
        {
                register_factory<BaseType,BaseType>( key );
        }

        /**
           Registers BaseType as an abstract base class. Use this for
           abstract base types, so that they get a working factory
           (i.e., one which always returns 0, since 'new BaseType'
           will not work for abstract types).
        */
        template <typename BaseType>
        void register_abstract_base( const key_type & key )
        {
                register_factory( key,
                                  PACKAGE_NAMESPACE::object_factory<BaseType>::no_instance
                                  );
        }

        /**
           Aliases thealias to be functionally equivalent to
           isthesameas for BaseType's classloader.

           This feature is useful for, e.g., assigning "friendly" or
           configurable names for a given application element. e.g.,
           consider the option of registering "DefaultDocumentClass"
           as an alias for some subtype of Document. This allows
           down-stream code to use symbolic names for loading
           application-specific types, as opposed to having to know
           the types real names.
        */
        template <typename BaseType>
        void alias( const key_type & thealias, const key_type & isthesameas )
        {
                classloader<BaseType>().alias( thealias, isthesameas );
        }


}

#undef LITECERR
#endif // cllite_CLLITE_H_INCLUDED
