////////////////////////////////////////////////////////////////////////
// serializers.cpp
// Author: stephan@s11n.net
// License: Public Domain
//
// Holds some impl code for various Serializers.
////////////////////////////////////////////////////////////////////////

#include "data_node_format.h" // core stuff
#include <S11N_NS/debuggering_macros.h> // CERR
#include <S11N_NS/class_name.h> // class_name helper
#include <S11N_NS/s11n_core.h> // classloader()

#include "serializers.h" // common Serializers

namespace S11N_NS {
        namespace io {

                typedef std::map<std::string,std::string> entity_translation_map;

                /**
                   Internal-use initializer for setting up a Serializer
                   translation map.
                */
                struct funtxt_serializer_translations_initializer
                {
                        void operator()( entity_translation_map & map )
                        {
                                // the order of these escapes is signifant.
                                // We only do double-backslashes to accomodate
                                // the case that the final char in a property
                                // is a backslash (yes, this has happened, and
                                // it hosed the input).
                                map["\\"] = "\\\\";
                                map["\n"] = "\\\n";
                                map["{"] = "\\{";
                                map["}"] = "\\}";
                        }
                };

                entity_translation_map &
                funtxt_serializer_translations()
                {
                        typedef phoenix<entity_translation_map,
                                sharing::funtxt_sharing_context,
                                funtxt_serializer_translations_initializer
                                > TMap;
                        return TMap::instance();
                }

                /**
                   Internal-use initializer for setting up the funtxt
                   Serializer translation map.
                */
                struct funxml_serializer_translations_initializer
                {
                        template <typename MapType>
                        void operator()( MapType & map )
                        {
                                map["&"] = "&amp;";
                                map["\""] = "&quot;";
                                map["'"] = "&apos;";
                                map[">"] = "&gt;";
                                map["<"] = "&lt;";
                        }
                };


                entity_translation_map &
                funxml_serializer_translations()
                {
                        typedef phoenix<
                                entity_translation_map,
                                sharing::funxml_sharing_context,
                                funxml_serializer_translations_initializer
                                > TheMap;
                        return TheMap::instance();
                }


                /**
                   Internal-use initializer for setting up a Serializer
                   translation map.
                */
                struct simplexml_serializer_translations_initializer
                {
                        void operator()( entity_translation_map & map )
                        {
                                // just the basics are covered here...
                                map["&"] = "&amp;";
                                map["\n"] = "&#010;";
                                map["\""] = "&quot;";
                                map["'"] = "&apos;";
                                map[">"] = "&gt;";
                                map["<"] = "&lt;";
                        }
                };

                entity_translation_map &
                simplexml_serializer_translations()
                {
                        typedef phoenix<entity_translation_map,
                                sharing::simplexml_sharing_context,
                                simplexml_serializer_translations_initializer
                                > TMap;
                        return TMap::instance();
                }


                /**
                   Internal-use initializer for setting up a Serializer
                   translation map.
                */
                struct paren_serializer_translations_initializer
                {
                        template <typename MapT>
                        void operator()( MapT & map )
                        {
                                // The order of these escapes is signifant.
                                // We only do double-backslashes to accomodate
                                // the case that the final chars in a property
                                // is a backslash (yes, this has happened, and
                                // it hosed the data when sending it via a
                                // format which uses \\\n to support multi-line
                                // values).
                                map["\\"] = "\\\\";
                                map[")"] = "\\)";
                                map["("] = "\\(";
                                // It is not strictly necessary to escape \(,
                                // but we do so because Parens is intended to
                                // be easy for hand-editing, and not escaping
                                // them confuses emacs when we have escaped
                                // closing parens. :)
                        }
                };

                entity_translation_map &
                parens_serializer_translations()
                {
                        typedef phoenix<
                                entity_translation_map,
                                sharing::parens_sharing_context,
                                paren_serializer_translations_initializer
                                > TheMap;
                        return TheMap::instance();
                }

                /**
                   This function exists to force instantiations of all
                   "known" Serializers.  We do this because if they
                   are registered via client-included header files
                   they add a huge amount of time to compiles.
                   Registrations for all Serializers for one NodeType
                   add about 23 seconds on my 2GHz PC!

                   This is so horrible. :/
                */
                void serializer_registration_kludge()
                {
                        compact_serializer<s11n::data_node> compact1;
                        funtxt_serializer<s11n::data_node> funtxt1;
                        funxml_serializer<s11n::data_node> funxml1;
                        parens_serializer<s11n::data_node> parens1;
                        simplexml_serializer<s11n::data_node> simplexml1;
                }

                int bogus_serializer_registration_placeholder =
                        ( serializer_registration_kludge(), 0 );



        } // namespace io
} // namespace S11N_NS



////////////////////////////////////////////////////////////////////////
// Register Serializers...
////////////////////////////////////////////////////////////////////////
#define S11N_DEFAULT_BASE_SERIALIZER s11n::io::data_node_serializer<s11n::data_node>

#define SERIALIZER_TYPE S11N_DEFAULT_BASE_SERIALIZER
#define SERIALIZER_BASE S11N_DEFAULT_BASE_SERIALIZER
#define SERIALIZER_ABSTRACT 1
#define SERIALIZER_NAME "s11n::io::data_node_serializer<s11n::data_node>"
#include <s11n/reg_serializer.h>


#define SERIALIZER_BASE S11N_DEFAULT_BASE_SERIALIZER
#define SERIALIZER_TYPE s11n::io::compact_serializer<s11n::data_node >
#define SERIALIZER_NAME "s11n::io::compact_serializer<s11n::data_node>"
#define SERIALIZER_ALIAS "compact"
#define SERIALIZER_MAGIC_COOKIE MAGIC_COOKIE_COMPACT
#include <s11n/reg_serializer.h>

#define SERIALIZER_BASE S11N_DEFAULT_BASE_SERIALIZER
#define SERIALIZER_TYPE s11n::io::funtxt_serializer< s11n::data_node >
#define SERIALIZER_NAME "s11n::io::funtxt_serializer<s11n::data_node>"
#define SERIALIZER_ALIAS "funtxt"
#define SERIALIZER_MAGIC_COOKIE MAGIC_COOKIE_FUNTXT
#include <s11n/reg_serializer.h>

#define SERIALIZER_BASE S11N_DEFAULT_BASE_SERIALIZER
#define SERIALIZER_TYPE s11n::io::funxml_serializer< s11n::data_node >
#define SERIALIZER_NAME "s11n::io::funxml_serializer<s11n::data_node>"
#define SERIALIZER_ALIAS "funxml"
#define SERIALIZER_MAGIC_COOKIE MAGIC_COOKIE_FUNXML
#include <s11n/reg_serializer.h>

#define SERIALIZER_BASE S11N_DEFAULT_BASE_SERIALIZER
#define SERIALIZER_TYPE s11n::io::parens_serializer< s11n::data_node >
#define SERIALIZER_NAME "s11n::io::parens_serializer<s11n::data_node>"
#define SERIALIZER_ALIAS "parens"
#define SERIALIZER_MAGIC_COOKIE MAGIC_COOKIE_PARENS
#include <s11n/reg_serializer.h>

#define SERIALIZER_BASE S11N_DEFAULT_BASE_SERIALIZER
#define SERIALIZER_TYPE s11n::io::simplexml_serializer< s11n::data_node >
#define SERIALIZER_NAME "s11n::io::simplexml_serializer<s11n::data_node>"
#define SERIALIZER_ALIAS "simplexml"
#define SERIALIZER_MAGIC_COOKIE MAGIC_COOKIE_SIMPLEXML
#include <s11n/reg_serializer.h>

#undef S11N_DEFAULT_BASE_SERIALIZER

