/*----------------------------------------------------------------------------

   libtunepimp -- The MusicBrainz tagging library.  
                  Let a thousand taggers bloom!
   
   Copyright (C) Robert Kaye 2003
   
   This file is part of libtunepimp.

   libtunepimp is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   libtunepimp is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with libtunepimp; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

   $Id: flac_trm.cpp,v 1.5 2004/03/09 23:27:55 robert Exp $

----------------------------------------------------------------------------*/
#include "../config.h"
#ifdef HAVE_FLAC

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "tunepimp.h"
#include "trm.h" 
#include "flac_trm.h"
#include <musicbrainz/mb_c.h>
#include <FLAC/file_decoder.h>

typedef struct { 
  trm_t         trm;
  unsigned long duration;
  bool          finished;
} clientdata_t;

void FLAC_errorcb(const FLAC__FileDecoder *decoder,
		  FLAC__StreamDecoderErrorStatus status,
		  void *client_data) {
}

void FLAC_metadatacb(const FLAC__FileDecoder *decoder,
		     const FLAC__StreamMetadata *metadata,
		     void *client_data) 
{
  if (metadata->type != FLAC__METADATA_TYPE_STREAMINFO)
    return;

  clientdata_t *cd = reinterpret_cast<clientdata_t *>(client_data);

  const FLAC__StreamMetadata_StreamInfo *si = &metadata->data.stream_info;
  trm_SetPCMDataInfo(cd->trm, si->sample_rate, si->channels, si->bits_per_sample);
  cd->duration = si->total_samples / si->sample_rate;
  trm_SetSongLength(cd->trm, cd->duration);
}

FLAC__StreamDecoderWriteStatus FLAC_writecb(const FLAC__FileDecoder *decoder,
					    const FLAC__Frame *frame,
					    const FLAC__int32 * const buffer[],
					    void *client_data)
{

  // No idea how TRM signatures are generated for more than two 
  // channels, so not going to try
  if (frame->header.channels > 2)
    return FLAC__STREAM_DECODER_WRITE_STATUS_ABORT;

  // TRM code doesn't support anything other than 8 and 16 bit samples
  if (frame->header.bits_per_sample != 8 && 
      frame->header.bits_per_sample != 16 &&
      frame->header.bits_per_sample != 24)
    return FLAC__STREAM_DECODER_WRITE_STATUS_ABORT;

  int samplesize = frame->header.bits_per_sample / 8;  
  int bufsize = samplesize * frame->header.channels * frame->header.blocksize;
  char *newdata = new char[bufsize];
  
  char *outputptr = newdata;
  for (unsigned samplenum = 0; samplenum < frame->header.blocksize; samplenum++) {
    for (unsigned channum = 0; channum < frame->header.channels; channum++) {
      memcpy(outputptr, &buffer[channum][samplenum], samplesize);
      outputptr += samplesize;
    }
  }
  clientdata_t *cd = reinterpret_cast<clientdata_t *>(client_data);

  bool done = trm_GenerateSignature(cd->trm, newdata, bufsize);
  delete [] newdata;

  if (done) {
    cd->finished = true;
    return FLAC__STREAM_DECODER_WRITE_STATUS_ABORT;
  }
    
  return FLAC__STREAM_DECODER_WRITE_STATUS_CONTINUE;
}

TRMResult TRMGeneratorFLAC::generate(const string &fileName, string &trmsig, 
				     unsigned long &durationArg)
{
  string         proxyServer;
  short          proxyPort;

  FLAC__FileDecoder *decoder = FLAC__file_decoder_new();
  if (!decoder)
    return eDecodeError;
  
  if (!FLAC__file_decoder_set_filename(decoder, fileName.c_str()))
    return eDecodeError;
  
  if (!FLAC__file_decoder_set_write_callback(decoder, &FLAC_writecb))  
    return eDecodeError;
  
  if (!FLAC__file_decoder_set_metadata_callback(decoder, &FLAC_metadatacb))  
    return eDecodeError;
  
  if (!FLAC__file_decoder_set_error_callback(decoder, &FLAC_errorcb))  
    return eDecodeError;
  
  clientdata_t cd;
  cd.trm = trm_New();
  cd.finished = false;
  
  tunepimp->getProxy(proxyServer, proxyPort);
  if (proxyServer.size() > 0 && proxyPort != 0)
    trm_SetProxy(cd.trm, (char *)proxyServer.c_str(), proxyPort);
  
  if (!FLAC__file_decoder_set_client_data(decoder, &cd)) {
    trm_Delete(cd.trm);
    return eDecodeError;
  }
  
  FLAC__FileDecoderState state;
  state = FLAC__file_decoder_init(decoder);
  if (state == FLAC__FILE_DECODER_ERROR_OPENING_FILE) {
    trm_Delete(cd.trm);    
    return eFileNotFound;
  }
  if (state != FLAC__FILE_DECODER_OK) {
    trm_Delete(cd.trm);
    return eDecodeError;
  }
  
  FLAC__file_decoder_process_until_end_of_file(decoder);
  state = FLAC__file_decoder_get_state(decoder);
  if (!cd.finished && state != FLAC__FILE_DECODER_OK) {
    trm_Delete(cd.trm);
    return eDecodeError;
  }
    
  if (!FLAC__file_decoder_finish(decoder)) {
    trm_Delete(cd.trm);
    return eDecodeError;
  }

  FLAC__file_decoder_delete(decoder);

  char             sig[17];
  if (trm_FinalizeSignature(cd.trm, sig, NULL) != 0) {
    trm_Delete(cd.trm);
    return eCannotConnect;
  }

  char ascii_sig[37];
  trm_ConvertSigToASCII (cd.trm, sig, ascii_sig);
  trmsig = string(ascii_sig);

  durationArg = cd.duration * 1000;
  
  trm_Delete(cd.trm);
  return eOk;
}

#endif // HAVE_FLAC
