/*!***************************************************************************

  module      : LVCSim_KernelInterface.hpp

  -------------------------------------------------------------------------

  responsible : IvanS

  special area: liveCache Simulator
  description : session free kernel interface implementation for simulator

  -------------------------------------------------------------------------





    ========== licence begin  GPL
    Copyright (c) 2000-2004 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end



*****************************************************************************/


#ifndef LVCSIM_KERNELINTERFACE_HPP
#define LVCSIM_KERNELINTERFACE_HPP

#include "liveCache/LVC_KernelInterface.hpp"
#include "liveCache/LVC_LockRequest.hpp"
#include "SAPDBCommon/MemoryManagement/SAPDBMem_DefaultRawAllocator.hpp"
#include "liveCache/LVC_RWLockManager.hpp"

class OmsStackTraceCallbackInterface;

/*!*****************************************************************************

   class:        LVCSim_KernelInterface
   description:  This class defines the methods, that can be called from the LibOms.
                 An Instance of this interface will made known to the LibOms
                 directly after the lib has been loaded.

*******************************************************************************/

class LVCSim_KernelInterface : public LVC_KernelInterface 
{
private:
	static tsp00_Int4	startupClock;	// save startup time here for Clock()

public:

	LVCSim_KernelInterface();

	/*!-------------------------------------------------------------------------
	function:	  Allocate
	description:  Allocates contiguous memory for 'count' bytes from LVCMem_Allocator.

	arguments:	  count [in] number of requested bytes

	return value: pointer to memory chunk
	--------------------------------------------------------------------------*/

	virtual void* Allocate(SAPDB_ULong count);

	/*!-------------------------------------------------------------------------
	function:	  Clock
	description:  returns the number of seconds since system start

	arguments:	  none
	return value: number of seconds
	--------------------------------------------------------------------------*/

	virtual tsp00_Int4 Clock();

	/*!-------------------------------------------------------------------------
	function:	  Deallocate
	description:  Deallocates contiguous memory identified by p from LVCMem_Allocator.

	arguments:	  p [in] memory to be deallocated

	return value: none
	--------------------------------------------------------------------------*/

	virtual void Deallocate(void* p);
  /*!-------------------------------------------------------------------------
	function:     DeregisterAllocator
	description:  deregisters an allocator in the kernel allocator registry.  
		   
	arguments:    allocatorInfo [in] a structure containing allocator info.
	return value: none
   --------------------------------------------------------------------------*/

    virtual void DeregisterAllocator(RTEMem_AllocatorInfo& allocatorInfo);

	/*!-------------------------------------------------------------------------
	function:	  GetOmsAllocator
	description:  returns a pointer to the global liveCache oms allocator 

	arguments:	  none
	return value: pointer to the liveCache oms allocator
	--------------------------------------------------------------------------*/

	virtual SAPDBMem_IRawAllocator* GetOmsAllocator();

	/*!-------------------------------------------------------------------------
	function:	  GetKnlVersion
	description:  fills a buffer with liveCache version information 

	arguments:	  pBuffer [out] address of the buffer to be filled
				  bufferLength [in] the size of the buffer
	return value: pointer to the buffer, if successful, NULL otherwise
	--------------------------------------------------------------------------*/

	virtual void* GetKnlVersion(void* pBuffer,int bufferLength);

	/*!-------------------------------------------------------------------------
	function:	  GetLockRequestTimeout
	description:  yields the value of the parameter REQUEST_TIMEOUT 

	arguments:	  none
	return value: the request timeout value
	--------------------------------------------------------------------------*/

	virtual tsp00_Int4 GetLockRequestTimeout();

	/*!-------------------------------------------------------------------------
	function:	  GetOmsRegionCnt
	description:  yields the value of the parameter _OMS_REGIONS 

	arguments:	  none
	return value: the value of oms regions
	--------------------------------------------------------------------------*/

	virtual int   GetOmsRegionCnt();

	/*!-------------------------------------------------------------------------
	function:	  GetPacketMaxUsableArea
	description:  yields the maximal usable area of the packet, i.e. the
				  number of bytes, that can be transferred via a packet
				  containing one data part.

	arguments:	  none
	return value: the number of bytes, that can be transferred via a packet
				  containing one data part.
	--------------------------------------------------------------------------*/

	virtual tsp00_Int4 GetPacketMaxUsableArea();

	/*!-------------------------------------------------------------------------
	function:	  GetSinkPtr
	description:  yields the address of the current kernel sink, i.e. the
				  kernel callback of the current session.

	arguments:	  none
	return value: the address of the kernel sink
	--------------------------------------------------------------------------*/

	virtual IliveCacheSink* GetSinkPtr();

	/*!-------------------------------------------------------------------------
	function:	  GetOmsVersionThreshold
	description:  yields the value of the parameter OMS_VERS_THRESHOLD 

	arguments:	  none
	return value: the oms version threshold value
	--------------------------------------------------------------------------*/

	virtual tsp00_Uint4 GetOmsVersionThreshold();

	/*!-------------------------------------------------------------------------
	function:	  HeapThresholdReached
	description:  returns true, if the heap threshold has been reached, i.e.
				  OMS_HEAP_THRESHOLD percent of the heap are in use 

	arguments:	  none
	return value: true, if threshold value has been reached, false otherwise
	--------------------------------------------------------------------------*/

	virtual bool  HeapThresholdReached();

    /*!-------------------------------------------------------------------------
    function:     OmsAllocate
    description:  Allocates contiguous memory for 'count' bytes from oms allocator.
  
    arguments:    count [in] number of requested bytes
    
    return value: pointer to memory chunk
    --------------------------------------------------------------------------*/

    virtual void* OmsAllocate(SAPDB_ULong count);

    /*!-------------------------------------------------------------------------
    function:     Deallocate
    description:  Deallocates contiguous memory identified by p from LVCMem_Allocator.
  
    arguments:    p [in] memory to be deallocated
    
    return value: none
    --------------------------------------------------------------------------*/
  
    virtual void OmsDeallocate(void* p);

    /*!-------------------------------------------------------------------------
    function:     IncOmsVersionUnloadCounter
    description:  increments the counter counting number of oms version unloads
   
    arguments:    none
     
    return value: none
    --------------------------------------------------------------------------*/
 
    virtual void IncOmsVersionUnloadCounter();

	/*!-----------------------------------------------------------------------
	 function:	   IsUnicodeInstance
	 description:  tells whether liveCache is runnung in UNICODE mode
	 arguments:    none
	 return value: true, if parameter _UNICODE='YES', otherwise false
	 -------------------------------------------------------------------------*/  

	virtual bool IsUnicodeInstance();

   /*!-------------------------------------------------------------------------
	function:     RegisterAllocator
	description:  registers an allocator in the kernel allocator registry.  
		   
	arguments:    allocatorInfo [in] a structure containing allocator info.
	return value: none
    --------------------------------------------------------------------------*/

    virtual void RegisterAllocator(RTEMem_AllocatorInfo& allocatorInfo);

	/*!-------------------------------------------------------------------------
	function:	  Signal
	description:  wakes up or timeouts a task waiting for an OMS Lock.	

	arguments:	  taskId [in] the task identification of the waiting task.
				  timeout[in] if true, waitung task is timed out
	return value: none
	--------------------------------------------------------------------------*/

	virtual void Signal(tsp00_TaskId taskId, bool timeout);

	/*!-------------------------------------------------------------------------
	function:	  SignalUnloadLiboms
	description:  the liboms calls this method, when it starts to become unloaded, i.e.
				  if it starts to destruct the globals. 

	arguments:	  none
	return value: none
	--------------------------------------------------------------------------*/

	virtual void SignalUnloadLiboms();

	/*!-------------------------------------------------------------------------
	function:	  TestBadAlloc
	description:  tests, if bad alloc count down has reached 0. (for test
				  purposes only (quickknl))

	arguments:	  none
	return value: none
	--------------------------------------------------------------------------*/

	virtual void TestBadAlloc();

	/*!-------------------------------------------------------------------------
	function:	  SimCtlAttachSink
	description:  attach a new sink to a handle in simulator to ease
				  writing tests

	arguments:	  handle [in]	OMS handle to attach new sink to
	return value: none
	--------------------------------------------------------------------------*/

	virtual void SimCtlAttachSink(class OmsHandle &handle);

	/*!-------------------------------------------------------------------------
	function:	  ConsoleMessage
	description:  print a console message, used in test programs in simulator

	arguments:	  msg [in]		message to print
	return value: none
	--------------------------------------------------------------------------*/

	virtual void ConsoleMessage(const char *msg);

	/*!-------------------------------------------------------------------------
	function:     AllocateArray
	description:  Allocates contiguous memory for 'count' bytes from LVCMem_Allocator
                  for use by array new[].

	arguments:  count [in] number of requested bytes
    
	return value: pointer to memory chunk
	--------------------------------------------------------------------------*/
  
	virtual void* AllocateArray(SAPDB_ULong count);

	/*!-------------------------------------------------------------------------
	function:     DeallocateArray
	description:  Deallocates contiguous memory identified by p from LVCMem_Allocator
                  for use by array delete[].

	arguments:  p [in] memory to be deallocated
    
	return value: none
	--------------------------------------------------------------------------*/

	virtual void DeallocateArray(void* p);

    static LVCSim_KernelInterface *Instance();

    virtual void DumpStackTrace();

    virtual unsigned int SaveStackTrace(void **buffer, unsigned int levels);

    virtual void EvalStackTrace(void **buffer, unsigned int levels,
      OmsStackTraceCallbackInterface *cb);

	static void DebugAllocatorErrorTrace(const char *errMsg, const void *blk);

  virtual SAPDB_UInt8 GetMicroTime();

  virtual bool StatFile(
    const char  *fn,
    SAPDB_UInt8 &ctime,
    SAPDB_UInt8 &fsize);

  virtual void FormatTime(
    char        *buffer,
    size_t      bufsize,
    SAPDB_UInt8 time);

  virtual short LockRequest(LVC_LockRequest &lockReq);

  virtual bool UseReaderWriterLocks();

protected:
  friend class LVCSim_liveCacheSink;

  static SAPDBMem_IRawAllocator *kernel_allocator;
  static LVCSim_KernelInterface *m_instance;

private:
  static SAPDBMem_IRawAllocator *getAllocator();

  /// Manager for R/W locks.
  LVC_RWLockManager m_rw_locks;
};


/*!*****************************************************************************

   endclass: LVCSim_KernelInterface

*******************************************************************************/

#endif