/*
 * medussa - a distributed cracking system
 * Copyright (C) 1999 Kostas Evangelinos <kos@bastard.net>
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

/*
 * $Id: bruteforce.c,v 1.2 2003/02/05 04:38:37 kos Exp $
 *
 * Bruteforce keyspace generator
 */

#include <stdio.h>
#include <string.h>
#include <math.h>
#include <stdlib.h>

#include "common.h"
#include "xmalloc.h"
#include "keyspace.h"
#include "generator.h"
#include "bruteforce.h"

#if 0
static void
npow(key_index_t x, int y, key_index_t z) {
  int i;
  
  for(keyspace_fromint(1, z),i=0; i<y; i++)
    keyspace_mul(x, x, z);
}
#endif

static void
nindex(key_index_t r, key_index_t x, int y) {
  int i;
  key_index_t t;

  keyspace_init(t);
  keyspace_fromint(0, r);

  for(i=1; i<=y; i++) {
    keyspace_set(t, x);
    keyspace_pow(t, i, t);
    keyspace_add(r, r, t);
  }
  keyspace_destroy(t);
}

static int
nroot(key_index_t index, key_index_t len, int min, int max) {
  int i;
  key_index_t lindex;

  keyspace_init(lindex);

  for(i=min; i<=max; i++) {
    keyspace_pow(lindex, i, len);    
    if(keyspace_cmp(lindex, index) > 0)
      goto out;
  }
  i = -1;
 out:
  
  keyspace_destroy(lindex);
  return i;
}

static int
bruteforce_uniq(bruteforce_t *k) {
  kchar *p;
  kchar *r;
  kchar newposs[BF_LINELEN];
  
  memset(newposs, '\0', BF_LINELEN);
  r = newposs;
  for(p=k->poss; *p; p++) {
    if(strchr(p+1, *p))
      continue;
    *r++ = *p;
  }

  strncpy(k->poss, newposs, BF_LINELEN);
  keyspace_fromint(strlen(k->poss), k->posslen);
  return 0;
}

key_index_t *
bruteforce_minindex(bruteforce_t *b) {
  return &b->minindex;
}

key_index_t *
bruteforce_maxindex(bruteforce_t *b) {
  return &b->maxindex;
}

key_index_t *
bruteforce_curindex(bruteforce_t *b) {
  return &b->index;
}

int
bruteforce_minlen(bruteforce_t *b) {
  return b->minlen;
}

int
bruteforce_maxlen(bruteforce_t *b) {
  return b->maxlen;
}

int
bruteforce_destroy(bruteforce_t *k) {
  keyspace_destroy(k->minindex);
  keyspace_destroy(k->maxindex);
  keyspace_destroy(k->index);
  keyspace_destroy(k->cur_min);
  keyspace_destroy(k->cur_max);
  keyspace_destroy(k->posslen);
  keyspace_destroy(k->kindex);
  keyspace_destroy(k->kmult);
  xfree(k);
  return 0;
}

bruteforce_t *
bruteforce_init(char *params) {
  bruteforce_t *k;
  key_index_t base;

  k = (bruteforce_t *)xcalloc(1, sizeof(bruteforce_t));

  keyspace_init(base);
  keyspace_init(k->posslen);
  keyspace_init(k->minindex);
  keyspace_init(k->maxindex);
  keyspace_init(k->index);
  keyspace_init(k->cur_min);
  keyspace_init(k->cur_max);
  keyspace_init(k->kindex);
  keyspace_init(k->kmult);

  param_get_int(params, "minlen", &k->minlen, BF_MINLEN);
  param_get_int(params, "maxlen", &k->maxlen, BF_MAXLEN);
  param_get_str(params, "poss", k->poss, BF_POSS);

  bruteforce_uniq(k);
  keyspace_fromint(strlen(k->poss), k->posslen);
    
  k->len = k->minlen;
  nindex(k->minindex, k->posslen, k->minlen-1);
  nindex(k->maxindex, k->posslen, k->maxlen);
  keyspace_set(k->index, k->minindex);  
  bruteforce_set(k, k->index);

  keyspace_destroy(base);
  return k;
}

/* Efficient next index operation */
int
bruteforce_fetch(bruteforce_t *k, kchar *buf, int userlen, int *len) {
  int i;

  if(keyspace_cmp(k->index, k->cur_max) == 0) {
    if(k->len == k->maxlen)
      return 1;
    keyspace_set(k->kindex, k->posslen);
    k->len++;    
    keyspace_set(k->cur_min, k->cur_max);
    nindex(k->cur_max, k->posslen, k->len);
    return bruteforce_fetch(k, buf, userlen, len);
  }

  keyspace_fromint(1, k->kmult);

  for(i=0; i<k->len && i<userlen; i++, keyspace_mul(k->kmult, k->kmult, k->posslen)) {
    keyspace_sub(k->kindex, k->index, k->cur_min);
    keyspace_div(k->kindex, k->kindex, k->kmult);    
    keyspace_mod(k->kindex, k->kindex, k->posslen);
    buf[i] = k->poss[keyspace_toint(k->kindex)];    
  }
  buf[i] = '\0';
  *len = k->len;

  keyspace_inc(k->index);
  return 0;
}

int 
bruteforce_done(bruteforce_t *k) {
  return (keyspace_cmp(k->index, k->maxindex) == 0);
}

int
bruteforce_set(bruteforce_t *k, key_index_t index) {

  if(keyspace_cmp(index, k->minindex) < 0 || keyspace_cmp(index, k->maxindex) > 0)
    return 1;

  k->len = nroot(index, k->posslen, k->minlen, k->maxlen);
  nindex(k->cur_max, k->posslen, k->len);
  nindex(k->cur_min, k->posslen, k->len-1);
  keyspace_set(k->index, index);
  return 0;
}

generator_impl_t bruteforce_impl = 
{
  "bruteforce",
    (gen_init)bruteforce_init,
    (gen_set)bruteforce_set,
    (gen_fetch)bruteforce_fetch,
    (gen_done)bruteforce_done,
    (gen_destroy)bruteforce_destroy,
    (gen_minindex)bruteforce_minindex,
    (gen_maxindex)bruteforce_maxindex,
    (gen_curindex)bruteforce_curindex,
    (gen_minlen)bruteforce_minlen,
    (gen_maxlen)bruteforce_maxlen
  };

#ifdef BRUTEFORCE_DEBUG
int
main(int argc, char **argv) {
  bruteforce_t *k;
  char buffer[1024];
  
  k = bruteforce_init(1, 5, "test01");

  while(!bruteforce_done(k)) {
    bruteforce_fetch(k, buffer, 1024);
    printf("%d ", k->len);
    printf("%s ", keyspace_ntoa(*bruteforce_curindex(k)));
    printf("%s ", keyspace_ntoa(*bruteforce_minindex(k)));
    printf("%s ", keyspace_ntoa(*bruteforce_maxindex(k)));
    printf("%s\n", buffer);
  }
  exit(0);
}

#endif
