/*
rfc822.c - MessageWall RFC822 parsing definitions
Copyright (C) 2002 Ian Gulliver

This program is free software; you can redistribute it and/or modify
it under the terms of version 2 of the GNU General Public License as
published by the Free Software Foundation.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <stdio.h>
#include <string.h>
#include <firestring.h>
#include "messagewall.h"
#include "mime.h"
#include "smtp.h"
#include "rfc822.h"

static const char tagstring[] = "$Id: rfc822.c,v 1.15 2002/07/12 17:45:03 ian Exp $";

int rfc822_split_message(struct firestring_estr_t *data, struct rfc822_message_t *message) {
	int i;
	
	i = firestring_estr_strstr(data,"\r\n\r\n",0);
	if (i == -1) {
		/*
		 * no message body
		 */
		message->header.a = message->header.l = data->l;
		message->header.s = data->s;
		message->body.a = message->body.l = 0;
		message->body.s = NULL;
		return 0;
	} else {
		/*
		 * message body
		 */
		message->header.a = message->header.l = i + 2;
		message->header.s = data->s;
		message->body.a = message->body.l = data->l - i - 4;
		message->body.s = &data->s[i+4];
		return 0;
	}
}

struct firestring_estr_t *rfc822_header_value(struct rfc822_message_t *message, char *header) {
	static struct firestring_estr_t ret;
	int i,j,l;
	int newline = 1;

	l = strlen(header);
	for (i = 0; i < message->header.l - l; i++) {
		if (newline == 1 && firestring_strncasecmp(&message->header.s[i],header,l) == 0) {
			/*
			 * got it
			 */
			i += l;
			while (i < message->header.l && strchr(RFC822_WHITESPACE,message->header.s[i++]) != NULL);
			i--;
			for (j = i; j <= message->header.l - 2; j++)
				if (memcmp(&message->header.s[j],"\r\n",2) == 0)
					if (j >= message->header.l - 3 || strchr(RFC822_WHITESPACE,message->header.s[j+2]) == NULL)
						break;
			ret.s = &message->header.s[i];
			ret.a = ret.l = j - i;
			return &ret;
		} else if (memcmp(&message->header.s[i],"\r\n",2) == 0) {
			/*
			 * new line
			 */
			newline = 1;
			i++;
		} else
			newline = 0;
	}
	return NULL;
}

struct firestring_estr_t *rfc822_eheader_value(struct rfc822_message_t *message, struct firestring_estr_t *header) {
	static struct firestring_estr_t ret;
	int i,j;
	int newline = 1;

	for (i = 0; i < message->header.l - header->l; i++) {
		if (newline == 1 && firestring_estr_estrncasecmp(&message->header,header,header->l,i) == 0) {
			/*
			 * got it
			 */
			i += header->l;
			while (i < message->header.l && strchr(RFC822_WHITESPACE,message->header.s[i++]) != NULL);
			i--;
			for (j = i; j <= message->header.l - 2; j++)
				if (memcmp(&message->header.s[j],"\r\n",2) == 0)
					if (j >= message->header.l - 3 || strchr(RFC822_WHITESPACE,message->header.s[j+2]) == NULL)
						break;
			ret.s = &message->header.s[i];
			ret.a = ret.l = j - i;
			return &ret;
		} else if (memcmp(&message->header.s[i],"\r\n",2) == 0) {
			/*
			 * new line
			 */
			newline = 1;
			i++;
		} else
			newline = 0;
	}
	return NULL;
}

struct firestring_estr_t *rfc822_parameter_value(struct firestring_estr_t *headervalue, char *parameter) {
	static struct firestring_estr_t ret;
	int i,j,l;

	l = strlen(parameter);

	/* 
	 * find the value/parameters delimiter
	 */
	i = firestring_estr_strchr(headervalue,';',0);
	if (i == -1)
		return NULL;

	while (1) {
		i = firestring_estr_stristr(headervalue,parameter,i+1);
		if (i == -1)
			return NULL;
		if (headervalue->s[i + l] != '=') {
			i++;
			continue;
		}
		break;
	}

	i += l + 1;
	/*
	 * now positioned at start of parameter value
	 */

	if (headervalue->s[i] == '"') {
		i++;
		j = i;
		while (j < headervalue->l &&
			headervalue->s[j] != '"')
			j++;
		if (headervalue->s[j] != '"')
			return NULL;
	} else {
		j = i;
		while (j < headervalue->l &&
			strchr(RFC822_WHITESPACE,headervalue->s[j]) == NULL)
			j++;
	}

	ret.s = &headervalue->s[i];
	ret.a = ret.l = j - i;
	return &ret;
}

int rfc822_to_cc_check(struct rfc822_message_t *message, struct firestring_estr_t *address) {
	struct firestring_estr_t *value;
	int i;

	value = rfc822_header_value(message,"To:");
	if (value == NULL)
		goto cc_check;
	i = firestring_estr_estristr(value,address,0);
	if (i >= 0)
		return 0;

cc_check:
	value = rfc822_header_value(message,"CC:");
	if (value == NULL)
		return 1;
	i = firestring_estr_estristr(value,address,0);
	if (i >= 0)
		return 0;

	return 1;
}

int rfc822_from_check(struct rfc822_message_t *message, struct firestring_estr_t *address) {
	struct firestring_estr_t *value;
	int i;

	value = rfc822_header_value(message,"From:");
	if (value == NULL)
		return 1;
	i = firestring_estr_estristr(value,address,0);
	if (i >= 0)
		return 0;

	return 1;
}

int rfc822_realname_check(struct rfc822_message_t *message) {
	struct firestring_estr_t *value;
	int i,j;

	value = rfc822_header_value(message,"From:");
	if (value == NULL)
		return 1;
	i = firestring_estr_strchr(value,'<',0);
	if (i == -1)
		return 1;
	for (j = 0; j < i; j++) {
		if (strchr(RFC822_WHITESPACE,value->s[j]) != NULL)
			/*
			 * non-whitespace character
			 */
			return 0;
	}

	/*
	 * there was no real name before <
	 */
	return 1;
}

int rfc822_header_check(struct rfc822_message_t *message, struct firestring_estr_t *header, struct firestring_estr_t *content) {
	struct firestring_estr_t *value;
	int i;

	value = rfc822_eheader_value(message,header);
	if (value == NULL)
		/*
		 * header isn't there, that's fine
		 */
		return 0;

	i = firestring_estr_estrstr(value,content,0);
	if (i == -1)
		/*
		 * doesn't contain content, that's fine
		 */
		return 0;

	return 1;
}

int rfc822_header_checki(struct rfc822_message_t *message, struct firestring_estr_t *header, struct firestring_estr_t *content) {
	struct firestring_estr_t *value;
	int i;

	value = rfc822_eheader_value(message,header);
	if (value == NULL)
		/*
		 * header isn't there, that's fine
		 */
		return 0;

	i = firestring_estr_estristr(value,content,0);
	if (i == -1)
		/*
		 * doesn't contain content, that's fine
		 */
		return 0;

	return 1;
}

int rfc822_header_reject_check(int client) {
	struct messagewall_header_reject_t *header_reject;

	header_reject = clients[client].profile->header_reject;
	while (header_reject != NULL) {
		if (rfc822_header_check(&clients[client].parts[0].message,&header_reject->header,&header_reject->content) != 0)
			if (smtp_reject(client,"RFC822","message failed header check for %e%e",SMTP_HEADER,header_reject->score,0,&header_reject->header,&header_reject->content) == 1)
				return 1;
		header_reject = header_reject->next;
	}

	return 0;
}

int rfc822_header_rejecti_check(int client) {
	struct messagewall_header_reject_t *header_reject;

	header_reject = clients[client].profile->header_rejecti;
	while (header_reject != NULL) {
		if (rfc822_header_checki(&clients[client].parts[0].message,&header_reject->header,&header_reject->content) != 0)
			if (smtp_reject(client,"RFC822","message failed header check for %e%e",SMTP_HEADER,header_reject->score,0,&header_reject->header,&header_reject->content) == 1)
				return 1;
		header_reject = header_reject->next;
	}

	return 0;
}

int rfc822_body_reject_check(int client, struct firestring_estr_t *part) {
	struct messagewall_estr_score_ll_t *estr_ll;

	estr_ll = clients[client].profile->body_reject;
	while (estr_ll != NULL) {
		if (firestring_estr_estrstr(part,&estr_ll->string,0) >= 0) {
			if (smtp_reject(client,"MIME","body text contains '%e'",SMTP_BADBODY,estr_ll->score,0,&estr_ll->string,NULL) == 1)
				return 1;
		}
		estr_ll = estr_ll->next;
	}

	return 0;
}

int rfc822_body_rejecti_check(int client, struct firestring_estr_t *part) {
	struct messagewall_estr_score_ll_t *estr_ll;

	estr_ll = clients[client].profile->body_rejecti;
	while (estr_ll != NULL) {
		if (firestring_estr_estristr(part,&estr_ll->string,0) >= 0)
			if (smtp_reject(client,"MIME","body text contains '%e'",SMTP_BADBODY,estr_ll->score,0,&estr_ll->string,NULL) == 1)
				return 1;
		estr_ll = estr_ll->next;
	}

	return 0;
}
