/*
security.c - MessageWall security functions
Copyright (C) 2002 Ian Gulliver

This program is free software; you can redistribute it and/or modify
it under the terms of version 2 of the GNU General Public License as
published by the Free Software Foundation.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <stdio.h>
#include <unistd.h>
#include <pwd.h>
#include <grp.h>
#include <sys/types.h>
#include <firestring.h>
#include "security.h"

static const char tagstring[] = "$Id: security.c,v 1.10 2002/07/11 03:10:38 ian Exp $";

void security_context(struct firestring_conf_t *config, int context) {
	const char *user;
	const char *group;
	const char *root;
	struct passwd *p;
	struct group *g;
 
	/*
	 * load config file variables
	 */
	user = firestring_conf_find(config,context == SECURITY_MWALL ? "user" : "auth_user");
	if (user == NULL) {
		fprintf(stderr,"STARTUP/FATAL: 'user' or 'auth_user' not set in config");
		exit(100);
	}

	group = firestring_conf_find(config,context == SECURITY_MWALL ? "group" : "auth_group");
	if (group == NULL) {
		fprintf(stderr,"STARTUP/FATAL: 'group' or 'auth_group' not set in config");
		exit(100);
	}

	root = firestring_conf_find(config,context == SECURITY_MWALL ? "root" : "auth_root");
	if (root == NULL) {
		fprintf(stderr,"STARTUP/FATAL: 'root' or 'auth_root' not set in config");
		exit(100);
	}

	/*
	 * check config file variables
	 */
	p = getpwnam(user);
	if (p == NULL) {
		fprintf(stderr,"STARTUP/FATAL: 'user' is set to an invalid username: %s\n",user);
		exit(100);
	}

	g = getgrnam(group);
	if (g == NULL) {
		fprintf(stderr,"STARTUP/FATAL: 'group' is set to an invalid group name: %s\n",group);
		exit(100);
	}

	if (p->pw_uid == 0 || g->gr_gid == 0) {
		fprintf(stderr,"STARTUP/FATAL: messagewall will not run as user or group root\n");
		exit(100);
	}

	/*
	 * chdir before chroot based on its4 warning
	 */
	/*
	 * root directory should be root controlled, chdirs and chroots should not be issues
	 */
	if (chdir(root) != 0) { /* ITS4: ignore chdir */
		perror("chdir");
		exit(100);
	}

	/*
	 * enter the chroot
	 */
	if (chroot(root) != 0) { /* ITS4: ignore chroot */
		perror("chroot");
		exit(100);
	}

	/*
	 * in case our working dir is outside the chroot
	 */
	if (chdir("/") != 0) { /* ITS4: ignore chdir */
		perror("chdir");
		exit(100);
	}

	/*
	 * drop groups
	 */
	if (setgid(g->gr_gid) != 0) {
		perror("setgid");
		exit(100);
	}

	if (setgroups(0,NULL) != 0) {
		perror("setgroups");
		exit(100);
	}

	/*
	 * drop uid
	 */
	if (setuid(p->pw_uid) != 0) {
		perror("setuid");
		exit(100);
	}
}
