#include <memory>

#include <cppunit/TestFixture.h>
#include <cppunit/extensions/HelperMacros.h>

#include "SDLFrameRate.h"
#include "Configuration.h"
#include "SoundInterface.h"

#include "Tiles.h"
#include "AllObjects.h"
#include "GameControlBase.h"
#include "PlayGround.h"

#include "GameBridge.h"


//----------------------------------------------------------------------------
class MagnetTest : public CppUnit::TestFixture
{
    CPPUNIT_TEST_SUITE(MagnetTest);

    CPPUNIT_TEST(testMagneticField);

    CPPUNIT_TEST_SUITE_END();

public:

    //------------------------------------------------------------------------
    void setUp()
    {
        Configuration::getInstance()->searchDataDir();

        SDLFrameRate::setFrameRate(72);

        // @todo: Find a way to test the objects without using libMoaggGame.
        Tiles::init();
        SurfacesBase::init();

        GameInterface::setInstance(GameBridge::getInstance());
        GameControlBase::init("test", "test_magnets.xml");
    }

    //------------------------------------------------------------------------
    void tearDown() 
    {
        GameControlBase::destroy();

        SurfacesBase::destroy();
        Tiles::destroy();
    }


protected:

    //------------------------------------------------------------------------
    void testMagneticField()
    {
        SDL_Rect r; r.w = 16; r.h = 16;

        // The initial gravity field in x direction.
        int initialXGravity[6][6] = {
            {  0,  0,  0,  0,  0,  0 },
            {  0,  0,  0,  0,  0,  0 },
            {  0,  0,  0,  0,  0,  0 },
            {  0,  0,  0,  0,  0,  0 },
            {  0,  0,  0,  0,  0,  0 },
            {  0,  0,  0,  0,  0,  0 }
        };

        // The initial gravity field in y direction.
        int initialYGravity[6][6] = {
            { 10, 10, 10, 10, 10, 10 },
            { 10, 10, 10, 10, 10, 10 },
            { 10, 10, 10, 10, 10, 10 },
            { 10, 10, 10, 10, 10, 10 },
            { 10, 10, 10, 10, 10, 10 },
            { 10, 10, 10, 10, 10, 10 }
        };
        
        // The first magnet's gravity field in y direction after activation.
        int magnet1YGravity[6][6] = {
            {  0,  0,  0,  0,  0,  0 },
            {  0, 10, 10, 10, 10,  0 },
            {  0, 10, 10, 10, 10,  0 },
            {  0, 10, 10, 10, 10,  0 },
            {  0, 10, 10, 10, 10,  0 },
            {  0,  0,  0,  0,  0,  0 }
        };

        // The second magnet's gravity field in x direction after activation.
        int magnet2XGravity[6][6] = {
            {  0,  0,  0,  0,  0,  0 },
            {  0,  0, 15, 15, 15,  0 },
            {  0,  0, 15, 15, 15,  0 },
            {  0,  0, 15, 15, 15,  0 },
            {  0,  0, 15, 15, 15,  0 },
            {  0,  0,  0,  0,  0,  0 }
        };

        // The first magnet's gravity field in y direction after activation.
        int magnet3YGravity[6][6] = {
            {  0,  0,  0,  0,  0,  0 },
            {  0, -5, -5, -5, -5,  0 },
            {  0, -5, -5, -5, -5,  0 },
            {  0, -5, -5, -5, -5,  0 },
            {  0,  0,  0,  0,  0,  0 },
            {  0,  0,  0,  0,  0,  0 }
        };

        // The fourth magnet's gravity field in x direction after activation.
        int magnet4XGravity[6][6] = {
            {  0,  0,  0,  0,  0,  0 },
            {  0,-10,-10,-10,  0,  0 },
            {  0,-10,-10,-10,  0,  0 },
            {  0,-10,-10,-10,  0,  0 },
            {  0,-10,-10,-10,  0,  0 },
            {  0,  0,  0,  0,  0,  0 }
        };

        PlayGround *playGround = PlayGround::getInstance();

        // Check, that the initial gravity field is active.
        for (Uint16 y=0; y<6; y++)
        {
            for (Uint16 x=0; x<6; x++)
            {
                r.x = 16 * x;
                r.y = 16 * y;

                CPPUNIT_ASSERT_EQUAL(
                    initialXGravity[y][x], playGround->getXGravity(r));
                CPPUNIT_ASSERT_EQUAL(
                    initialYGravity[y][x], playGround->getYGravity(r));
            }
        }

        // The first update activates the magnets.
        playGround->update();

        // Now check the gravity field again.
        for (Uint16 y=0; y<6; y++)
        {
            for (Uint16 x=0; x<6; x++)
            {
                r.x = 16 * x;
                r.y = 16 * y;

                CPPUNIT_ASSERT_EQUAL(
                    initialXGravity[y][x]+
                    magnet2XGravity[y][x]+
                    magnet4XGravity[y][x],
                    playGround->getXGravity(r));
                CPPUNIT_ASSERT_EQUAL(
                    initialYGravity[y][x]+
                    magnet1YGravity[y][x]+
                    magnet3YGravity[y][x],
                    playGround->getYGravity(r));
            }
        }
        
        // After fps-1 update()'s, the magnets will be turned off again.
        for (unsigned i=1; i<SDLFrameRate::getFrameRate(); i++)
        {
            playGround->update();
        }

        // Now, only the initial gravity field remains.
        for (Uint16 y=0; y<6; y++)
        {
            for (Uint16 x=0; x<6; x++)
            {
                r.x = 16 * x;
                r.y = 16 * y;

                CPPUNIT_ASSERT_EQUAL(
                    initialXGravity[y][x], playGround->getXGravity(r));
                CPPUNIT_ASSERT_EQUAL(
                    initialYGravity[y][x], playGround->getYGravity(r));
            }
        }
    }
};



//----------------------------------------------------------------------------
class TurretTest : public CppUnit::TestFixture
{
    CPPUNIT_TEST_SUITE(TurretTest);

    CPPUNIT_TEST(testNoCollisionWithCreatedProjectiles);

    CPPUNIT_TEST_SUITE_END();

public:

    //------------------------------------------------------------------------
    void setUp()
    {
        Configuration::getInstance()->searchDataDir();

        SDLFrameRate::setFrameRate(72);

        // @todo: Find a way to test the objects without using libMoaggGame.
        Tiles::init();
        SurfacesBase::init();

        GameInterface::setInstance(GameBridge::getInstance());
    }

    //------------------------------------------------------------------------
    void tearDown() 
    {
        SurfacesBase::destroy();
        Tiles::destroy();
    }


protected:

    //------------------------------------------------------------------------
    void testNoCollisionWithCreatedProjectiles()
    {
        XMLNode turretNode("turret");
        turretNode.addProperty(new XMLProperty("orientation", ""));
        turretNode.addProperty(new XMLProperty("x", "0"));
        turretNode.addProperty(new XMLProperty("y", "0"));
        turretNode.addProperty(new XMLProperty("type", "tube"));
        turretNode.addProperty(new XMLProperty("delay", "10"));

        XMLNode barrelNode("barrel");
        barrelNode.addProperty(new XMLProperty("type", "fixed"));
        barrelNode.addProperty(new XMLProperty("angle", "0"));

        do
        {
            turretNode.getProperty("orientation")->setValue("top");
            TurretBase::InitializationData turretInit(&turretNode);
            TurretBarrel::InitializationData barrelInit(&barrelNode);

            TopTurret turret(turretInit);
            TurretBarrel barrel(barrelInit);

            TurretProjectile projectile(&turret, &barrel);

            CPPUNIT_ASSERT(!ObjectBase::isCollision(&turret, &projectile));
        }
        while (0);

        do
        {
            turretNode.getProperty("orientation")->setValue("bottom");
            TurretBase::InitializationData turretInit(&turretNode);
            TurretBarrel::InitializationData barrelInit(&barrelNode);

            BottomTurret turret(turretInit);
            TurretBarrel barrel(barrelInit);

            TurretProjectile projectile(&turret, &barrel);

            CPPUNIT_ASSERT(!ObjectBase::isCollision(&turret, &projectile));
        }
        while (0);

        do
        {
            turretNode.getProperty("orientation")->setValue("left");
            TurretBase::InitializationData turretInit(&turretNode);
            TurretBarrel::InitializationData barrelInit(&barrelNode);

            LeftTurret turret(turretInit);
            TurretBarrel barrel(barrelInit);

            TurretProjectile projectile(&turret, &barrel);

            CPPUNIT_ASSERT(!ObjectBase::isCollision(&turret, &projectile));
        }
        while (0);

        do
        {
            turretNode.getProperty("orientation")->setValue("right");
            TurretBase::InitializationData turretInit(&turretNode);
            TurretBarrel::InitializationData barrelInit(&barrelNode);

            RightTurret turret(turretInit);
            TurretBarrel barrel(barrelInit);

            TurretProjectile projectile(&turret, &barrel);

            CPPUNIT_ASSERT(!ObjectBase::isCollision(&turret, &projectile));
        }
        while (0);
    }
};



//----------------------------------------------------------------------------
class MortarTest : public CppUnit::TestFixture
{
    CPPUNIT_TEST_SUITE(MortarTest);

    CPPUNIT_TEST(testNoCollisionWithCreatedGrenades);

    CPPUNIT_TEST_SUITE_END();

public:

    //------------------------------------------------------------------------
    void setUp()
    {
        Configuration::getInstance()->searchDataDir();

        SDLFrameRate::setFrameRate(72);

        // @todo: Find a way to test the objects without using libMoaggGame.
        Tiles::init();
        SurfacesBase::init();

        GameInterface::setInstance(GameBridge::getInstance());
    }

    //------------------------------------------------------------------------
    void tearDown() 
    {
        SurfacesBase::destroy();
        Tiles::destroy();
    }


protected:

    //------------------------------------------------------------------------
    void testNoCollisionWithCreatedGrenades()
    {
        XMLNode mortarNode("mortar");
        mortarNode.addProperty(new XMLProperty("orientation", ""));
        mortarNode.addProperty(new XMLProperty("x", "0"));
        mortarNode.addProperty(new XMLProperty("y", "0"));
        mortarNode.addProperty(new XMLProperty("delay", "10"));

        XMLNode barrelNode("barrel");
        barrelNode.addProperty(new XMLProperty("type", "fixed"));
        barrelNode.addProperty(new XMLProperty("angle", "0"));
        barrelNode.addProperty(new XMLProperty("warhead", "none"));
        barrelNode.addProperty(new XMLProperty("delay", "10"));

        do
        {
            mortarNode.getProperty("orientation")->setValue("top");
            MortarBase::InitializationData mortarInit(&mortarNode);
            MortarBarrel::InitializationData barrelInit(&barrelNode);

            TopMortar mortar(mortarInit);
            MortarBarrel barrel(barrelInit);

            std::auto_ptr<Grenade> grenade(barrel.createGrenade(&mortar));

            CPPUNIT_ASSERT(!ObjectBase::isCollision(&mortar, grenade.get()));
        }
        while (0);

        do
        {
            mortarNode.getProperty("orientation")->setValue("bottom");
            MortarBase::InitializationData mortarInit(&mortarNode);
            MortarBarrel::InitializationData barrelInit(&barrelNode);

            BottomMortar mortar(mortarInit);
            MortarBarrel barrel(barrelInit);

            std::auto_ptr<Grenade> grenade(barrel.createGrenade(&mortar));

            CPPUNIT_ASSERT(!ObjectBase::isCollision(&mortar, grenade.get()));
        }
        while (0);

        do
        {
            mortarNode.getProperty("orientation")->setValue("left");
            MortarBase::InitializationData mortarInit(&mortarNode);
            MortarBarrel::InitializationData barrelInit(&barrelNode);

            LeftMortar mortar(mortarInit);
            MortarBarrel barrel(barrelInit);

            std::auto_ptr<Grenade> grenade(barrel.createGrenade(&mortar));

            CPPUNIT_ASSERT(!ObjectBase::isCollision(&mortar, grenade.get()));
        }
        while (0);

        do
        {
            mortarNode.getProperty("orientation")->setValue("right");
            MortarBase::InitializationData mortarInit(&mortarNode);
            MortarBarrel::InitializationData barrelInit(&barrelNode);

            RightMortar mortar(mortarInit);
            MortarBarrel barrel(barrelInit);

            std::auto_ptr<Grenade> grenade(barrel.createGrenade(&mortar));

            CPPUNIT_ASSERT(!ObjectBase::isCollision(&mortar, grenade.get()));
        }
        while (0);
    }
};



//----------------------------------------------------------------------------
class MissileTest : public CppUnit::TestFixture
{
    CPPUNIT_TEST_SUITE(MissileTest);

    CPPUNIT_TEST(testSmartHeadingStrategy);

    CPPUNIT_TEST_SUITE_END();

public:

    //------------------------------------------------------------------------
    void setUp()
    {
    }

    //------------------------------------------------------------------------
    void tearDown() 
    {
    }


protected:

    //------------------------------------------------------------------------
    void testSmartHeadingStrategy()
    {
        Missile::SmartHeadingStrategy *s =
            Missile::SmartHeadingStrategy::getInstance();

        // Tests without gravity.

        for (Sint16 i=0; i<360; i+=15)
        {
            CPPUNIT_ASSERT_EQUAL(
                (Sint16)i, s->calculateHeadingAngle(100, i, 0, 0));
        }

        // Tests with |a| > |g|.

        CPPUNIT_ASSERT_EQUAL(
            (Sint16)0, s->calculateHeadingAngle(200, 0, 0, 100));
        CPPUNIT_ASSERT_EQUAL(
            (Sint16)66, s->calculateHeadingAngle(200, 45, 0, 100));
        CPPUNIT_ASSERT_EQUAL(
            (Sint16)120, s->calculateHeadingAngle(200, 90, 0, 100));
        CPPUNIT_ASSERT_EQUAL(
            (Sint16)156, s->calculateHeadingAngle(200, 135, 0, 100));
        CPPUNIT_ASSERT_EQUAL(
            (Sint16)180, s->calculateHeadingAngle(200, 180, 0, 100));
        CPPUNIT_ASSERT_EQUAL(
            (Sint16)204, s->calculateHeadingAngle(200, 225, 0, 100));
        CPPUNIT_ASSERT_EQUAL(
            (Sint16)240, s->calculateHeadingAngle(200, 270, 0, 100));
        CPPUNIT_ASSERT_EQUAL(
            (Sint16)294, s->calculateHeadingAngle(200, 315, 0, 100));

        // Tests with |a| == |g|.

        CPPUNIT_ASSERT_EQUAL(
            (Sint16)0, s->calculateHeadingAngle(100, 0, 0, 100));
        CPPUNIT_ASSERT_EQUAL(
            (Sint16)90, s->calculateHeadingAngle(100, 45, 0, 100));
        CPPUNIT_ASSERT_EQUAL(
            (Sint16)180, s->calculateHeadingAngle(100, 90, 0, 100));
        CPPUNIT_ASSERT_EQUAL(
            (Sint16)180, s->calculateHeadingAngle(100, 135, 0, 100));
        CPPUNIT_ASSERT_EQUAL(
            (Sint16)180, s->calculateHeadingAngle(100, 180, 0, 100));
        CPPUNIT_ASSERT_EQUAL(
            (Sint16)180, s->calculateHeadingAngle(100, 225, 0, 100));
        CPPUNIT_ASSERT_EQUAL(
            (Sint16)180, s->calculateHeadingAngle(100, 270, 0, 100));
        CPPUNIT_ASSERT_EQUAL(
            (Sint16)270, s->calculateHeadingAngle(100, 315, 0, 100));

        CPPUNIT_ASSERT_EQUAL(
            (Sint16)270, s->calculateHeadingAngle(100, 0, 100, 0));
        CPPUNIT_ASSERT_EQUAL(
            (Sint16)0, s->calculateHeadingAngle(100, 45, 100, 0));
        CPPUNIT_ASSERT_EQUAL(
            (Sint16)90, s->calculateHeadingAngle(100, 90, 100, 0));
        CPPUNIT_ASSERT_EQUAL(
            (Sint16)180, s->calculateHeadingAngle(100, 135, 100, 0));
        CPPUNIT_ASSERT_EQUAL(
            (Sint16)270, s->calculateHeadingAngle(100, 180, 100, 0));
        CPPUNIT_ASSERT_EQUAL(
            (Sint16)270, s->calculateHeadingAngle(100, 225, 100, 0));
        CPPUNIT_ASSERT_EQUAL(
            (Sint16)270, s->calculateHeadingAngle(100, 270, 100, 0));
        CPPUNIT_ASSERT_EQUAL(
            (Sint16)270, s->calculateHeadingAngle(100, 315, 100, 0));

        // Tests with |a| < |g|.
        
        CPPUNIT_ASSERT_EQUAL(
            (Sint16)0, s->calculateHeadingAngle(100, 0, 0, 200));
        CPPUNIT_ASSERT_EQUAL(
            (Sint16)180, s->calculateHeadingAngle(100, 45, 0, 200));
        CPPUNIT_ASSERT_EQUAL(
            (Sint16)180, s->calculateHeadingAngle(100, 90, 0, 200));
        CPPUNIT_ASSERT_EQUAL(
            (Sint16)180, s->calculateHeadingAngle(100, 135, 0, 200));
        CPPUNIT_ASSERT_EQUAL(
            (Sint16)180, s->calculateHeadingAngle(100, 180, 0, 200));
        CPPUNIT_ASSERT_EQUAL(
            (Sint16)180, s->calculateHeadingAngle(100, 225, 0, 200));
        CPPUNIT_ASSERT_EQUAL(
            (Sint16)180, s->calculateHeadingAngle(100, 270, 0, 200));
        CPPUNIT_ASSERT_EQUAL(
            (Sint16)180, s->calculateHeadingAngle(100, 315, 0, 200));
    }
};


//----------------------------------------------------------------------------
class ShipTest : public CppUnit::TestFixture
{
    CPPUNIT_TEST_SUITE(ShipTest);

    CPPUNIT_TEST(testRotation);
    CPPUNIT_TEST(testThrust);
    CPPUNIT_TEST(testIsInLandingZone);

    CPPUNIT_TEST_SUITE_END();

public:

    //------------------------------------------------------------------------
    void setUp()
    {
        Configuration::getInstance()->searchDataDir();
        SoundInterface::setInstance(NullSoundInterface::getInstance());

        SDLFrameRate::setFrameRate(72);

        ShipSurfaces::init();

        m_ship = new Ship(ShipSurfaces::S_DEFAULT);

        GameInterface::setInstance(NullGameInterface::getInstance());
    }

    //------------------------------------------------------------------------
    void tearDown() 
    {
        ZAP_POINTER(m_ship);

        ShipSurfaces::destroy();
    }


protected:

    //------------------------------------------------------------------------
    void testRotation()
    {
        size_t fr = SDLFrameRate::getFrameRate();

        Uint16 updatesPerRotationStep =
            SURFACES_ROTATION_ANGLE * fr /
            m_ship->getRotationDegreesPerSecond();

        m_ship->setRotation(Ship::ROT_RIGHT);
        for (Uint16 j=1; j<=fr; j++)
        {
            static Uint16 desiredAngles[] = {
                174, 168, 162, 156, 150, 144, 138, 132, 126, 120, 114, 108,
                102,  96,  90,  84,  78,  72,  66,  60,  54,  48,  42,  36,
                 30,  24,  18,  12,   6,   0, 354, 348, 342, 336, 330, 324
            };

            m_ship->update();
            CPPUNIT_ASSERT_EQUAL(
                (Uint16)(desiredAngles[(j-1) / updatesPerRotationStep]),
                m_ship->getAngle());
        }

        m_ship->setRotation(Ship::ROT_NONE);
        for (Uint16 j=1; j<=fr; j++)
        {
            m_ship->update();
            CPPUNIT_ASSERT_EQUAL((Uint16)324, m_ship->getAngle());
        }

        m_ship->setRotation(Ship::ROT_LEFT);
        for (Uint16 j=1; j<=fr; j++)
        {
            static Uint16 desiredAngles[] = {
                324, 330, 336, 342, 348, 354,   0,   6,  12,  18,  24,  30,
                 36,  42,  48,  54,  60,  66,  72,  78,  84,  90,  96, 102,
                108, 114, 120, 126, 132, 138, 144, 150, 156, 162, 168, 174,
                180
            };

            m_ship->update();
            CPPUNIT_ASSERT_EQUAL(
                (Uint16)(desiredAngles[j / updatesPerRotationStep]),
                m_ship->getAngle());
        }
    }

    //------------------------------------------------------------------------
    void testThrust()
    {
        helperThrust();

        MATH_TOOLS::Vector finalFinePosition = m_ship->getFinePosition();
        MATH_TOOLS::Vector finalFineVelocity = m_ship->getFineVelocity();

        // Fly again with the double ship's mass.
        // Only the half velocity and position will be reached.

        m_ship->m_mass = 2*m_ship->getMass();

        helperThrust();

        CPPUNIT_ASSERT_EQUAL(
            finalFinePosition.getX()/2, m_ship->getFinePosition().getX());
        CPPUNIT_ASSERT_EQUAL(
            finalFineVelocity.getX()/2, m_ship->getFineVelocity().getX());

        // Double the ship's mass again.
        // Now, only the quarter velocity and position will be reached.

        m_ship->m_mass = 2*m_ship->getMass();

        helperThrust();

        CPPUNIT_ASSERT_EQUAL(
            finalFinePosition.getX()/4, m_ship->getFinePosition().getX());
        CPPUNIT_ASSERT_EQUAL(
            finalFineVelocity.getX()/4, m_ship->getFineVelocity().getX());
    }

    //------------------------------------------------------------------------
    void testIsInLandingZone()
    {
        m_ship->setPosition(0, 0);
        m_ship->getFineVelocity().set(0.0, 0.0);

        SDL_Rect landingZone = m_ship->getBottomPixels();
        Sint16 x = landingZone.x;
        Sint16 y = landingZone.y;
        Uint16 w = landingZone.w;

        // First, the landing zone's size is equal to the ship's bottom.
        // When sweeping, there's exactly one position,
        // where isInLandingZone() returns true.
        for (landingZone.y = y-4; landingZone.y <= y+4; landingZone.y++)
        {
            for (landingZone.x = x-48; landingZone.x <= x+48; landingZone.x++)
            {
                CPPUNIT_ASSERT_EQUAL(landingZone.x == x && landingZone.y == y,
                                     m_ship->isInLandingZone(landingZone));
            }
        }

        // Now let the landing zone be one pixel less
        // than the ship's bottom width.
        landingZone.w--;
        for (landingZone.y = y-4; landingZone.y <= y+4; landingZone.y++)
        {
            for (landingZone.x = x-48; landingZone.x <= x+48; landingZone.x++)
            {
                CPPUNIT_ASSERT(!m_ship->isInLandingZone(landingZone));
            }
        }

        // Now let the landing zone be four pixel wider
        // than the ship's bottom width.
        landingZone.w = w+4;
        for (landingZone.y = y-4; landingZone.y <= y+4; landingZone.y++)
        {
            for (landingZone.x = x-48; landingZone.x <= x+48; landingZone.x++)
            {
                CPPUNIT_ASSERT_EQUAL(
                    landingZone.x >= x-4 && landingZone.x <= x &&
                    landingZone.y == y,
                    m_ship->isInLandingZone(landingZone));
            }
        }
    }



    //------------------------------------------------------------------------
    void helperThrust()
    {
        unsigned fr = SDLFrameRate::getFrameRate();

        m_ship->setPosition(0, 0);
        m_ship->getFineVelocity().set(0.0, 0.0);

        // Head to the right (90).
        m_ship->m_fineAngle = 90.0;

        // Fly two second with thrust ...
        m_ship->setThrust(true);
        for (Uint16 j=0; j<2*fr; j++)
        {
            m_ship->update();
        }

        // Turn of the thrust and drift five seconds.
        m_ship->setThrust(false);
        for (Uint16 j=0; j<5*fr; j++)
        {
            m_ship->update();
        }

        // Head back to the left (90).
        m_ship->m_fineAngle = 90.0;

        // Then fly three seconds with thrust.
        m_ship->setThrust(true);
        for (Uint16 j=0; j<3*fr; j++)
        {
            m_ship->update();
        }
    }


    //------------------------------------------------------------------------
    Ship *m_ship;
};


//----------------------------------------------------------------------------
CPPUNIT_TEST_SUITE_NAMED_REGISTRATION(MagnetTest, "Magnet");
CPPUNIT_TEST_SUITE_NAMED_REGISTRATION(TurretTest, "Turret");
CPPUNIT_TEST_SUITE_NAMED_REGISTRATION(MortarTest, "Mortar");
CPPUNIT_TEST_SUITE_NAMED_REGISTRATION(MissileTest, "Missile");
CPPUNIT_TEST_SUITE_NAMED_REGISTRATION(ShipTest, "Ship");
