/*
 *	netcam.c
 *
 *	Copyright 2002 by Jeroen Vreeken (pe1rxq@amsat.org)
 *	Additional copyright 2005 Angel Carpintero and Christopher Price
 *	(only main contributors of this file listed).
 *	This software is distributed under the GNU Public License Version 2
 *	See also the file 'COPYING'.
 *
 */

#include "motion.h"

#ifdef __freebsd__
#include "video_freebsd.h"
#else
#include "video.h"
#endif /* __freebsd__ */

#include <netdb.h>
#include <netinet/in.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <jpeglib.h>
#include <setjmp.h>

/* for rotation */
#include "rotate.h"
#include "netcam_wget.h"

#define REQUEST_AUTH "GET %s HTTP/1.0\r\nAuthorization: Basic %s\r\n\r\n"
#define REQUEST "GET %s HTTP/1.0\r\n\r\n"
#define TIMEOUT 30

typedef struct {
	struct jpeg_source_mgr pub;
	char *data;
	int length;
	JOCTET *buffer;
	boolean start_of_file;
} my_source_mgr;

typedef my_source_mgr *my_src_ptr;

struct my_error_mgr {
	struct jpeg_error_mgr pub;
	jmp_buf setjmp_buffer;
};
 
void my_init_source (j_decompress_ptr); 
boolean my_fill_input_buffer (j_decompress_ptr); 
void my_skip_input_data (j_decompress_ptr, long);
void my_term_source (j_decompress_ptr);
void jpeg_memory_src(j_decompress_ptr, char *, int);
void my_error_exit (j_common_ptr);

int netcam_check_JPEG(struct netcam_context *, int, int);
static void *netcam_loop(void *);
int netcam_read_header(struct netcam_context *, int);
int netcam_connect(struct netcam_context *);
int netcam_close(struct netcam_context *);
int netcam_read_image(struct netcam_context *);
int netcam_single_read(struct netcam_context *,int);
int netcam_read_image_no_content_length(struct netcam_context *,int);
int netcam_read_stream_no_content_length(struct netcam_context *,int);
int netcam_stream_read(struct netcam_context *,int);
int netcam_stream_reconnection(struct netcam_context *,int);


int netcam_start(struct context *);
unsigned char *netcam_next(struct context *, char *);

/*
	parses the url
*/

int netcam_parse_url (struct url_t *parse_url, char *text_url)
{
	char *dup;
	char *parse;
	char *p;

	dup = strdup(text_url);
	parse = dup;
	parse_url->service = NULL;
	parse_url->host = NULL;
	parse_url->port = 80;
	parse_url->path = NULL;

	if ((p = strstr (parse, ":/"))) {
		*p++ = '\0';
		if (p[0] == '/' && p[1] == '/') p += 2;
		parse_url->service = strdup (dup);
		parse = p;
	}
	
	p = strchr (parse, '/');
	
	if (!p || parse < p){
		char *p2;
		
		parse_url->host = strdup (parse);
		if (p) parse_url->host[p - parse] = '\0';
		if (*parse_url->host == '[') {
			p2 = strchr (parse_url->host, ']');
			if (p2)	p2 = strchr (p2, ':');
		} else {
			p2 = strchr (parse_url->host, ':');
		}
		if (p2) {
			*p2++ = '\0';
			parse_url->port = atoi (p2);
		}
	}
	parse = p;
	if (parse) parse_url->path = strdup (parse);

	free(parse_url->service);	
	free (dup);
	
	return 1;
}


/* 
	jpeglib init_source 

	overrided by own init_source routine.
	Initialize source --- called by jpeg_read_header
	before any data is actually read.
*/

void my_init_source (j_decompress_ptr cinfo) 
{
	my_src_ptr src = (my_src_ptr) cinfo->src;
	/* We reset the empty-input-file flag for each image,
	but we don't clear the input buffer.
	This is correct behavior for reading a series of images from one source.
	*/
	src->start_of_file = TRUE;
}

/*
	jpeglib fill_input_buffer

	override by own fill_input_buffer routine.
	Fill the input buffer --- called whenever buffer is emptied
*/

boolean my_fill_input_buffer (j_decompress_ptr cinfo)
{
	my_src_ptr src = (my_src_ptr) cinfo->src;
	size_t nbytes;
	
	if (src->start_of_file) {
		nbytes=src->length;
		src->buffer=src->data;
	} else {
	// Insert a fake EOI marker - as per jpeglib recommendation
		src->buffer[0] = (JOCTET) 0xFF;
		src->buffer[1] = (JOCTET) JPEG_EOI; // 0xD9
		nbytes = 2;
	}
	
	src->pub.next_input_byte = src->buffer;
	src->pub.bytes_in_buffer = nbytes;
	src->start_of_file = FALSE;
	
	return TRUE;
}

/*
	jpeglib skip_input_data 

	override by own skip_input_data routine. 
	Skip data --- used to skip over a potentially large amount of
	uninteresting data
*/

void my_skip_input_data (j_decompress_ptr cinfo, long num_bytes)
{
my_src_ptr src = (my_src_ptr) cinfo->src;

	if (num_bytes > 0) {
		while (num_bytes > (long) src->pub.bytes_in_buffer) {
			num_bytes -= (long) src->pub.bytes_in_buffer;
			(void) my_fill_input_buffer(cinfo);
		}
	src->pub.next_input_byte += (size_t) num_bytes;
	src->pub.bytes_in_buffer -= (size_t) num_bytes;
	}
}

/*
	jpeglib term_source 

	Terminate source --- called by jpeg_finish_decompress
	after all data has been read.  Often a no-op.
	NOT called by jpeg_abort or jpeg_destroy; surrounding
	application must deal with any cleanup that should happen even
	for error exit.
*/
void my_term_source (j_decompress_ptr cinfo) { }

/*
	jpeg_memory_src 

	Prepare for input from memory .
	The caller must have already allocated memory, and is responsible
	for freed it after finishing decompression.
*/

void jpeg_memory_src(j_decompress_ptr cinfo, char *data, int length)
{
	my_src_ptr src;

	if (cinfo->src == NULL) {
		cinfo->src = (struct jpeg_source_mgr *) (*cinfo->mem->alloc_small)
			((j_common_ptr) cinfo, JPOOL_PERMANENT,sizeof(my_source_mgr));
		src = (my_src_ptr) cinfo->src;
	}

	src = (my_src_ptr) cinfo->src;
	src->data = data;
	src->length = length;
	src->pub.init_source = my_init_source;
	src->pub.fill_input_buffer = my_fill_input_buffer;
	src->pub.skip_input_data = my_skip_input_data;
	src->pub.resync_to_restart = jpeg_resync_to_restart;
	src->pub.term_source = my_term_source;
	src->pub.bytes_in_buffer = 0;
	src->pub.next_input_byte = NULL;
}


/*
	jpeglib error_exit

	overrided by our own error_exit routine. 
	Triggered when a fatal error occurs.
*/

void my_error_exit (j_common_ptr cinfo)
{
	struct my_error_mgr *myerr = (struct my_error_mgr *) cinfo->err;
	(*cinfo->err->output_message) (cinfo);
	/* Return control to the setjmp point */
	longjmp (myerr->setjmp_buffer, 1);
}

/*
	This function will do a sanity check over jpeg image , if an error is found
	read the header and return -1.
	Reference JFIF format http://www.w3.org/Graphics/JPEG/jfif3.pdf 
	
	If JPEG start or end are incorrect or connection error returns -1
	Else return 0
*/
int netcam_check_JPEG(struct netcam_context *netcam, int which_image, int connection){
	int error=0,ret;

	/* check jfif start */
	if (!memmem(netcam->image[which_image].buffer,3,"\xFF\xD8\xFF",3)) {
		fprintf(stderr,"Netcam thread %d : Error checking JPEG not start code \n",netcam->threadnr);
		error=1;
	}
	/* check jfif end */
	else if (!memmem(&netcam->image[which_image].buffer[netcam->image[which_image].size -2],
		2,"\xFF\xD9",2)) {
		fprintf(stderr,"Netcam thread %d : Error checking JPEG not end code \n",netcam->threadnr);
		error=1;
	}
		
	if (error) {
		if (connection) {
			sleep(1);
			if (netcam_connect(netcam) != 0) return -1;
		}
		do {
			if (netcam->content_length) ret = netcam_read_header(netcam, 0);
			else ret = netcam_read_header(netcam, 2);
		} while (ret == 0);
	}
	
	return error;
}

/*
	main thread
*/

static void *netcam_loop(void *arg)
{
	struct context *cnt = arg;
	syslog(LOG_DEBUG, "Netcam: thread %d: thread PID: %d",cnt->threadnr, getpid());
	fprintf(stderr, "Netcam: thread %d: thread PID: %d\n", cnt->threadnr, getpid());
	do {
		if (cnt->netcam->read(cnt->netcam,0) == -1) {
			syslog(LOG_DEBUG, "Netcam: thread %d: broken connection, FATAL error", cnt->threadnr);
			fprintf(stderr,"Netcam: thread %d: broken connection, FATAL error\n", cnt->threadnr);
		}
	} while (!cnt->finish);


	pthread_mutex_lock(&cnt->netcam->image[0].mutex);
	pthread_mutex_lock(&cnt->netcam->image[1].mutex);

	/* Clean up Memory */
	if (cnt->netcam->image[0].buffer) free(cnt->netcam->image[0].buffer);
	if (cnt->netcam->image[1].buffer) free(cnt->netcam->image[1].buffer);
	if ((cnt->conf.netcam_userpass) && (cnt->netcam->userpass)) free(cnt->netcam->userpass);
	if (cnt->netcam->boundary_string) free(cnt->netcam->boundary_string);
	if (cnt->netcam->url.host) free(cnt->netcam->url.host);
	if (cnt->netcam->url.path) free(cnt->netcam->url.path);
	if (cnt->netcam->response) free(cnt->netcam->response);

	pthread_mutex_unlock(&cnt->netcam->image[0].mutex);
	pthread_mutex_unlock(&cnt->netcam->image[1].mutex);

	fprintf(stderr, "Netcam: thread %d: Exiting\n",cnt->netcam->threadnr);
	if (cnt->netcam) free(cnt->netcam);

	
	return 0;
}

/*
	if Content-Length is not supplied i.ex Axis 2100 , this function have to get 
	the boundary string for mjpeg.
	i.ex "Content-Type: multipart/x-mixed-replace; boundary=--myboundary".

	On Error Return  -1 Header Uknown
	                 -2 Header Error
	On Success       > 0
*/

int netcam_read_header(struct netcam_context *netcam, int initial)
{
	char *header,*conttype;
	int status,hcount=0;
	
	while (1) {
		status = header_get (netcam->response, &header, HG_NONE);
	
		if (status == HG_ERROR) {
#if 0
			syslog(LOG_ERR, "Netcam: thread %d: error reading header: [%s]", netcam->threadnr, header);
			fprintf (stderr,"Netcam: thread %d: error reading header\nheader:\n [%s]\n",netcam->threadnr, header);
#endif			
			free(header);
			return -2;
		}
#if 0
		else if (status == HG_EOF) {
			syslog(LOG_ERR, "Netcam: thread %d: error reading header: [%s]", netcam->threadnr, header);
			fprintf (stderr,"Netcam: thread %d: error reading header\nheader:\n [%s]\n", netcam->threadnr, header);
		}
#endif		
		
		if (hcount == 0 && initial == 1) {
			/* process http ok ... or any other error header response */
			fprintf(stderr,"[%s]\n",header);
		}
		
		if ( (header_process (header, "Content-Length", header_extract_number,
			&netcam->image[netcam->which_image].size) ) ) {
			netcam->content_length=1;
		} else if ((initial == 1) && (header_process (header, "Content-Type",
			http_process_type, &conttype) ))
		{
			if (strcmp("multipart/x-mixed-replace", conttype) == 0) {
				char *bound;
				fprintf(stderr, "Netcam: thread %d, mjpeg stream\n", netcam->threadnr);
				/* Set the bondary string following Content-Type */
				if ( (!netcam->boundary_string) && (bound = strstr(header,"boundary=")) ) {
					bound = bound + 9;
					netcam->boundary_string = strdup(bound);
					fprintf(stderr, "Netcam: thread %d: netcam->boundary_string [%s]\n", netcam->threadnr, netcam->boundary_string);
				}
				netcam->read = (void *) netcam_stream_read;
			} else if (strcmp("image/jpeg", conttype) == 0) {
				fprintf(stderr, "Netcam: thread %d: jpeg file\n", netcam->threadnr);
				netcam->read = (void *) netcam_single_read;
			} else {
				syslog(LOG_ERR, "Netcam: thread %d: unknown Content-Type", netcam->threadnr);
				fprintf (stderr,"Netcam: thread %d: unknown Content-Type\n", netcam->threadnr);
				free(header);
				return -1;
			}
			if (conttype) free(conttype);
		} else if (!*header) {
			free (header);
			break;
		}
		
		hcount++;
		free (header);
	}
	
	return hcount;
}

/*
	Establish netcam connection , it has a 5 seconds timeout
	On Success return 0
	On Error   return -1
*/

int netcam_connect(struct netcam_context *netcam)
{
	int sock;
	struct sockaddr_in server;
	struct hostent *host_info;
	struct timeval tv;	
	char *request;
	
	sock = socket(PF_INET, SOCK_STREAM, 0);
	if (sock < 0) {
		syslog(LOG_ERR, "Netcam: thread %d: failed to create socket : %s", netcam->threadnr, strerror(errno));
		fprintf(stderr,"Netcam: thread %d: failed to create socket : %s\n", netcam->threadnr, strerror(errno));	
		return -1;
	}
	
	memset(&server, 0, sizeof(server));
	host_info = gethostbyname(netcam->url.host);
	if (host_info == NULL) {
		syslog(LOG_ERR, "Netcam: thread %d: unknown server: %s", netcam->threadnr, netcam->url.host);
		fprintf(stderr,"Netcam: thread %d: unknown server: %s\n", netcam->threadnr, netcam->url.host);
		close(sock);
		return -1;
	}
	
	memcpy((char *) &server.sin_addr, host_info->h_addr, host_info->h_length);
	
	server.sin_family = AF_INET;
	server.sin_port = htons(netcam->url.port);
	
	/* 5 seconds timeout */
	tv.tv_sec = 5;
	tv.tv_usec = 0;
	setsockopt(sock, SOL_SOCKET, SO_RCVTIMEO, &tv, sizeof(tv));
	
	if (connect (sock, (struct sockaddr *) &server, sizeof(server)) < 0) {
		syslog(LOG_ERR, "Netcam: thread %d: can't connect to server", netcam->threadnr);
		fprintf(stderr,"Netcam: thread %d: can't connect to server\n", netcam->threadnr);
		return -1;
	}
	
	if (netcam->userpass != NULL) {
		request = (char *) mymalloc(strlen(REQUEST_AUTH) +
			strlen(netcam->url.path) + strlen(netcam->userpass) - 4 + 1);
		sprintf(request, REQUEST_AUTH, netcam->url.path, netcam->userpass);
	} else {
		request = (char *) mymalloc(strlen(REQUEST) +
		          strlen(netcam->url.path) + 1);
		sprintf(request, REQUEST, netcam->url.path);
	}
	
	rbuf_initialize(netcam->response, sock);
	
	send(RBUF_FD(netcam->response), request, strlen(request), 0);
	
	free(request);
	
	return 0;
}

/*
	Close netcam connection , only for single image.
*/
int netcam_close(struct netcam_context *netcam)
{
	if (netcam->response->fd == -1) return 0;
	close(RBUF_FD(netcam->response));
	netcam->response->fd = -1;
	return 0;
}


/*
	Try to reconnect to the stream and get a frame

	On Success return  1
	On Error   return -1
*/
int netcam_stream_reconnection(struct netcam_context *netcam,int which_image)
{
	int ret=-1,num_errors=-1;
	
	netcam_close(netcam);

	if (netcam->boundary_string) {
		free(netcam->boundary_string);
 		netcam->boundary_string=NULL;
	}	

	netcam->content_length=0;	

	while ((netcam_connect(netcam) == -1) && (num_errors < 10)) {
		num_errors++;
		syslog(LOG_ERR,"Netcam: thread %d: error connecting , retrying [%d]", netcam->threadnr, num_errors);
		fprintf(stderr,"Netcam: thread %d: error connecting , retrying [%d]\n", netcam->threadnr, num_errors);
		sleep(5);
	}

	if (num_errors < 10) {
		syslog(LOG_ERR,"Netcam: thread %d: re-connection done", netcam->threadnr);
		fprintf(stderr,"Netcam: thread %d: re-connection done\n", netcam->threadnr);
	
		do {
			ret = netcam_read_header(netcam, 1);
		} while (ret == 0);
	
		do {
			if (netcam->content_length)
				ret = netcam_read_header(netcam, 0);
			else
				ret = netcam_read_header(netcam, 2);
		} while (ret == 0);
		
		if (netcam->content_length)
			ret = netcam_read_image(netcam);
		else
			ret = netcam_read_stream_no_content_length(netcam,which_image);
	
		if (ret) {
			syslog(LOG_ERR,"Netcam: thread %d: re-connection got frame", netcam->threadnr);
			fprintf(stderr,"Netcam: thread %d: re-connection got frame\n", netcam->threadnr);
		}
		else if (!netcam_check_JPEG(netcam,which_image,0)) {
			syslog(LOG_ERR,"Netcam: thread %d: re-connection did not get frame", netcam->threadnr);
			fprintf(stderr,"Netcam: thread %d: re-connection did not get frame\n", netcam->threadnr);
			ret = -1;
		}
	}
	
	return ret;
}

/*
	AXIS 2100 
	--myboundary^M
	Content-Type: image/jpeg^M
	<image>^M
	--myboundary^M
	Content-Type: image/jpeg^M
	<image>^M
	[ .... ]
	
	netcam stream read function for Non Content-Length support devices.
	On Success returns 1
	On Error returns -1
*/
int netcam_read_stream_no_content_length(struct netcam_context *netcam,int which_image)
{
	int cur_size=0,readb=0;
	size_t boundary_string_length;
	char *pointer_boundary=NULL,*pointer_init=NULL;
	

	/* FIXME !? */
	cur_size = rbuf_flush(netcam->response, netcam->image[which_image].buffer,100000);
	boundary_string_length = strlen(netcam->boundary_string); 
	
	/*
		Get the image until find jpeg header ( JFIF v3 )
	*/

	// Get the init of the image
	if (cur_size != 0){
		pointer_init = memmem(netcam->image[which_image].buffer,cur_size,"\xFF\xD8\xFF",3);
		if (pointer_init){
			int init_read=0,end_read=0;
			init_read = pointer_init - netcam->image[which_image].buffer;
			if (init_read != 0){
				end_read = cur_size - init_read;
				memmove(netcam->image[which_image].buffer,&netcam->image[which_image].buffer[init_read],end_read);
				cur_size = end_read;
			}	
		}
	}
	
	while (!pointer_init){
		readb = read(netcam->response->fd, &netcam->image[which_image].buffer[cur_size], 4096);
		cur_size+=readb;
		pointer_init = memmem(netcam->image[which_image].buffer,cur_size,"\xFF\xD8\xFF",3);
		if (pointer_init){
			int init_read=0,end_read=0;
			init_read = pointer_init - netcam->image[which_image].buffer;
			end_read = cur_size - init_read;
			memmove(netcam->image[which_image].buffer,&netcam->image[which_image].buffer[init_read],end_read);
			cur_size = end_read;
		}
	}
	
	

	/*
	 * Get the image until find the next STREAM HEADER
	 */
	
	do {
		readb = read(netcam->response->fd, &netcam->image[which_image].buffer[cur_size], 4096);
		if (readb < 0) {
			syslog(LOG_ERR,"Netcam: thread %d: error reading stream : %s", netcam->threadnr, strerror(errno));
			fprintf(stderr,"Netcam: thread %d: error reading stream : %s\n", netcam->threadnr, strerror(errno));
			return -1;
		}
		cur_size += readb;
		if ( (pointer_boundary = memmem(netcam->image[which_image].buffer,
			cur_size,netcam->boundary_string, boundary_string_length)) ) {
			cur_size = pointer_boundary - netcam->image[which_image].buffer;
		}
	} while (!pointer_boundary);

	if ( (pointer_boundary = memmem(netcam->image[which_image].buffer,cur_size,"\xFF\xD9",2)) == NULL){
		fprintf(stderr,"Netcam: thread %d: Bad Jpeg end\n",netcam->threadnr);
		netcam->image[which_image].buffer_size = cur_size -4;
		netcam->image[which_image].size = cur_size -4;
	}else{
		netcam->image[which_image].buffer_size = (pointer_boundary - netcam->image[which_image].buffer)+2;
		netcam->image[which_image].size = netcam->image[which_image].buffer_size; 
	}		 


	return 1;
}


/*
	netcam image read function for Non Content-Length support devices
	On Success returns 1
	On Error   returns -1
*/
int netcam_read_image_no_content_length(struct netcam_context *netcam,int which_image)
{
	int cur_size,readb;
	
	
	/* FIXME !? */
	cur_size = rbuf_flush(netcam->response, netcam->image[which_image].buffer, 100000);
	
	do {
		readb = read(netcam->response->fd, &netcam->image[which_image].buffer[cur_size], 4096);
		cur_size += readb;
	} while ((readb > 0) && (cur_size < 100000));
	
	if ((readb < 0) || (cur_size > 100000)){
		syslog(LOG_ERR, "Netcam: thread %d: error reading image : %s", netcam->threadnr, strerror(errno));
		fprintf(stderr,"Netcam: thread %d: error reading image : %s\n", netcam->threadnr, strerror(errno));
		return -1;
	}
	netcam->image[which_image].buffer_size = cur_size;
	netcam->image[which_image].size = cur_size;
	
	return 1;
}


/*
	netcam stream/image read function for Content-Length support devices
	On Success returns 1
	On Error   returns -1
*/
int netcam_read_image(struct netcam_context *netcam)
{
	int which_image,cur_size,cur_left;
	
	which_image = netcam->which_image;
	if (netcam->image[which_image].size > netcam->image[which_image].buffer_size) {
		netcam->image[which_image].buffer_size = netcam->image[which_image].size;
		netcam->image[which_image].buffer =
		     myrealloc(netcam->image[which_image].buffer,
		               netcam->image[which_image].buffer_size,"netcam_stream_read");
	}
	
	cur_size = rbuf_flush(netcam->response, netcam->image[which_image].buffer,
	                      netcam->image[which_image].buffer_size);
	
	while (netcam->image[which_image].size > cur_size) {
		cur_left = recv(netcam->response->fd,&netcam->image[which_image].buffer[cur_size],
	                        netcam->image[which_image].size - cur_size, 0);
		if (cur_left < 0) {
			syslog(LOG_ERR, "Netcam: thread %d: error reading image, not complete", netcam->threadnr);
			fprintf (stderr,"Netcam: thread %d: error reading image, not complete\n", netcam->threadnr);
			netcam->image[which_image].size = 0;
			return -1;
		}
		cur_size += cur_left;
	}
	
	return 1;
}	

/*
	netcam image read function
	On Success returns 1
	On Error   returns -1
*/
int netcam_single_read(struct netcam_context *netcam,int initial)
{
	int which_image,ret,num_errors=-1;

	if (initial) netcam_close(netcam); 
	
	while (netcam_connect(netcam) == -1) {
		num_errors++;
		syslog(LOG_ERR,"Netcam: thread %d: error connecting, retrying [%d]", netcam->threadnr, num_errors);
		fprintf(stderr,"Netcam: thread %d: error connecting, retrying [%d]\n", netcam->threadnr, num_errors);
		if (num_errors == 10) {
			syslog(LOG_ERR,"Netcam: thread %d: Error Connection LOST , waiting 1 minute", netcam->threadnr);
			fprintf(stderr,"Netcam: thread %d: Error Connection LOST waiting 1 minute\n", netcam->threadnr);
			sleep(60);
		}
		sleep(5);
	}
	

	pthread_mutex_lock(&netcam->mutex);
	which_image = netcam->which_image = !netcam->which_image;
	pthread_mutex_unlock(&netcam->mutex);
	
	do {
		if (initial){
			netcam->content_length=0;
			ret = netcam_read_header(netcam, 2);
			if (netcam->content_length){
				fprintf(stderr,"Netcam single: thread %d: Content-length supported \n", netcam->threadnr);
			}else{
				fprintf(stderr,"Netcam single: thread %d : Content-length NO supported \n", netcam->threadnr);
			}
		}
		else {
			ret = netcam_read_header(netcam, 0);
		}
	} while (ret == 0);

	pthread_mutex_lock(&netcam->image[which_image].mutex);
	
	if (netcam->content_length)
		ret = netcam_read_image(netcam);
	else
		ret = netcam_read_image_no_content_length(netcam,which_image);
	netcam_close(netcam);
	
	if ( (ret != -1) && (netcam_check_JPEG(netcam,which_image,1) != -1) ) {
		netcam->image[which_image].filled = 1;
		pthread_cond_signal(&netcam->image[which_image].buffer_filled);
		netcam_close(netcam);
	} else {
		netcam->error_frames++;
		syslog(LOG_ERR,"Netcam: thread %d: error frame [%d]", netcam->threadnr, netcam->error_frames);
		fprintf(stderr,"Netcam: thread %d: error frame [%d]\n", netcam->threadnr, netcam->error_frames);
		if (netcam->error_frames == 100){ 
			netcam->error_frames=0;
			syslog(LOG_INFO,"Netcam: thread %d: error frame reset to 0", netcam->threadnr);
			fprintf(stderr,"Netcam: thread %d: error frame reset to 0\n", netcam->threadnr);
		}
		ret=1;
		netcam_close(netcam);
	}
	
	pthread_mutex_unlock(&netcam->image[which_image].mutex);
	
	return ret;
}

/*
	netcam stream read function
	On Success returns 1 <- !?
	On Error returns -1
*/
int netcam_stream_read(struct netcam_context *netcam,int initial)
{
	int which_image,ret,num_errors=-1;
	
	pthread_mutex_lock(&netcam->mutex);
	which_image = netcam->which_image = !netcam->which_image;
	pthread_mutex_unlock(&netcam->mutex);

	do {
		if (initial) {
			netcam->content_length=0;
			ret = netcam_read_header(netcam, 0);
			if (netcam->content_length){
				fprintf(stderr,"Netcam stream : thread %d: Content-length supported \n", netcam->threadnr);
			}else{
				fprintf(stderr,"Netcam stream : thread %d: Content-length NO supported \n", netcam->threadnr);
			}
			// Check if boundary_string exits
			if (!netcam->boundary_string){
				netcam->image[which_image].size=0;
				netcam_close(netcam);
				syslog(LOG_ERR,"Netcam: thread %d: BOUNDARY STRING not found", netcam->threadnr);
				fprintf(stderr,"Netcam: thread %d: BOUNDARY STRING not found\n", netcam->threadnr);
				return -1;
			}
		}
		else {
			if (netcam->content_length)
				ret = netcam_read_header(netcam, 0);
			else
				ret = netcam_read_header(netcam, 2);
		}
	} while (ret == 0);
	
	pthread_mutex_lock(&netcam->image[which_image].mutex);
	
	do {
		if (netcam->content_length)
			ret = netcam_read_image(netcam);
		else
			ret = netcam_read_stream_no_content_length(netcam,which_image);
		num_errors++;
		if (ret == -1)
			break;
	} while ( ((netcam_check_JPEG(netcam,which_image,0)) == -1) && (num_errors < 10));
	
	if ((ret == -1) || (num_errors == 10)){
		if ( num_errors == 10) {
			syslog(LOG_ERR,"Netcam: thread %d: too many JPEG errors", netcam->threadnr);
			fprintf(stderr,"Netcam: thread %d: too many JPEG errors\n", netcam->threadnr);
			num_errors = -1;
		}
		while (netcam_stream_reconnection(netcam,which_image) == -1) {
			syslog(LOG_ERR,"Netcam: thread %d: error reconnecting, waiting a minute", netcam->threadnr);
			fprintf(stderr,"Netcam: thread %d: errors reconnecting, waiting a minute\n", netcam->threadnr);
			sleep(60); 
		}
		ret=1;
	} 
	
	netcam->image[which_image].filled=1;	
	pthread_cond_signal(&netcam->image[which_image].buffer_filled);
	pthread_mutex_unlock(&netcam->image[which_image].mutex);
	
	return ret;
}


/*
	netcam init called function , it establishes connection to network device, gets
	the image or the stream and creates the read thread.
*/

int netcam_start(struct context *cnt)
{
	struct jpeg_decompress_struct cinfo;
	struct jpeg_error_mgr jerr;
	pthread_attr_t thread_attr;
	pthread_t thread_id;
	struct netcam_context *netcam;
	
	
	syslog(LOG_INFO, "Netcam: thread %d: starting...", cnt->threadnr);
	fprintf(stderr,"Netcam: thread %d: starting...\n", cnt->threadnr);
	
	cnt->netcam = (struct netcam_context *) mymalloc(sizeof(struct netcam_context));
	netcam = cnt->netcam;
	netcam->threadnr = cnt->threadnr;
	netcam->boundary_string = NULL;
	
	pthread_mutex_init(&netcam->mutex, NULL);
	pthread_mutex_init(&netcam->image[0].mutex, NULL);
	pthread_mutex_init(&netcam->image[1].mutex, NULL);
	// Conditional pthread variables ( replaces semaphores in pthread )
	pthread_cond_init(&netcam->image[0].buffer_filled,NULL);
	pthread_cond_init(&netcam->image[1].buffer_filled,NULL);
	netcam->image[0].filled=0;
	netcam->image[1].filled=0;
	netcam->error_frames=0;
	netcam->userpass = NULL;
	
	netcam_parse_url (&netcam->url, cnt->conf.netcam_url);
	
	if (cnt->conf.netcam_userpass) {
		netcam->userpass = (char *) mymalloc(BASE64_LENGTH(strlen(cnt->conf.netcam_userpass))+1);
		{
			char *userpass=mymalloc(strlen(cnt->conf.netcam_userpass)+4);
			/* base64_encode can read 3 bytes after the end of the string,
			initialize it */
			memset(userpass,0,strlen(cnt->conf.netcam_userpass)+4);
			strcpy(userpass,cnt->conf.netcam_userpass);
			base64_encode(userpass, netcam->userpass, strlen(cnt->conf.netcam_userpass));
			free(userpass);
		}
	}	
	
	netcam->response = (struct rbuf *) mymalloc(sizeof(struct rbuf));
	
	/* 
		image buffer should be set for Non Content-Length support devices.
		FIXME : Maybe it should not be hardcoded here !?
	*/
	
	netcam->image[0].buffer_size = 100000;
	netcam->image[0].size = 0;
	netcam->image[0].buffer = (char *) mymalloc(netcam->image[0].buffer_size);
	netcam->image[1].buffer_size = 100000;
	netcam->image[1].size = 0;
	netcam->image[1].buffer = (char *) mymalloc(netcam->image[1].buffer_size);
	netcam->which_image = 0;
	
	/* Initial Connection to Netcam */
	if (netcam_connect(netcam) != 0){
		if ((cnt->conf.netcam_userpass) && (netcam->userpass)) free(netcam->userpass);
		if (netcam->response) free(netcam->response);
		free(netcam->image[0].buffer);
		free(netcam->image[1].buffer);
		if (netcam->url.path) free(netcam->url.path);
		if (netcam->url.host) free(netcam->url.host);
		free(netcam);
		return -1;
	}
	
	/* Initial header read , as specified passing as parameter 1 */	
	if ( netcam_read_header(netcam, 1) < 0){
		if ((cnt->conf.netcam_userpass) && (netcam->userpass)) free(netcam->userpass);
		if (netcam->response) free(netcam->response);
		free(netcam->image[0].buffer);
		free(netcam->image[1].buffer);
		if (netcam->boundary_string) free(netcam->boundary_string);
		if (netcam->url.path) free(netcam->url.path);
		if (netcam->url.host) free(netcam->url.host);
		free(netcam);	
		return -1;
	}
	
	/* Call to fetch image or stream function */
	netcam->read(netcam,1);

	if (netcam->image[netcam->which_image].size > 0) {
		netcam->image[netcam->which_image].filled=0;
		cinfo.err = jpeg_std_error(&jerr);
		jpeg_create_decompress(&cinfo);
		jpeg_memory_src(&cinfo,	netcam->image[netcam->which_image].buffer,
		                netcam->image[netcam->which_image].size);
		jpeg_read_header(&cinfo, TRUE);
		
		cnt->imgs.width=cinfo.image_width;
		cnt->imgs.height=cinfo.image_height;
		cnt->imgs.size=cnt->imgs.width*cnt->imgs.height*3/2;
		cnt->imgs.motionsize=cnt->imgs.width*cnt->imgs.height;
		cnt->imgs.type=VIDEO_PALETTE_YUV420P;
		
		/* Create the netcam loop thread */
		pthread_attr_init(&thread_attr);
		pthread_attr_setdetachstate(&thread_attr, PTHREAD_CREATE_DETACHED);
		pthread_create(&thread_id, &thread_attr, &netcam_loop, cnt);
	} else {
		fprintf(stderr,"Netcam netcam_start(): thread %d: Image size %d\n",
		        netcam->threadnr, (int)netcam->image[netcam->which_image].size);
		if (netcam->image[0].buffer) free(netcam->image[0].buffer);
		if (netcam->image[1].buffer) free(netcam->image[1].buffer);
		if ((cnt->conf.netcam_userpass) && (netcam->userpass)) free(netcam->userpass);
		if (netcam->boundary_string) free(netcam->boundary_string);
		if (netcam->url.path) free(netcam->url.path);
		if (netcam->url.host) free(netcam->url.host);
		if (netcam->response) free(netcam->response);
		if (netcam) free(netcam);
		return -1;
	}
	return 0;
}


/*
	netcam process image function , is called by motion thread to get an image
	from netcam read thread.
*/

unsigned char *netcam_next(struct context *cnt, char *image)
{
	struct jpeg_decompress_struct cinfo;
	struct my_error_mgr jerr;
	unsigned char *pic, *upic, *vpic;
	JSAMPARRAY line;
	int i, line_size, y,which_image,ret;
	JSAMPROW row[1];
	int width, height;                 /* for rotation */
	struct netcam_context *netcam;
	struct timespec now;

	pic = NULL;
	netcam = cnt->netcam;
	
	pthread_mutex_lock(&netcam->mutex);
	which_image = !netcam->which_image;
	if (!netcam->image[which_image].filled)
		which_image = netcam->which_image;
	pthread_mutex_unlock(&netcam->mutex);
	
	pthread_mutex_lock(&netcam->image[which_image].mutex);

	while ((!netcam->image[which_image].filled) && (!cnt->finish)) {
		memset(&now, 0, sizeof now);
		now.tv_sec=time(0)+5;
		now.tv_nsec=0;
		ret = pthread_cond_timedwait(&netcam->image[which_image].buffer_filled,&netcam->image[which_image].mutex,
					&now);
		 if (ret == ETIMEDOUT) { 
			if ((!netcam->image[which_image].filled) && (!cnt->finish)) { 
			/* Code for time-out condition */
				fprintf(stderr,"Thread %d Condiction timeout\n",netcam->threadnr); 
			} else { 
			/* success condition */ 
			break; 
			}
		} 
	}

	if (cnt->finish){
		pthread_mutex_unlock(&netcam->image[which_image].mutex);
		return NULL;	
	}

	netcam->image[which_image].filled = 0;
	if (netcam->image[which_image].size > 0){
		cinfo.err = jpeg_std_error(&jerr.pub);
		jerr.pub.error_exit = my_error_exit;
		
		if (setjmp(jerr.setjmp_buffer)) {
			jpeg_destroy_decompress(&cinfo);
			pthread_mutex_unlock(&netcam->image[which_image].mutex);
			return pic;
		}
		
		jpeg_create_decompress(&cinfo);
		jpeg_memory_src(&cinfo, netcam->image[which_image].buffer,netcam->image[which_image].size);
		jpeg_read_header(&cinfo, TRUE);
		
		cinfo.out_color_space = JCS_YCbCr;
		
		/* NOTE: Since this is a capture, we need to use the capture dimensions. */
		/* POSSIBLE BUG: What if the netcam changes resolution? */
		
		width = cnt->rotate_data.cap_width;
		height = cnt->rotate_data.cap_height;
		
		jpeg_start_decompress(&cinfo);
		line_size = width * 3;
		
		line = (*cinfo.mem->alloc_sarray) ((j_common_ptr) & cinfo,
			JPOOL_IMAGE, cinfo.output_width * cinfo.output_components, 1);
		pic = image;
		upic = pic + width * height;
		vpic = upic + (width * height) / 4;
		row[0] = (unsigned char *) line;
		y = 0;
		
		while (cinfo.output_scanline < height) {
			jpeg_read_scanlines(&cinfo, row, 1);
			for (i = 0; i < line_size; i += 3) {
				pic[i / 3] = ((unsigned char *) line)[i];
				if (i & 1) {
					upic[(i / 3) / 2] = ((unsigned char *) line)[i + 1];
					vpic[(i / 3) / 2] = ((unsigned char *) line)[i + 2];
				}
			}
			pic += line_size / 3;
			if (y++ & 1) {
				upic += width / 2;
				vpic += width / 2;
			}
		}
		
		jpeg_finish_decompress(&cinfo);
		jpeg_destroy_decompress(&cinfo);
		
		if(cnt->rotate_data.degrees > 0) {
			/* rotate as specified */
			rotate_map(image, cnt);
		}
	} else {
		fprintf(stderr,"Netcam netcam_next(): thread %d: Image size %d\n",netcam->threadnr, (int)netcam->image[netcam->which_image].size);
		pthread_mutex_unlock(&netcam->image[which_image].mutex);
		if (netcam->image[0].buffer) free(netcam->image[0].buffer);
		if (netcam->image[1].buffer) free(netcam->image[1].buffer);
		if ((cnt->conf.netcam_userpass) && (netcam->userpass)) free(netcam->userpass);
		if (netcam->boundary_string) free(netcam->boundary_string);
		if (netcam->url.host) free(netcam->url.host);
		if (netcam->url.path) free(netcam->url.path);
		if (netcam->response) free(netcam->response);
		if (netcam) free(netcam);
		return NULL;
	}
	pthread_mutex_unlock(&netcam->image[which_image].mutex);
	
	return pic;
}
