/***************************************************************************
 Mutella - A commandline/HTTP client for the Gnutella filesharing network.

 This program is free software; you can redistribute it and/or
 modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation; either version 2
 of the License, or (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 gnudirector.h  -  Central point of control for all connections and transfers

 the original version of this file was taken from Gnucleus (http://gnucleus.sourceforge.net)

    begin                : Sat Sep 1 2001
    copyright            : (C) 2001 by
    email                : maksik@gmx.co.uk
 ***************************************************************************/

#if !defined(__GNUDIRECTOR_H_INCLUDED__)
#define __GNUDIRECTOR_H_INCLUDED__

#include "gnuhash.h"

struct key_Value;

class MController;
class MGnuPreferences;
class MGnuSock;
class MGnuNode;
class MGnuDownload;
class MGnuUpload;
class MGnuCache;
class MGnuShare;
class MGnuSearch;
class MGnuWebCache;
class MGnuWordHash;
class MGnuMarkedFiles;
class MAsyncDns;
class MUI;

struct packet_Ping;
struct packet_Pong;
struct packet_Push;
struct packet_Query;
struct packet_QueryHit;

struct QueuedHit
{
	int   size;
	char* pPacket;
	DWORD dwNodeID;
};

class MGnuDirector : public MAsyncSocket
{
public:
	MGnuDirector(MController*);
	virtual ~MGnuDirector();

	// service
	MGnuCache* GetHostCatcher(){ASSERT(m_pHostCatcher);return m_pHostCatcher;}
	MGnuCache* GetUltraCatcher(){ASSERT(m_pUltraCatcher);return m_pUltraCatcher;}
	MGnuCache* GetHostStore(){ASSERT(m_pHostStore);return m_pHostStore;}
	MGnuShare* GetShare(){ASSERT(m_pShare);return m_pShare;}
	MGnuPreferences* GetPrefs(){ASSERT(m_pPrefs);return m_pPrefs;}
	void AttachShare(MGnuShare*);
	void DetachShare(MGnuShare*);
	
	const GUID* GetClientID(){return &m_ClientID;}
	WORD GetLocalPort(){return m_nPort;}
	// more management functions
	bool BehindFirewall();
	bool NotLocal(Node);
	bool IsOkForDirectConnect(const IP& host);
	// custom http folders
	bool RegisterHttpFolder(MUI* pUI, const CString& sFolder);
	bool UnregisterHttpFolder(MUI* pUI, const CString& sFolder);
	bool OnExternalHttpRequest(const IP& ipRemoteHost, LPCSTR szFolder, LPCSTR szHandshake, const char* pBuffRest, HANDLE hSocket);
	// search
	MGnuSearch* AddSearch(LPCSTR szSearch, int type, int size, int sizeMode, bool bAutoGet = false);
	bool AddSearch(MGnuSearch* pSearch);
	void ForEachSearch(void* pData, tEachSearch callback);
	bool ModifySearch(DWORD dwID, LPCSTR szSearch);
	MGnuSearch* GetSearchByID(DWORD dwID);
	bool GetSearchByID(DWORD dwID, SGnuSearch& gs, vector<Result>& rv, vector<ResultGroup>& gv);
	int  GetSearchesCount();
	bool GetResultsByID(DWORD dwID, vector<Result>& rv);
	bool ClearSearchByID(DWORD dwID);
	bool RemoveSearchByID(DWORD dwID, bool bRemovePartial = false);
	MGnuSearch* LookUpSearch(const CString& search, int size);
	MGnuSearch* GetSearchByFile(const CString& name, int size);
	void RemoveSearchesByType(int nType);
	//
	void ForEachConnection(void* pData, tEachConnection callback);
	bool CloseConnectionByID(DWORD dwID);
	//
	void ForEachUpload(void* pData, tEachUpload callback);
	void ForEachDownload(void* pData, tEachDownload callback);
	bool RemoveTransferByID(DWORD dwID, bool bDelPart);
	// download queuing
	void OnDownloadHostBusy(const IP& ip, int nNextValidTime);
	bool IsDownloadHostBusy(const IP& ip);
	//
	void OnQueryHit(packet_QueryHit* pHit, MGnuNode* pOrigin, bool bOur);
	// Listening control
	bool StartListening();
	void StopListening();
    //
	bool AllowMakingConnections();
	bool AllowIncomingConnections(){return true;}

	//////////////////////////////////////////
	// NEW
	bool IsLeaf();
	bool IsUltraPeer();
	bool IsForcedUltraPeer() { return m_bForcedSupernode; }

	bool IsShieldedLeaf(); // true if we are in any leaf mode and we have connections to UP
	
	int  GetHostMode() { return m_nHostMode; }
	void SetHostMode(int nMode);
	int  HostModeValidUntil(); // time after which it is allowed to upgrade/downgrade

	void IncUltrapeerCredit() { ++m_nUltrapeerCredit; if (m_nUltrapeerCredit > 10) m_nUltrapeerCredit = 10; }
	void DecUltrapeerCredit() { --m_nUltrapeerCredit; }
	int  GetUltrapeerCredit() { return m_nUltrapeerCredit; }
	
	int GetClientStartTime() { return m_nStartTime; }
	int CountNormalConnects();
	int CountUltrapeerLeaves();
	int GetNormalConnectsApprox();

	void DropSomeNode(int nTimeStamp);
	void DropSomeLeaf(int nTimeStamp);

	void CloseAllOtherNodes(MGnuNode* pExcept);
	
	int  GetExtPongBytes() { return m_nExtPongBytes; }
	void IncExtPongBytes(int nBy) { m_nExtPongBytes += nBy; }
	
	void Route_Pong(packet_Pong*, int, DWORD);
	void Route_UltraPong(packet_Pong*, int, DWORD);
	void Route_Push(packet_Push*, int, DWORD);
	void Route_QueryHit(packet_QueryHit *, int, DWORD);

	DWORD GetLocalSharedFiles();
	DWORD GetLocalSharedSize();
	DWORD GetUltrapeerSizeMarker();
	void  OnShareChanged();

	MGnuMarkedFiles* GetMarkedFiles() { return m_pKnownFiles; }

	double FreeUltraCapacity(int nRemoteTotal);
	double RunningUltraCapacity(int nRemoteTotal);

	MGnuWordHash* GetWordTable() { ASSERT(m_pWordHashTable); return m_pWordHashTable; }

	void EnableSupernodePromotion(bool bForce = false);
	void DisableSupernodePromotion();

	void SetReceivedIncomingConnections() {m_bReceivedIncomingConnections = true;}

protected:
	bool m_bForcedSupernode;
	int m_nHostMode; // SuperNode or Child
	int m_nHostModeChangeTime;
	int m_nUltrapeerCredit;
	int m_nLastNormalConnects;
	MGnuWordHash* m_pWordHashTable;
	int m_nExtPongBytes;
	int m_nStartTime;
	int m_nUltraPongBytes;

	bool m_bLastIsShieldedLeaf;

	bool m_bLastEnableSuperNode;
	bool m_bReceivedIncomingConnections;

	MAsyncDns* m_pAsyncDNS;
	
	///////////////////////////////////////////////////////
	///////////////////////////////////////////////////////

public:
	// Traffic control
	void Post_QueryHit(QueryComp* pSearchQuery, const BYTE* QueryReply, DWORD QueryReplyLength, BYTE ReplyCount);
	void Broadcast_Ping(packet_Ping*,   int, MGnuNode*);
	void Broadcast_Query(packet_Query*, int, MGnuNode*);
	void Send_ForwardQuery(const QueryComp&, const list<DWORD>& NodeIDs );
	GUID Broadcast_LocalQuery(BYTE*, int);
	void SendAllLocalQueries(MGnuNode* pNode);

	void Route_Pong(packet_Pong*, int, key_Value*);
	void Route_QueryHit(packet_QueryHit *, DWORD, key_Value*);
	void Route_Push(packet_Push*, int, key_Value*);
	bool Route_LocalPush(const Result&);
	// Node control
	void AddNode(IP ipHost, UINT port, bool bForceV4 = false);
	void RemoveNode(MGnuNode*);
	void RemoveNode(DWORD dwID);
	MGnuNode* FindNode(IP ip, UINT port);
	// net stats
	void GetNetStats(int& nHosts, int& nSharingHosts, int& nFiles, int& nSize);
	void GetCacheStats(int& nCatcherHosts, int& nUltraCatcherHosts, int& nStoreHosts, int& nWebCaches);
	// Socket Counts
	int  CountConnections();
	int  CountIncomingConns();
	int  CountConnPerSubnet(int SubnetBits, IP ipHost);
	int  GetLastConnCount(){ return m_nLastConnCount;}
	//
	int  CountUploads();
	int  CountUploads(const IP& host, MGnuUpload* pExcept);
	int  CountUploads(int nFileIndex, MGnuUpload* pExcept);
	//
	int  CountTotalDownloads();
	int  CountActiveDownloads();
	int  CountActiveDownloads(const HOST& from, MGnuDownload* pExclude);
	//
	void AddSock(MGnuSock*);
	void AddNode(MGnuNode*);
	void AddUpload(MGnuUpload*);
	bool CheckIfPushung(packet_Push* Push);
	//
	void AddAndStartDownload(MGnuDownload*);
	bool AddDownload(const ResultVec& results);
	bool RenameDownload(DWORD dwID, LPCSTR szNewFilename);
	MGnuDownload* LookUpDownload(IP ipHost, CString FileName, int Index);
	bool OnIncomingPush(const CString& sFileName, int nIndex, SOCKET hPushSock); // should return 'true' if socket was attached
	void BuildDownloadPaths(CString sName, int nFileSize, CString* psFileName, CString* psFilePath, CString* psPartPath);
	bool IsDownloading(const CString& sPartPath);

	public:
	virtual void OnAccept(int nErrorCode);
	void OnTimer(int nTimeStamp); // called once a second by the MGnutellaThread
	
	void RemoveNode_LeastReplyRate(int nTimeStamp);
	void RemoveNode_MaxDrop(int nTimeStamp);
	void AddConnect(bool bSuperpeer, bool bUseStore = false);
	void ManageConnects(int nTimeStamp);
	
	bool IsInTheList(MGnuNode* pNode);

	void AddWebCacheUrl(const CString& sURL, bool bChecked = false);
	void OnWebCacheReply(const CString& sURL);
	void RemoveWebCacheUrl(const CString& sURL);
	void OpenWebCacheConnection(const CString& sURL, int nRequestType, bool bLock = true);
	const CString& GetLastGoodWebCacheUrl() { return m_sLastGoodWebCacheUrl; }
	
	bool SaveWebCaches(const CString sPath);
	bool LoadWebCaches(const CString sPath);

	IP    GetPublicIP();
	DWORD GetPublicPort();
protected:
	MGnuDirector();                         // no implementation
	MGnuDirector(const MGnuDirector& rSrc);  // no implementation

protected:
	MController* m_pController;
	MGnuPreferences* m_pPrefs;
	MGnuCache* m_pHostCatcher;  // this is an 'immediate' host cache
	MGnuCache* m_pUltraCatcher; // this is an 'immediate' ultrapeer cache
	MGnuCache* m_pHostStore;    // this cache consists of hosts that we succeeded to connect to
	MGnuShare* m_pShare;
	
	// Local Client Data
	GUID    m_ClientID;
	DWORD   m_dwFiles;
	DWORD   m_dwFilesSize;

	//int		m_nSecsAboveLimitConn; // Seconds over max conn bandwidth
	
	double	m_dCurrentReceiveLimit;
	double m_dDownloadRateLimit;

	int     m_nPort;
	// Used to get hostname from an IP, (one per thread)
	hostent* m_HostEnt;
	DWORD m_dwReSearchID;
	int m_nReSearchTimer;
	//

	map<DWORD, MGnuNode*>   m_NodeMap;
	vector<MGnuSock*>       m_SockList;
	vector<MGnuDownload*>   m_DownloadList;
	vector<MGnuUpload*>     m_UploadList;
	map<DWORD, MGnuSearch*> m_SearchMap;
	list<DWORD>             m_PendingQueries;
	vector<MGnuWebCache*>   m_WebCacheList;
	set<CString>            m_WebCachesUrlSet;
	std::deque<CString>     m_WebCachesUrlQueue;
	CString                 m_sLastGoodWebCacheUrl;
	int                     m_anInitialWebCacheRequests[3];
	CString                 m_asInitialWebCaches[3];
	bool                    m_bNoValidWebCacheUrlsMessageSent;
	// custom web folders
	map<CString, MUI*>    m_mapWebFolders;
	// access control to the lists above
	MMutex m_listmutex;
	//
	std::queue<QueuedHit> m_hitQueue;
	MMutex m_hitqueuemutex;

	MGnuMarkedFiles* m_pKnownFiles;
	//
	int m_nLastConnCount;
	// download queuing
	map <IP, int> m_mapBusyHosts;
public:
	// Hash tables are thread safe, nothing bad should happen to them
	MGnuHash m_TableQuery;
	MGnuHash m_TablePing;
	MGnuHash m_TablePush;
	MGnuHash m_TableLocal;
	MGnuHash m_TableRouting;
	//
	int m_nQGood;
	int m_nQRepeat;
	int m_nQError;
	
	DWORD m_dwConnBytesRecv;
	DWORD m_dwConnBytesSend;
	DWORD m_dwConnRecvRate;
	DWORD m_dwConnSendRate;
	
	double m_dDownloadRate;
	double m_dUploadRate;
};

/////////////////////////////////////////////////////////////////////////////

#endif // __GNUDIRECTOR_H_INCLUDED__

