/* Copyright (C) 2004 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/**
 * @file myx_gc_figure.cpp 
 * @brief Implementation of the model element class.
 * 
 */

#include "myx_gc.h"

#include <libxml/tree.h>
#include "myx_gc_figure.h"
#include "myx_gc_model.h"
#include "myx_gc_layer.h"
#include "myx_gc_canvas.h"
#include "myx_gc_font_manager.h"

//----------------- CFigureElementTemplate ------------------------------------------------------------------------------

CFigureElementTemplate::CFigureElementTemplate(wstring ID)
{
  FID = ID;
  FLayout = GC_LAYOUT_ROW;
  FResizable = false; 
  FParent = NULL;
  FFigure = NULL;
  FUpdateCount = 0;
  FCaption = NULL;
}

//----------------------------------------------------------------------------------------------------------------------

CFigureElementTemplate::~CFigureElementTemplate(void)
{
  for (CElementTemplateList::iterator Iterator = FChildren.begin(); Iterator != FChildren.end(); ++Iterator)
    delete *Iterator;
  delete FCaption;
}

//----------------------------------------------------------------------------------------------------------------------

void CFigureElementTemplate::AddChild(CFigureElementTemplate* Child)
{
  Child->FParent = this;
  Child->FFigure = NULL;
  FChildren.push_back(Child);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Increases the update counter to avoid change propagations.
 */
void CFigureElementTemplate::BeginUpdate(void)
{
  ++FUpdateCount;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Decreases the update counter until it reaches 0. If it becomes null then change propagations are enabled again
 * and the element as well as its parent is revalidated.
 */
void CFigureElementTemplate::EndUpdate(void)
{
  if (FUpdateCount > 0)
  {
    --FUpdateCount;
    if (FUpdateCount == 0)
      MakeDirty();
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Helper method used by the figure parser to initialize some members.
 *
 * @param Layout The the layout to be used in the element.
 * @param Resizable Resizable flag.
 * @param Constraints The resize constraints.
 */
void CFigureElementTemplate::Initialize(TFigureElementLayout Layout, bool Resizable, wstring StyleName, 
                                        const TConstraints& Constraints, int MinOccurs, int MaxOccurs)
{
  FLayout = Layout;
  FResizable = Resizable;
  FStyleName = StyleName;
  FConstraints = Constraints;
  FMinOccurs = MinOccurs;
  FMaxOccurs = MaxOccurs;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Called when either a child element or the style changes. The event is propagated to the parent element.
 */
void CFigureElementTemplate::MakeDirty(void)
{
  if (FUpdateCount == 0)
  {
    if (FParent != NULL)
      FParent->MakeDirty();
    else
      if (FFigure != NULL)
        FFigure->MakeDirty();
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Sets the (optional) caption for this element. If there is already one then it is freed.
 *
 * @param Caption The caption to be used from now on.
 */
void CFigureElementTemplate::SetCaption(CCaptionElementTemplate* Caption)
{
  if (FCaption != NULL)
    delete FCaption;
  FCaption = Caption;
  if (FCaption != NULL)
    FCaption->FParent = this;
}

//----------------- CCaptionElementTemplate ------------------------------------------------------------------------------------

CCaptionElementTemplate::CCaptionElementTemplate(wstring ID)
{
  FFontSize = 20;
  FWeight = 400;
  FHorizontalAlignment = GC_ALIGN_LEFT_TOP;
  FVerticalAlignment = GC_ALIGN_LEFT_TOP;
  FHasColor = false;
}

//----------------------------------------------------------------------------------------------------------------------

void CCaptionElementTemplate::Initialize(wstring Text, float X, float Y, string FontFamily, int FontSize, int Weight, 
                                         string FontStyle, TAlignment HorizontalAlignment, TAlignment VerticalAlignment, 
                                         GLubyte* Color, const TConstraints& Constraints)
{
  FText = Text;
  FOffsetX = X;
  FOffsetY = Y;
  FFontFamily = FontFamily;
  FFontSize = FontSize;
  FWeight = Weight;
  FFontStyle = FontStyle;
  FHorizontalAlignment = HorizontalAlignment;
  FVerticalAlignment = VerticalAlignment;
  FHasColor = Color != NULL;
  if (FHasColor)
  {
    for (int I = 0; I < 4; ++I)
      FColor[I] = Color[I];
  };
  FConstraints = Constraints;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Called when a property is changed.
 */
void CCaptionElementTemplate::MakeDirty(void)
{
  if (FParent != NULL)
    FParent->MakeDirty();
}

//----------------- CFigureTemplate ------------------------------------------------------------------------------------

CFigureTemplate::CFigureTemplate(wstring ID)
{
  FID = ID;
  FContent = NULL;
}

//----------------------------------------------------------------------------------------------------------------------

CFigureTemplate::~CFigureTemplate(void)
{
  delete FContent;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Returns the content of this element template.
 *
 * @return The content (can be NULL).
 */
CFigureElementTemplate* CFigureTemplate::Content(void)
{
  return FContent;
}

//----------------- CCaptionElement ------------------------------------------------------------------------------------

CCaptionElement::CCaptionElement(void)
{
  FFontFamily = "Arial";
  FFontSize = 20;
  FWeight = 400;
  FFontStyle = "normal";
  FHorizontalAlignment = GC_ALIGN_LEFT_TOP;
  FVerticalAlignment = GC_ALIGN_LEFT_TOP;
  FParent = NULL;
  FDirty = true;
  FHasColor = false;
  FBidiMode = GC_BIDI_LEFT_TO_RIGHT;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Recomputes the offsets for the caption to maintain the current alignments.
 */
void CCaptionElement::ApplyAlignment(void)
{
  // -- Horizontal alignment
  // If the text is larger than the available space then shorten it.
  float ParentWidth = FParent->FBoundingBox.LowerRightX - FParent->FBoundingBox.UpperLeftX;
  float Width = FBoundingBox.LowerRightX - FBoundingBox.UpperLeftX;
  if (Width > ParentWidth)
  {
    // There is not enough space to display the full text.
    FAlignOffsetX = 0;
    int Len = FText.size();
    if ((Len == 0) || (Width <= 0))
      FDisplayText.clear();
    else
    {
      // Determine width of triple point using the current font settings.
      TBoundingBox LocalBox;
      FontManager()->BoundingBox(L"...", FFontFamily, FFontStyle, FWeight, FFontSize, &LocalBox);
      float EllipsisWidth = LocalBox.LowerRightX - LocalBox.UpperLeftX;

      if (Width <= EllipsisWidth)
        FDisplayText.clear();
      else
      {
        // Do a binary search for the optimal string length which fits into the given width.
        int L = 0;
        int H = Len - 1;
        if (FBidiMode == GC_BIDI_RIGHT_TO_LEFT)
        {
          while (L < H)
          {
            int N = (L + H) >> 1;
            FontManager()->BoundingBox(FText.substr(N), FFontFamily, FFontStyle, FWeight, FFontSize, &LocalBox);
            float W = LocalBox.LowerRightX - LocalBox.UpperLeftX + EllipsisWidth;
            if (W <= ParentWidth)
              H = N;
            else
              L = N + 1;
          };
          FDisplayText = L"..." + FText.substr(L + 1);
        }
        else
        {
          while (L < H)
          {
            int N = (L + H + 1) >> 1;
            FontManager()->BoundingBox(FText.substr(0, N), FFontFamily, FFontStyle, FWeight, FFontSize, &LocalBox);
            float W = LocalBox.LowerRightX - LocalBox.UpperLeftX + EllipsisWidth;
            if (W <= ParentWidth)
              L = N;
            else
              H = N - 1;
          };
          FDisplayText = FText.substr(0, L - 1) + L"...";
        };
      };
    };
  }
  else
  {
    FDisplayText = FText;

    TAlignment Alignment = FHorizontalAlignment;
    // In right-to-left mode alignment is reverted.
    if (FBidiMode == GC_BIDI_RIGHT_TO_LEFT)
      if (Alignment == GC_ALIGN_LEFT_TOP)
        Alignment = GC_ALIGN_RIGHT_BOTTOM;
      else
        if (Alignment == GC_ALIGN_RIGHT_BOTTOM)
          Alignment = GC_ALIGN_LEFT_TOP;

    switch (FHorizontalAlignment)
    {
      case GC_ALIGN_LEFT_TOP: // Left aligned.
        {
          FAlignOffsetX = 0;
          break;
        };
      case GC_ALIGN_CENTER: // Centered.
        {
          FAlignOffsetX = (ParentWidth - Width) / 2;
          break;
        };
      case GC_ALIGN_RIGHT_BOTTOM: // Right aligned.
        {
          FAlignOffsetX = ParentWidth - Width;
          break;
        };
    };

  };
  
  // -- Vertical alignment
  float ParentHeight = FParent->FBoundingBox.LowerRightY - FParent->FBoundingBox.UpperLeftY;
  float Height = FBoundingBox.LowerRightY - FBoundingBox.UpperLeftY;
  switch (FVerticalAlignment)
  {
    case GC_ALIGN_LEFT_TOP: // Left aligned.
      {
        FAlignOffsetY = 0;
        break;
      };
    case GC_ALIGN_CENTER: // Centered.
      {
        FAlignOffsetY = (ParentHeight - Height) / 2;
        break;
      };
    case GC_ALIGN_RIGHT_BOTTOM: // Right aligned.
      {
        FAlignOffsetY = ParentHeight - Height;
        break;
      };
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Returns the current bounding box of the text.
 *
 * @return The current bounding box.
 */
TBoundingBox* CCaptionElement::BoundingBox(void)
{
  if (FDirty)
    Validate();
  return &FBoundingBox;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Marks the element as changed so its bounding box it is validated next time it is used.
 */
void CCaptionElement::MakeDirty(void)
{
  if (!FDirty)
  {
    FDirty = true;
    if (FParent != NULL)
      FParent->MakeDirty();
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Renders the caption element.
 */
void CCaptionElement::Render(void)
{
  // Apply color if we have been one given.
  if (FHasColor)
    glColor4ubv(FColor);

  // FTGL uses the lower left corner as origin, we however have the upper left corner for this task.
  // Adjust the vertical coordinate accordingly.
  glRasterPos2f(FOffsetX + FAlignOffsetX, FOffsetY + FAlignOffsetY + FBoundingBox.LowerRightY);

  // Compute the actual font size out of the current scale factor.
  GLfloat Matrix[16];
  glGetFloatv(GL_MODELVIEW_MATRIX, Matrix);
  float Zoom = Matrix[0];
  if (Matrix[5] < Zoom)
    Zoom = Matrix[5];
  int ActualFontSize = ROUND(Zoom * FFontSize);

  if (ActualFontSize > 1)
    FontManager()->TextOut(FDisplayText, FFontFamily, FFontStyle, FWeight, ActualFontSize);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Validates the bounding box.
 */
void CCaptionElement::Validate(void)
{
  if (FDirty)
  {
    FDirty = false;
    FontManager()->BoundingBox(FText, FFontFamily, FFontStyle, FWeight, FFontSize, &FBoundingBox);

    // Apply constraints to the bounding box.
    if (FConstraints.MaxHeight > -1)
    {
      if (FBoundingBox.LowerRightY - FBoundingBox.UpperLeftY > FConstraints.MaxHeight)
        FBoundingBox.LowerRightY = FBoundingBox.UpperLeftY + FConstraints.MaxHeight;
    };
    if (FConstraints.MaxWidth > -1)
    {
      if (FBoundingBox.LowerRightX - FBoundingBox.UpperLeftX > FConstraints.MaxWidth)
        FBoundingBox.LowerRightX = FBoundingBox.UpperLeftX + FConstraints.MaxWidth;
    };
    if (FConstraints.MinHeight > -1)
    {
      if (FBoundingBox.LowerRightY - FBoundingBox.UpperLeftY < FConstraints.MinHeight)
        FBoundingBox.LowerRightY = FBoundingBox.UpperLeftY + FConstraints.MinHeight;
    };
    if (FConstraints.MinWidth > -1)
    {
      if (FBoundingBox.LowerRightX - FBoundingBox.UpperLeftX < FConstraints.MinWidth)
        FBoundingBox.LowerRightX = FBoundingBox.UpperLeftX + FConstraints.MinWidth;
    };
  };
}

//----------------- CFigureElement -------------------------------------------------------------------------------------

CFigureElement::CFigureElement(void)
{
  FTranslation[0] = 0;
  FTranslation[1] = 0;
  FTranslation[2] = 0;
  FTranslation[3] = 0;

  FLayout = GC_LAYOUT_COLUMN;
  FResizable = false; 
  FStyle = NULL;
  FParent = NULL;
  FFigure = NULL;
  FUpdateCount = 0;
  FCaption = NULL;
  FLayouter = new CColumnLayouter(this);
  FExpanded = true;
}

//----------------------------------------------------------------------------------------------------------------------

CFigureElement::~CFigureElement(void)
{
  if (FStyle != NULL)
  {
    CGCStyle* OldStyle = FStyle;
    // Avoid recursion.
    FStyle = NULL;
    OldStyle->RemoveListener(this);
  };

  for (CElementList::iterator Iterator = FChildren.begin(); Iterator != FChildren.end(); ++Iterator)
    delete *Iterator;
  delete FCaption;
  delete FLayouter;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * (Re) computes the overall bounding box for this element. This includes the bounds of the style as well as all children.
 * It is assumed that the caller has already validated the owning figure (and so all contained elements).
 */
void CFigureElement::ComputeBoundingBox(void)
{
  CBoundingBoxComputer BBComputer;

  if (FStyle != NULL)
    BBComputer.Include(NULL, FStyle->BoundingBox());

  if (FExpanded)
  {
    TBoundingBox LocalBox;
    FLayouter->Reset();
    while (FLayouter->HasNext())
    {
      FLayouter->NextBoundingBox(&LocalBox);
      BBComputer.Include(NULL, &LocalBox);
    };
  };

  FBoundingBox = BBComputer.BoundingBox();

  // Apply constraints to the bounding box.
  if (FConstraints.MaxHeight > -1)
  {
    if (FBoundingBox.LowerRightY - FBoundingBox.UpperLeftY > FConstraints.MaxHeight)
      FBoundingBox.LowerRightY = FBoundingBox.UpperLeftY + FConstraints.MaxHeight;
  };
  if (FConstraints.MaxWidth > -1)
  {
    if (FBoundingBox.LowerRightX - FBoundingBox.UpperLeftX > FConstraints.MaxWidth)
      FBoundingBox.LowerRightX = FBoundingBox.UpperLeftX + FConstraints.MaxWidth;
  };
  if (FConstraints.MinHeight > -1)
  {
    if (FBoundingBox.LowerRightY - FBoundingBox.UpperLeftY < FConstraints.MinHeight)
      FBoundingBox.LowerRightY = FBoundingBox.UpperLeftY + FConstraints.MinHeight;
  };
  if (FConstraints.MinWidth > -1)
  {
    if (FBoundingBox.LowerRightX - FBoundingBox.UpperLeftX < FConstraints.MinWidth)
      FBoundingBox.LowerRightX = FBoundingBox.UpperLeftX + FConstraints.MinWidth;
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Increases the update counter to avoid change propagations.
 */
void CFigureElement::BeginUpdate(void)
{
  ++FUpdateCount;
}

//----------------------------------------------------------------------------------------------------------------------

TBoundingBox* CFigureElement::BoundingBox(void)
{
  if (FDirty)
    Validate();
  return &FBoundingBox;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Creates a figure element from a template and returns it.
 * 
 * @param Owner The controller for the new figure element. It is responsible to free the returned instance.
 * @param Template The template to be used when creating the figure element.
 * @param Model The model to which the new element belongs.
 *
 * @return The new figure element instance.
 */
void CFigureElement::CreateFromTemplate(CFigure* Owner, CFigureElementTemplate* Template, CGCModel* Model, 
                                        CElementList* NewElements)
{
  NewElements->clear();

  if (Template != NULL && Template->FMinOccurs > 0)
  {
    for (int I = 0; I < Template->FMinOccurs; ++I)
    {
      CFigureElement* NewElement = new CFigureElement();
      NewElement->FLayout = Template->FLayout;
      NewElement->FLayouter = LayoutMapper::LayouterForElement(NewElement);
      NewElement->FResizable = Template->FResizable;
      NewElement->UseStyle(Model->Style(Template->FStyleName));
      NewElement->FConstraints = Template->FConstraints;

      CCaptionElementTemplate* CaptionTemplate = Template->FCaption;
      if (CaptionTemplate != NULL)
      {
        NewElement->FCaption = new CCaptionElement();
        NewElement->FCaption->FText = CaptionTemplate->FText;
        NewElement->FCaption->FFontFamily = CaptionTemplate->FFontFamily;
        NewElement->FCaption->FFontSize = CaptionTemplate->FFontSize;
        NewElement->FCaption->FWeight = CaptionTemplate->FWeight;
        NewElement->FCaption->FFontStyle = CaptionTemplate->FFontStyle;
        NewElement->FCaption->FHorizontalAlignment = CaptionTemplate->FHorizontalAlignment;
        NewElement->FCaption->FVerticalAlignment = CaptionTemplate->FVerticalAlignment;
        NewElement->FCaption->FParent = NewElement;
        NewElement->FCaption->FHasColor = CaptionTemplate->FHasColor;
        for (int I = 0; I < 4; ++I)
          NewElement->FCaption->FColor[I] = CaptionTemplate->FColor[I];
        NewElement->FCaption->FConstraints = CaptionTemplate->FConstraints;
        NewElement->FCaption->FOffsetX = CaptionTemplate->FOffsetX;
        NewElement->FCaption->FOffsetY = CaptionTemplate->FOffsetY;
      }
      else
        NewElement->FCaption = NULL;

      // Child elements.
      for (CElementTemplateList::iterator Iterator = Template->FChildren.begin(); Iterator != Template->FChildren.end(); ++Iterator)
      {
        CElementList ChildList;
        CFigureElement::CreateFromTemplate(NULL, *Iterator, Model, &ChildList);
        for (CElementList::const_iterator Iterator = ChildList.begin(); Iterator != ChildList.end(); ++Iterator)
        {
          (*Iterator)->FParent = NewElement;
          NewElement->FChildren.push_back(*Iterator);
        };
      };

      NewElement->FFigure = Owner;

      NewElements->push_back(NewElement);
    };
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Decreases the update counter until it reaches 0. If it becomes null then change propagations are enabled again
 * and the element as well as its parent is revalidated.
 */
void CFigureElement::EndUpdate()
{
  if (FUpdateCount > 0)
  {
    --FUpdateCount;
    if (FUpdateCount == 0)
      MakeDirty();
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Returns the owning figure for this element. Involves a parent traversal if necessary.
 *
 * @return The owner of this figure element (or its (grand etc.) parent if not directly owned by a figure.
 */
CFigure* CFigureElement::Figure(void)
{
  if (FFigure != NULL)
    return FFigure;
  else
    return FParent->Figure();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Called when either a child element or the style changes. The event is propagated to the parent element.
 */
void CFigureElement::MakeDirty()
{
  if (FUpdateCount == 0 && !FDirty)
  {
    FDirty = true;
    if (FParent != NULL)
      FParent->MakeDirty();
    else
      if (FFigure != NULL)
        FFigure->MakeDirty();
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * This method renders this element and triggers rendering of its (potential) caption as well as its child elements.
 */
void CFigureElement::Render()
{
  if (FDirty)
    Validate();

  if (FStyle != NULL)
    glCallList(FStyle->DisplayList());

  if (FExpanded)
  {
    FLayouter->Reset();
    while (FLayouter->HasNext())
      FLayouter->RenderNext();
  };

  if (FCaption != NULL)
    FCaption->Render();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Sets a new style to be used for this element.
 *
 * @param NewStyle The new style to be used.
 * @return The previously used style (can be NULL).
 */
CGCStyle* CFigureElement::UseStyle(CGCStyle* NewStyle)
{
  CGCStyle* OldStyle = FStyle;
  if (FStyle != NewStyle)
  {
    if (OldStyle != NULL)
      OldStyle->RemoveListener(this);
    FStyle = NewStyle;
    if (FStyle != NULL)
      FStyle->AddListener(this);

    MakeDirty();
  };

  return OldStyle;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Called before the owner figure creates its display list, so it can be used to create anything necessary that
 * must not be done while a display list is being compiled.
 */
void CFigureElement::Validate(void)
{
  if (FDirty)
  {
    FDirty = false;
    if (FCaption != NULL)
      FCaption->Validate();
    for (CElementList::iterator Iterator = FChildren.begin(); Iterator != FChildren.end(); ++Iterator)
      (*Iterator)->Validate();
    ComputeBoundingBox();

    if (FCaption != NULL)
      FCaption->ApplyAlignment();
  };
}

//----------------- CFigure --------------------------------------------------------------------------------------------

CFigure::CFigure(CGCModel* Owner, CFigureTemplate* Template)
{
  FDestroying = false;
  FModel = Owner;
  FContent = NULL;

  // Initialize with useful values.
  FRotation[0] = 0;
  FRotation[1] = 0;
  FRotation[2] = 0;
  FRotation[3] = 1;

  FScaling[0] = 1;
  FScaling[1] = 1;
  FScaling[2] = 1;

  FTranslation[0] = 0;
  FTranslation[1] = 0;
  FTranslation[2] = 0;

  BuildFromTemplate(Template);

  FDirty = true;
}

//----------------------------------------------------------------------------------------------------------------------

CFigure::~CFigure(void)
{
  FDestroying = true;
  for (CFigureInstanceList::iterator Iterator = FNotificationList.begin(); Iterator != FNotificationList.end(); ++Iterator)
    (*Iterator)->FreeNotification(this);
  FNotificationList.clear();

  delete FContent;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Adds a new figure instance to the notification list to tell them when this figure is about to be destroyed.
 *
 * @param Instance The instance to add.
 */
void CFigure::AddFreeNotification(CFigureInstance* Instance)
{
  FNotificationList.insert(Instance);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Applies the current translation, rotation and scalie factors.
 */
void CFigure::ApplyTransformations(void)
{
  glTranslated(FTranslation[0], FTranslation[1], FTranslation[2]);
  // Index 0 contains the angle, while the other three coordinates form the axis to rotate around.
  glRotated(FRotation[0], FRotation[1], FRotation[2], FRotation[3]);
  glScaled(FScaling[0], FScaling[1], FScaling[2]);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Parses the given template and creates its child structure.
 */
void CFigure::BuildFromTemplate(CFigureTemplate* Template)
{
  CElementList NewElements;
  CFigureElement::CreateFromTemplate(this, Template->Content(), FModel, &NewElements);

  // There should really only be one occurence of the root element. It does not make any sense to have more.
  CElementList::const_iterator Iterator = NewElements.begin();
  FContent = NULL;
  if (NewElements.size() > 0)
  {
    FContent = *Iterator;
    ++Iterator;
  };

  while (Iterator != NewElements.end())
  {
    delete *Iterator;
    ++Iterator;
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Removes the given figure instance from the notification list.
 *
 * @param Instance The instance to remove.
 */
void CFigure::RemoveFreeNotification(CFigureInstance* Instance)
{
  for (CFigureInstanceList::iterator Iterator = FNotificationList.begin(); Iterator != FNotificationList.end(); ++Iterator)
  {
    if (*Iterator == Instance)
    {
      FNotificationList.erase(Iterator);
      break;
    };
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Renders the figure elements.
 */
void CFigure::RenderContent(void)
{
  if (FContent != NULL)
    FContent->Render();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Creates the display list of this figure (and all child figures) if necessary.
 */
void CFigure::Validate(void)
{
  if (FDirty)
  {
    FDirty = false;

    // Let the content prepare itself if necessary.
    if (FContent != NULL)
      FContent->Validate();
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Marks the display list for this figure as invalid, hence it will be recreated next time Validate is called.
 */
void CFigure::MakeDirty(void)
{
  if (!FDirty)
  {
    FDirty = true;

    // Notfiy all instances of this figure.
    for (CFigureInstanceList::iterator Iterator = FNotificationList.begin(); Iterator != FNotificationList.end(); ++Iterator)
      (*Iterator)->MakeDirty();
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Checks the validity of the figure display list and executes it.
 */
void CFigure::Render(void)
{
  ApplyTransformations();
  RenderContent();
}

//----------------------------------------------------------------------------------------------------------------------

/** 
 * Turns the figure around the given axis by the angle Angle (in radians). This version of Rotate uses a single float 
 * values in the parameter list.
 *
 * @param Angle The angle in radians to turn the figure.
 * @param Rx The x part of the rotation axis.
 * @param Ry The y part of the rotation axis.
 * @param Rz The z part of the rotation axis.
 * @note Currently there is no accumulative version of Rotate available (requires a quaternion lib, which we don't have yet).
 */
void CFigure::Rotate(double Angle, double Rx, double Ry, double Rz)
{
  FRotation[0] = Angle;
  FRotation[1] = Rx;
  FRotation[2] = Ry;
  FRotation[3] = Rz;
  MakeDirty();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Turns the figure around the given axis by the angle Angle (in radians). This version of Rotate uses a vector for the
 * rotation axis in the parameter list.
 *
 * @param Angle The angle in radians to turn the figure.
 * @param Axis The axis around which the figure is to be rotated.
 * @note Currently there is no accumulative version of Rotate available (requires a quaternion lib, which we don't have yet).
 */
void CFigure::RotateV(double Angle, const double Axis[3])
{
  FRotation[0] = Angle;
  FRotation[1] = Axis[0];
  FRotation[2] = Axis[1];
  FRotation[3] = Axis[2];
  MakeDirty();
}

//----------------------------------------------------------------------------------------------------------------------

/** 
 * Scales the figure by the amount given in Factor. If Accumulative is true then the new scale factors are multiplied
 * with the existing values. This version of Scale uses single float values as parameters.
 *
 * @param Sx The scale factor in x direction.
 * @param Sy The scale factor in y direction.
 * @param Sz The scale factor in z direction.
 * @param Accumulative If true then the given values are added to any existing values otherwiese they are used as given.
 */
void CFigure::Scale(double Sx, double Sy, double Sz, bool Accumulative)
{
  if (Accumulative)
  {
    FScaling[0] += Sx;
    FScaling[1] += Sy;
    FScaling[2] += Sz;
  }
  else
  {
    FScaling[0] = Sx;
    FScaling[1] = Sy;
    FScaling[2] = Sz;
  };
  MakeDirty();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Scales the figure by the amount given in Factor. If Accumulative is true then the new scale factors are multiplied
 * with the existing values. This version of Scale uses an array of values in the parameter list.
 *
 * @param Factor An array containing the three scale values for x, y and z.
 * @param Accumulative If true then the given values are added to any existing values otherwiese they are used as given.
 */
void CFigure::ScaleV(const double Factor[3], bool Accumulative)
{
  if (Accumulative)
  {
    FScaling[0] += Factor[0];
    FScaling[1] += Factor[1];
    FScaling[2] += Factor[2];
  }
  else
  {
    FScaling[0] = Factor[0];
    FScaling[1] = Factor[1];
    FScaling[2] = Factor[2];
  };
  MakeDirty();
}

//----------------------------------------------------------------------------------------------------------------------

/** 
 * Moves the figure by the amount given in Factor. If Accumulative is true then the new translation factors are 
 * multiplied with the existing values. This version of Translate uses an array for the values in the parameter list.
 *
 * @param Tx Scale factor for the x axis.
 * @param Ty Scale factor for the y axis.
 * @param Tz Scale factor for the z axis.
 * @param Accumulative If true then the given values are added to any existing values otherwiese they are used as given.
 */
void CFigure::Translate(double Tx, double Ty, double Tz, bool Accumulative)
{
  if (Accumulative)
  {
    FTranslation[0] += Tx;
    FTranslation[1] += Ty;
    FTranslation[2] += Tz;
  }
  else
  {
    FTranslation[0] = Tx;
    FTranslation[1] = Ty;
    FTranslation[2] = Tz;
  };
  MakeDirty();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Moves the figure by the amount given in Factor. If Accumulative is true then the new translation factors are 
 * multiplied with the existing values. This version of Translate uses an array for the values in the parameter list.
 *
 * @param Factor An array of translation values, for each axis one.
 * @param Accumulative If true then the given values are added to any existing values otherwiese they are used as given.
 */
void CFigure::TranslateV(const double Factor[3], bool Accumulative)
{
  if (Accumulative)
  {
    FTranslation[0] += Factor[0];
    FTranslation[1] += Factor[1];
    FTranslation[2] += Factor[2];
  }
  else
  {
    FTranslation[0] = Factor[0];
    FTranslation[1] = Factor[1];
    FTranslation[2] = Factor[2];
  };
  MakeDirty();
}

//----------------- CFigureInstance ------------------------------------------------------------------------------------

CFigureInstance::CFigureInstance(CLayer* Owner, CFigure* Figure)
{
  FLayer = NULL;
  FFigure = Figure;
  FSelected = false;
  FValidBounds = false;

  FRotation[0] = 0;
  FRotation[1] = 0;
  FRotation[2] = 0;
  FRotation[3] = 1;

  FScaling[0] = 1;
  FScaling[1] = 1;
  FScaling[2] = 1;

  FTranslation[0] = 0;
  FTranslation[1] = 0;
  FTranslation[2] = 0;

  FCurrentBounds.Left = 0;
  FCurrentBounds.Top = 0;
  FCurrentBounds.Right = 0;
  FCurrentBounds.Bottom = 0;

  FDirty = true;
  if (Owner != NULL)
    Owner->AddInstance(this);
  if (Figure != NULL)
    Figure->AddFreeNotification(this);
}

//----------------------------------------------------------------------------------------------------------------------

CFigureInstance::~CFigureInstance(void)
{
  if (FLayer != NULL)
  {
    if (FSelected) 
      FLayer->GetCanvas()->RemoveFromSelection(this);
    if (!FLayer->IsUpdating())
      FLayer->RemoveInstance(this);
  };

  if ((FFigure != NULL) && (!FFigure->FDestroying))
    FFigure->RemoveFreeNotification(this);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Applies the current translation, rotation and scalie factors.
 */
void CFigureInstance::ApplyTransformations(void)
{
  glTranslated(FTranslation[0], FTranslation[1], FTranslation[2]);
  // Index 0 contains the angle, while the other three coordinates form the axis to rotate around.
  glRotated(FRotation[0], FRotation[1], FRotation[2], FRotation[3]);
  glScaled(FScaling[0], FScaling[1], FScaling[2]);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Called by a class (usually CFigure) with which we registered us as notification sink and which is
 * about to be destroyed.
 */
void CFigureInstance::FreeNotification(CFigure* Figure)
{
  if (Figure == FFigure)
    delete this;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Marks the display list for this figure instance as invalid, hence it will be recreated next time Validate 
 * is called. If a list already exists then it is freed. If the figure is connected to a parent figure then this parent 
 * is invalidated as well.
 */
void CFigureInstance::MakeDirty(void)
{
  if (!FDirty)
  {
    FDirty = true;
    FValidBounds = false;
    if (FLayer != NULL)
    {
      if (FSelected) 
        FLayer->GetCanvas()->InvalidateSelectionBounds(this);
      if (!FLayer->IsUpdating())
        FLayer->MakeDirty();
    };
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Validates the associated figure if it is dirty.
 */
void CFigureInstance::Validate(void)
{
  if (FFigure->FDirty)
    FFigure->Validate();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Returns the bounds of this figure instance according to the figure it represents and all transformations.
 *
 * @return The smallest rectangle enclosing the whole content of the figure instance in the coordinate system of
 *         the layer this instance belongs to.
 */
void CFigureInstance::GetBounds(TBounds* Bounds)
{
  if (!FValidBounds)
  {
    // (Re)compute the current bounding box if it is not valid currently.
    FValidBounds = true;
    GLsizei BufferSize = 1000;
    GLfloat* Buffer = NULL;
    int Hits = 0;

    GLint Viewport[4];
    glGetIntegerv(GL_VIEWPORT, Viewport);
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    glOrtho(0, Viewport[2], Viewport[3], 0, -1000, 1000);

    // Feedback might be requested while normal rendering is already in progress.
    // So make sure to restore projection and modelview matrix once we are finished.
    glMatrixMode(GL_MODELVIEW);
    glPushMatrix();
    do
    {
      Buffer = (GLfloat*) realloc(Buffer, BufferSize * sizeof(GLfloat));
      glFeedbackBuffer(BufferSize, GL_2D, Buffer);
      glRenderMode(GL_FEEDBACK);
      glMatrixMode(GL_MODELVIEW);
      glLoadIdentity();

      /*
      glBegin(GL_POLYGON);
        glVertex2d(0, 0);
        glVertex2d(0, 3000);
        glVertex2d(1000, 3000);
        glVertex2d(1000, 0);
      glEnd();
      */
      FLayer->RenderFeedback(this);
  
      Hits = glRenderMode(GL_RENDER);
      BufferSize <<= 1;
    }
    while (Hits < 0);

    glPopMatrix();

    if (Hits > 0)
    {
      #define CHECK_VERTEX(X, Y) \
        if (X < FCurrentBounds.Left) \
          FCurrentBounds.Left = X; \
        if (Y > FCurrentBounds.Top) \
          FCurrentBounds.Top = Y; \
        if (X > FCurrentBounds.Right) \
          FCurrentBounds.Right = X; \
        if (Y < FCurrentBounds.Bottom) \
          FCurrentBounds.Bottom = Y;

      int Offset = 0;
      FCurrentBounds.Left = 1e10;
      FCurrentBounds.Top = -1e10;
      FCurrentBounds.Right = -1e10;
      FCurrentBounds.Bottom = 1e10;
      while (Offset < Hits)
      {
        switch ((unsigned int) Buffer[Offset++])
        {
          /*
          case GL_PASS_THROUGH_TOKEN:
            {
              float Token = Buffer[Offset++]; // Just for debugging.
              break;
            };
          // */
          case GL_POINT_TOKEN:
          case GL_BITMAP_TOKEN:
          case GL_DRAW_PIXEL_TOKEN:
          case GL_COPY_PIXEL_TOKEN:
            {
              float X = Buffer[Offset++];
              float Y = Buffer[Offset++];
              CHECK_VERTEX(X, Y);
              break;
            };
          case GL_POLYGON_TOKEN:
            {
              int Count = (int) Buffer[Offset++];
              for (int I = 0; I < Count; I++)
              {
                float X = Buffer[Offset++];
                float Y = Buffer[Offset++];
                CHECK_VERTEX(X, Y);
              };
              break;
            };
          case GL_LINE_TOKEN:
          case GL_LINE_RESET_TOKEN:
            {
              float X = Buffer[Offset++];
              float Y = Buffer[Offset++];
              CHECK_VERTEX(X, Y);

              X = Buffer[Offset++];
              Y = Buffer[Offset++];
              CHECK_VERTEX(X, Y);

              break;
            };
        };
      };
      #undef CHECK_VERTEX
    };

    free(Buffer);

    FLayer->CheckError();
  };

  *Bounds = FCurrentBounds;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * @brief Destroys this object.
 */
void CFigureInstance::Release(void)
{
  delete this;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Checks the validity of the figure instance display list and executes it.
 */
void CFigureInstance::Render(void)
{
  // Place a reference of this class onto the name stack. This is only used when picking is enabled.
//  glLoadName((GLuint) this);

  glPushMatrix();

  ApplyTransformations();
  if (FFigure != NULL)
    FFigure->Render();

  glPopMatrix();
}

//----------------------------------------------------------------------------------------------------------------------

/** 
 * Rotates the figure around the given axis by the angle Angle (in radians). This version of Rotate uses a vector for the
 * rotation axis in the parameter list.
 *
 * @param Angle The rotation angle in radians.
 * @param Rx The x part of the axis around which to rotate the figure instance.
 * @param Ry The y part of the axis around which to rotate the figure instance.
 * @param Rz The z part of the axis around which to rotate the figure instance.
 * @note: Currently there is no accumulative version of Rotate available (requires a quaternion lib, which we don't have yet).
 */
void CFigureInstance::Rotate(double Angle, double Rx, double Ry, double Rz)
{
  FRotation[0] = Angle;
  FRotation[1] = Rx;
  FRotation[2] = Ry;
  FRotation[3] = Rz;
  MakeDirty();
}

//----------------------------------------------------------------------------------------------------------------------

/** 
 * Rotates the figure around the given axis by the angle Angle (in radians). This version of Rotate uses a vector for the
 * rotation axis in the parameter list.
 *
 * @param Angle The rotation angle in radians.
 * @param Axis The axis around which to rotate the figure instance.
 * @note: Currently there is no accumulative version of Rotate available (requires a quaternion lib, which we don't have yet).
 */
void CFigureInstance::RotateV(double Angle, const double Axis[3])
{
  FRotation[0] = Angle;
  FRotation[1] = Axis[0];
  FRotation[2] = Axis[1];
  FRotation[3] = Axis[2];
  MakeDirty();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Scales the figure by the amount given in Factor. If Accumulative is true then the new scale factors are multiplied
 * with the existing values. This version of Scale uses an array of values in the parameter list.
 *
 * @param Sx The scale value for the x-axis
 * @param Sy The scale value for the y-axis
 * @param Sz The scale value for the z-axis
 * @param Accumulative If true then the new scale values are added to any previously assigned values.
 */
void CFigureInstance::Scale(double Sx, double Sy, double Sz, bool Accumulative)
{
  if (Accumulative)
  {
    FScaling[0] += Sx;
    FScaling[1] += Sy;
    FScaling[2] += Sz;
  }
  else
  {
    FScaling[0] = Sx;
    FScaling[1] = Sy;
    FScaling[2] = Sz;
  };
  MakeDirty();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Scales the figure by the amount given in Factor. If Accumulative is true then the new scale factors are multiplied
 * with the existing values. This version of Scale uses an array of values in the parameter list.
 *
 * @param Factor Contains the scaling factors for all three axes. Index 0 contains the value for the x-axis, index 1
 *               that for the y-axis and index 2 for z.
 * @param Accumulative If true then the new scale values are added to any previously assigned values.
 */
void CFigureInstance::ScaleV(const double Factor[3], bool Accumulative)
{
  if (Accumulative)
  {
    FScaling[0] += Factor[0];
    FScaling[1] += Factor[1];
    FScaling[2] += Factor[2];
  }
  else
  {
    FScaling[0] = Factor[0];
    FScaling[1] = Factor[1];
    FScaling[2] = Factor[2];
  };
  MakeDirty();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Tells the caller whether this instance is currently selected.
 * 
 * @return True if this figure instance is currently selected, otherwise false.
 */
bool CFigureInstance::Selected(void)
{
  return FSelected;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Moves the figure by the amount given in Factor. If Accumulative is true then the new translation factors are multiplied
 * with the existing values.
 *
 * @param Tx The scale factor to apply on the x-axis.
 * @param Ty The scale factor to apply on the y-axis.
 * @param Tz The scale factor to apply on the z-axis.
 * @param Accumulative If true scaling factors are added to the values already set previously.
 */
void CFigureInstance::Translate(double Tx, double Ty, double Tz, bool Accumulative)
{
  if (Accumulative)
  {
    FTranslation[0] += Tx;
    FTranslation[1] += Ty;
    FTranslation[2] += Tz;
  }
  else
  {
    FTranslation[0] = Tx;
    FTranslation[1] = Ty;
    FTranslation[2] = Tz;
  };
  MakeDirty();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Moves the figure by the amount given in Factor. If Accumulative is true then the new translation factors are multiplied
 * with the existing values.
 *
 * @param Factor The scale factor to apply. Index 0 contains the factor for the x-axis etc.
 * @param Accumulative If true scaling factors are added to the values already set previously.
 */
void CFigureInstance::TranslateV(const double Factor[3], bool Accumulative)
{
  if (Accumulative)
  {
    FTranslation[0] += Factor[0];
    FTranslation[1] += Factor[1];
    FTranslation[2] += Factor[2];
  }
  else
  {
    FTranslation[0] = Factor[0];
    FTranslation[1] = Factor[1];
    FTranslation[2] = Factor[2];
  };
  MakeDirty();
}

//----------------------------------------------------------------------------------------------------------------------

