/* Copyright (C) 2004 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/**
 * @file myx_gc_svgparser.cpp
 * @brief Parser for SVG elements, which are converted to OpenGL calls.
 * 
 */

#include "myx_gc.h"

#include "myx_gc_svgparser.h"
#include "myx_gc_gl_helper.h"
#include "myx_gc_texture.h"
#include "myx_gc_font_manager.h"
#include "myx_gc_const.h"

//----------------------------------------------------------------------------------------------------------------------

typedef enum
{
  GC_PRIMITIVE_UNKNOWN = -1,
  GC_PRIMITIVE_RECT,
  GC_PRIMITIVE_LINE,
  GC_PRIMITIVE_POLYLINE,
  GC_PRIMITIVE_POLYGON,
  GC_PRIMITIVE_CIRCLE,
  GC_PRIMITIVE_TEXT,
  GC_PRIMITIVE_TSPAN,
  GC_PRIMITIVE_GROUP,
  GC_PRIMITIVE_IMAGE
} GC_PRIMITIVE;

//----------------------------------------------------------------------------------------------------------------------

// Convert a primitive name to a number.
GC_PRIMITIVE FindPrimitive(string Name)
{
  static map<string, GC_PRIMITIVE> Primitives;

  if (Primitives.size() == 0)
  {
    // Initialize the map first if necessary.
    Primitives["rect"] = GC_PRIMITIVE_RECT;
    Primitives["line"] = GC_PRIMITIVE_LINE;
    Primitives["polyline"] = GC_PRIMITIVE_POLYLINE;
    Primitives["polygon"] = GC_PRIMITIVE_POLYGON;
    Primitives["circle"] = GC_PRIMITIVE_CIRCLE;
    Primitives["text"] = GC_PRIMITIVE_TEXT;
    Primitives["tspan"] = GC_PRIMITIVE_TSPAN;
    Primitives["g"] = GC_PRIMITIVE_GROUP;
    Primitives["image"] = GC_PRIMITIVE_IMAGE;
  };

  map<string, GC_PRIMITIVE>::iterator Iterator = Primitives.find(Name);
  if (Iterator == Primitives.end())
    return GC_PRIMITIVE_UNKNOWN;
  else
    return Iterator->second;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Scans the given string for coordinate values and fills the Points vector with them.
 */
void ParsePoints(xmlChar* Raw, CVertexVector* Points)
{
  Points->clear();

  StringTokenizer Tokenizer((char*) Raw, " ,\t\n");

  while (Tokenizer.HasMoreTokens())
  {
    // Invalid input results in a value of 0, so we don't need to take special
    // care for such cases.
    TVertex V;
    V.X = Tokenizer.NextTokenAsFloat();
    if (Tokenizer.HasMoreTokens())
    {
      V.Y = Tokenizer.NextTokenAsFloat();
      Points->push_back(V);
    };
  };
}

//----------------- CSVGParser -----------------------------------------------------------------------------------------

CSVGParser::CSVGParser(void)
{
}

//----------------------------------------------------------------------------------------------------------------------

CSVGParser::~CSVGParser(void)
{
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Parses the content of a circle definition.
 *
 * @see ParseElement for a description of the parameters.
 */
void CSVGParser::ParseCircle(xmlNodePtr SVG, bool DoFill, GLubyte* FillColor, bool DoStroke, GLubyte* StrokeColor, 
                             float StrokeWidth, CBoundingBoxComputer* BB)
{
  GLUquadricObj *Quadric = gluNewQuadric();
  float InnerRadius = GetFloatAttributeDef(SVG, "inner-radius", 0);
  float OuterRadius = GetFloatAttributeDef(SVG, "outer-radius", 1);
  float StartAngle = GetFloatAttributeDef(SVG, "start-angle", 0);
  float SweepAngle = GetFloatAttributeDef(SVG, "sweep-angle", 360);
  GLint Slices = GetIntAttributeDef(SVG, "slices", 10);
  GLint Loops = GetIntAttributeDef(SVG, "rings", 3);

  // Update the accumulated bounding box.
  GLfloat Matrix[16];
  glGetFloatv(GL_MODELVIEW_MATRIX, Matrix);
  BB->Include(Matrix, -OuterRadius, -OuterRadius);
  BB->Include(Matrix, OuterRadius, OuterRadius);

  // Disable polygon smoothing. We cannot have both here, alpha blended figures and polygon smoothing.
  glPushAttrib(GL_ENABLE_BIT);
  glDisable(GL_POLYGON_SMOOTH);

  gluQuadricOrientation(Quadric, GLU_INSIDE);

  if (FillColor != NULL)
    glColor4ubv(FillColor);

  if (DoFill)
  {
    gluQuadricDrawStyle(Quadric, GLU_FILL);
    gluQuadricNormals(Quadric, GLU_SMOOTH);
    gluQuadricTexture(Quadric, GL_TRUE);

    gluPartialDisk(Quadric, InnerRadius, OuterRadius, Slices, Loops, StartAngle, SweepAngle);
  };

  if (DoStroke)
  {
    if (StrokeColor != NULL)
      glColor4ubv(StrokeColor);

    gluQuadricDrawStyle(Quadric, GLU_SILHOUETTE);
    gluPartialDisk(Quadric, InnerRadius, OuterRadius, Slices, Loops, StartAngle, SweepAngle);
  };

  gluDeleteQuadric(Quadric);
  glPopAttrib();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Recursively called function that parses the given SVG xml element and converts it to OpenGL calls.
 *
 * @param SVG Any of the supported primitives.
 * @param DoFill The parent's fill flag. If this is true then also this element is filled.
 * @param FillColor The parent's fill color. Only used if the we have a local opacity.
 * @param DoStroke The parent's outline flag. If this true then also this element is outlined.
 * @param StrokeColor The parent's stroke color. Only used if the we have a local opacity.
 * @param StrokeWidth The parent's stroke width. Used only if we draw an outline at all and no local width is given.
 * @param MasterAlpha The accumulated alpha value, which is currently active. Used in conjunction with a local opacity.
 * @param BB This calculator is used to determine the overall bounding box of the element being parsed.
 * @return Returns a new display list for the content of this element.
 */
GLuint CSVGParser::ParseElement(xmlNodePtr SVG, bool DoFill, GLubyte* FillColor, bool DoStroke, GLubyte* StrokeColor, 
                                float StrokeWidth, float MasterAlpha, CBoundingBoxComputer* BB)
{
  GC_PRIMITIVE Primitive = FindPrimitive((char*) SVG->name);
  CGCTexture* Texture = NULL;
  GLuint ResultList = 0;
                                    
  if (Primitive != GC_PRIMITIVE_UNKNOWN)
  {
    bool Display = true;

    xmlChar* Attribute = xmlGetProp(SVG, (xmlChar*) "display");
    if (Attribute != NULL)
    {
      Display = xmlStrcmp(Attribute, (const xmlChar *) "none") != 0;
      xmlFree(Attribute);
    };
    
    if (Display)
    {
      Attribute = xmlGetProp(SVG, (xmlChar*) "texture");
      if (Attribute != NULL)
      {
        Texture = TextureManager()->FindTexture(string((char*) Attribute));
        xmlFree(Attribute);
      };

      // See if we have a local color and/or a local opacity.
      float Opacity = MasterAlpha;
      bool HasOpacity = GetFloatAttribute(SVG, "opacity", Opacity);
      if (HasOpacity)
        Opacity *= MasterAlpha;

      GLubyte LocalFillColor[4];
      GLubyte* FillColorPtr = NULL;

      int Conversion = ConvertColor(SVG, "fill", LocalFillColor);
      // Fill this element if either filling is not completely disabled (fill ="none")
      // and either a local color is given or the parent element used filling already.
      bool LocalFill = (Conversion != 3) && (DoFill || FillColor != NULL || Conversion == 0);
      if (LocalFill)
      {
        // If there is no valid local color then use that of the parent.
        if (Conversion != 0)
        {
          // Combine the parent color with this opacity to create a new color.
          LocalFillColor[0] = FillColor[0];
          LocalFillColor[1] = FillColor[1];
          LocalFillColor[2] = FillColor[2];
        };
        LocalFillColor[3] = GLubyte(255 * Opacity);
        FillColorPtr = LocalFillColor;
      };

      GLubyte LocalStrokeColor[4];
      GLubyte* StrokeColorPtr = NULL;

      Conversion = ConvertColor(SVG, "stroke", LocalStrokeColor);
      bool LocalStroke = (Conversion != 3) && (DoStroke || StrokeColor != NULL || Conversion == 0);
      if (LocalStroke)
      {
        // If there is no valid local color then use that of the parent.
        if (Conversion != 0)
        {
          // Combine the parent color with this opacity to create a new color.
          LocalStrokeColor[0] = StrokeColor[0];
          LocalStrokeColor[1] = StrokeColor[1];
          LocalStrokeColor[2] = StrokeColor[2];
        };
        LocalStrokeColor[3] = GLubyte(255 * Opacity);
        StrokeColorPtr = LocalStrokeColor;
      };

      float LocalStrokeWidth = GetFloatAttributeDef(SVG, "stroke-width", StrokeWidth);

      // Prepare child display lists before starting with the one for this element.
      GLuint LocalList = 0;
      switch (Primitive)
      {
        case GC_PRIMITIVE_TEXT:
        case GC_PRIMITIVE_TSPAN:
          {
            LocalList = ParseText(SVG, LocalFill, FillColorPtr, LocalStroke, StrokeColorPtr, LocalStrokeWidth, Opacity, BB);
            break;
          };
        case GC_PRIMITIVE_GROUP:
          {
            LocalList = ParseGroup(SVG, LocalFill, FillColorPtr, LocalStroke, StrokeColorPtr, LocalStrokeWidth, Opacity, BB);
            break;
          };
      };

      ResultList = glGenLists(1);
      // We need the display listed executed while being compiled to have the modelview matrix updated correctly.
      // This is needed to compute the bounding box correctly.
      glNewList(ResultList, GL_COMPILE_AND_EXECUTE);

      // Save current transformation matrix, color and texture values.
      glPushMatrix();
      glPushAttrib(GL_CURRENT_BIT);

      if (Texture != NULL)
        Texture->ActivateTexture();

      Attribute = xmlGetProp(SVG, (xmlChar*) "transform");
      if (Attribute != NULL)
      {
        ParseTransformation((char*) Attribute);
        xmlFree(Attribute);
      };

      switch (Primitive) 
      {
        case GC_PRIMITIVE_RECT:
          {
            ParseRectangle(SVG, LocalFill, FillColorPtr, LocalStroke, StrokeColorPtr, LocalStrokeWidth, BB);
            break;
          };
        case GC_PRIMITIVE_LINE:
          {
            ParseLine(SVG, LocalStroke, StrokeColorPtr, LocalStrokeWidth, BB);
            break;
          };
        case GC_PRIMITIVE_POLYLINE:
          {
            ParsePolyline(SVG, LocalFill, FillColorPtr, LocalStroke, StrokeColorPtr, LocalStrokeWidth, BB);
            break;
          };
        case GC_PRIMITIVE_POLYGON:
          {
            ParsePolygon(SVG, LocalFill, FillColorPtr, LocalStroke, StrokeColorPtr, LocalStrokeWidth, BB);
            break;
          };
        case GC_PRIMITIVE_CIRCLE:
          {
            ParseCircle(SVG, LocalFill, FillColorPtr, LocalStroke, StrokeColorPtr, LocalStrokeWidth, BB);
            break;
          };
        case GC_PRIMITIVE_TEXT:
        case GC_PRIMITIVE_TSPAN:
        case GC_PRIMITIVE_GROUP:
          {
            if (LocalList != 0)
              glCallList(LocalList);
            break;
          };
        case GC_PRIMITIVE_IMAGE:
          {
            ParseImage(SVG, BB);
            break;
          };
      };

      if (Texture != NULL)
        Texture->DeactivateTexture();

      glPopAttrib();
      glPopMatrix();

      glEndList();
    };
  };

  return ResultList;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Collects all data in an <svg:g> element.
 *
 * @return A new display list comprising all subelements.
 * @see ParseElement for the description of the parameters.
 */
GLuint CSVGParser::ParseGroup(xmlNodePtr SVG, bool DoFill, GLubyte* FillColor, bool DoStroke, GLubyte* StrokeColor, 
                              float StrokeWidth, float MasterAlpha, CBoundingBoxComputer* BB)
{
  xmlNodePtr Child = SVG->children;
  vector<GLuint> Lists;
  while (Child != NULL)
  {
    if (Child->type == XML_ELEMENT_NODE)
    {
      GLuint List = ParseElement(Child, DoFill, FillColor, DoStroke, StrokeColor, StrokeWidth, MasterAlpha, BB);
      Lists.push_back(List);
    };
    Child = Child->next;
  };

  GLuint ResultList = glGenLists(1);
  glNewList(ResultList, GL_COMPILE);
  for (vector<GLuint>::iterator Iterator = Lists.begin(); Iterator != Lists.end(); Iterator++)
    glCallList(*Iterator);
  glEndList();

  return ResultList;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Parses the content of an image definition.
 *
 * @param SVG The XML SVG element containing the definition.
 */
void CSVGParser::ParseImage(xmlNodePtr SVG, CBoundingBoxComputer* BB)
{
  float X = GetFloatAttributeDef(SVG, "x", 0);
  float Y = GetFloatAttributeDef(SVG, "y", 0);
  float Width = GetFloatAttributeDef(SVG, "width", 0);
  float Height = GetFloatAttributeDef(SVG, "height", 0);
  string Filename = GetStringAttributeDef(SVG, "href", "");
  string ID = GetStringAttributeDef(SVG, "id", Filename);

  // Update the accumulated bounding box.
  GLfloat Matrix[16];
  glGetFloatv(GL_MODELVIEW_MATRIX, Matrix);
  BB->Include(Matrix, X, Y);
  BB->Include(Matrix, X + Width, Y + Height);

  // Reuse an existing image.
  CGCTexture* Texture = TextureManager()->FindTexture(ID);
  if (Texture == NULL)
  {
    TLODList LODs;
    LODs.push_back(Filename);

    Texture = TextureManager()->CreateTextureEntry(LODs, ID, "clamp-to-edge", "clamp-to-edge", "linear-mipmap-linear", 
      "linear", 2, "modulate");
  };
  Texture->ActivateTexture();

  glColor4f(1, 1, 1, 1);
  glTranslatef(X, Y, 0);
  glBegin(GL_POLYGON);
    glTexCoord2d(0, 0);
    glVertex2f(0, 0);
    glTexCoord2d(1, 0);
    glVertex2f(Width, 0);
    glTexCoord2d(1, 1);
    glVertex2f(Width, Height);
    glTexCoord2d(0, 1);
    glVertex2f(0, Height);
  glEnd();
  Texture->DeactivateTexture();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Parses the content of a line definition.
 *
 * @param SVG The XML SVG element containing the definition.
 * @param DoStroke Flag to indicate if the line is to be drawn or not.
 * @param StrokeColor The color to be used for the line (if set).
 * @param StrokeWidth The width of the line.
 * @param BB This calculator is used to determine the overall bounding box of the element being parsed.
 */
void CSVGParser::ParseLine(xmlNodePtr SVG, bool DoStroke, GLubyte* StrokeColor, float StrokeWidth, CBoundingBoxComputer* BB)
{
  float X1 = GetFloatAttributeDef(SVG, "x1", 0);
  float Y1 = GetFloatAttributeDef(SVG, "y1", 0);
  float X2 = GetFloatAttributeDef(SVG, "x2", 0);
  float Y2 = GetFloatAttributeDef(SVG, "y2", 0);

  // Update the accumulated bounding box.
  GLfloat Matrix[16];
  glGetFloatv(GL_MODELVIEW_MATRIX, Matrix);
  BB->Include(Matrix, X1, Y1);
  BB->Include(Matrix, X2, Y2);

  if (DoStroke)
  {
    if (StrokeColor != NULL)
      glColor4ubv(StrokeColor);
    glLineWidth(StrokeWidth);

    glBegin(GL_LINES);
      glVertex2f(X1, Y1);
      glVertex2f(X2, Y2);
    glEnd();
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Parses the content of a polygon definition.
 *
 * @see ParseElement for a description of the parameters.
 */
void CSVGParser::ParsePolygon(xmlNodePtr SVG, bool DoFill, GLubyte* FillColor, bool DoStroke, GLubyte* StrokeColor, 
                              float StrokeWidth, CBoundingBoxComputer* BB)
{
  xmlChar* Raw = xmlGetProp(SVG, (xmlChar*) "points");
  if (Raw != NULL)
  {
    CVertexVector Vertices;
    ParsePoints(Raw, &Vertices);
    xmlFree(Raw);

    RenderVertices(DoFill, FillColor, DoStroke, StrokeColor, Vertices, NULL, StrokeWidth, true, BB);
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Parses the content of a polyline definition.
 *
 * @see ParseElement for a description of the parameters.
 */
void CSVGParser::ParsePolyline(xmlNodePtr SVG, bool DoFill, GLubyte* FillColor, bool DoStroke, GLubyte* StrokeColor, 
                               float StrokeWidth, CBoundingBoxComputer* BB)
{
  xmlChar* Raw = xmlGetProp(SVG, (xmlChar*) "points");
  if (Raw != NULL)
  {
    CVertexVector Vertices;
    ParsePoints(Raw, &Vertices);
    xmlFree(Raw);

    RenderVertices(DoFill, FillColor, DoStroke, StrokeColor, Vertices, NULL, StrokeWidth, false, BB);
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Parses the content of a rectangle definition.
 *
 * @see ParseElement for a description of the parameters.
 */
void CSVGParser::ParseRectangle(xmlNodePtr SVG, bool DoFill, GLubyte* FillColor, bool DoStroke, GLubyte* StrokeColor, 
                                float StrokeWidth, CBoundingBoxComputer* BB)
{
  float X, Y, Width, Height, RX, RY;

  if (!GetFloatAttribute(SVG, "x", X))
    X = 0;
  if (!GetFloatAttribute(SVG, "y", Y))
    Y = 0;
  if (!GetFloatAttribute(SVG, "width", Width))
    Width = 1;
  if (!GetFloatAttribute(SVG, "height", Height))
    Height = 1;

  // There are a few more things to check with the corner radii.
  bool RXfound = GetFloatAttribute(SVG, "rx", RX);
  bool RYfound = GetFloatAttribute(SVG, "ry", RY);
  if (!RXfound && !RYfound)
  {
    RX = 0;
    RY = 0;
  }
  else
  {
    if (!RXfound)
      RX = RY;
    else
      if (!RYfound)
        RY = RX;
    if (RX > Width / 2)
      RX = Width / 2;
    if (RY > Width / 2)
      RY = Width / 2;
  };

  // TODO: Consider rounded corners.
  CVertexVector Vertices;
  TVertex V;
  V.X = X;
  V.Y = Y;
  Vertices.push_back(V);
  V.X = X + Width;
  Vertices.push_back(V);
  V.Y = Y + Height;
  Vertices.push_back(V);
  V.X = X;
  Vertices.push_back(V);

  CVertexVector TextureCoordinates;
  V.X = 0;
  V.Y = 0;
  TextureCoordinates.push_back(V);
  V.X = Width;
  TextureCoordinates.push_back(V);
  V.Y = Height;
  TextureCoordinates.push_back(V);
  V.X = 0;
  TextureCoordinates.push_back(V);

  RenderVertices(DoFill, FillColor, DoStroke, StrokeColor, Vertices, &TextureCoordinates, StrokeWidth, true, BB);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Takes a text node and gets all attributes for direct or dynamic rendering. This function is called recursively
 * and can take either a <text> or a <tspan> node.
 *
 * @param SVG The text or tspan node to parse.
 * @param FillColor The color for the text interior.
 * @param StrokeColor The color of the outline.
 * @param The width of the outlines.
 * @param MasterAlpha Only passed through because text nodes can have children.
 * @param BB This calculator is used to determine the overall bounding box of the element being parsed.
 * @return A new display list comprising all subelements.
 */
GLuint CSVGParser::ParseText(xmlNodePtr SVG, bool DoFill, GLubyte* FillColor, bool DoStroke, GLubyte* StrokeColor, 
                             float StrokeWidth, float MasterAlpha, CBoundingBoxComputer* BB)
{
  // Collect font information.
  string FontFamily = GetStringAttributeDef(SVG, "font-family", DefaultFontFamily);
  string FontStyle = GetStringAttributeDef(SVG, "font-style", DefaultFontStyle);
  string FontWeightString = GetStringAttributeDef(SVG, "font-weight", DefaultFontWeight);
  int Weight = ConvertFontWeight(FontWeightString);
  int FontSize = GetIntAttributeDef(SVG, "font-size", DefaultFontSize);

  vector<GLuint> Lists;
  xmlNodePtr Child = SVG->children;
  while (Child != NULL)
  {
    if (XML_IS(Child, "tspan"))
    {
      GLuint LocalList = ParseElement(Child, DoFill, FillColor, DoStroke, StrokeColor, StrokeWidth, MasterAlpha, BB);
      Lists.push_back(LocalList);
    }
    else
    {
      string Source((char*) Child->content);
      wstring Output = Utf8ToUtf16(Source);

      // Make the font manager create all display lists before we build ours.
      TBoundingBox Box;
      FontManager()->BoundingBox(Output, FontFamily, FontStyle, Weight, FontSize, &Box);

      // Update the accumulated bounding box.
      GLfloat Matrix[16];
      glGetFloatv(GL_MODELVIEW_MATRIX, Matrix);
      BB->Include(Matrix, &Box);

      GLuint LocalList = glGenLists(1);
      glNewList(LocalList, GL_COMPILE);
        FontManager()->TextOut(Output, FontFamily, FontStyle, Weight, FontSize);
      glEndList();
      Lists.push_back(LocalList);
    };

    Child = Child->next;
  };

  // Create final display list.
  GLuint DisplayList = glGenLists(1);
  glNewList(DisplayList, GL_COMPILE);

  float X = GetFloatAttributeDef(SVG, "x", 0);
  float Y = GetFloatAttributeDef(SVG, "y", 0);

  glTranslatef(X, Y, 0);

  if (FillColor !=  NULL)
    glColor4ubv(FillColor);

  for (vector<GLuint>::iterator Iterator = Lists.begin(); Iterator != Lists.end(); Iterator++)
    glCallList(*Iterator);
  glEndList();

  return DisplayList;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Parsers the given string and interprets the content as one or more transformation of the form:
 *   translate(x, y, z) scale(x, y, z) etc.
 *
 * @param Transformation The textual representation of the transformation to parse and perform.
 */
void CSVGParser::ParseTransformation(char* Transformation)
{
  string Delimiters(" ,()\t\n");
  StringTokenizer Tokenizer(Transformation, Delimiters);

  // Each run in the outer loop is one transformation.
  while (Tokenizer.HasMoreTokens())
  {
    // Skip leading white spaces.
    string Token = Tokenizer.NextToken();

    int Type = 0;
    if (Token == "matrix")
      Type = 1;
    else
      if (Token == "translate")
        Type = 2;
      else
        if (Token == "scale")
          Type = 3;
        else
          if (Token == "rotate")
            Type = 4;
          else
            if (Token == "skewX")
              Type = 5;
            else
              if (Token == "skewY")
                Type = 6;
              else
                break;

    switch (Type)
    {
      case 1: // Matrix
        {
          // Argument list found. Read 6 float numbers.
          // | a  d  0  0 |
          // | b  e  0  0 |
          // | 0  0  1  0 |
          // | c  f  0  1 |
          GLfloat Matrix[16] = {
            1, 0, 0, 0, 
            0, 1, 0, 0, 
            0, 0, 1, 0, 
            0, 0, 0, 1
          };

          for (int J = 0; J < 2; J++)
            for (int I = 0; I < 2; I++)
              Matrix[J * 4 + I] = Tokenizer.NextTokenAsFloat();
          Matrix[12] = Tokenizer.NextTokenAsFloat();
          Matrix[13] = Tokenizer.NextTokenAsFloat();
          glMultMatrixf(Matrix);

          break;
        };
      case 2: // Translation
        {
          // One or two floats are possible here.
          float TX = Tokenizer.NextTokenAsFloat();
          float TY = 0;
          if (Tokenizer.HasMoreTokens())
            TY = Tokenizer.NextTokenAsFloat();
          glTranslatef(TX, TY, 0);

          break;
        };
      case 3: // Scaling
        {
          // One or two floats are possible here.
          float SX = Tokenizer.NextTokenAsFloat();
          float SY = SX;
          if (Tokenizer.HasMoreTokens())
            SY = Tokenizer.NextTokenAsFloat();
          glScalef(SX, SY, 0);

          break;
        };
      case 4: // Rotation
        {
          // An angle (in degrees) and an optional rotation point are possible here.
          float Angle = Tokenizer.NextTokenAsFloat();
          float X = 0;
          float Y = 0;
          if (Tokenizer.HasMoreTokens())
          {
            X = Tokenizer.NextTokenAsFloat();
            Y = Tokenizer.NextTokenAsFloat();
          };

          glTranslatef(X, Y, 0);
          glRotatef(Angle, 0, 0, 1);
          glTranslatef(-X, -Y, 0);

          break;
        };
      case 5: // X Skewing
      case 6: // Y Skewing
        {
          // Only one value is expected, which gives the skewing angle in degrees.
          GLdouble Matrix[16] = {
            1, 0, 0, 0, 
            0, 1, 0, 0, 
            0, 0, 1, 0, 
            0, 0, 0, 1
          };

          int Index = (Type == 5) ? 4 : 1;
          Matrix[Index] = tan(Tokenizer.NextTokenAsFloat() * M_PI / 180);
          glMultMatrixd(Matrix);
          break;
        };
    };
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Renders the given set of vertices.
 *
 * @param DoFill Filled output is only done if this flag is true. For strokes the existance of the color as such is used
 *               as indicator.
 * @param FillColor If not NULL then it gives the local color for this call, otherwise the current color stays untouched.
 * @param StrokeColor If not nULL then the vertices are render again as lines but using that color.
 * @param Vertices The vertex data to render.
 * @param TextureCoordinates If given (can be NULL) then there must be exactly the same number of texture coordinates
 *                           as there are vertices.
 * @param StrokeWidth The width of the border if one is rendered.
 * @param CloseShape Determines whether the border line (if StrokeColor is given) is closed (that is, the last point is
 *                   connected to the first point).
 * @param BB This calculator is used to determine the overall bounding box of the element being parsed.
 */
void CSVGParser::RenderVertices(bool DoFill, GLubyte* FillColor, bool DoStroke, GLubyte* StrokeColor, 
                                const CVertexVector& Vertices, CVertexVector* TextureCoordinates, float StrokeWidth, 
                                bool CloseShape, CBoundingBoxComputer* BB)
{
  glDisable(GL_POLYGON_SMOOTH);

  GLfloat Matrix[16];
  glGetFloatv(GL_MODELVIEW_MATRIX, Matrix);

  if (DoStroke)
  {
    if (StrokeColor != NULL)
      glColor4ubv(StrokeColor);
    if (StrokeWidth <= 1)
    {
      glBegin(CloseShape ? GL_LINE_LOOP : GL_LINE_STRIP);
      for (int I = 0; I < (int) Vertices.size(); I++)
        glVertex2f(Vertices[I].X, Vertices[I].Y);
      glEnd();
    }
    else
    {
      // Render small rectangles for each line.
      float HalfWidth = StrokeWidth / 2;
      vector<TVertex>::size_type Index = 0;
      vector<TVertex>::size_type Count = Vertices.size();
      if (!CloseShape)
        Count--;
      while (Index < Count)
      {
        vector<TVertex>::size_type NextIndex = Index + 1;
        if (Index == Vertices.size() - 1)
          NextIndex = 0;
        // We need the angle of the current line relative to the horizontal axis.
        float dX = Vertices[NextIndex].X - Vertices[Index].X;
        float dY = Vertices[NextIndex].Y - Vertices[Index].Y;
        float Angle = atan2(dY, dX);

        // Compute the four corners for the rectangle. We can use symmetry to speed up computation.
        dX = HalfWidth * sin(Angle);
        dY = HalfWidth * cos(Angle);
        glBegin(GL_POLYGON);
          glVertex2f(Vertices[Index].X - dX, Vertices[Index].Y + dY);
          // Update the accumulated bounding box.
          BB->Include(Matrix, Vertices[Index]);

          glVertex2f(Vertices[NextIndex].X - dX, Vertices[NextIndex].Y + dY);
          BB->Include(Matrix, Vertices[Index]);

          glVertex2f(Vertices[NextIndex].X + dX, Vertices[NextIndex].Y - dY);
          BB->Include(Matrix, Vertices[Index]);

          glVertex2f(Vertices[Index].X + dX, Vertices[Index].Y - dY);
          BB->Include(Matrix, Vertices[Index]);
        glEnd();

        Index++;
      };
    };
  };

  if (FillColor != NULL)
    glColor4ubv(FillColor);

  if (DoFill)
  {
    glBegin(GL_POLYGON);
    for (int I = 0; I < (int) Vertices.size(); I++)
    {
      if (TextureCoordinates != NULL)
        glTexCoord2d((*TextureCoordinates)[I].X, (*TextureCoordinates)[I].Y);
      glVertex2f(Vertices[I].X, Vertices[I].Y);

      // Update the accumulated bounding box.
      BB->Include(Matrix, Vertices[I]);
    };
    glEnd();
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Parses the given SVG xml description and converts it to OpenGL calls. This method can be called within an active
 * OpenGL display list compilation (but not between glBegin/glEnd).
 *
 * @param SVG The SVG top level element (<svg:svg> </svg:svg>).
 * @param DisplayList The OpenGL display list to render into.
 * @param BB The bounding box computer to use to compute the overall bounding box of the element.
 * @note In order to render correctly the svg:svg element must have width and height attributes set. Particularly the 
 *       height attribute is needed to convert from SVG's top-down to OpenGL's bottom-up coordinate system.
 *       Without height attribute the element will be drawn head-down.
 */
void CSVGParser::Convert(xmlNodePtr SVG, GLuint DisplayList, CBoundingBoxComputer* BB)
{
  vector<GLuint> Lists;
  xmlNodePtr Child = SVG->children;
  while (Child != NULL)
  {
    if (Child->type == XML_ELEMENT_NODE)
    {
      GLuint LocalList = ParseElement(Child, false, NULL, false, NULL, 1, 1, BB);
      Lists.push_back(LocalList);
    };
    Child = Child->next;
  };

  // Not necessary for creating the list but used to create a clean starting point to transform bounding boxes.
  glLoadIdentity();

  glNewList(DisplayList, GL_COMPILE);

  glColor4f(0, 0, 0, 1);

  for (vector<GLuint>::iterator Iterator = Lists.begin(); Iterator != Lists.end(); Iterator++)
    glCallList(*Iterator);
  glEndList();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Parses the given style definition and creates a new style, which is then added to the given model.
 *
 * @param Definition The definition to parse.
 * @Model The model to which the new style is to be added.
 */
void CSVGParser::ParseDefinition(xmlNodePtr Definition, CGCModel* Model)
{
  string ID; 
  if (GetStringAttribute(Definition, "id", ID))
  {
    // Handle sub entries of the layout definition.
    xmlNodePtr Element = Definition->children;

    // This call will create an entry if there is none yet.
    CGCStyle* Style = Model->Style(Utf8ToUtf16(ID));
    if (Style->FDisplayList == 0)
      Style->FDisplayList = glGenLists(1);
    while (Element != NULL)
    {
      if (Element->type == XML_ELEMENT_NODE)
      {
        if (XML_IS(Element, "svg"))
        {
          CBoundingBoxComputer BBComputer;
          Convert(Element, Style->FDisplayList, &BBComputer);
          Style->FBoundingBox = BBComputer.BoundingBox();
          Model->CheckError();
        };

        // Ignore everything else.
        break;
      };
      Element = Element->next;
    };
  };
}

//----------------------------------------------------------------------------------------------------------------------

