/*

Copyright (C) 2000, 2001, 2002 Christian Kreibich <christian@whoop.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/

#include <netdude/nd_debug.h>
#include <netdude/nd_gui.h>
#include <netdude/nd_dialog.h>
#include <netdude/nd_packet_iterator.h>
#include <netdude/nd_dialog.h>

#include <nd_tcp.h>
#include <nd_tcp_callbacks.h>

void    
nd_tcp_sport_cb(ND_Packet   *packet,
		guchar      *header,
		guchar      *data)
{
  static ND_ProtoCallbackData cb_data;
  struct tcphdr *tcphdr;

  tcphdr = (struct tcphdr *) header;
  nd_proto_cb_data_init(&cb_data, nd_tcp_get(), data, 0, FALSE);

  nd_dialog_number(_("Enter source port:"),
		   ND_BASE_DEC,
		   ntohs(tcphdr->th_sport), 65535,
		   nd_proto_iterate_16bit_cb,
		   NULL, packet, &cb_data);
}


void    
nd_tcp_dport_cb(ND_Packet   *packet,
		guchar      *header,
		guchar      *data)
{
  static ND_ProtoCallbackData cb_data;
  struct tcphdr *tcphdr;

  tcphdr = (struct tcphdr *) header;
  nd_proto_cb_data_init(&cb_data, nd_tcp_get(), data, data-header, TRUE);

  nd_dialog_number(_("Enter destination port:"),
		   ND_BASE_DEC,
		   ntohs(tcphdr->th_dport), 65535,
		   nd_proto_iterate_16bit_cb,
		   NULL, packet, &cb_data);
}


static void
tcp_seq_ok_cb(ND_Packet *packet, void *user_data, guint value)
{
  ND_PacketIterator     pit;
  struct tcphdr        *tcphdr;

  for (nd_pit_init(&pit, packet->trace, TRUE); nd_pit_get(&pit); nd_pit_next(&pit))
    {
      tcphdr = (struct tcphdr *) nd_packet_get_data(nd_pit_get(&pit), nd_tcp_get(), 0);
      if (!tcphdr)
	continue;
  
      tcphdr->th_seq = htonl(value);
      nd_tcp_update_state(packet, nd_pit_get_index(&pit));
      nd_packet_modified_at_index(nd_pit_get(&pit), nd_pit_get_index(&pit));
    }

  return;
  TOUCH(user_data);
}


void    
nd_tcp_seq_cb(ND_Packet   *packet,
	      guchar      *header,
	      guchar      *data)
{
  struct tcphdr *tcphdr;

  tcphdr = (struct tcphdr *) header;

  nd_dialog_number(_("Enter sequence number:"),
		   ND_BASE_DEC,
		   ntohl(tcphdr->th_seq), (guint) -1,
		   tcp_seq_ok_cb,
		   NULL, packet, data);

  return;
  TOUCH(data);
}


static void
tcp_ack_ok_cb(ND_Packet *packet, void *user_data, guint value)
{
  ND_PacketIterator     pit;
  struct tcphdr        *tcphdr;

  for (nd_pit_init(&pit, packet->trace, TRUE); nd_pit_get(&pit); nd_pit_next(&pit))
    {
      tcphdr = (struct tcphdr *) nd_packet_get_data(nd_pit_get(&pit), nd_tcp_get(), 0);
      if (!tcphdr)
	continue;
  
      tcphdr->th_ack = htonl(value);
      nd_tcp_update_state(packet, nd_pit_get_index(&pit));
      nd_packet_modified_at_index(nd_pit_get(&pit), nd_pit_get_index(&pit));
    }

  return;
  TOUCH(user_data);
}


void    
nd_tcp_ack_cb(ND_Packet   *packet,
	      guchar      *header,
	      guchar      *data)
{
  struct tcphdr *tcphdr;

  tcphdr = (struct tcphdr *) header;

  nd_dialog_number(_("Enter acknowledgement number:"),
		   ND_BASE_DEC,
		   ntohl(tcphdr->th_seq), (guint) -1,
		   tcp_ack_ok_cb,
		   NULL, packet, data);

  return;
  TOUCH(data);
}


void    
nd_tcp_off_cb(ND_Packet   *packet,
	      guchar      *header,
	      guchar      *data)
{
  static ND_ProtoCallbackData cb_data;
  struct tcphdr *tcphdr;

  tcphdr = (struct tcphdr *) header;
  nd_proto_cb_data_init(&cb_data, nd_tcp_get(), data, data-header, FALSE);

  nd_dialog_number(_("Enter header size in 32-bit words:"),
		   ND_BASE_DEC,
		   tcphdr->th_off, 15,
		   nd_proto_iterate_4bit_hi_cb,
		   NULL, packet, &cb_data);
}


static void
tcp_x2_ok_cb(ND_Packet *packet, void *user_data, guint value)
{
  ND_PacketIterator     pit;
  struct tcphdr        *tcphdr;

  for (nd_pit_init(&pit, packet->trace, TRUE); nd_pit_get(&pit); nd_pit_next(&pit))
    {
      tcphdr = (struct tcphdr *) nd_packet_get_data(nd_pit_get(&pit), nd_tcp_get(), 0);
      if (!tcphdr)
	continue;
  
      tcphdr->th_flags &= 0x3f;
      tcphdr->th_flags |= ((value & 0x03) << 6);
      tcphdr->th_x2 = ((value & 0x3c) >> 2);
      nd_packet_modified_at_index(nd_pit_get(&pit), nd_pit_get_index(&pit));
    }

  return;
  TOUCH(user_data);
}


void    
nd_tcp_unused_cb(ND_Packet   *packet,
		 guchar      *header,
		 guchar      *data)
{
  struct tcphdr *tcphdr;

  tcphdr = (struct tcphdr *) header;
  nd_dialog_number(_("Enter 6-bit unused value:"),
		   ND_BASE_DEC,
		   ((tcphdr->th_x2 << 2) | ((tcphdr->th_flags & 0xc0) >> 6)),
		   63,
		   tcp_x2_ok_cb,
		   NULL, packet, data);
}

void    
nd_tcp_u_cb(ND_Packet   *packet,
	    guchar      *header,
	    guchar      *data)
{
  struct tcphdr        *tcphdr;
  ND_PacketIterator     pit;

  for (nd_pit_init(&pit, packet->trace, TRUE); nd_pit_get(&pit); nd_pit_next(&pit))
    {
      tcphdr = (struct tcphdr *) nd_packet_get_data(nd_pit_get(&pit), nd_tcp_get(), 0);
      if (!tcphdr)
	continue;
  
      tcphdr->th_flags = tcphdr->th_flags ^ TH_URG;
      nd_packet_modified_at_index(nd_pit_get(&pit), nd_pit_get_index(&pit));
    }

  return;
  TOUCH(header);
  TOUCH(data);
}

void    
nd_tcp_a_cb(ND_Packet   *packet,
	    guchar      *header,
	    guchar      *data)
{
  struct tcphdr        *tcphdr;
  ND_PacketIterator     pit;

  for (nd_pit_init(&pit, packet->trace, TRUE); nd_pit_get(&pit); nd_pit_next(&pit))
    {
      tcphdr = (struct tcphdr *) nd_packet_get_data(nd_pit_get(&pit), nd_tcp_get(), 0);
      if (!tcphdr)
	continue;
  
      tcphdr->th_flags = tcphdr->th_flags ^ TH_ACK;
      nd_packet_modified_at_index(nd_pit_get(&pit), nd_pit_get_index(&pit));
    }

  return;
  TOUCH(header);
  TOUCH(data);
}


void    
nd_tcp_p_cb(ND_Packet   *packet,
	    guchar      *header,
	    guchar      *data)
{
  struct tcphdr        *tcphdr;
  ND_PacketIterator     pit;

  for (nd_pit_init(&pit, packet->trace, TRUE); nd_pit_get(&pit); nd_pit_next(&pit))
    {
      tcphdr = (struct tcphdr *) nd_packet_get_data(nd_pit_get(&pit), nd_tcp_get(), 0);
      if (!tcphdr)
	continue;
  
      tcphdr->th_flags = tcphdr->th_flags ^ TH_PUSH;
      nd_packet_modified_at_index(nd_pit_get(&pit), nd_pit_get_index(&pit));
    }

  return;
  TOUCH(header);
  TOUCH(data);
}


void    
nd_tcp_r_cb(ND_Packet   *packet,
	    guchar      *header,
	    guchar      *data)
{
  struct tcphdr        *tcphdr;
  ND_PacketIterator     pit;

  for (nd_pit_init(&pit, packet->trace, TRUE); nd_pit_get(&pit); nd_pit_next(&pit))
    {
      tcphdr = (struct tcphdr *) nd_packet_get_data(nd_pit_get(&pit), nd_tcp_get(), 0);
      if (!tcphdr)
	continue;
  
      tcphdr->th_flags = tcphdr->th_flags ^ TH_RST;
      nd_packet_modified_at_index(nd_pit_get(&pit), nd_pit_get_index(&pit));
    }

  return;
  TOUCH(header);
  TOUCH(data);
}


void    
nd_tcp_s_cb(ND_Packet   *packet,
	    guchar      *header,
	    guchar      *data)
{
  struct tcphdr        *tcphdr;
  ND_PacketIterator     pit;

  for (nd_pit_init(&pit, packet->trace, TRUE); nd_pit_get(&pit); nd_pit_next(&pit))
    {
      tcphdr = (struct tcphdr *) nd_packet_get_data(nd_pit_get(&pit), nd_tcp_get(), 0);
      if (!tcphdr)
	continue;
  
      tcphdr->th_flags = tcphdr->th_flags ^ TH_SYN;
      nd_packet_modified_at_index(nd_pit_get(&pit), nd_pit_get_index(&pit));
    }

  return;
  TOUCH(header);
  TOUCH(data);
}


void    
nd_tcp_f_cb(ND_Packet   *packet,
	    guchar      *header,
	    guchar      *data)
{
  struct tcphdr        *tcphdr;
  ND_PacketIterator     pit;

  for (nd_pit_init(&pit, packet->trace, TRUE); nd_pit_get(&pit); nd_pit_next(&pit))
    {
      tcphdr = (struct tcphdr *) nd_packet_get_data(nd_pit_get(&pit), nd_tcp_get(), 0);
      if (!tcphdr)
	continue;
  
      tcphdr->th_flags = tcphdr->th_flags ^ TH_FIN;
      nd_packet_modified_at_index(nd_pit_get(&pit), nd_pit_get_index(&pit));
    }

  return;
  TOUCH(header);
  TOUCH(data);
}


void    
nd_tcp_win_cb(ND_Packet   *packet,
	      guchar      *header,
	      guchar      *data)
{
  static ND_ProtoCallbackData cb_data;
  struct tcphdr *tcphdr;

  tcphdr = (struct tcphdr *) header;
  nd_proto_cb_data_init(&cb_data, nd_tcp_get(), data, data-header, FALSE);

  nd_dialog_number(_("Enter window size:"),
		   ND_BASE_DEC,
		   ntohs(tcphdr->th_win), 65535,
		   nd_proto_iterate_16bit_cb,
		   NULL, packet, &cb_data);
}


void    
nd_tcp_cksum_cb(ND_Packet   *packet,
		guchar      *header,
		guchar      *data)
{
  static ND_ProtoCallbackData cb_data;
  struct tcphdr *tcphdr;
  
  tcphdr = (struct tcphdr *) header;
  nd_proto_cb_data_init(&cb_data, nd_tcp_get(), data, data-header, FALSE);
  
  nd_dialog_number(_("Enter checksum value:"),
		   ND_BASE_HEX,
		   ntohs(tcphdr->th_sum), 65535,
		   nd_proto_iterate_16bit_cb,
		   NULL, packet, &cb_data);
}


void    
nd_tcp_urp_cb(ND_Packet   *packet,
	      guchar      *header,
	      guchar      *data)
{
  static ND_ProtoCallbackData cb_data;
  struct tcphdr *tcphdr;
  
  tcphdr = (struct tcphdr *) header;
  nd_proto_cb_data_init(&cb_data, nd_tcp_get(), data, data-header, FALSE);

  nd_dialog_number(_("Enter urgent pointer value:"),
		   ND_BASE_DEC,
		   ntohs(tcphdr->th_urp), 65535,
		   nd_proto_iterate_16bit_cb,
		   NULL, packet, &cb_data);
}

void    
nd_tcp_cksum_fix_cb(ND_Packet   *packet,
		    guchar      *header,
		    int          value)
{
  guint16               correct_sum;
  struct tcphdr        *tcphdr;
  ND_PacketIterator     pit;

  for (nd_pit_init(&pit, packet->trace, TRUE); nd_pit_get(&pit); nd_pit_next(&pit))
    {
      tcphdr = (struct tcphdr *) nd_packet_get_data(nd_pit_get(&pit), nd_tcp_get(), 0);
      if (!tcphdr)
	continue;
      
      if (!nd_tcp_csum_correct(packet, &correct_sum))
	{
	  tcphdr->th_sum = correct_sum;
	  nd_packet_modified_at_index(nd_pit_get(&pit), nd_pit_get_index(&pit));
	}
    }

  return;
  TOUCH(value);
  TOUCH(header);
}
