/* $Id: cliques.h,v 1.24 2004/04/19 03:54:50 graziano Exp $ */

#ifndef CLIQUES_H
#define CLIQUES_H

/*
** A "clique" comprises a set of hosts (clique members) running a particular
** set of experiments.  Presently, only sensors participate in cliques and only
** interhost experiments (e.g., bandwidth) are managed through cliques, but
** there's no particular reason that the concept couldn't be extended to
** include non-sensor hosts (e.g., memory) or single-host experiments.
*/


#include <stddef.h>		/* offsetof() */
#include "nws_messages.h"	/* messages ID */
#include "messages.h"		/* DataDescriptor */

#ifdef __cplusplus 
extern "C" {
#endif

#define MAX_CLIQUE_NAME_SIZE (31 + 1)
#define MAX_MEMBERS          (100)
#define MAX_OPTIONS_SIZE     (63 + 1)
#define MAX_SKILL_SIZE       (31 + 1)

/*
** A clique token passed between member hosts.  One host generates the token in
** response to a call to StartCliqueActivity(); from then on, the token is
** passed round-robin between members as part of a CLIQUE_TOKEN_FWD message.
**
** #name# is the name of the clique registered with the NWS name server.
** #whenGenerated# and #instance# may be thought of as the major and minor
**   version numbers of the token.  #whenGenerated# holds the time when
**   the token was spawned, #instance# a sequential update count.  When
**   presented with contradictory tokens, hosts will discard the older.
** #skill# is an identifier describing the experiment(s) being performed by
**   the clique.
** #options# contains any skill-specific configuration options.
** #period# indicates how often the clique will be activated on a particular
**   member host.  Clique activation causes the member host to conduct the
**   experiment(s) associated with the clique with each other member host.
** #timeOut# holds the amount of time that should be allowed to lapse before a
**   clique token should be considered lost.  The first clique member who
**   detects a time-out on the clique will generate a new token.
** #count# contains the number of member hosts participating in the clique;
**   i.e., the number of used elements in the #members# field.
** #leader# is the member index of the clique member who initiates passing the
**   token.
** #hosts# contains the ip address and port number of each clique member host.
*/
typedef struct {
  unsigned long address;
  unsigned long port;
} CliqueMember;
static const DataDescriptor cliqueMemberDescriptor[] =
  {SIMPLE_MEMBER(UNSIGNED_LONG_TYPE, 1, offsetof(CliqueMember, address)),
   SIMPLE_MEMBER(UNSIGNED_LONG_TYPE, 1, offsetof(CliqueMember, port))};
#define cliqueMemberDescriptorLength (2)


typedef struct {
	char name[MAX_CLIQUE_NAME_SIZE];
	double whenGenerated;
	double instance;
	char skill[MAX_SKILL_SIZE];
	char options[MAX_OPTIONS_SIZE];
	double period;
	double timeOut;
	CliqueMember members[MAX_MEMBERS];
	unsigned int count;
	unsigned int leader;
} Clique;
static const DataDescriptor cliqueDescriptor[] =
  {SIMPLE_MEMBER(CHAR_TYPE, MAX_CLIQUE_NAME_SIZE, offsetof(Clique, name)),
   SIMPLE_MEMBER(DOUBLE_TYPE, 1, offsetof(Clique, whenGenerated)),
   SIMPLE_MEMBER(DOUBLE_TYPE, 1, offsetof(Clique, instance)),
   SIMPLE_MEMBER(CHAR_TYPE, MAX_SKILL_SIZE, offsetof(Clique, skill)),
   SIMPLE_MEMBER(CHAR_TYPE, MAX_OPTIONS_SIZE, offsetof(Clique, options)),
   SIMPLE_MEMBER(DOUBLE_TYPE, 1, offsetof(Clique, period)),
   SIMPLE_MEMBER(DOUBLE_TYPE, 1, offsetof(Clique, timeOut)),
   {STRUCT_TYPE, MAX_MEMBERS, offsetof(Clique, members),
    (DataDescriptor *)&cliqueMemberDescriptor, cliqueMemberDescriptorLength,
    PAD_BYTES(CliqueMember, port, unsigned long, 1)},
   SIMPLE_MEMBER(UNSIGNED_INT_TYPE, 1, offsetof(Clique, count)),
   SIMPLE_MEMBER(UNSIGNED_INT_TYPE, 1, offsetof(Clique, leader))};
#define cliqueDescriptorLength (10)

#ifdef __cplusplus 
}
#endif

#endif
