/*
 * rtp-record.cc --
 *
 *      RTP Recorder for archive
 *
 * Copyright (c) 1997-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * @(#) $Header: /usr/mash/src/repository/mash/mash-1/archive/rtp-record.cc,v 1.34 2002/02/03 03:09:26 lim Exp $
 */


#include "source.h"
#include "tclcl.h"
#include "archive/rtp-record.h"
#include "archive/archive-file.h"
#include "archive/archive-stream.h"
#include "misc/mtrace.h"
#include "misc/nethost.h"

#ifndef WIN32
#include "sys/time.h"
#else
#include "time.h"
#endif

#ifdef sun
#include <limits.h>
#endif

#include "archive/rtp-play.h"
#include "rtp/ntp-time.h"


char *safe_strncpy(char *dest, const char *src, size_t n)
{
  dest[n-1] = '\0';
  return strncpy(dest, src, n-1);
}

// Rcvr is attached to stream and vice-versa in rtp-record-agent.tcl


DEFINE_OTCL_CLASS(RTPRecordRcvr, "Module/RTPRecord") {
	INSTPROC(attach);

}

DEFINE_OTCL_CLASS(RTCPRecordRcvr, "Module/RTPRecordCtrl") {
	INSTPROC(attach);

}

DEFINE_OTCL_CLASS(RTPRecordStream, "ArchiveStream/Record/RTP") {
	INSTPROC(attach);
	INSTPROC(init_file_header);
	INSTPROC(write_headers);
	INSTPROC(source);
	INSTPROC(set_trigger);
	INSTPROC(error_rate);
	INSTPROC(ssrc);
}


RTPRecordRcvr::RTPRecordRcvr() {


}


RTPRecordRcvr::~RTPRecordRcvr() {

}
int RTPRecordRcvr::attach(int argc, const char * const *argv)
{
	TclObject *stream;

	BEGIN_PARSE_ARGS(argc, argv);
	ARG(stream);
	END_PARSE_ARGS;

	MTrace(trcArchive, ("inside RTPRecordStream::attach"));
	stream_=(RTPRecordStream *)stream;
	return TCL_OK;
}


void RTPRecordRcvr::recv(pktbuf* pb)
{

	if (stream_!=NULL) stream_->Record(pb);
}

/**********************************************************/

RTCPRecordRcvr::RTCPRecordRcvr() {


}


RTCPRecordRcvr::~RTCPRecordRcvr() {

}

int RTCPRecordRcvr::attach(int argc, const char * const *argv)
{
	TclObject *stream;

	BEGIN_PARSE_ARGS(argc, argv);
	ARG(stream);
	END_PARSE_ARGS;

	MTrace(trcArchive, ("inside RTCPRecordStream::attach"));
	stream_=(RTPRecordStream *)stream;
	return TCL_OK;
}

void RTCPRecordRcvr::recv(pktbuf* pb)
{
	rtcphdr* rh = (rtcphdr*)pb->data;
	//int cc = 2 * RTP_MTU; // ???
	int cc = pb->len;
	//printf("cc=%d size=%d\n", cc, sizeof(pktbuf));

	//int layer = pb->layer;

	u_char* epack = (u_char*)rh + cc;
	//printf("record cc=%d layer=%d\n",cc, layer);
	while ((u_char*)rh < epack) {
		u_int len = (ntohs(rh->rh_len) << 2) + 4;
		u_char* ep = (u_char*)rh + len;
		if (ep > epack) {
			return;
		}
		u_int flags = ntohs(rh->rh_flags);
		if (flags >> 14 != RTP_VERSION) {
			return;
		}
		//printf("%d\n", flags & 0xff);
		//all* nall = (all*)pb->data;
		int num_rr;
		switch (flags & 0xff) {
		case RTCP_PT_SR:
			num_rr= ((len/4)-7)/6;
			if (stream_!=NULL) stream_->Record_sr((rtcp_sr*)(rh+1), rh->rh_ssrc, num_rr);

			//if (stream_!=NULL) stream_->Record_sr(pb);
			break;
		case RTCP_PT_SDES:
			if (stream_!=NULL) stream_->Record_sdes();
			break;

		case RTCP_PT_RR:
			num_rr = ((len/4)-2)/6;
			if (stream_!=NULL) stream_->Record_rr((rtcp_rr*)(rh+1), num_rr);
			break;
		default:
			break;
		}
		rh = (rtcphdr*)ep;
	}
}



/*******************************************************************/

RTPRecordStream::RTPRecordStream()
	: rcvr_(NULL), crcvr_(NULL), firstTime_(TRUE), source_(NULL),
	  bytesWritten_(0), first_rtp_(0), first_ntp_(0), sent_pkts_(50),
	  lost_pkts_(0), above_below_(-1), lastNotify_(0.0)
{
}



RTPRecordStream::~RTPRecordStream()
{

	//printf("In RTPRecordStream destructor\n");
	WriteHeaders();
	ArchiveFile* file = DataFile_();

	ArchiveFile* ifile = IndexFile_();

	delete file;
	delete ifile;
}

int RTPRecordStream::attach(int argc, const char * const *argv)
{
	TclObject *rcvr;
	TclObject *crcvr;

	BEGIN_PARSE_ARGS(argc, argv);
	ARG(rcvr);
	ARG(crcvr);
	END_PARSE_ARGS;

	MTrace(trcArchive, ("inside RTPRecordStream::attach"));
	rcvr_=(RTPRecordRcvr *)rcvr;
	crcvr_=(RTCPRecordRcvr *)crcvr;
	return TCL_OK;
}

int RTPRecordStream::source(int argc, const char * const *argv)
{
	TclObject *source;
	BEGIN_PARSE_ARGS(argc, argv);
	ARG(source);
	END_PARSE_ARGS;

	source_=(Source *)source;
	return TCL_OK;
}

// error_rate should be a number from 0 to 100, representing a
// percentage error rate.  above_below indicates whether the notification
// should come when we're above or below that rate.  0 for below, 1 for above.
int RTPRecordStream::set_trigger(int argc, const char * const *argv)
{
	int trigger_rate, above_below;
	BEGIN_PARSE_ARGS(argc, argv);
	ARG(trigger_rate);
	ARG(above_below);
	END_PARSE_ARGS;

	trigger_rate_=(int) trigger_rate;
	above_below_=(int) above_below;
	return TCL_OK;
}

// Should only be called by aries
int RTPRecordStream::error_rate(int argc, const char * const *argv)
{
	Tcl& tcl = Tcl::instance();

	BEGIN_PARSE_ARGS(argc, argv);
	END_PARSE_ARGS;
	//printf("error_rate_=%f\n", error_rate_);
	tcl.resultf("%f", error_rate_);
	//error_rate_ = 0;
	//lost_pkts_ = 0;
	//sent_pkts_ =0;
	return TCL_OK;
}


int RTPRecordStream::ssrc(int argc, const char * const *argv)
{
	Tcl& tcl = Tcl::instance();

	BEGIN_PARSE_ARGS(argc, argv);
	END_PARSE_ARGS;
	//printf("ssrc=%u\n", privatehdr_.ssrc);
	tcl.resultf("%u", privatehdr_.ssrc);
	return TCL_OK;
}


int
RTPRecordStream::init_file_header(int argc, const char * const *argv)
{
	BEGIN_PARSE_ARGS(argc,argv);
	END_PARSE_ARGS;

	memset(&hdr_, 0, sizeof(FileHeader));
	if (Invoke("media", NULL)==TCL_ERROR) {
		MTrace(trcArchive, ("Cannot determine media of stream %s",
		       name()));
		strcpy(hdr_.media, "unknown");
	} else {
		strcpy(hdr_.media, Tcl::instance().result());
	}

	strcpy(hdr_.protocol, PROTO_RTP);
	strcpy(hdr_.cname, "");
	strcpy(hdr_.name,  "");

	hdr_.start_sec = hdr_.start_usec = 0;
	hdr_.end_sec   = hdr_.end_usec   = 0;
	hdr_.privateLen = sizeof(RTPprivatehdr);

	memset(&privatehdr_, 0, sizeof(RTPprivatehdr));
	if (strcmp(hdr_.media, "video")==0) {
		privatehdr_.scale = 90000;
	} else if (strcmp(hdr_.media, "audio")==0) {
		privatehdr_.scale = 8000;
	} else {
		// huh?
		privatehdr_.scale = 90000;
	}
	return TCL_OK;
}



int
RTPRecordStream::WriteHeaders()
{
	FileHeader hdrNet;
	RTPprivatehdr privateHdrNet;
	ArchiveFile *file;

	host2net(hdr_, hdrNet);
	MTrace(trcArchive, ("ssrcHost %u", privatehdr_.ssrc));
	MTrace(trcArchive, ("scaleHost %u", privatehdr_.scale));
	host2net(privatehdr_, privateHdrNet);
	MTrace(trcArchive, ("ssrcNet %u", privateHdrNet.ssrc));
	MTrace(trcArchive, ("scaleNet %u", privateHdrNet.scale));

	hdrNet.privateLen = host2net((u_int32_t) sizeof(RTPprivatehdr));

	strcpy(hdrNet.version, DATA_VERSION);
	file = DataFile_();
	if (file!=NULL) {
		if (file->Write(&hdrNet, (Byte*)&privateHdrNet) < 0)
			return TCL_ERROR;
	}

	strcpy(hdrNet.version, INDEX_VERSION);
	hdrNet.privateLen = host2net((u_int32_t) 0);
	file = IndexFile_();
	if (file!=NULL) {
		if (file->Write(&hdrNet) < 0) return TCL_ERROR;
	}

	return TCL_OK;
}



int RTPRecordStream::write_headers(int argc, const char * const *argv)
{
	BEGIN_PARSE_ARGS(argc, argv);
	END_PARSE_ARGS;
	return WriteHeaders();
}


/***********************************************************/

void RTPRecordStream::parse_rr(rtcp_rr* rr, int num)
{
	int i;


	//Source *s;

	for(i=0;i<num; i++) {
		//u_int32_t ssrc = rr->rr_srcid;
		//s = sm_->consult(ssrc);  // ?
		//printf("recvr name = %s\n", s->sdes(RTCP_SDES_CNAME));
		rr=(rtcp_rr*)(rr+1);

	}


}


void RTPRecordStream::Record_rr(rtcp_rr* rr, int num)


{
	//Source *ps = source_;
	int x;

	for (x=0; x<num; x++) {
		//u_int32_t loss=net2host(rr->rr_loss);
		//u_int32_t loss=ntohl(rr->rr_loss & 0x00ffffff);

		//u_int32_t cum_loss=loss & 0x00ffffff;
		//u_int32_t frac_loss= ((loss>>24) & 0x000000ff);

		//u_int32_t high_seq=net2host(rr->rr_ehsr);

		//printf("RR Sender name=%s ssrc=%u pkts lost= %u frac_loss=%u high_seq=%u\n", ps->sdes(RTCP_SDES_NAME), rr->rr_srcid, cum_loss, frac_loss, high_seq);
		rr=(rtcp_rr*)(rr+1);
	}

}


void RTPRecordStream::Record_sr(rtcp_sr* sr, u_int32_t /* ssrc */, int num)
{

	//rtcp_sr* sr = (rtcp_sr*)(rh + 1);

	//printf("cname=%s myssrc=%u ssrc=%u\n", source_->sdes(RTCP_SDES_CNAME), source_->srcid(), ssrc);

	//printf("sender report packet count=%u bytes=%u\n", sr->sr_np, sr->sr_nb);

	rtcp_rr *rr;
	rr = (rtcp_rr*)(sr+1);
	Record_rr(rr, num);
	//parse_rr(rr,1);




	//Doesn't work
	//u_int32_t rtp_time = net2host(sr->sr_ts);
	//timeval ntp;
	//ntp.tv_sec = ntohl(sr->sr_ntp.upper);
	//ntp.tv_usec = ntohl(sr->sr_ntp.lower);

	//MTrace(trcArchive, ("Record_sr %u %u ",ntp.tv_sec , ntp.tv_usec));

	//double ntp_time = (ntohl(sr->sr_ntp.upper) + (ntohl(sr->sr_ntp.lower) / 1000000.0));

	//u_int32_t i_ntp = (( ntohl(sr->sr_ntp.upper) << 16) | (ntohl(sr->sr_ntp.lower) >> 16));

	//double d_ntp = i_ntp * 65536.0 / 1000000.0;

	//u_int32_t i_ntp = ntptime(ntp);

	//MTrace(trcArchive, ("%u %f", i_ntp, d_ntp));
	//if (first_rtp_==0) {
	//	first_rtp_ = rtp_time;
	//	first_ntp_ = d_ntp;
	//}
	//else {
	//	double scale = (i_ntp - first_ntp_) /
	//		(rtp_time - first_rtp_);
	//	MTrace(trcArchive, ("Record_sr %f - %f / %u - %u = %f",
	//		    d_ntp, first_ntp_, rtp_time,
	//		    first_rtp_, scale));

	//}

}


/*
 * parse a list of sdes items
 * len is the number of 'chunks'
 */
void RTPRecordStream::Record_sdes()
{
	const char *cname, *name, *email, *phone, *loc, *tool;
	Bool changed=FALSE;

	MTrace(trcArchive, ("Record_sdes"));
	cname  = source_->sdes(RTCP_SDES_CNAME);
	if (cname !=0 ) {
		if (strcmp(cname, (const char *) &(hdr_.cname))!=0){
			safe_strncpy(hdr_.cname, cname, sizeof(hdr_.cname));
			changed=TRUE;
		}
	}

	name = source_->sdes(RTCP_SDES_NAME);
	if (name!=0) {

		if (strcmp(name, (const char *) &hdr_.name)!=0) {
			safe_strncpy(hdr_.name, name, sizeof(hdr_.name));
			Invoke("notify_observers", "name", name, NULL);
			changed=TRUE;
		}
	}

	email = source_->sdes(RTCP_SDES_EMAIL);
	if (email != 0) {
		if (strcmp(email, (const char *)&privatehdr_.email)!=0) {
			safe_strncpy(privatehdr_.email, email,
				     sizeof(privatehdr_.email));
			changed=TRUE;
		}
	}

	phone = source_->sdes(RTCP_SDES_PHONE);
	if (phone != 0) {
		if (strcmp(phone, (const char *)&privatehdr_.phone)!=0) {
			safe_strncpy(privatehdr_.phone, email,
				     sizeof(privatehdr_.phone));
			changed=TRUE;
		}
	}

	loc = source_->sdes(RTCP_SDES_LOC);
	if (loc != 0) {
		if (strcmp(loc, (const char *)&privatehdr_.loc)!=0) {
			safe_strncpy(privatehdr_.loc, loc, sizeof(privatehdr_.loc));
		char c_ssrc[16];
		sprintf(c_ssrc, "%u", privatehdr_.ssrc);
		Invoke("notify_observers", "loc", loc, c_ssrc, NULL);
			changed=TRUE;
		}
	}

	tool = source_->sdes(RTCP_SDES_TOOL);
	if (tool != 0) {
		if (strcmp(tool, (const char *)&privatehdr_.tool)!=0) {
			safe_strncpy(privatehdr_.tool, tool, sizeof(privatehdr_.tool));
			changed=TRUE;
		}
	}

	if (changed) {
		WriteHeaders();
	}
}

/***********************************************************/


int RTPRecordStream::Record(pktbuf* pb)
{
	u_char buffer[16];
	DataFile *dataFile;
	IndexFile *indexFile;


	rtphdr* rh = (rtphdr*)pb->data;


	dataFile = DataFile_();
	indexFile = IndexFile_();


	struct IndexRecord idxHost, idxNet;

	timeval now;
	gettimeofday(&now, NULL);

	u_int32_t ts = net2host(rh->rh_ts);
	timeval logical_time;

	if (firstTime_) {
		privatehdr_.ref_rtp = ts;
		privatehdr_.ref_tv_sec  = now.tv_sec;
		privatehdr_.ref_tv_usec = now.tv_usec;
		privatehdr_.ssrc = rh->rh_ssrc;
		logical_time = rtp2logical(ts, privatehdr_.scale,
					   privatehdr_.ref_rtp, now);
		hdr_.start_sec = logical_time.tv_sec;
		hdr_.start_usec = logical_time.tv_usec;
		// FIXME: must check the return value
		WriteHeaders();
		SeekToStartOfData();
		firstTime_=FALSE;
		last_seqno_=(net2host(rh->rh_seqno))-1;
		MTrace(trcArchive, ("Wrote start header scale=%d %u.%u",
				    privatehdr_.scale, hdr_.start_sec,
				    hdr_.start_usec));
	} else {
		timeval ref_tv = { privatehdr_.ref_tv_sec,
				   privatehdr_.ref_tv_usec };
		logical_time = rtp2logical(ts, privatehdr_.scale,
					   privatehdr_.ref_rtp,
					   ref_tv);
	}
	int cur_seqno = net2host(rh->rh_seqno);
	//double r = random() / double(INT_MAX);
	//printf("r=%f\n", r);
	//if (r > .5) {
	MTrace(trcArchive|trcVerbose,
	       ("Index rec %u %u", logical_time.tv_sec, logical_time.tv_usec));
	idxHost.recvTS_sec = now.tv_sec;
	idxHost.recvTS_usec= now.tv_usec;
	idxHost.sentTS_sec = logical_time.tv_sec;
	idxHost.sentTS_usec= logical_time.tv_usec;
	idxHost.seqno      = net2host(rh->rh_seqno);
	idxHost.filePointer= dataFile->Tell();

	hdr_.end_sec = logical_time.tv_sec;
	hdr_.end_usec = logical_time.tv_usec;

	host2net(idxHost, idxNet);
	if (indexFile->Write(&idxNet) < 0) {
		MTrace(trcArchive, ("Error writing to index file"));
			return FALSE;
	}

	recordhdr* hdr= (recordhdr *)&buffer[0];
	u_int temp = pb->len;
	u_int temp2 = host2net(temp);
	hdr->len = temp2;
	hdr->type = 0;
	hdr->d2 = 0;

	if (dataFile->Write((u_char *)hdr, sizeof(recordhdr)) < 0) {
		MTrace(trcArchive, ("Error writing pkt header to data file"));
		return FALSE;
	}
	if (dataFile->Write(pb->data, pb->len) < 0) {
		MTrace(trcArchive, ("Error writing packet to data file"));
		return FALSE;
	}

	// notify all observers
	bytesWritten_ += sizeof(recordhdr) + pb->len;
	double fnow;
	fnow = tvtof(now);
	if (fnow - lastNotify_ > 0.5) {
		Invokef("notify_observers bytes_rcvd %lu",
			(unsigned long) bytesWritten_);
		lastNotify_ = fnow;
	}

	int i, missing;
	missing=0;
	for (i=last_seqno_ +1 ; i<cur_seqno ; i++ ) {
		missing++;
	//Invokef("notify_observers report_missing %u %d",
	//		(unsigned) (net2host(rh->rh_ssrc)), i);
	}

	sent_pkts_++;
	lost_pkts_ += missing;
	/// FIX THIS
	error_rate_ = ((double)lost_pkts_/(double)(lost_pkts_+sent_pkts_)*100);
        //printf("er=%f seqno=%u lost=%d sent= %d \n", error_rate_, cur_seqno, lost_pkts_, sent_pkts_);
	if (above_below_ == 0 ) {
		if (error_rate_ < trigger_rate_) {
			Invokef("notify_observers below_thresh %u ", (unsigned (net2host(rh->rh_ssrc))));
		}
	} else if (above_below_ == 1) {
		if (error_rate_ > trigger_rate_) {
			Invokef("notify_observers above_thresh %u ", (unsigned (net2host(rh->rh_ssrc))));
		}
	}
	if (sent_pkts_ >= 200) {
		sent_pkts_=100;
		lost_pkts_=lost_pkts_ / 2;
	}
	last_seqno_ = cur_seqno;
	timeval temptime;
	gettimeofday(&temptime, NULL);
	MTrace(trcArchive|trcVerbose, ("Wrote %u %u %u %u", pb->len,
				    ts,temptime.tv_sec, temptime.tv_usec ));
	//}
	pb->release();
	return TCL_OK;
}


int
RTPRecordStream::SeekToStartOfData()
{
	ArchiveFile *file;
	file = DataFile_();
	if (file!=NULL) {
		if (file->Seek(sizeof(FileHeader) + sizeof(RTPprivatehdr),
			       SEEK_SET)==TCL_ERROR)
			return TCL_ERROR;
	}

	file = IndexFile_();
	if (file!=NULL) {
		if (file->Seek(sizeof(FileHeader), SEEK_SET)==TCL_ERROR)
			return TCL_ERROR;
	}

	return TCL_OK;
}

