/*
 * mb-cmd.h --
 *
 *      header file for commands in mb
 *
 * Copyright (c) 1996-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * $Header: /usr/mash/src/repository/mash/mash-1/mb/mb-cmd.h,v 1.17 2002/02/03 03:16:30 lim Exp $
 */

#ifndef MBCMD_H_
#define MBCMD_H_

#include "mb/mb-obj.h"
#include "mb/mb-file.h"
#include "mb/mb-items.h"
#include "mb/mb-page.h"


#define MBUI

typedef enum {
	CNop = 0,
	CCreateItem,                // 1 MBCmdCreate
	CGroup,                     // 2 MBCmdGroup
	CDel,                       // 3 MBCmdDel
	CCoord,                     // 4 MBCmdCoord  #not used!!! not idempotent
	CMove,                      // 5 MBCmdMove
	CDup,                       // 6 MBCmdDup
	CPgName,                    // 7 MBCmdPgName
	CChar,                      // 8 MBCmdChar
	CFrag,                      // 9 MBCmdFrag   #fragment
	CInvalid                    // 10 *invalid*, used as stopper
} MBCmdType;

extern const char* achCmdNames[];

//
// Generic command that relates to a page. Similar to drawops in wb
//
class MBCmd {
protected:
	MBCmd(MBCmdType cmdType, n_long timestamp)
		: cmdType_(cmdType), seqno_(0), timestamp_(timestamp),
		  pRcvr_(0) {}

public:
	MBCmd(MBCmdType cmdType) : cmdType_(cmdType), seqno_(0),
			timestamp_(0), pRcvr_(0) {}
	virtual ~MBCmd() {}

	static MBCmd* Create(ulong seqno, Byte* pb);

#ifdef MBUI
	virtual int executeInteractive(Page* /*pPage*/) {
		disperr("Executing non-interactive function interactively!");
		return MB_EXE_ERROR;
	}
	virtual int execute(Page* pPage, const MBTime& currTime,
			    const MBTime& targetTime);
	virtual int apply(Page* pPage, const MBTime& t)=0;
	virtual int reverse(Page* pPage, const MBTime& t)=0;

protected: /* convenience functions */
	int errorDup(Page* pPage);
	int removeAssocItem(Page* pPage, int reportErr=1);

	virtual int Instantiate(Page* pPage) {
		// will set the new seqno if it is not set.
		if (!seqno_) seqno_ = pPage->getNewSeqno();
		return MB_EXE_OK;
	}
#endif
public:
	// returns 0 if it is completed, or the first
	virtual ulong Incomplete(MBPageObject* /*pPage*/) const { return 0; }
	virtual Byte* Packetize(Byte* pb);
	virtual short getPacketLen() { return sizeof(Pkt_CmdHdr); }
	ulong  getSeqno() const { return seqno_; }
	virtual Byte* Extract(Byte* /*pb*/)  // use create to get whole command
		{ return (Byte*)NULL; }

	void SetTimeStamp(ulong timestamp) { timestamp_ = timestamp; }
	virtual ulong getTimeStamp() { return timestamp_; }
	virtual ulong startTime() { return timestamp_; }
	virtual ulong endTime() { return cMBTimePosInf; }
	// Returns nothing by default
	// Other commands will return the id # of the item
	// that the command is associated with.
	virtual ulong getItemId() { return 0; }
	// Returns nothing by default.
	// Commands that has a copy of the pageitem will return the item.
	// This is to facilitate transformations (e.g. move) of items.
	virtual const PageItem* getItem() const { return NULL; }
	MBCmdType getType() { return cmdType_; }

	// returns whether the command produce any visual effect
	// this is useful to determine if we need to switch over to
	// a active page
	virtual Bool HasVisualEffect() { return TRUE; }

	void rcvr(MBBaseRcvr* pRcvr) { pRcvr_ = pRcvr; }
	MBBaseRcvr* rcvr() { return pRcvr_; }
	virtual Bool activeAt(const MBTime& t) {
		return (t == cMBTimeAny ||
			(t >= startTime() && t <= endTime()));
	}

#ifdef MB_DEBUG
	void DumpTS(Tcl_Obj* pObj);
#endif

private:
	MBCmdType cmdType_;
	ulong seqno_;
	ulong timestamp_;
	MBBaseRcvr* pRcvr_; /* receiver that generated the command */
};



//
// creation of an item
//
class MBCmdCreate: public MBCmd {
public:
	MBCmdCreate(n_long timestamp, PageItem *pItem)
		: MBCmd(CCreateItem,timestamp), pItem_(pItem)
		{ assert(pItem_!=0); }

	MBCmdCreate() : MBCmd(CCreateItem), pItem_(0) { }

	virtual ~MBCmdCreate() { delete pItem_; }

#ifdef MBUI
	virtual int apply(Page* pPage, const MBTime& t);
	virtual int reverse(Page* pPage, const MBTime& /*t*/) {
		return removeAssocItem(pPage, 0);
	}
#endif

	// the item the command refers to is identified by the seq no
	virtual n_long getItemId() { return getSeqno(); }
	virtual const PageItem* getItem() const { return pItem_; }

	// returns 0 if it is completed, or the first
	virtual n_long Incomplete(MBPageObject* /*pPage*/) const { return 0; }

	// packetization
	virtual Byte* Extract(Byte* pb);
	virtual Byte* Packetize(Byte* pb);
	virtual short getPacketLen() {
		short sz=sizeof(Pkt_CmdCreate)+pItem_->getPacketLen(TRUE);
		return (sz+MBCmd::getPacketLen());
	}

	// creation of text produce no effects for now...
	virtual Bool HasVisualEffect() {
		return pItem_->getType()==PgItemText? FALSE : TRUE;
	}

private:
	PageItem *pItem_;
};


//
// Creation of a group from a range
//
class MBCmdGroup: public MBCmd {
public:
	MBCmdGroup(n_long timestamp, PageItemType groupType,
		   n_long snStart, n_long snEnd, PageItem *pItem)
		: MBCmd(CGroup,timestamp), groupType_(groupType),
		  pItem_(pItem), snStart_(snStart), snEnd_(snEnd) { }

	static MBCmdGroup* Create(n_long timestamp, PageItemType groupType,
				  n_long snStart, n_long snEnd,
				  PageItem *pItem, int argc=0,
				  const char*const* argv= 0);

	MBCmdGroup() : MBCmd(CGroup), pItem_(NULL), snStart_(0), snEnd_(0) { }

	// returns nothing by default
	virtual ~MBCmdGroup() { delete pItem_; }

#ifdef MBUI
	virtual int apply(Page* pPage, const MBTime& t);
	virtual int reverse(Page* pPage, const MBTime& t);
	PageItemType getGroupType() { return groupType_; }
#endif

	virtual const PageItem* getItem() const { return pItem_; }

	// the item the command refers to is identified by the seq no
	virtual n_long getItemId() { return getSeqno(); }

	// Returns 0 if it is completed, or the first item that we depend on
	virtual n_long Incomplete(MBPageObject* pPage) const;

	// packetization
	static MBCmdGroup* Create(Byte* pb);
	virtual Byte* Extract(Byte* pb);
	virtual Byte* Packetize(Byte* pb);
	virtual short getPacketLen() {
		return (sizeof(Pkt_CmdGroup)+MBCmd::getPacketLen());
	}
	// creation of group has no visual effect
	virtual Bool HasVisualEffect() { return FALSE; }

	n_long snStart() { return snStart_; }
	n_long snEnd() { return snEnd_; }
protected:
	// note: we need the (redundant) grouptype_ 'cos pItem_ is not
	// packetized
	PageItemType groupType_;
	PageItem *pItem_;
	n_long snStart_, snEnd_;
};


//
// A Text Command groups a range of MBCmdChar into a text item
//
class MBCmdText: public MBCmdGroup {
public:
	MBCmdText(n_long timestamp, PageItemType groupType,
		  n_long snStart, n_long snEnd, PageItem *pItem)
		: MBCmdGroup(timestamp, groupType, snStart, snEnd, pItem) { }
	MBCmdText() : MBCmdGroup() { }

#ifdef MBUI
	virtual int apply(Page* pPage, const MBTime& t);
	virtual int reverse(Page* pPage, const MBTime& t);
#endif
};


//
// MBCmdMLine groups a range of MBCmdCreate (Lines) into
//    a multi-line item.
//
class MBCmdMLine: public MBCmdGroup {
public:
	MBCmdMLine(n_long timestamp, PageItemType groupType,
		   n_long snStart, n_long snEnd, PageItem *pItem)
		: MBCmdGroup(timestamp, groupType, snStart, snEnd, pItem) { }
	MBCmdMLine() : MBCmdGroup() { }

#ifdef MBUI
	virtual int apply(Page* pPage, const MBTime& targetTime);
	virtual int reverse(Page* pPage, const MBTime& t);
#endif
};


//
// A MBCmdImage groups a range of MBCmdFrags of the graphics into
//    an image item
//
class MBCmdImage: public MBCmdGroup {
public:
	MBCmdImage(n_long timestamp, n_long snStart, n_long snEnd,
		   PageItem *pItem, int /*argc*/, const char*const* argv)
		: MBCmdGroup(timestamp, PgItemImage, snStart, snEnd, pItem) {
		if ( (!*argv) || (*argv)[0]==cchNull) {
			SignalError(("Invalid File Name!"));
			return;
		}
		::AllocNCopy(&szFileName_, *argv);
	}
	MBCmdImage() : MBCmdGroup(), szFileName_(NULL) {
		groupType_ = PgItemImage;
	}
	virtual ~MBCmdImage() { delete[] szFileName_; }

	const char *getFileName() { return szFileName_; }
#ifdef MBUI
	virtual int apply(Page* pPage, const MBTime& targetTime);
	virtual int reverse(Page* pPage, const MBTime& /*t*/) {
		return removeAssocItem(pPage);
	}
#endif

	// Image adds the pItem_ to the packet
	virtual Byte* Packetize(Byte* pb) {
		Byte* pbCurr = MBCmdGroup::Packetize(pb);
		assert(pItem_!=NULL);
		return (pItem_->packetize(TRUE,pbCurr));
	}
	virtual Byte* Extract(Byte* pb) {
		Byte *pbCurr = MBCmdGroup::Extract(pb);
		pItem_ = PageItem::create(groupType_, TRUE, pbCurr);
		return (Byte*) NULL; // return value should not be needed
	}
	virtual short getPacketLen() {
		return MBCmdGroup::getPacketLen()+pItem_->getPacketLen(TRUE);
	}
private:
	// if the file has been assembled store file name here
	char *szFileName_;
};

//
// MBCmdPS groups a range of MBCmdFrag into a PS file
//
// Note: Postscript file is assumed to contain only a
//       Page, additional pages are ignored.
//
// identical to a generic group except for the execute function
//
class MBCmdPS: public MBCmdGroup {
public:
	MBCmdPS(n_long timestamp, n_long snStart, n_long snEnd,
		PageItem *pItem)
		: MBCmdGroup(timestamp, PgItemPS, snStart, snEnd, pItem) { }

	MBCmdPS() : MBCmdGroup() { groupType_ = PgItemPS; }
	virtual ~MBCmdPS() { }

#ifdef MBUI
	virtual int apply(Page* pPage, const MBTime& targetTime);
	virtual int reverse(Page* pPage, const MBTime& /*t*/) {
		return removeAssocItem(pPage);
	}
#endif

	// Image adds the pItem_ to the packet
	virtual Byte* Packetize(Byte* pb) {
		Byte* pbCurr = MBCmdGroup::Packetize(pb);
		assert(pItem_!=NULL);
		return (pItem_->packetize(TRUE,pbCurr));
	}
	virtual Byte* Extract(Byte* pb) {
		Byte *pbCurr = MBCmdGroup::Extract(pb);
		pItem_ = PageItem::create(groupType_, TRUE, pbCurr);
		return (Byte*) NULL; // return value should not be needed
	}
	virtual short getPacketLen() {
		return MBCmdGroup::getPacketLen()+pItem_->getPacketLen(TRUE);
	}
};


//
// Insert a character into a group
//
class MBCmdChar: public MBCmd {
public:
	MBCmdChar(n_long timestamp, u_long targetItem, u_short index, char chr)
		: MBCmd(CChar, timestamp), targetItem_(targetItem),
		  index_(index), char_(chr) { }
	MBCmdChar() : MBCmd(CChar), targetItem_(0), index_(0),
			  char_(cchNull) { }

	// returns nothing by default
	virtual ~MBCmdChar() { }


#ifdef MBUI
	virtual int apply(Page* pPage, const MBTime& targetTime);
	virtual int reverse(Page* pPage, const MBTime& targetTime);
#endif

	// the item the command refers to is identified by the seq no
	virtual n_long getItemId() { return targetItem_; }

	// Returns 0 if it is completed, or the first item that we depend on
	virtual n_long Incomplete(MBPageObject* pPage) const;

	// accessor functions
	u_int getIndex() { return index_; }
	char getChar() { return char_; }

	// packetization
	virtual Byte* Extract(Byte* pb) {
		Pkt_CmdChar *pPkt = (Pkt_CmdChar *) pb;
		targetItem_ = net2host(pPkt->targetItem);
		index_  = net2host(pPkt->index);
		char_ = pPkt->chr;
		return (Byte*) NULL; /* not used */
	}
	virtual Byte* Packetize(Byte* pb) {
		Byte* pbCurr = MBCmd::Packetize(pb);

		Pkt_CmdChar *pPkt = (Pkt_CmdChar*) pbCurr;
		pPkt->targetItem = host2net(targetItem_);
		pPkt->index = host2net(index_);
		pPkt->chr = char_;
		return (Byte *)(pPkt+1);
	}
	virtual short getPacketLen() {
		short sz=sizeof(Pkt_CmdChar);
		return (sz+MBCmd::getPacketLen());
	}

private:
	u_long targetItem_;
	u_short index_;
	char char_;
};

//
// Represents a fragment of a larger command
//   Used for: 1) files (image+text)
//
class MBCmdFrag: public MBCmd, public MBFrag {
public:
	MBCmdFrag(n_long timestamp) : MBCmd(CFrag,timestamp), MBFrag() {}
	MBCmdFrag() : MBCmd(CFrag,0), MBFrag() {}
	virtual ~MBCmdFrag() {}

#ifdef MBUI
	virtual int apply(Page* pPage, const MBTime& /*targetTime*/) {
		return  (MBCmd::Instantiate(pPage));
	}
	virtual int reverse(Page* /*pPage*/, const MBTime& /*targetTime*/) {
		return MB_EXE_OK;
	}
#endif

	// packetization
	virtual Byte* Extract(Byte* pb) {
		Pkt_CmdFrag *pPkt = (Pkt_CmdFrag *) pb;
		Read(pPkt->pData, net2host(pPkt->len));
		return (Byte*) NULL; /* not used */
	}
	virtual Byte* Packetize(Byte* pb) {
		Byte* pbCurr = MBCmd::Packetize(pb);
		Pkt_CmdFrag *pPkt = (Pkt_CmdFrag*) pbCurr;
		pPkt->len = host2net( Write(pPkt->pData) );
		return (Byte *)PKT_ROUNDUP( pbCurr + sizeof(Pkt_CmdFrag) +
					    len_*sizeof(Byte) );
	}
	virtual short getPacketLen() {
		return PKT_ROUNDUP( sizeof(Pkt_CmdFrag) + sizeof(Byte)*len_ +
				    MBCmd::getPacketLen() );
	}
};


// The delete command hides an item from view
//
// The only operation (for now) permitted on a command is undelete.
// Therefore, on initial delete, we save the ASCII version of the item
// (MBCanvItem).
// For reverse of delete, we put the item back into the canvas, and
// keep a copy of the binary version of the item (PageItem).
//
class MBCmdDel: public MBCmd {
public:
	MBCmdDel(n_long timestamp, n_long ItemId)
		: MBCmd(CDel,timestamp), itemId_(ItemId), pPageItem_(NULL) { }

	MBCmdDel(): MBCmd(CDel), itemId_(0), pPageItem_(NULL) { }
	virtual ~MBCmdDel() { delete pPageItem_; }

#ifdef MBUI
	virtual int apply(Page* pPage, const MBTime& targetTime);
	virtual int reverse(Page* pPage, const MBTime& targetTime);
#endif

	virtual n_long getItemId() { return pPageItem_ ? getSeqno() : 0; }
	virtual const PageItem* getItem() const { return pPageItem_; }

	// Returns 0 if it is completed, or the first item that we depend on
	virtual n_long Incomplete(MBPageObject* pPage) const;

	// packetization
	virtual Byte*  Extract(Byte* pb);
	virtual Byte* Packetize(Byte* pb);
	virtual short getPacketLen() {
		return (sizeof(Pkt_CmdDel)+MBCmd::getPacketLen());
	}

private:
	n_long itemId_ ;
	// stores the item if this is an undelete
	PageItem *pPageItem_;
};

#ifdef OLD
//
// Change of the coords of an item
//  REVIEW: decide whether this is needed
//
class MBCmdCoord: public MBCmd {
public:
	MBCmdCoord(n_long timestamp, n_long targetId,
		   Coord x1, Coord y1, Coord x2, Coord y2)
		: MBCmd(CCoord,timestamp), targetId_(targetId),
		  x1_(x1), y1_(y1), x2_(x2), y2_(y2)
		{ }
	MBCmdCoord() :targetId_(0), x1_(0), y1_(0), x2_(0), y2_(0)
		{ cmdType_=CCoord; }
	virtual      ~MBCmdCoord() {}
	virtual int execute(Page* pPage, const MBTime& targetTime);
	virtual int executeInteractive(Page* pPage);

	virtual n_long getItemId() { return targetId_; }
	// Returns 0 if it is completed, or the first item that we depend on
	virtual n_long Incomplete(Page* pPage) const;

	// packetization
	virtual Byte* Extract(Byte* pb);
	virtual Byte* Packetize(Byte* pb);
	virtual short GetPacketLen() {
		return (sizeof(Pkt_CmdCoord)+MBCmd::GetPacketLen());
	}

private:
	// item that the command acts on
	n_long targetId_ ;
	Coord x1_, y1_, x2_, y2_;
};
#endif


//
// Move an item
//
class MBCmdMove: public MBCmd {
public:
	MBCmdMove(n_long timestamp, n_long targetId, Coord dx, Coord dy)
		: MBCmd(CMove, timestamp), targetId_(targetId),
		  dx_(dx), dy_(dy)
		{}

	MBCmdMove() : MBCmd(CMove) {}
	virtual ~MBCmdMove() {}

#ifdef MBUI
	virtual int apply(Page* pPage, const MBTime& targetTime);
	virtual int reverse(Page* pPage, const MBTime& targetTime);

	virtual int executeInteractive(Page* pPage);
#endif

	virtual n_long getItemId() { return getSeqno(); }
	n_long getTargetId() { return targetId_; }
	Bool Apply(PageItem* pItem);

	// Returns 0 if it is completed, or the first item that we depend on
	virtual n_long Incomplete(MBPageObject* pPage) const;

	// packetization
	virtual Byte* Extract(Byte* pb);
	virtual Byte* Packetize(Byte* pb);
	virtual short getPacketLen() {
		return (sizeof(Pkt_CmdMove)+MBCmd::getPacketLen());
	}

	void getOffsets(Coord &dx, Coord &dy) { dx = dx_; dy = dy_; }
	// added by Yatin for TGMB

private:
	// item that the command acts on
	n_long targetId_ ;
	Coord dx_, dy_;
};


//
// Copy an item
//
class MBCmdDup: public MBCmd {
public:
	MBCmdDup(n_long timestamp, n_long targetId)
		: MBCmd(CDup,timestamp), targetId_(targetId), pItem_(NULL) { }
	MBCmdDup() : MBCmd(CDup,0), targetId_(0), pItem_(NULL) {
		assert(!pItem_);
	}
	virtual ~MBCmdDup() { delete pItem_; }

#ifdef MBUI
	virtual int apply(Page* pPage, const MBTime& targetTime);
	virtual int reverse(Page* pPage, const MBTime& targetTime);
#endif

	// the item the command refered to is identified by the seq no
	virtual n_long getItemId() { return getSeqno(); }
	virtual const PageItem* getItem() const { return pItem_; }

	// Returns 0 if it is completed, or the first item that we depend on
	virtual n_long Incomplete(MBPageObject* pPage) const;

	// packetization
	virtual Byte* Extract(Byte* pb);
	virtual Byte* Packetize(Byte* pb);
	virtual short getPacketLen() {
		return (sizeof(Pkt_CmdDup)+MBCmd::getPacketLen());
	}

private:
	// item that the command acts on
	n_long targetId_ ;
	PageItem* pItem_;
};


//
// Change of a page name
//      Note: Creation of a page is automatic when a new id
//      arrives
//
class MBCmdPgName: public MBCmd {
public:
	MBCmdPgName(n_long timestamp, const char *szName )
		: MBCmd(CPgName,timestamp)
		{ ::AllocNCopy(&szPgName_, szName); }
	MBCmdPgName(): MBCmd(CPgName), szPgName_(NULL) { }
	virtual      ~MBCmdPgName() { delete [] szPgName_; }

#ifdef MBUI
	virtual int apply(Page* pPage, const MBTime& targetTime);
	virtual int reverse(Page* pPage, const MBTime& targetTime);
#endif

	// packetization
	virtual Byte* Extract(Byte* pb);
	virtual Byte* Packetize(Byte* pb);
	virtual short getPacketLen();
	virtual Bool HasVisualEffect() { return FALSE; }

private:
	char *szPgName_;
};

#endif /* #ifdef MBCMD_H */
