/*
 * logger.cc --
 *
 *      This file contains the methods of the LogService class. The LogService
 *      is used as a debugging tool to keep track of when certain events are
 *      called. A global instance of the LogService named "logger" is
 *      instantiated and is intended as a well known global variable name that
 *      can be used by any part of the system. A tcl interface to this
 *      instantiation can be obtained with the OTcl command: "new LogService".
 *
 * Copyright (c) 1998-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/time.h>
#include "tclcl.h"
#include "pktbuf.h"
#include "logger.h"

/*
 * Global, well-known instantiation of LogService.
 */

LogService logger;

/*
 * Log an rtp packet. Records time of log as well as
 * standard RTP headers found in the packet (flags,
 * sequence number, timestamp, ssrc).
 */

void
LogService::log_rtp_pkt(pktbuf *pb) {
  rtphdr *rh = (rtphdr *) pb->dp;
  RTPPktEntry *entry;

  if (!on_flag_) {
    return;
  }

  entry = &(rtp_entries_[cur_rtp_record_]);
  if (cur_rtp_record_ < 8999)
    cur_rtp_record_++;

  ::gettimeofday(&(entry->log_ts_), 0);

  entry->flags_ = ntohs(rh->rh_flags);
  entry->seq_no_ = ntohs(rh->rh_seqno);
  entry->ts_ = ntohl(rh->rh_ts);
  entry->ssrc_ = ntohl(rh->rh_ssrc);

  entry->length_ = pb->len;
}

/*
 * OTcl interface to LogService.
 * Methods available include:
 *     on  -- turns logger on
 *     off -- turns logger off
 *     reset -- clears all previous entries
 *     dump <filename> -- writes log out to file
 *     log_graph_comm <in_out_flag> <delivered_flag> <mesg> -- logs control
 *                                                             message.
 */

int
LogService::command(int argc, const char*const* argv) {

  if (argc == 2) {
    if (strcmp(argv[1], "on") == 0) {
      on_flag_ = 1;
      return (TCL_OK);
    }
    if (strcmp(argv[1], "off") == 0) {
      on_flag_ = 0;
      return (TCL_OK);
    }
    if (strcmp(argv[1], "reset") == 0) {
      cur_rtp_record_ = 0;
      cur_comm_record_ = 0;
      return (TCL_OK);
    }
  }
  if (argc == 3) {
    if (strcmp(argv[1], "dump") == 0) {
      FILE *fp;

      fp = fopen(argv[2], "w");

      int rtp_cntr = 0;
      int comm_cntr = 0;

      while ((rtp_cntr < cur_rtp_record_) ||
	     (comm_cntr < cur_comm_record_)) {
	if (check_time(rtp_cntr, comm_cntr) == 1) {
	  RTPPktEntry *e = &(rtp_entries_[rtp_cntr++]);

	  /* RTP Counter wins. */

	  fprintf(fp, "%lu %lu rtp %u %u %u %u %u %d\n",
		  e->log_ts_.tv_sec,
		  e->log_ts_.tv_usec,
		  (e->flags_ & 0x7f),
		  ((e->flags_ & 0x80) >> 7),
		  e->seq_no_,
		  e->ts_,
		  e->ssrc_,
		  e->length_);
	} else {
	  GraphCommEntry *e = &(gcomm_entries_[comm_cntr++]);

	  fprintf(fp, "%lu %lu comm %u %u {%s}\n",
		  e->log_ts_.tv_sec,
		  e->log_ts_.tv_usec,
		  e->in_or_out_,
		  e->delivered_,
		  e->mesg_);
	}
      }
      fclose(fp);
      return (TCL_OK);
    }
  }
  if (argc == 5) {
    if (strcmp(argv[1], "log_graph_comm") == 0) {
      if (on_flag_) {
	GraphCommEntry *entry;
	entry = &(gcomm_entries_[cur_comm_record_]);

	if (cur_comm_record_ < 8999)
	  cur_comm_record_++;

	::gettimeofday(&(entry->log_ts_), 0);

	entry->in_or_out_ = atoi(argv[2]);
	entry->delivered_ = atoi(argv[3]);
	strncpy(entry->mesg_, argv[4], 99);
	entry->mesg_[99] = 0;
      }
      return (TCL_OK);
    }
  }

  return TclObject::command(argc, argv);
}

/*
 * Log generic string entry.
 */

void
LogService::gen_log(char *mesg)
{
  if (on_flag_) {
    GraphCommEntry *entry;
    entry = &(gcomm_entries_[cur_comm_record_]);

    if (cur_comm_record_ < 8999)
      cur_comm_record_++;

    ::gettimeofday(&(entry->log_ts_), 0);

    entry->in_or_out_ = 1;
    entry->delivered_ = 1;
    strncpy(entry->mesg_, mesg, 99);
    entry->mesg_[99] = 0;
  }
}

/*
 * Compares timestamps from rtp log and control log to
 * determine which came first. Used in dumping log to disk.
 */

int
LogService::check_time(int rtp_index, int comm_index)
{
  /* Assumes that at least one of two is valid. */

  /* Returns 1 if rtp is valid and earlier, otherwise returns 0 */

  if (comm_index >= cur_comm_record_) {
    return 1;
  }
  if (rtp_index >= cur_rtp_record_) {
    return 0;
  }

  /* Both valid, must check time */

  long rtp_sec, rtp_usec, comm_sec, comm_usec;

  rtp_sec = rtp_entries_[rtp_index].log_ts_.tv_sec;
  comm_sec = gcomm_entries_[comm_index].log_ts_.tv_sec;

  if (rtp_sec < comm_sec) {
    return 1;
  }
  if (rtp_sec > comm_sec) {
    return 0;
  }

  rtp_usec = rtp_entries_[rtp_index].log_ts_.tv_usec;
  comm_usec = gcomm_entries_[comm_index].log_ts_.tv_usec;

  if (rtp_usec < comm_usec) {
    return 1;
  }
  if (rtp_usec > comm_usec) {
    return 0;
  }
  return 1;
}

/*
 * Binding between C++ and OTcl
 */
static class LogServiceClass : public TclClass {
public:
  LogServiceClass() : TclClass("LogService") {}
  TclObject *create(int argc, const char*const* argv) {
    return ((TclObject *) (&logger));
  }
} log_service_class_;
