# agent-studio.tcl --
#
#       Derived from VideoAgent.  DcStudioAgent takes video packets and
#       uses the PacketSplitter, sending one packet to a decoder and the
#       other to a PacketSwitcher for (possible) broadcasting.
#
# Copyright (c) 2000-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

import VideoAgent

#-----------------------------------------------------------------------------
# Class:
#   DcStudioAgent
#
# Description:
#   Derived from VideoAgent.  DcStudioAgent takes video packets and
#   uses the PacketSplitter, sending one packet to a decoder and the
#   other to a PacketSwitcher for (possible) broadcasting.
#
# Members:
#   splitter_ -- an array PackerSplitter for copying packets to two
#                destinations.  Indexed by Source/RTP objects.
#   switcher_ -- the Module/RTPSwitcher object.
#-----------------------------------------------------------------------------
Class DcStudioAgent -superclass VideoAgent

#-----------------------------------------------------------------------------
# Method:
#   DcStudioAgent init
#
# Arguments:
#   dcApp -- the Application object for DC
#   inetAddr,iPort,iTTL -- mcast specification of the studio video session
#   packetSwitcher -- a Module/RTPSwitcher object.
#
# Description:
#   Set members according to the input values and call constructor for
#   VideoAgent.  Initialize the splitter to "".
#
#-----------------------------------------------------------------------------
DcStudioAgent instproc init {dcApp inetAddr iPort iTTL packetSwitcher} {
    $self next $dcApp "$inetAddr/$iPort//$iTTL"
    $self instvar switcher_

    set switcher_ $packetSwitcher
}


#-----------------------------------------------------------------------------
# Method:
#   DcStudioAgent create_decoder
#
# Arguments:
#   src -- the Source/RTP object which we want to process/decode.
#
# Description:
#   This overrides VideoAgent's create_decoder{}, which creates a 
#   decoder, and returns it so that it is used as the data-handler of 
#   $src in activate{} and reactivate{}.  Here, we want the data
#   handler of $src to be the splitter instead, so we return $splitter_.
#   We also initialize (or re-initialize) both targets of the splitter.
#
#-----------------------------------------------------------------------------
DcStudioAgent instproc create_decoder {src} {
    $self instvar switcher_ splitter_

    #
    # This method could be call when a new source is detected,
    # or when an existing source change its format.  If it is
    # the latter, we do not need to create a splitter again.
    #
    if {![info exists splitter_($src)]} {
	set splitter_($src) [new Module/PacketSplitter]
    }

    # This will actually create the decoder.
    set decoder [$self next $src]

    #
    # FIXME: This is nasty. We're storing the decoder object
    #      in the Sourcre/RTP object. This is so that the
    #      handler is the splitter but the code for rendering
    #      and stuff can still find the decoder (normally,
    #      it is assumed that the data-handler for Source/RTP object
    #      is the decoder, but this isn't the case here!)
    #
    #      THIS MEANS YOU HAVE TO BE CAREFUL. Everywhere a decoder is
    #      assumed to be a Source/RTP's data handler, you have to do
    #      a set decoder to get the actual decoder object.
    #
    $src set decoder_ $decoder

    $splitter_($src) target $decoder
    $splitter_($src) target2 $switcher_

    return $splitter_($src)
}


#-----------------------------------------------------------------------------
# Method:
#   DcStudioAgent reactivate
#
# Arguments:
#   src -- the Source/RTP object which we want to process/decode.
#
# Description:
#   We need to override VideoAgent's reactivate{} here since the 
#   data-handler for $src is not a decoder as VideoAgent expects, 
#   but is a packet splitter instead.
#   
#   It removes the current decoder for $src, creates a new decoder, 
#   and returns the splitter.
# 
#   NOTE: This return value is not used. (?)
#
#-----------------------------------------------------------------------------
DcStudioAgent instproc reactivate {src} {

    $self instvar splitter_

    # 
    # Delete current decoder if it exists
    #
    set old_decoder [$splitter_($src) target]
    if {$old_decoder != ""} {
	delete $old_decoder
    }

    # 
    # Replace current_decoder in the decoder list.
    #
    $self instvar decoders_
    set k [lsearch -exact $decoders_ $old_decoder]
    set decoders_ [lreplace $decoders_ $k $k]

    # 
    # This will create the decoder and add it as
    # first target of $splitter_
    #
    $self create_decoder $src


    lappend decoders_ [$splitter_($src) target]
    return $splitter_($src)
}


#-----------------------------------------------------------------------------
# Method:
#   DcStudioAgent switch_src
#
# Arguments:
#   src -- the Source/RTP object which we want to switch
#
# Description:
#   Set the input source id of the switcher to $src.
#-----------------------------------------------------------------------------
DcStudioAgent instproc switch_src {src} {
    $self instvar switcher_
    $switcher_ set isrcid_ [$src srcid]
}
