# scope.tcl --
#
#       A ScopeZone object describes a range of multicast addresses that make
#       up an administratively scoped zone.
#
# Copyright (c) 1998-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

# A ScopeZone object describes a range of multicast addresses that
# make up an administratively scoped zone (as described in the
# Internet Draft draft-ietf-mboned-admin-ip-space).
Class ScopeZone

# Creates a new ScopeZone object for the address block <i>range</i>.
# <i>range</i> should be in CIDR format -- a base address followed
# by the number of bits that are significant.  For example, the
# block <tt>239.255.255.0/24</tt> describes the block of address
# from <tt>239.255.255.0</tt> through <tt>239.255.255.255</tt>.
# The optional parameters <i>bw</i> and <i>name</i> contain the
# bandwidth for SAP announcements in this zone and a name that
# describes how the region (e.g., "CAIRN")
ScopeZone public init {range {bw 200} {name ""}} {
    $self instvar range_ bw_

    set range_ $range
    set bw_ $bw

    # global is handled differently...
    if {$range == "224.2.128.0/17"} {
	set o [$self options]
	set addr [$o get_option SAPaddress]
        set port [$o get_option SAPport]
	$self set sapAddr_ "$addr/$port"
	$self set name_ "Global"
	return
    }

    if {$name != ""} {
	$self set name_ $name
    } else {
	$self set name_ "Admin Zone $range"
    }

    $self set sapAddr_ [$self addr $range]
}

# Determines the address and port on which SAP announcements are
# broadcast in this scope zone.  In general, this is the highest
# address in the zone and udp port 9875.
ScopeZone private addr {spec} {

    # global sap doesn't use the same convention as admin scoped zones.
    if {$spec == "224.2.128.0/17"} {
        set o [$self options]
        set addr [$o get_option SAPaddress]
        set port [$o get_option SAPport]
	puts "returning $addr/$port"
	return "$addr/$port"
    }

    set l [split $spec /]
    set len [llength $l]
    if { $len < 2 || $len > 3} {
	$self warn "Bogus scope zone spec $spec"
	exit 1
    }

    set base [lindex $l 0]
    set mask [lindex $l 1]
    set comps [split $base .]
    if {[llength $comps] != 4 || $mask > 24} {
	$self warn "Bogus scope zone spec $spec"
	exit 1
    }
    set a [lindex $comps 0]
    set b [lindex $comps 1]
    set c [lindex $comps 2]
    set d [lindex $comps 3]
    if {$a<224 || $a>239 || $b<0 || $b>255 || $c<0 || $c>255 || $d<0 || $d>255} {
	$self warn "Bogus scope zone spec $spec"
	exit 1
    }

    if {$mask < 16} {
	set b [expr $b | ~((-1)<<(16-$mask))]
	set mask 16
    }
    if {$mask < 24} {
	set c [expr $c | ~((-1)<<(24-$mask))]
	set mask 24
    }
    set d [expr $d | ~((-1)<<(32-$mask))]

    if {$len == 3} {
	set port [lindex $l 2]
    } else {
	set port 9875
    }

    set addr "$a.$b.$c.$d/$port"
    #puts "$spec -> $addr"
    return $addr
}

# Returns a string that describes this range.
ScopeZone public name {} {
    return [$self set name_]
}

# Returns the total bandwidth for SAP announcements in this region
ScopeZone public bw {} {
    return [$self set bw_]
}


# Returns a string in CIDR format (i.e., baseaddress/bits) indicating
# the range of addresses in this zone.
ScopeZone public range {} {
    return [$self set range_]
}

# Returns the address and port on which SAP announcements are transmitted
# in this zone, as calculated by the <i>addr</i> method.
ScopeZone public sapAddr {} {
    return [$self set sapAddr_]
}

# FIXME duplicated in alloc.tcl
ScopeZone private inet_addr {a} {
    set l [split $a .]
    set addr [expr [lindex $l 0] <<24]
    incr addr [expr [lindex $l 1] <<16]
    incr addr [expr [lindex $l 2] <<8]
    incr addr [lindex $l 3]
    return $addr
}

#
ScopeZone public contains {addr} {

    $self instvar range_
    set l [split $range_ /]
    set base [$self inet_addr [lindex $l 0]]
    set mask [lindex $l 1]
    set addr [$self inet_addr $addr]
    if {$base == [expr $addr & ((-1)<<(32-$mask))]} {
	return 1
    }
    return 0
}
