# announce.tcl --
#
#       FIXME: This file needs a description here.
#
# Copyright (c) 1998-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.


import AnnounceListenManager SDPParser
#import AnnounceListenManager/SAP

# A SAPAnnouncer implements the sending portion of the Session
# Announcement Protocol (SAP).  This functionality should eventually
# be moved in to the ProgramSource/SAP class but until then, it is
# the easiest way to implement the SAPForwarder class.<p>
#
# This class implements the <i>addprog</i>, <i>updateprog</i>, and
# <i>removeprog</i> methods that a ProgramSource object requires
# so it may be the designated receiver for a ProgramSource, effectively
# creating a SAP ``soft state gateway''.
Class SAPAnnouncer -superclass AnnounceListenManager/SAP

# Creates a new SAPAnnouncer object that sends and listens to SAP
# announcements on the multicast group and port given by <i>spec</i>.
# Aggregate bandwidth consumed by SAP on this channel is kept to
# <i>bw</i> bits per second as described in the SAP specification.
# <i>mtu</i> is the MTU for SAP announcements on this channel and
# defaults to 2 kilobytes if not specified.
SAPAnnouncer public init { zone {mtu 2048} } {
    set ttl [$self get_option sapTTL]
    $self next $mtu [$zone sapAddr]/$ttl

    $self set bw_ [$zone bw]
    $self set nsrcs_ 0
    $self set avgsize_ 0
    $self set sdp_ [new SDPParser]
}

#
SAPAnnouncer public destroy {} {
    $self next
    $self instvar timers_
    foreach t [array names timers_] {
	delete $timers_($t)
    }
}

# Invoked when an announcement is received on the SAP channel
# (i.e., when an announcement is sent by some other user of this
# channel).  Updates local state needed for the SAP timer algorithm.
SAPAnnouncer private recv_announcement {addr data size}  {
    $self instvar avgsize_ sdp_ progs_ nsrcs_

    set avgsize_ [expr $avgsize_ + (($size - $avgsize_)>>3)]
    foreach msg [$sdp_ parse [string trim $data]] {
	set o [join [$msg field_value o] :]
	if ![info exists progs_($o)] {
	    incr nsrcs_
	    set progs_($o) [after [$self calc_timeout] "$self timeout $o"]
	} else {
	    after cancel $progs_($o)
	    set progs_($o) [after [$self calc_timeout] "$self timeout $o"]
	}

	delete $msg
    }
}

# Calculates the expected interval between SAP announcement.
# If the parameter <i>rand</i> is 0 (or if it is unspecified), the
# expected interval is returned.  If <i>rand</i> is non-zero, the
# calculated interval is multiplied by a number drawn from a uniform
# distribution over the interval [0.5,1.5].
SAPAnnouncer private interval {{rand 0}} {
    $self instvar nsrcs_ avgsize_ bw_

    set i [expr (($avgsize_+28)*8) * $nsrcs_ / double($bw_)]
    if {$rand != 0} {
	# random in U[0.5,1.5]
	set r [expr [random]/double(0x7fffffff)+0.5]
	set i [expr $r * $i]
    }
    if {$i < 5} { set i 5 }
    return [expr int(1000 * $i)]
}

# Calculates the interval until an announcement should be timed out.
# As per the SAP specification, this is the maximum of 10 times the
# announcement interval and 30 minutes.
SAPAnnouncer private calc_timeout {} {
    set t [expr 10*[$self interval]]
    if {$t < 1800000} { set t 1800000 }
    return $t
}

# Invoked when an announcement has timed out.  The parameter
# <i>o</i> is the <tt>o=</tt> field from the SDP message with
# the fields joined by colons.  When this method is called, the
# specified program is no longer considered when calculating
# SAP timers.
SAPAnnouncer private timeout {o} {
    $self instvar progs_ nsrcs_

    if ![info exists progs_($o)] {
	return
    }
    after cancel $progs_($o)
    unset progs_($o)
    incr nsrcs_ -1
}

# This method is present only so that a SAPAnnouncer object can be
# used as the receiver of program announcements from a
# ProgramSource/SAP object.  It is a no-op.
SAPAnnouncer public addsource {src} { }

#
SAPAnnouncer public addprog {source p} {
    #puts "got [$p field_value s]"
    $self instvar nsrcs_
    incr nsrcs_
    foreach msg [$p set msgs_] {
	$self doannounce $msg
    }
}

# Invoked when the announcement for the SDPMesage <i>msg</i> needs
# to be transmitted.  Does the actual announcement and then
# schedules a timer for the next announcement.
SAPAnnouncer private doannounce {msg} {
    $self instvar timers_ avgsize_

    set text [$msg set msgtext_]
    set size [string length $text]
    set avgsize_ [expr $avgsize_ + (($size - $avgsize_) >> 3)]

    $self announce $text
    set o [$msg unique_key]
    set timers_($o) [after [$self interval 1] "$self doannounce $msg"]
}

# Notifies this SAPAnnouncer that an SDP program that it is
# announcing, described in the SDPMesage <i>old</i> has been changed.
# The old announcement is replaced by the SDPMesage <i>new</i>.
SAPAnnouncer public updateprog {source p} {
    $self instvar timers_
    set o [$p unique_key]
    if [info exists timers_($o)] {
	after cancel $timers_($o)
	unset timers_($o)
    }
    foreach msg [$p set msgs_] {
	$self doannounce $msg
    }
}

# Asks this SAPAnnouncer to stop announcing the program described
# by the SDPMesage <i>msg</i>.
SAPAnnouncer public removeprog {source p} {
    $self instvar timers_ nsrcs_
    set o [$p unique_key]
    if ![info exists timers_($o)] {
	$self warn "SAPAnnouncer::removeprog inconsistency!"
	return
    }
    after cancel $timers_($o)
    unset timers_($o)
    incr nsrcs_ -1
}

