# cf.tcl --
#
#       Provides content follower.  Causes one RVC to show the same thing that
#       is on the projector in 405.
#
# Copyright (c) 2000-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

Import enable

import 405Client

# use the Tcl namespace facility to prevent conflicts
namespace eval ContentFollower {
    variable self
    variable enabled
    variable outputSource
}

Class ContentFollower

ContentFollower instproc init {base {extendedGUI 1}} {
    $self instvar 405_

    set 405_ [new 405Client]

    # so that the callback can refer to this object
    set ContentFollower::self $self

    set callback "$self processAmxMsg"
    set filter [$self getFilter]
    $405_ callback_register $callback $filter

    $self initUI $base $extendedGUI
}

ContentFollower instproc initUI {base extendedGUI} {
    frame $base.controls -borderwidth 3
    pack $base.controls -side top -fill x

    checkbutton $base.controls.onOff -text "Enable Content Following" -variable ContentFollower::enabled -command "$self toggleOnOff"
    pack $base.controls.onOff -side left

    if {$extendedGUI} {
	frame $base.outputs -borderwidth 3
	pack $base.outputs -side top -fill x

	# I have to give each button its own frame to make it look pretty and
	#  line up correctly
	frame $base.outputs.labelFrame
	pack $base.outputs.labelFrame -side top -fill x
	label $base.outputs.labelFrame.outputLabel -text "Output Sources"
	pack $base.outputs.labelFrame.outputLabel -side left

	frame $base.outputs.htsrBut
	pack $base.outputs.htsrBut -side top -fill x
	radiobutton $base.outputs.htsrBut.htsr -text "htsr" -variable ContentFollower::outputSource -value "htsr"
	pack $base.outputs.htsrBut.htsr -side left

	frame $base.outputs.htsr2But
	pack $base.outputs.htsr2But -side top -fill x
	radiobutton $base.outputs.htsr2But.htsr2 -text "htsr2" -variable ContentFollower::outputSource -value "htsr2"
	pack $base.outputs.htsr2But.htsr2 -side left
    }


     # put all the inits down here in case we call toggleOnOff, which needs
    #   ContentFollower::outputSource to be defined already

    set ContentFollower::outputSource "htsr"

    # if you want the default to be on, have to set ContentFollower::enabled
    #   to 1 and call toggleOnOff manually
    set ContentFollower::enabled 0
#    set ContentFollower::enabled 1
#    $self toggleOnOff
}

ContentFollower public toggleOnOff {} {
    $self instvar 405_

    set callback "$self processAmxMsg"
    if {$ContentFollower::enabled == 0} {
	$405_ callback_disable $callback
    } else {
	$405_ callback_enable $callback
    }

}

ContentFollower public AMXSwitch { target output } {
    $self instvar 405_

    $405_ matrix_switchVideoStream $target $output
}

ContentFollower public processAmxMsg { amxMsg } {
#    puts stdout "amxMsg is $amxMsg"

    if {[llength $amxMsg] != 0} {
	set eventInfo [lindex $amxMsg 0]
	set type [lindex $eventInfo 0]
	set eventData [lindex $amxMsg 1]
	set cmd [lindex $eventData 0]
	set dev [lindex $eventData 1]
	set chan_str [lindex $eventData 2]
#	puts stdout "processAmxMsg: received type=$type, cmd=$cmd, dev=$dev, chan_str=$chan_str"
    } else {
	puts stderr "processAmxMsg: nothing received"
	return
    }

    if {$type != "amxResponse"} {
	# it couldn't possibly be a button push
	return
    }

    if {$dev == "remotePanel" || $dev == "touchPanel"} {
	if {$cmd == "push"} {
	    # it is a user pushing a button on the touch panel
	    #		puts stdout "matrix command"
	    switch -exact -- $chan_str {
		pcToProj {
		    # PC
#		    puts stdout "switching to PC"
		    $self AMXSwitch frontPC $ContentFollower::outputSource
		}
		vcrToProj {
		    # vcr
#		    puts stdout "switching to VCR"
		    $self AMXSwitch frontVCR $ContentFollower::outputSource
		}
		elmoToProj {
		    # elmo
#		    puts stdout "switching to elmo"
		    $self AMXSwitch elmo $ContentFollower::outputSource
		}
		laptopToProj {
		    # laptop
#		    puts stdout "switching to laptop"
		    $self AMXSwitch laptop $ContentFollower::outputSource
		}
		sgiToProj {
		    # sgi
#		    puts stdout "switching to sgi"
		    $self AMXSwitch sgiPC $ContentFollower::outputSource
		}
		default {
		    # something not related to matrix
#		    puts stdout "non-matrix button pushed -- autoswitch ignoring"
		}
	    }
	}
    }

    return ""
}

ContentFollower public setOutput {output} {
    switch -exact -- $output {
	htsr -
	htsr2 {
	    set ContentFollower::outputSource $output
	}
	default {
	    return -code error "ContentFollower::setOutput: output must be \[htsr, htsr2\]"
	}
    }
}

ContentFollower public enable {} {
    set ContentFollower::enabled 1
    $self toggleOnOff
}

ContentFollower public disable {} {
    set ContentFollower::enabled 0
    $self toggleOnOff
}

ContentFollower instproc getFilter {} {
    # this filters out everything but button pushes on the touch panel or
    #   remote panel
    #
    # we don't try to filter out for specific buttons, since that is too much
    #   work!
    set filter ""
    set filter "$filter set doCallback 0\n"
    set filter "$filter set info \[lindex \$arg 0\]\n"
    set filter "$filter set type \[lindex \$info 0\]\n"
    set filter "$filter set data \[lindex \$arg 1\]\n"
    set filter "$filter if \{\$type == \"amxResponse\"\} \{\n"
    set filter "$filter set cmd \[lindex \$data 0\]\n"
    set filter "$filter set dev \[lindex \$data 1\]\n"
    set filter "$filter set chan \[lindex \$data 2\]\n"
    set filter "$filter if \{\$dev == \"remotePanel\" || \$dev == \"touchPanel\"\} \{\n"
    set filter "$filter if \{\$cmd == \"push\"\} \{\n"
    set filter "$filter set doCallback 1\n"
    set filter "$filter \}\n"
    set filter "$filter \}\n"
    set filter "$filter \}\n"

    return $filter
}

