/*
 * image2tcl.cc --
 *
 *      This is a stand-alone utility that reads an image file from stdin and
 *      writes Tcl code to stdout.  The Tcl code contains the image data as a
 *      string and creates the image with the given name.
 *
 *      The syntax is:
 *
 *          image2tcl image-name < image-file > tcl-file
 *
 *      Currently, Tk only supports GIF and PPM/PGM files.
 *
 * Copyright (c) 1997-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h>

#ifdef WIN32
#   include <fcntl.h>
#   include <io.h>
#else
#   include <unistd.h>
#endif


void
Image2Tcl(const char *imageName)
{
    unsigned char buffer[1024], *ptr;
    int i, bytesRead, stdin_FD = 0;

    /*
     * The usual way to create an image from data is:
     *
     *  image create photo imageName -data $imageData
     *
     * However, Jan Nijtmans noted in comp.lang.tcl that the -data option
     * stores the data in memory even though it's never needed later.  The
     * following method does the same thing without using extra memory:
     *
     *  image create photo imageName
     *  imageName put $imageData
     */

    printf("image create photo %s\n", imageName);
    printf("%s put ", imageName);

    while ((bytesRead = read(stdin_FD, buffer, sizeof(buffer))) > 0) {
        for (i = 0, ptr = buffer; i < bytesRead; i++, ptr++) {
            printf("\\x%X", *ptr);
        }
    }

    printf("\n");
}


int
main(int argc, char *argv[])
{
#ifdef WIN32
    _setmode(_fileno(stdin), _O_BINARY);
#endif
    if (argc != 2) {
        fprintf(stderr,
                "Usage: image2tcl image-name < image-file > tcl-file\n");
        return -1;
    }
    Image2Tcl(argv[1]);
    return 0;
}
