/* --------------------------------------------------------------------------
 * module_cdrom.c
 * code for module cdrom
 *
 * Copyright 2002 Matthias Grimm
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version
 * 2 of the License, or (at your option) any later version.
 * -------------------------------------------------------------------------*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <string.h>
#include <unistd.h>
#include <sys/time.h>
#include <linux/cdrom.h>
#include <errno.h>
#include <pbb.h>
#include <time.h>

#include "pbbinput.h"
#include "gettext_macros.h"
#include "input_manager.h"
#include "module_cdrom.h"
#include "support.h"
#include "debug.h"

struct moddata_cdrom {
	char *cdromdev;		/* pathname of the cdrom device */
	unsigned short keyejectcd;
	unsigned short modejectcd;
	struct timeval tv;      /* time of key press */
	int keydelaytime;      /* delay in ms */
	int cdejected;
} modbase_cdrom;

int
cdrom_init (struct tagitem *taglist)
{
	struct moddata_cdrom *base = &modbase_cdrom;
	static char devbuffer_cdrom[STDBUFFERLEN];
	int rc;

	sprintf(devbuffer_cdrom, DEFAULT_CDROM);

	base->cdromdev		= devbuffer_cdrom;
	base->keyejectcd	= KEY_EJECTCD;
	base->modejectcd	= MOD_NONE;
	base->keydelaytime	= 0;

	if ((rc = cdrom_handle_tags (MODE_CONFIG, taglist))) /* interprete configfile */
		return rc;                 /* critcal error occured, abort program */

#if defined(DEBUG) && CDROMPERF
	cdrom_testperformance ();
#endif	
	
	register_function (KBDQUEUE, cdrom_keyboard);
	register_function (QUERYQUEUE, cdrom_query);
	register_function (CONFIGQUEUE, cdrom_configure);
	return 0;
}

int
cdrom_exit ()
{
	return 0;
}

void
cdrom_keyboard (struct tagitem *taglist)
{
	struct moddata_cdrom *base = &modbase_cdrom;
	int code, value, mod;

	code = (int) tagfind (taglist, TAG_KEYCODE, 0);
	value = (int) tagfind (taglist, TAG_KEYREPEAT, 0);
	mod = (int) tagfind (taglist, TAG_MODIFIER, 0);

	if (value) {
		if (value == 1) base->cdejected = 0;
		if ((code == base->keyejectcd) && (mod == base->modejectcd))
			if ((base->keydelaytime == 0) || (keydelayms(&base->tv, value, base->keydelaytime)))
				if (base->cdejected == 0) {
					base->cdejected = 1;
					cdrom_eject ();
				}
	}
}

void
cdrom_query (struct tagitem *taglist)
{
	cdrom_handle_tags(MODE_QUERY, taglist);
}

void
cdrom_configure (struct tagitem *taglist)
{
	cdrom_handle_tags(MODE_CONFIG, taglist);
}

int
cdrom_handle_tags (int cfgure, struct tagitem *taglist)
{
	struct moddata_cdrom *base = &modbase_cdrom;
	int err, rc = 0;

	while (taglist->tag != TAG_END) {
		switch (taglist->tag) {
		case TAG_EJECTCD:
			if (cfgure) {
				singletag_to_clients(CHANGEVALUE, TAG_EJECTCD, 1);
				if ((cdrom_eject()) == 0)
					singletag_to_clients(CHANGEVALUE, TAG_EJECTCD, 0);
			} else
				tagerror (taglist, E_NOREAD);
			break;
		case TAG_CDROMDEVICE:
			if (cfgure) {
				if ((err = copy_path ((char *) taglist->data, base->cdromdev, TYPE_BLKDEV, CPFLG_MAYBEMISSED)))
					rc = tagerror (taglist, err);
			} else
				taglist->data = (long) base->cdromdev;
			break;
		case TAG_EJECTCDKEY:
			if (cfgure)	base->keyejectcd = taglist->data;
			else		taglist->data = base->keyejectcd;
			break;
		case TAG_EJECTCDMOD:
			if (cfgure)	base->modejectcd = taglist->data;
			else		taglist->data = base->modejectcd;
			break;
		case TAG_EJECTCDKEYDELAY:
			if (cfgure)	base->keydelaytime = taglist->data;
			else		taglist->data = base->keydelaytime;
			break;
		}
		taglist++;
	}
	return rc;
}

char *
cdrom_getmountpoint(char *device)
{
	char *mountpoint = NULL, *token;
	static char buffer[100];
	FILE *fd;

	if ((fd = fopen ("/proc/mounts","r"))) {
		while (fgets (buffer, sizeof (buffer), fd))
			if ((token = strtok (buffer," \n"))) {
				if (!strcmp (device, token)) {
					mountpoint = strtok(0," \n");
					break;
				} else
					strtok (0,"\n");
			}
		fclose(fd);
	}
	return mountpoint;
}

/* This function eject the cdrom or close the tray if the CDROM is
 * already ejected. If neseccary the cdrom will be unmounted first.
 * Following return codes are possible:
 *   0: Drive is busy,
 *   1: CDROM has been ejected or the tray was opened,
 *   2: The Tray has been closed,
 * The external helper program umount is used because the system
 * call umount() doesn't update /etc/mtab and this conflicts with
 * umount's normal operation.
 */
int
cdrom_eject()
{
	struct moddata_cdrom *base = &modbase_cdrom;
	int fd, err = 0, rc = 0;
	unsigned int t1, t2;
	char *mp;

	singletag_to_clients(CHANGEVALUE, TAG_EJECTCD, 0); /* busy */
	if ((mp = cdrom_getmountpoint(base->cdromdev)))
		err = call_script ("/bin/umount %.30s", mp);

#if defined(DEBUG) && CDROM
	printf ("mp=%s, err=%d, %s (%d)\n", mp, err, strerror(errno), errno);
#endif

	if (mp == NULL || err == 0 || errno == EINVAL)
		if ((fd = open(base->cdromdev, O_RDONLY | O_NONBLOCK | O_NDELAY)) >= 0) {
			
			err = ioctl (fd, CDROM_DRIVE_STATUS);
			switch (err) {
			case 2:	
			default:
				singletag_to_clients(CHANGEVALUE, TAG_EJECTCD, 2); /* close */
				t1 = cdrom_gettime ();
				ioctl (fd, CDROMCLOSETRAY);
				t2 = cdrom_gettime ();
#if defined(DEBUG) && CDROMPERF
				printf ("close Tray: %4d (%4u-%4u)\n", t2-t1, t2, t1);
#endif
				rc = 2;
				if (t2-t1 > 100)
					break;
			case 1:
			case 4:
				singletag_to_clients(CHANGEVALUE, TAG_EJECTCD, 1); /* eject */
#if defined(DEBUG) && CDROMPERF
				t1 = cdrom_gettime ();
#endif
				ioctl (fd, CDROMEJECT);
#if defined(DEBUG) && CDROMPERF
				t2 = cdrom_gettime ();
				printf ("open Tray : %4d (%4u-%4u)\n", t2-t1, t2, t1);
#endif
				rc = 1;
			}
			close (fd);
		}
	return rc;
}

unsigned int
cdrom_gettime ()
{
	struct timeval tv;

	gettimeofday(&tv, NULL);
	return (tv.tv_sec * 1000 + tv.tv_usec / 1000 );
}

#if defined(DEBUG) && CDROMPERF
void
cdrom_testperformance ()
{
	struct moddata_cdrom *base = &modbase_cdrom;
	int fd;
	unsigned int t1, t2;

	printf ("\nCDROM Performancedata: \n");
	if ((fd = open(base->cdromdev, O_RDONLY | O_NONBLOCK | O_NDELAY)) >= 0) {

		printf ("Close already closed Tray: ");
		t1 = cdrom_gettime();
		ioctl (fd, CDROMCLOSETRAY);
		t2 = cdrom_gettime();
		printf ("%4d (%4u-%4u)\n", t2-t1, t2, t1);
		sleep(1);
		
		printf ("Open closed Tray         : ");
		t1 = cdrom_gettime();
		ioctl (fd, CDROMEJECT);
		t2 = cdrom_gettime();
		printf ("%4d (%4u-%4u)\n", t2-t1, t2, t1);
		sleep(1);
		
		printf ("Open already open Tray   : ");
		t1 = cdrom_gettime();
		ioctl (fd, CDROMEJECT);
		t2 = cdrom_gettime();
		printf ("%4d (%4u-%4u)\n", t2-t1, t2, t1);
		sleep(1);
		
		printf ("Close open Tray          : ");
		t1 = cdrom_gettime();
		ioctl (fd, CDROMCLOSETRAY);
		t2 = cdrom_gettime();
		printf ("%4d (%4u-%4u)\n", t2-t1, t2, t1);
		sleep(1);
		
		printf ("Close already closed Tray: ");
		t1 = cdrom_gettime();
		ioctl (fd, CDROMCLOSETRAY);
		t2 = cdrom_gettime();
		printf ("%4d (%4u-%4u)\n\n", t2-t1, t2, t1);
		
		close (fd);
	} else
		printf ("CDROM open failed!\n\n");
}
#endif

