/* --------------------------------------------------------------------------
 * module_system.c
 * code for general system functions
 *
 * Copyright 2002 Matthias Grimm
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version
 * 2 of the License, or (at your option) any later version.
 * -------------------------------------------------------------------------*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <string.h>
#include <errno.h>
#include <unistd.h>
#include <pbb.h>

#include "gettext_macros.h"
#include "input_manager.h"
#include "module_system.h"
#include "configfile.h"

struct moddata_system {
	char *userallowed;  /* remember system tag for config file saving */
	char *configwritefile; /* configfile to write: eg. /var/lib/pbbuttonsd/pbbuttonsd.conf */
	int  autorescan;
	int  rescantimer;
} modbase_system;

int
system_init (struct tagitem *taglist)
{
	struct moddata_system *base = &modbase_system;
	static char buf_userallowed[STDBUFFERLEN];
	static char buf_configwritefile[STDBUFFERLEN];
	int rc;

	base->autorescan = 0;   /* disable by default */
	base->rescantimer = 0;
	base->userallowed = buf_userallowed;
	base->userallowed[0] = '\0';
	base->configwritefile = buf_configwritefile;
	base->configwritefile[0] = '\0';

	if ((rc = system_handle_tags (MODE_CONFIG, taglist))) /* interprete configfile */
		return rc;     /* critcal error occured, abort program */

	ipc_protect_tag (TAG_SAVECONFIG);  /* tags only for privileged use */
	register_function (QUERYQUEUE, system_query);
	register_function (CONFIGQUEUE, system_configure);
	register_function (T1000QUEUE, system_timer1000);
	return 0;
}

int
system_exit ()
{
	return 0;
}

void
system_query (struct tagitem *taglist)
{
	system_handle_tags (MODE_QUERY, taglist);
}

void
system_configure (struct tagitem *taglist)
{
	system_handle_tags (MODE_CONFIG, taglist);
}

int
system_handle_tags (int cfgure, struct tagitem *taglist)
{
	struct moddata_system *base = &modbase_system;
	struct tagitem args[] = {{ TAG_AMBIENTLIGHT, 0 },
		                 { TAG_SLEEPSUPPORTED, 0 },
	                         { TAG_END, 0 }};
	static char *version = VERSION;
	int rc = 0, val;

	while (taglist->tag != TAG_END) {
		switch (taglist->tag) {
		case TAG_SAVECONFIG:
			if (cfgure) {
				if (strlen(base->configwritefile) > 0)
					write_configfile(base->configwritefile);
				else
					tagerror (taglist, E_NOSUPPORT);
			} else
				tagerror (taglist, E_NOREAD);
			break;
		case TAG_USERALLOWED:  /* private tag */
			if (cfgure)	strncpy(base->userallowed, (char*) taglist->data, STDBUFFERLEN);
			else		taglist->data = (long) base->userallowed;
			break;
		case TAG_VERSION:
			if (cfgure)	tagerror (taglist, E_NOWRITE);
			else		taglist->data = (long) version;
			break;
		case TAG_AUTORESCAN:
			if (cfgure)	base->autorescan = taglist->data;
			else		taglist->data = (long) base->autorescan;
			break;
		case TAG_TIMEFORCMD:
			if (cfgure)	set_timeforcmd(taglist->data);
			else		taglist->data = (long) get_timeforcmd();
			break;
		case TAG_SYSINFO:
			if (cfgure)	tagerror (taglist, E_NOWRITE);
			else {
				val = 0;
				if (strlen(base->configwritefile) > 0)
					val |= SYSINFO_CONFIGWRITABLE;
				process_queue (QUERYQUEUE, args);
				if (tagfind (args, TAG_AMBIENTLIGHT, -1) != -1)
					val |= SYSINFO_HAVELMU;
				if (tagfind (args, TAG_SLEEPSUPPORTED, -1) != -1)
					val |= SYSINFO_SLEEPSUPPORTED;
#ifdef WITH_PMUD
				val |= SYSINFO_PMUDSUPPORT;
#endif
				taglist->data = val;
			}
			break;
		case TAG_CFGNAME:    /* private tag */
			if (cfgure)
				system_config_writable ((char*) taglist->data, base->configwritefile);
			break;
		}
		taglist++;
	}
	return rc;
}

void
system_timer1000 (struct tagitem *taglist)
{
	struct moddata_system *base = &modbase_system;

	if (base->autorescan) {
		base->rescantimer++;
		if (base->rescantimer > RESCANTIME) {
			base->rescantimer = 0;
			scan_for_kbdevdevs();
		}
	}
}

void
system_config_writable (char *configreadfile, char *configwritefile)
{
	int perms, owner;

	configwritefile[0] = '\0';
	owner = get_owner(configreadfile);
	perms = get_permissions(configreadfile);
	if (owner != getuid() || (perms & 022))
		print_error (_("WARNING: config file [%s] is insecure, saving of config disabled.\n"), configreadfile);
	else if (owner == getuid() && !(perms & 0200))
		print_error (_("WARNING: config file [%s] is not writable, saving of config disabled.\n"), configreadfile);
	else {
		strcpy(configwritefile, configreadfile);
		print_error (_("INFO: saving of config enabled to %s.\n"), configwritefile);
		return; /* config is writable, name in configwritefile */
	}
}

