/*
 * PIMPPA - Create all assign patterns
 *
 *
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <ctype.h>
#include <unistd.h>

#include "pimppa.h"
	
MYSQL *src_db, *dst_db;

void usage(char *name)
{
	fprintf(stderr, "Usage: %s <options>\n\n"
				"-c            Clear all previous assigns. Does not affect\n"
				"              patterns with destinations 0 or -1.\n"
				"-f <filename> Show current assign for <filename>\n"
				"-r            Refresh (recreate) all assign patterns\n"
				"-0            Delete all assign patterns with target area 0.\n"
				"-n            Delete all assign patterns with target area -1.\n"
				"-t <area_id>  With -f, set a new target area id instead\n"
				,name);
}	

int create_assigns(void)
{
	MYSQL_RES *sql_res;
	MYSQL_ROW sql_row;
	unsigned long total_files=0;
	char *value;
	int minassnamelen;

	value=p_getmisc(src_db, P_KEY_MINASSNAMELEN);
	if(value)
		minassnamelen=atoi(value);
	else
		minassnamelen=atoi(P_MIN_ASS_NAMELENGTH);

/******** check and assign *******/

	fprintf(stderr, "Processing filenames...\n");

	p_query(src_db, "SELECT file_name, file_area, area_context "
			"FROM p_files, p_areas "
			"WHERE NOT (area_flags & %ld) "
			"  AND NOT (area_flags & %ld) "
			"  AND area_id=file_area "
			"  AND NOT (file_backup=0 AND (file_flags & %ld))",
		AREA_INCOMING, AREA_NOASSIGN, FILE_OFFLINE);
	if(mysql_error(src_db)[0])
		return(-1);
	sql_res=mysql_use_result(src_db);

	while((sql_row=mysql_fetch_row(sql_res)))
	{
		if(p_assign(dst_db, sql_row[0], minassnamelen, atoi(sql_row[1]),
					atoi(sql_row[2])))
		{
			total_files++;
		
			if(total_files%500==0)
				fprintf(stderr, "%ld done...\n", total_files);
		}
	}

	mysql_free_result(sql_res);

	fprintf(stderr, "Counting totals...\n");

	p_query(src_db, "SELECT count(*) FROM p_assign "
			"WHERE a_dest>=0");
	sql_res=mysql_store_result(src_db);
	if(sql_res)
	{
		sql_row=mysql_fetch_row(sql_res);
		
		fprintf(stderr, "Parsed %ld files mapping to %s patterns.\n",
			total_files, sql_row[0]);
		
		mysql_free_result(sql_res);
	}


	return(total_files);
}
	

int main(int argc, char *argv[])
{
	int go=1;
	int destarea=0,gotdest=0;
	char filename[P_LEN_FILE_NAME];

	filename[0]=0;

	if(argc==1)
	{
		usage(argv[0]);
		return(0);
	}

	/********* Connect databases ****************/

	src_db=p_connect();
	if(!src_db)
		return(-1);

	dst_db=p_connect();
	if(!dst_db)
	{
		mysql_close(src_db);
		return(-1);
	}

	while(go)
	{
		switch(getopt(argc, argv, "0cf:hnrt:V"))
		{
			case '0':
				fprintf(stderr, "Deleting assign patterns with target 0...\n");
				p_query(dst_db, "DELETE FROM p_assign "
						"WHERE a_dest=0");
				break;
			case 'n':
				fprintf(stderr, "Deleting assign patterns with target -1...\n");
				p_query(dst_db, "DELETE FROM p_assign "
						"WHERE a_dest=-1");
				break;
			case 'c':
				fprintf(stderr, "Clearing previous assign patterns...\n");
				p_query(dst_db, "DELETE FROM p_assign "
						"WHERE a_dest>0");
				break;
			case 'f':
				strcpy(filename, optarg);
				break;
			case 'h':
				usage(argv[0]);
				return(0);
				break;
			case 'r':
				create_assigns();
				break;
			case 't':
				destarea=atoi(optarg);
				gotdest=1;
				break;
			case 'V':
				printf("%s %s %s\n", PACKAGE, argv[0], VERSION);
				return(0);
				break;
			default:
				go=0;
				break;
		}
	}	

	// Do we wish just to print the current assign?
	if(filename[0] && !gotdest)
	{
		char pattern[P_LEN_FILE_NAME];
		char escaped_pat[2*P_LEN_FILE_NAME+1];
		int retval;

		retval=p_makepattern(pattern, filename);
		if(retval)
		{
			MYSQL_ROW sql_row;
			MYSQL_RES *sql_res;
			
			mysql_escape_string(escaped_pat, pattern, strlen(pattern));
			
			p_query(dst_db, "SELECT area_name, a_dest "
					"FROM p_assign,p_areas "
					"WHERE a_pattern='%s' "
					"  AND a_dest=area_id", 
				escaped_pat);
			sql_res=mysql_store_result(dst_db);
			if(sql_res)
			{
				if((sql_row=mysql_fetch_row(sql_res)))
					printf("%s => ( %s , %s ) => %s\n", filename, pattern, sql_row[1], sql_row[0]);
				else
					printf("%s => ( %s , UNSET )\n", filename, pattern);
			
				mysql_free_result(sql_res);
			}	
		}
		else
		{
			printf("%d: Pattern not made (invalid filename?)\n", retval);
		}
	}

	// Do we wish to assign a new destination?
	if(filename[0] && gotdest)
	{
		char pattern[P_LEN_FILE_NAME];
		char escaped_pat[2*P_LEN_FILE_NAME+1];
		int retval;

		retval=p_makepattern(pattern, filename);
		if(retval)
		{
			printf("%s => ( %s , %d )\n", filename, pattern, destarea);

			mysql_escape_string(escaped_pat, pattern, strlen(pattern));
			
			p_query(dst_db, "REPLACE INTO p_assign "
					"  (a_pattern,a_dest) "
					"VALUES ('%s',%d)", 
				escaped_pat, destarea);
		}
		else
			printf("Urk %d\n", retval);
	}
	
	mysql_close(src_db);
	mysql_close(dst_db);
	
	return(0);
}

