#include <string.h>
#include <sys/types.h>
#include <termios.h>
#include <sys/ioctl.h>

#include <config.h>
#include <support.h>
#include <xcio.h>
#include "info.h"

struct number_string_s {
    int number;
    char *string;
};

static int
f2s(int flags, struct number_string_s *p, char *buf, int buflen)
{
    int len;

    buf[0] = '\0';
    len = 0;
    for(; p->string; p++){
	if(flags & p->number){
	    len += strlen(p->string) + 1;
	    if(len < buflen){
		strncat(buf, p->string, buflen - len);
		buf[len - 1] = ':';
		buf[len] = '\0';
	    }else{
		break;
	    }
	}
    }
    if(len > 0){
	len--;
	buf[len] = '\0';	/* delete last ":" */
    }
    return(len);
}

static int
n2s(int num, struct number_string_s *p, char *buf, int buflen)
{
    for(; *p->string; p++)
	if(num == p->number)
	    break;

    if(p->string){
	strncpy(buf, p->string, buflen);
    }else{
	buf[0] = '\0';
    }
    return(strlen(buf));
}

/************************************************************/
static struct number_string_s l_stat[] = {
    { LSTAT_PROMPT,	"prompt" },
    { LSTAT_TTY,	"tty"    },
    { LSTAT_CHAT,	"chat"   },
    { LSTAT_PPP,	"ppp"    },
    { LSTAT_NLINK,	"nlink"  },
    { LSTAT_DIAL,	"dial"   },
    { 0, NULL }
};

static struct number_string_s m_flag[] = {
    { MFLAG_AUTO,	"auto" },
    { 0, NULL }
};

static struct number_string_s n_stat[] = {
    { NSTAT_IP,	"ip"  },
    { NSTAT_IPX,	"ipx" },
    { 0, NULL }
};

static struct number_string_s minfo[] = {
    { TIOCM_DTR,	"DTR" },
    { TIOCM_RTS,	"RTS" },
    { TIOCM_CD,		"CD"  },
    { TIOCM_RNG,	"RNG" },
    { TIOCM_DSR,	"DSR" },
    { 0, NULL }
};

static struct number_string_s phase[] = {
    { PS_DEAD,		"dead"		},
    { PS_ESTABLISH,	"establish"	},
    { PS_AUTHENTICATE,	"authenticate"	},
    { PS_CALLBACK,	"callback"	},
    { PS_NETWORK,	"network"	},
    { PS_TERMINATE,	"terminate"	},
    { 0, NULL }
};

#define PPPSTR_BUF_SIZE 64

static void
ppp_setkeyval(struct keyval_s *kv, char *key, int type)
{
    kv->keyval[0] = key;
    kv->type = type;
    if(type == PPXPINFO_STR){
	kv->keyval[1] = (char *)malloc(sizeof(char)*PPPSTR_BUF_SIZE);
    }else if(type == PPXPINFO_INT){
#if PPXP_TCL
	kv->keyval[1] = (char *)malloc(sizeof(char)*PPPSTR_BUF_SIZE);
#endif
    }else if(type == PPXPINFO_LONG){
#if PPXP_TCL
	kv->keyval[1] = (char *)malloc(sizeof(char)*PPPSTR_BUF_SIZE);
#endif
    }
}

static struct keyval_s *
pppinfo_new(void)
{
    struct keyval_s *info;
    int i;

    info = (struct keyval_s *) malloc(sizeof(struct keyval_s)*N_PPPINFO);

    i = 0;	ppp_setkeyval(&info[i],	"phase",	PPXPINFO_STR);
    i++;	ppp_setkeyval(&info[i],	"idle",		PPXPINFO_LONG);
    i++;	ppp_setkeyval(&info[i],	"connect",	PPXPINFO_LONG);
    i++;	ppp_setkeyval(&info[i],	"send.count",	PPXPINFO_INT);
    i++;	ppp_setkeyval(&info[i],	"send.error",	PPXPINFO_INT);
    i++;	ppp_setkeyval(&info[i],	"send.lsize",	PPXPINFO_INT);
    i++;	ppp_setkeyval(&info[i],	"send.nsize",	PPXPINFO_INT);
    i++;	ppp_setkeyval(&info[i],	"recv.count",	PPXPINFO_INT);
    i++;	ppp_setkeyval(&info[i],	"recv.error",	PPXPINFO_INT);
    i++;	ppp_setkeyval(&info[i],	"recv.lsize",	PPXPINFO_INT);
    i++;	ppp_setkeyval(&info[i],	"recv.nsize",	PPXPINFO_INT);
    i++;	ppp_setkeyval(&info[i],	"modeminfo",	PPXPINFO_STR);
    i++;	ppp_setkeyval(&info[i],	"linestat",	PPXPINFO_STR);
    i++;	ppp_setkeyval(&info[i],	"netstat",	PPXPINFO_STR);
    i++;	ppp_setkeyval(&info[i],	"mode",		PPXPINFO_STR);

    return(info);
}
static int
pppinfo_parse(struct keyval_s *info, char *buf, int buflen)
{
    struct pppinfo_s p;
    /* struct pppinfo_s *p = (struct pppinfo_s *)buf; */
    int i;

    memcpy(&p, buf, sizeof(p));

#if PPXP_TCL
#define SETINT(i, s) (sprintf((s), "%d", (i)))
#define SETLONG(i, s) (sprintf((s), "%ld", (i)))
#elif PPXP_PERL
#define SETINT(i, s) (s = (char *)&i)
#define SETLONG(i, s) (s = (char *)&i)
#endif

    i = 0;	n2s(p.phase, phase, info[i].keyval[1], PPPSTR_BUF_SIZE);
    i++;	SETLONG(p.idle, info[i].keyval[1]);
    i++;	SETLONG(p.connect, info[i].keyval[1]);
    i++;	SETINT(p.s.count, info[i].keyval[1]);
    i++;	SETINT(p.s.error, info[i].keyval[1]);
    i++;	SETINT(p.s.lsize, info[i].keyval[1]);
    i++;	SETINT(p.s.nsize, info[i].keyval[1]);
    i++;	SETINT(p.r.count, info[i].keyval[1]);
    i++;	SETINT(p.r.error, info[i].keyval[1]);
    i++;	SETINT(p.r.lsize, info[i].keyval[1]);
    i++;	SETINT(p.r.nsize, info[i].keyval[1]);
    i++;	f2s(p.minfo, minfo, info[i].keyval[1], PPPSTR_BUF_SIZE);
    i++;	f2s(p.l_stat, l_stat, info[i].keyval[1], PPPSTR_BUF_SIZE);
    i++;	f2s(p.n_stat, n_stat, info[i].keyval[1], PPPSTR_BUF_SIZE);
    i++;	f2s(p.m_flag, m_flag, info[i].keyval[1], PPPSTR_BUF_SIZE);
    i++;

    return(i);
}

static void
pppinfo_clear(struct keyval_s *info)
{
}

static void
pppinfo_delete(struct keyval_s *info)
{
    int i;

    for(i = 0; i < N_PPPINFO; i++){
	if(
#if PPXP_TCL
	    1
#elif PPXP_PERL
	    info[i].type == PPXPINFO_STR
#endif
	    )
	    free(info[i].keyval[1]);
    }
    free(info);
}

struct ppxpinfo_ops PPxP_pppInfo = {
    pppinfo_new,
    pppinfo_parse,
    pppinfo_clear,
    pppinfo_delete
};

/************************************************************/
static struct keyval_s *
pwdinfo_new(void)
{
    static char *keywords[N_PWDINFO] = {
	"name", "password", "entry", "group", "script"
    };
    struct keyval_s *pwdinfo;
    int i;

    pwdinfo = (struct keyval_s *)
	malloc(sizeof(struct keyval_s)*N_PWDINFO);
    for(i = 0; i < N_PWDINFO; i++){
	pwdinfo[i].keyval[0] = keywords[i];
	pwdinfo[i].type = PPXPINFO_STR;
    }
    return(pwdinfo);
}

static int
pwdinfo_parse(struct keyval_s *pwdinfo, char *buf, int buflen)
{
    char *cp, *ep;
    int i;

    cp = buf;
    ep = buf + buflen;

    for(i = 0; i < N_PWDINFO && cp < ep; cp += strlen(cp) + 1, i++){
	pwdinfo[i].keyval[1] = cp;
    }
    return(i);
}

static void
pwdinfo_clear(struct keyval_s *pwdinfo)
{
}

static void
pwdinfo_delete(struct keyval_s *pwdinfo)
{
    free(pwdinfo);
}

struct ppxpinfo_ops PPxP_PwdInfo = {
    pwdinfo_new,
    pwdinfo_parse,
    pwdinfo_clear,
    pwdinfo_delete
};

/************************************************************/
#define CONFLAG_BUF_SIZE 32
#define CONID_BUF_SIZE 32

static struct keyval_s *
coninfo_new(void)
{
    static char *keywords[N_CONINFO] = {
	"name", "from", "flags", "cfd"
    };
    struct keyval_s *info;
    int i;

    info = (struct keyval_s *) malloc(sizeof(struct keyval_s)*N_CONINFO);
    for(i = 0; i < N_CONINFO; i++){
	info[i].keyval[0] = keywords[i];
	info[i].type = PPXPINFO_STR;
    }
    info[2].keyval[1] = (char *)malloc(sizeof(char)*CONFLAG_BUF_SIZE);
#if PPXP_TCL
    info[3].keyval[1] = (char *)malloc(sizeof(char)*CONID_BUF_SIZE);
#endif
    info[3].type = PPXPINFO_INT;

    return(info);
}

static struct number_string_s f_console[] = {
    { CONSOLE_AUTOF,	"autof" },
    { CONSOLE_AUTOC,	"autoc" },
    { CONSOLE_CURRENT,	"current" },
    { 0, NULL }
};

static int
coninfo_parse(struct keyval_s *info, char *buf, int buflen)
{
    char *cp, *ep;
    int i, cfd;

    cp = buf;
    info[0].keyval[1] = cp;	/* name */
    cp += strlen(cp) + 1;
    info[1].keyval[1] = cp;	/* from */
    cp += strlen(cp) + 1;
    f2s(*cp, f_console, info[2].keyval[1], CONFLAG_BUF_SIZE);
    cp++;
#if PPXP_PERL
    info[3].keyval[1] = cp;
#elif PPXP_TCL
    memcpy(&cfd, cp, sizeof(cfd));
    sprintf(info[3].keyval[1], "%d", cfd);
#endif
    return(N_CONINFO);
}

static void
coninfo_clear(struct keyval_s *info)
{
}

static void
coninfo_delete(struct keyval_s *info)
{
    free(info[2].keyval[1]);
#if PPXP_TCL
    free(info[2].keyval[1]);
#endif
    free(info);
}

struct ppxpinfo_ops PPxP_ConInfo = {
    coninfo_new,
    coninfo_parse,
    coninfo_clear,
    coninfo_delete
};
