
#include <Python.h>
#include <stdio.h>
#include <math.h>
#include "libnumarray.h"

#ifdef MS_WIN32
#pragma warning(once : 4244)
#endif

#define logical_and(arg1, arg2) (arg1 != 0) & (arg2 != 0)
#define logical_or(arg1, arg2)  (arg1 != 0) | (arg2 != 0)
#define logical_xor(arg1, arg2) ((arg1 != 0) ^ (arg2 != 0)) & 1
#define ufmaximum(arg1, arg2) (((temp1=arg1) > (temp2=arg2)) ? temp1 : temp2)
#define ufminimum(arg1, arg2) (((temp1=arg1) < (temp2=arg2)) ? temp1 : temp2)

static PyObject *_Error;
/*********************  minus  *********************/

static int minus_Int8_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8 *tin = (Int8 *) buffers[0];
    Int8 *tout = (Int8 *) buffers[1];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = -*tin;
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR2(minus_Int8_vector, sizeof(Int8), sizeof(Int8));
/*********************  add  *********************/

static int add_Int8_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8  *tin1  = (Int8 *) buffers[0];
    Int8  *tin2  = (Int8 *) buffers[1];
    Int8 *tout  = (Int8 *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = *tin1 + *tin2;
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3(add_Int8_vector_vector, sizeof(Int8), sizeof(Int8), sizeof(Int8));


static int add_Int8_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8 *tin1     = (Int8 *) buffers[0];
    Int8 tscalar   = *(Int8 *) buffers[1];
    Int8 *tout    = (Int8 *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = *tin1 + tscalar;
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3vs(add_Int8_vector_scalar, sizeof(Int8), sizeof(Int8), sizeof(Int8));


static int add_Int8_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8 tscalar = *(Int8 *) buffers[0];
    Int8 *tin2   =  (Int8 *) buffers[1];
    Int8 *tout  = (Int8 *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = tscalar + *tin2;
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3sv(add_Int8_scalar_vector, sizeof(Int8), sizeof(Int8), sizeof(Int8));


static void _add_Int8_accumulate(
            long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    Int8 *tin   = (Int8 *) ((char *) input  + inboffset);
    Int8 *tout = (Int8 *) ((char *) output + outboffset);
    Int8 lastval;
    
    if (dim == 0) {
        for (i=1; i<niters[dim]; i++) {
            lastval = *tout;
            tin = (Int8 *) ((char *) tin + inbstrides[dim]);
            tout = (Int8 *) ((char *) tout + outbstrides[dim]);
            *tout = lastval + *tin;
        }
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            _add_Int8_accumulate(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
}

static int add_Int8_accumulate(
            long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
            BEGIN_THREADS
            _add_Int8_accumulate(dim, dummy, niters,
                  input, inboffset, inbstrides, output, outboffset, outbstrides);
            END_THREADS
            return 0;
}

STRIDING_DESCR2(add_Int8_accumulate, CHECK_ALIGN, sizeof(Int8), sizeof(Int8));

static void _add_Int8_reduce(long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    Int8  *tin   = (Int8 *) ((char *) input  + inboffset);
    Int8 *tout  = (Int8 *) ((char *) output + outboffset);
    Int8 net;
    
    if (dim == 0) {
        net = *tout;
        for (i=1; i<niters[dim]; i++) {
            tin = (Int8 *) ((char *) tin + inbstrides[dim]);
            net = net + *tin;
        }
        *tout = net;
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            _add_Int8_reduce(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
}

static int add_Int8_reduce(long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
            BEGIN_THREADS
            _add_Int8_reduce(dim, dummy, niters,
                             input, inboffset, inbstrides, output, outboffset, outbstrides );
            END_THREADS
            return 0;
}

STRIDING_DESCR2(add_Int8_reduce, CHECK_ALIGN, sizeof(Int8), sizeof(Int8));
/*********************  subtract  *********************/

static int subtract_Int8_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8  *tin1  = (Int8 *) buffers[0];
    Int8  *tin2  = (Int8 *) buffers[1];
    Int8 *tout  = (Int8 *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = *tin1 - *tin2;
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3(subtract_Int8_vector_vector, sizeof(Int8), sizeof(Int8), sizeof(Int8));


static int subtract_Int8_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8 *tin1     = (Int8 *) buffers[0];
    Int8 tscalar   = *(Int8 *) buffers[1];
    Int8 *tout    = (Int8 *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = *tin1 - tscalar;
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3vs(subtract_Int8_vector_scalar, sizeof(Int8), sizeof(Int8), sizeof(Int8));


static int subtract_Int8_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8 tscalar = *(Int8 *) buffers[0];
    Int8 *tin2   =  (Int8 *) buffers[1];
    Int8 *tout  = (Int8 *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = tscalar - *tin2;
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3sv(subtract_Int8_scalar_vector, sizeof(Int8), sizeof(Int8), sizeof(Int8));


static void _subtract_Int8_accumulate(
            long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    Int8 *tin   = (Int8 *) ((char *) input  + inboffset);
    Int8 *tout = (Int8 *) ((char *) output + outboffset);
    Int8 lastval;
    
    if (dim == 0) {
        for (i=1; i<niters[dim]; i++) {
            lastval = *tout;
            tin = (Int8 *) ((char *) tin + inbstrides[dim]);
            tout = (Int8 *) ((char *) tout + outbstrides[dim]);
            *tout = lastval - *tin;
        }
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            _subtract_Int8_accumulate(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
}

static int subtract_Int8_accumulate(
            long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
            BEGIN_THREADS
            _subtract_Int8_accumulate(dim, dummy, niters,
                  input, inboffset, inbstrides, output, outboffset, outbstrides);
            END_THREADS
            return 0;
}

STRIDING_DESCR2(subtract_Int8_accumulate, CHECK_ALIGN, sizeof(Int8), sizeof(Int8));

static void _subtract_Int8_reduce(long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    Int8  *tin   = (Int8 *) ((char *) input  + inboffset);
    Int8 *tout  = (Int8 *) ((char *) output + outboffset);
    Int8 net;
    
    if (dim == 0) {
        net = *tout;
        for (i=1; i<niters[dim]; i++) {
            tin = (Int8 *) ((char *) tin + inbstrides[dim]);
            net = net - *tin;
        }
        *tout = net;
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            _subtract_Int8_reduce(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
}

static int subtract_Int8_reduce(long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
            BEGIN_THREADS
            _subtract_Int8_reduce(dim, dummy, niters,
                             input, inboffset, inbstrides, output, outboffset, outbstrides );
            END_THREADS
            return 0;
}

STRIDING_DESCR2(subtract_Int8_reduce, CHECK_ALIGN, sizeof(Int8), sizeof(Int8));
/*********************  multiply  *********************/

static int multiply_Int8_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8  *tin1  = (Int8 *) buffers[0];
    Int8  *tin2  = (Int8 *) buffers[1];
    Int8 *tout  = (Int8 *) buffers[2];
    Int32 temp;
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        temp = ((Int32) *tin1) * ((Int32) *tin2);
    if (temp > 127) temp = int_overflow_error(127.);
if (temp < -128) temp = int_overflow_error(-128.);
    *tout = (Int8) temp;
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3(multiply_Int8_vector_vector, sizeof(Int8), sizeof(Int8), sizeof(Int8));


static int multiply_Int8_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8 *tin1     = (Int8 *) buffers[0];
    Int8 tscalar   = *(Int8 *) buffers[1];
    Int8 *tout    = (Int8 *) buffers[2];
    Int32 temp;
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tout++) {
        temp = ((Int32) *tin1) * ((Int32) tscalar);
    if (temp > 127) temp = int_overflow_error(127.);
if (temp < -128) temp = int_overflow_error(-128.);
    *tout = (Int8) temp;
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3vs(multiply_Int8_vector_scalar, sizeof(Int8), sizeof(Int8), sizeof(Int8));


static int multiply_Int8_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8 tscalar = *(Int8 *) buffers[0];
    Int8 *tin2   =  (Int8 *) buffers[1];
    Int8 *tout  = (Int8 *) buffers[2];
    Int32 temp;
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin2++, tout++) {
        temp = ((Int32) tscalar) * ((Int32) *tin2);
    if (temp > 127) temp = int_overflow_error(127.);
if (temp < -128) temp = int_overflow_error(-128.);
    *tout = (Int8) temp;
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3sv(multiply_Int8_scalar_vector, sizeof(Int8), sizeof(Int8), sizeof(Int8));


static void _multiply_Int8_accumulate(
            long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    Int8 *tin   = (Int8 *) ((char *) input  + inboffset);
    Int8 *tout = (Int8 *) ((char *) output + outboffset);
    Int8 lastval;
    Int32 temp;
    if (dim == 0) {
        for (i=1; i<niters[dim]; i++) {
            lastval = *tout;
            tin = (Int8 *) ((char *) tin + inbstrides[dim]);
            tout = (Int8 *) ((char *) tout + outbstrides[dim]);
            temp = ((Int32) lastval) * ((Int32) *tin);
    if (temp > 127) temp = int_overflow_error(127.);
if (temp < -128) temp = int_overflow_error(-128.);
    *tout = (Int8) temp;
        }
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            _multiply_Int8_accumulate(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
}

static int multiply_Int8_accumulate(
            long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
            BEGIN_THREADS
            _multiply_Int8_accumulate(dim, dummy, niters,
                  input, inboffset, inbstrides, output, outboffset, outbstrides);
            END_THREADS
            return 0;
}

STRIDING_DESCR2(multiply_Int8_accumulate, CHECK_ALIGN, sizeof(Int8), sizeof(Int8));

static void _multiply_Int8_reduce(long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    Int8  *tin   = (Int8 *) ((char *) input  + inboffset);
    Int8 *tout  = (Int8 *) ((char *) output + outboffset);
    Int8 net;
    Int32 temp;
    if (dim == 0) {
        net = *tout;
        for (i=1; i<niters[dim]; i++) {
            tin = (Int8 *) ((char *) tin + inbstrides[dim]);
            temp = ((Int32) net) * ((Int32) *tin);
    if (temp > 127) temp = int_overflow_error(127.);
if (temp < -128) temp = int_overflow_error(-128.);
    net = (Int8) temp;
        }
        *tout = net;
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            _multiply_Int8_reduce(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
}

static int multiply_Int8_reduce(long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
            BEGIN_THREADS
            _multiply_Int8_reduce(dim, dummy, niters,
                             input, inboffset, inbstrides, output, outboffset, outbstrides );
            END_THREADS
            return 0;
}

STRIDING_DESCR2(multiply_Int8_reduce, CHECK_ALIGN, sizeof(Int8), sizeof(Int8));
/*********************  multiply  *********************/
/*********************  multiply  *********************/
/*********************  multiply  *********************/
/*********************  multiply  *********************/
/*********************  multiply  *********************/
/*********************  multiply  *********************/
/*********************  multiply  *********************/
/*********************  multiply  *********************/
/*********************  divide  *********************/

static int divide_Int8_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8  *tin1  = (Int8 *) buffers[0];
    Int8  *tin2  = (Int8 *) buffers[1];
    Int8 *tout  = (Int8 *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = ((*tin2==0) ? int_dividebyzero_error(*tin2, *tin1) : *tin1 / *tin2);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3(divide_Int8_vector_vector, sizeof(Int8), sizeof(Int8), sizeof(Int8));


static int divide_Int8_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8 *tin1     = (Int8 *) buffers[0];
    Int8 tscalar   = *(Int8 *) buffers[1];
    Int8 *tout    = (Int8 *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = ((tscalar==0) ? int_dividebyzero_error(tscalar, *tin1) : *tin1 / tscalar);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3vs(divide_Int8_vector_scalar, sizeof(Int8), sizeof(Int8), sizeof(Int8));


static int divide_Int8_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8 tscalar = *(Int8 *) buffers[0];
    Int8 *tin2   =  (Int8 *) buffers[1];
    Int8 *tout  = (Int8 *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = ((*tin2==0) ? int_dividebyzero_error(*tin2, 0) : tscalar / *tin2);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3sv(divide_Int8_scalar_vector, sizeof(Int8), sizeof(Int8), sizeof(Int8));


static void _divide_Int8_accumulate(
            long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    Int8 *tin   = (Int8 *) ((char *) input  + inboffset);
    Int8 *tout = (Int8 *) ((char *) output + outboffset);
    Int8 lastval;
    
    if (dim == 0) {
        for (i=1; i<niters[dim]; i++) {
            lastval = *tout;
            tin = (Int8 *) ((char *) tin + inbstrides[dim]);
            tout = (Int8 *) ((char *) tout + outbstrides[dim]);
            *tout = ((*tin==0) ? int_dividebyzero_error(*tin, 0) : lastval / *tin);
        }
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            _divide_Int8_accumulate(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
}

static int divide_Int8_accumulate(
            long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
            BEGIN_THREADS
            _divide_Int8_accumulate(dim, dummy, niters,
                  input, inboffset, inbstrides, output, outboffset, outbstrides);
            END_THREADS
            return 0;
}

STRIDING_DESCR2(divide_Int8_accumulate, CHECK_ALIGN, sizeof(Int8), sizeof(Int8));

static void _divide_Int8_reduce(long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    Int8  *tin   = (Int8 *) ((char *) input  + inboffset);
    Int8 *tout  = (Int8 *) ((char *) output + outboffset);
    Int8 net;
    
    if (dim == 0) {
        net = *tout;
        for (i=1; i<niters[dim]; i++) {
            tin = (Int8 *) ((char *) tin + inbstrides[dim]);
            net = ((*tin==0) ? int_dividebyzero_error(*tin, 0) : net / *tin);
        }
        *tout = net;
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            _divide_Int8_reduce(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
}

static int divide_Int8_reduce(long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
            BEGIN_THREADS
            _divide_Int8_reduce(dim, dummy, niters,
                             input, inboffset, inbstrides, output, outboffset, outbstrides );
            END_THREADS
            return 0;
}

STRIDING_DESCR2(divide_Int8_reduce, CHECK_ALIGN, sizeof(Int8), sizeof(Int8));
/*********************  divide  *********************/
/*********************  floor_divide  *********************/

static int floor_divide_Int8_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8  *tin1  = (Int8 *) buffers[0];
    Int8  *tin2  = (Int8 *) buffers[1];
    Int8 *tout  = (Int8 *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = ((*tin2==0) ? int_dividebyzero_error(*tin2, *tin1) : floor(*tin1 / (double) *tin2));
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3(floor_divide_Int8_vector_vector, sizeof(Int8), sizeof(Int8), sizeof(Int8));


static int floor_divide_Int8_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8 *tin1     = (Int8 *) buffers[0];
    Int8 tscalar   = *(Int8 *) buffers[1];
    Int8 *tout    = (Int8 *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = ((tscalar==0) ? int_dividebyzero_error(tscalar, *tin1) : floor(*tin1 / (double) tscalar));
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3vs(floor_divide_Int8_vector_scalar, sizeof(Int8), sizeof(Int8), sizeof(Int8));


static int floor_divide_Int8_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8 tscalar = *(Int8 *) buffers[0];
    Int8 *tin2   =  (Int8 *) buffers[1];
    Int8 *tout  = (Int8 *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = ((*tin2==0) ? int_dividebyzero_error(*tin2, 0) : floor(tscalar / (double) *tin2));
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3sv(floor_divide_Int8_scalar_vector, sizeof(Int8), sizeof(Int8), sizeof(Int8));


static void _floor_divide_Int8_accumulate(
            long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    Int8 *tin   = (Int8 *) ((char *) input  + inboffset);
    Int8 *tout = (Int8 *) ((char *) output + outboffset);
    Int8 lastval;
    
    if (dim == 0) {
        for (i=1; i<niters[dim]; i++) {
            lastval = *tout;
            tin = (Int8 *) ((char *) tin + inbstrides[dim]);
            tout = (Int8 *) ((char *) tout + outbstrides[dim]);
            *tout = ((*tin==0) ? int_dividebyzero_error(*tin, 0) : floor(lastval / (double) *tin));
        }
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            _floor_divide_Int8_accumulate(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
}

static int floor_divide_Int8_accumulate(
            long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
            BEGIN_THREADS
            _floor_divide_Int8_accumulate(dim, dummy, niters,
                  input, inboffset, inbstrides, output, outboffset, outbstrides);
            END_THREADS
            return 0;
}

STRIDING_DESCR2(floor_divide_Int8_accumulate, CHECK_ALIGN, sizeof(Int8), sizeof(Int8));

static void _floor_divide_Int8_reduce(long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    Int8  *tin   = (Int8 *) ((char *) input  + inboffset);
    Int8 *tout  = (Int8 *) ((char *) output + outboffset);
    Int8 net;
    
    if (dim == 0) {
        net = *tout;
        for (i=1; i<niters[dim]; i++) {
            tin = (Int8 *) ((char *) tin + inbstrides[dim]);
            net = ((*tin==0) ? int_dividebyzero_error(*tin, 0) : floor(net / (double) *tin));
        }
        *tout = net;
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            _floor_divide_Int8_reduce(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
}

static int floor_divide_Int8_reduce(long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
            BEGIN_THREADS
            _floor_divide_Int8_reduce(dim, dummy, niters,
                             input, inboffset, inbstrides, output, outboffset, outbstrides );
            END_THREADS
            return 0;
}

STRIDING_DESCR2(floor_divide_Int8_reduce, CHECK_ALIGN, sizeof(Int8), sizeof(Int8));
/*********************  floor_divide  *********************/
/*********************  true_divide  *********************/

static int true_divide_Int8_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8  *tin1  = (Int8 *) buffers[0];
    Int8  *tin2  = (Int8 *) buffers[1];
    Float32 *tout  = (Float32 *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = ((*tin2==0) ? int_dividebyzero_error(*tin2, *tin1) : *tin1 / (double) *tin2);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3(true_divide_Int8_vector_vector, sizeof(Int8), sizeof(Int8), sizeof(Float32));


static int true_divide_Int8_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8 *tin1     = (Int8 *) buffers[0];
    Int8 tscalar   = *(Int8 *) buffers[1];
    Float32 *tout    = (Float32 *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = ((tscalar==0) ? int_dividebyzero_error(tscalar, *tin1) : *tin1 / (double) tscalar);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3vs(true_divide_Int8_vector_scalar, sizeof(Int8), sizeof(Int8), sizeof(Float32));


static int true_divide_Int8_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8 tscalar = *(Int8 *) buffers[0];
    Int8 *tin2   =  (Int8 *) buffers[1];
    Float32 *tout  = (Float32 *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = ((*tin2==0) ? int_dividebyzero_error(*tin2, 0) : tscalar / (double) *tin2);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3sv(true_divide_Int8_scalar_vector, sizeof(Int8), sizeof(Int8), sizeof(Float32));

/*********************  true_divide  *********************/
/*********************  remainder  *********************/

static int remainder_Int8_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8  *tin1  = (Int8 *) buffers[0];
    Int8  *tin2  = (Int8 *) buffers[1];
    Int8 *tout  = (Int8 *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = ((*tin2==0) ? int_dividebyzero_error(*tin2, *tin1) : *tin1  %  *tin2);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3(remainder_Int8_vector_vector, sizeof(Int8), sizeof(Int8), sizeof(Int8));


static int remainder_Int8_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8 *tin1     = (Int8 *) buffers[0];
    Int8 tscalar   = *(Int8 *) buffers[1];
    Int8 *tout    = (Int8 *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = ((tscalar==0) ? int_dividebyzero_error(tscalar, *tin1) : *tin1  %  tscalar);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3vs(remainder_Int8_vector_scalar, sizeof(Int8), sizeof(Int8), sizeof(Int8));


static int remainder_Int8_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8 tscalar = *(Int8 *) buffers[0];
    Int8 *tin2   =  (Int8 *) buffers[1];
    Int8 *tout  = (Int8 *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = ((*tin2==0) ? int_dividebyzero_error(*tin2, 0) : tscalar  %  *tin2);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3sv(remainder_Int8_scalar_vector, sizeof(Int8), sizeof(Int8), sizeof(Int8));


static void _remainder_Int8_accumulate(
            long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    Int8 *tin   = (Int8 *) ((char *) input  + inboffset);
    Int8 *tout = (Int8 *) ((char *) output + outboffset);
    Int8 lastval;
    
    if (dim == 0) {
        for (i=1; i<niters[dim]; i++) {
            lastval = *tout;
            tin = (Int8 *) ((char *) tin + inbstrides[dim]);
            tout = (Int8 *) ((char *) tout + outbstrides[dim]);
            *tout = ((*tin==0) ? int_dividebyzero_error(*tin, 0) : lastval  %  *tin);
        }
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            _remainder_Int8_accumulate(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
}

static int remainder_Int8_accumulate(
            long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
            BEGIN_THREADS
            _remainder_Int8_accumulate(dim, dummy, niters,
                  input, inboffset, inbstrides, output, outboffset, outbstrides);
            END_THREADS
            return 0;
}

STRIDING_DESCR2(remainder_Int8_accumulate, CHECK_ALIGN, sizeof(Int8), sizeof(Int8));

static void _remainder_Int8_reduce(long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    Int8  *tin   = (Int8 *) ((char *) input  + inboffset);
    Int8 *tout  = (Int8 *) ((char *) output + outboffset);
    Int8 net;
    
    if (dim == 0) {
        net = *tout;
        for (i=1; i<niters[dim]; i++) {
            tin = (Int8 *) ((char *) tin + inbstrides[dim]);
            net = ((*tin==0) ? int_dividebyzero_error(*tin, 0) : net  %  *tin);
        }
        *tout = net;
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            _remainder_Int8_reduce(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
}

static int remainder_Int8_reduce(long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
            BEGIN_THREADS
            _remainder_Int8_reduce(dim, dummy, niters,
                             input, inboffset, inbstrides, output, outboffset, outbstrides );
            END_THREADS
            return 0;
}

STRIDING_DESCR2(remainder_Int8_reduce, CHECK_ALIGN, sizeof(Int8), sizeof(Int8));
/*********************  remainder  *********************/
/*********************  power  *********************/

static int power_Int8_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8  *tin1  = (Int8 *) buffers[0];
    Int8  *tin2  = (Int8 *) buffers[1];
    Int8 *tout  = (Int8 *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = num_pow(*tin1, *tin2);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3(power_Int8_vector_vector, sizeof(Int8), sizeof(Int8), sizeof(Int8));


static int power_Int8_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8 *tin1     = (Int8 *) buffers[0];
    Int8 tscalar   = *(Int8 *) buffers[1];
    Int8 *tout    = (Int8 *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = num_pow(*tin1, tscalar);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3vs(power_Int8_vector_scalar, sizeof(Int8), sizeof(Int8), sizeof(Int8));


static int power_Int8_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8 tscalar = *(Int8 *) buffers[0];
    Int8 *tin2   =  (Int8 *) buffers[1];
    Int8 *tout  = (Int8 *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = num_pow(tscalar, *tin2);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3sv(power_Int8_scalar_vector, sizeof(Int8), sizeof(Int8), sizeof(Int8));


static void _power_Int8_accumulate(
            long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    Int8 *tin   = (Int8 *) ((char *) input  + inboffset);
    Int8 *tout = (Int8 *) ((char *) output + outboffset);
    Int8 lastval;
    
    if (dim == 0) {
        for (i=1; i<niters[dim]; i++) {
            lastval = *tout;
            tin = (Int8 *) ((char *) tin + inbstrides[dim]);
            tout = (Int8 *) ((char *) tout + outbstrides[dim]);
            *tout = num_pow(lastval ,*tin);
        }
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            _power_Int8_accumulate(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
}

static int power_Int8_accumulate(
            long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
            BEGIN_THREADS
            _power_Int8_accumulate(dim, dummy, niters,
                  input, inboffset, inbstrides, output, outboffset, outbstrides);
            END_THREADS
            return 0;
}

STRIDING_DESCR2(power_Int8_accumulate, CHECK_ALIGN, sizeof(Int8), sizeof(Int8));

static void _power_Int8_reduce(long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    Int8  *tin   = (Int8 *) ((char *) input  + inboffset);
    Int8 *tout  = (Int8 *) ((char *) output + outboffset);
    Int8 net;
    
    if (dim == 0) {
        net = *tout;
        for (i=1; i<niters[dim]; i++) {
            tin = (Int8 *) ((char *) tin + inbstrides[dim]);
            net = num_pow(net, *tin);
        }
        *tout = net;
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            _power_Int8_reduce(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
}

static int power_Int8_reduce(long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
            BEGIN_THREADS
            _power_Int8_reduce(dim, dummy, niters,
                             input, inboffset, inbstrides, output, outboffset, outbstrides );
            END_THREADS
            return 0;
}

STRIDING_DESCR2(power_Int8_reduce, CHECK_ALIGN, sizeof(Int8), sizeof(Int8));
/*********************  abs  *********************/

static int abs_Int8_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8 *tin = (Int8 *) buffers[0];
    Int8 *tout = (Int8 *) buffers[1];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = fabs(*tin);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR2(abs_Int8_vector, sizeof(Int8), sizeof(Int8));
/*********************  sin  *********************/

static int sin_Int8_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8 *tin = (Int8 *) buffers[0];
    Float64 *tout = (Float64 *) buffers[1];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = sin(*tin);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR2(sin_Int8_vector, sizeof(Int8), sizeof(Float64));
/*********************  cos  *********************/

static int cos_Int8_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8 *tin = (Int8 *) buffers[0];
    Float64 *tout = (Float64 *) buffers[1];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = cos(*tin);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR2(cos_Int8_vector, sizeof(Int8), sizeof(Float64));
/*********************  tan  *********************/

static int tan_Int8_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8 *tin = (Int8 *) buffers[0];
    Float64 *tout = (Float64 *) buffers[1];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = tan(*tin);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR2(tan_Int8_vector, sizeof(Int8), sizeof(Float64));
/*********************  arcsin  *********************/

static int arcsin_Int8_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8 *tin = (Int8 *) buffers[0];
    Float64 *tout = (Float64 *) buffers[1];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = asin(*tin);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR2(arcsin_Int8_vector, sizeof(Int8), sizeof(Float64));
/*********************  arccos  *********************/

static int arccos_Int8_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8 *tin = (Int8 *) buffers[0];
    Float64 *tout = (Float64 *) buffers[1];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = acos(*tin);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR2(arccos_Int8_vector, sizeof(Int8), sizeof(Float64));
/*********************  arctan  *********************/

static int arctan_Int8_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8 *tin = (Int8 *) buffers[0];
    Float64 *tout = (Float64 *) buffers[1];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = atan(*tin);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR2(arctan_Int8_vector, sizeof(Int8), sizeof(Float64));
/*********************  arctan2  *********************/

static int arctan2_Int8_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8  *tin1  = (Int8 *) buffers[0];
    Int8  *tin2  = (Int8 *) buffers[1];
    Float64 *tout  = (Float64 *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = atan2(*tin1, *tin2);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3(arctan2_Int8_vector_vector, sizeof(Int8), sizeof(Int8), sizeof(Float64));


static int arctan2_Int8_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8 *tin1     = (Int8 *) buffers[0];
    Int8 tscalar   = *(Int8 *) buffers[1];
    Float64 *tout    = (Float64 *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = atan2(*tin1, tscalar);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3vs(arctan2_Int8_vector_scalar, sizeof(Int8), sizeof(Int8), sizeof(Float64));


static int arctan2_Int8_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8 tscalar = *(Int8 *) buffers[0];
    Int8 *tin2   =  (Int8 *) buffers[1];
    Float64 *tout  = (Float64 *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = atan2(tscalar, *tin2);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3sv(arctan2_Int8_scalar_vector, sizeof(Int8), sizeof(Int8), sizeof(Float64));

/*********************  log  *********************/

static int log_Int8_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8 *tin = (Int8 *) buffers[0];
    Float64 *tout = (Float64 *) buffers[1];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = num_log(*tin);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR2(log_Int8_vector, sizeof(Int8), sizeof(Float64));
/*********************  log10  *********************/

static int log10_Int8_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8 *tin = (Int8 *) buffers[0];
    Float64 *tout = (Float64 *) buffers[1];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = num_log10(*tin);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR2(log10_Int8_vector, sizeof(Int8), sizeof(Float64));
/*********************  exp  *********************/

static int exp_Int8_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8 *tin = (Int8 *) buffers[0];
    Float64 *tout = (Float64 *) buffers[1];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = exp(*tin);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR2(exp_Int8_vector, sizeof(Int8), sizeof(Float64));
/*********************  sinh  *********************/

static int sinh_Int8_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8 *tin = (Int8 *) buffers[0];
    Float64 *tout = (Float64 *) buffers[1];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = sinh(*tin);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR2(sinh_Int8_vector, sizeof(Int8), sizeof(Float64));
/*********************  cosh  *********************/

static int cosh_Int8_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8 *tin = (Int8 *) buffers[0];
    Float64 *tout = (Float64 *) buffers[1];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = cosh(*tin);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR2(cosh_Int8_vector, sizeof(Int8), sizeof(Float64));
/*********************  tanh  *********************/

static int tanh_Int8_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8 *tin = (Int8 *) buffers[0];
    Float64 *tout = (Float64 *) buffers[1];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = tanh(*tin);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR2(tanh_Int8_vector, sizeof(Int8), sizeof(Float64));
/*********************  arcsinh  *********************/

static int arcsinh_Int8_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8 *tin = (Int8 *) buffers[0];
    Float64 *tout = (Float64 *) buffers[1];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = num_asinh(*tin);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR2(arcsinh_Int8_vector, sizeof(Int8), sizeof(Float64));
/*********************  arccosh  *********************/

static int arccosh_Int8_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8 *tin = (Int8 *) buffers[0];
    Float64 *tout = (Float64 *) buffers[1];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = num_acosh(*tin);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR2(arccosh_Int8_vector, sizeof(Int8), sizeof(Float64));
/*********************  arctanh  *********************/

static int arctanh_Int8_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8 *tin = (Int8 *) buffers[0];
    Float64 *tout = (Float64 *) buffers[1];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = num_atanh(*tin);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR2(arctanh_Int8_vector, sizeof(Int8), sizeof(Float64));
/*********************  ieeemask  *********************/
/*********************  ieeemask  *********************/
/*********************  sqrt  *********************/

static int sqrt_Int8_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8 *tin = (Int8 *) buffers[0];
    Float64 *tout = (Float64 *) buffers[1];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = sqrt(*tin);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR2(sqrt_Int8_vector, sizeof(Int8), sizeof(Float64));
/*********************  equal  *********************/

static int equal_Int8_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8  *tin1  = (Int8 *) buffers[0];
    Int8  *tin2  = (Int8 *) buffers[1];
    Bool *tout  = (Bool *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = *tin1 == *tin2;
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3(equal_Int8_vector_vector, sizeof(Int8), sizeof(Int8), sizeof(Bool));


static int equal_Int8_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8 *tin1     = (Int8 *) buffers[0];
    Int8 tscalar   = *(Int8 *) buffers[1];
    Bool *tout    = (Bool *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = *tin1 == tscalar;
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3vs(equal_Int8_vector_scalar, sizeof(Int8), sizeof(Int8), sizeof(Bool));


static int equal_Int8_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8 tscalar = *(Int8 *) buffers[0];
    Int8 *tin2   =  (Int8 *) buffers[1];
    Bool *tout  = (Bool *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = tscalar == *tin2;
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3sv(equal_Int8_scalar_vector, sizeof(Int8), sizeof(Int8), sizeof(Bool));

/*********************  not_equal  *********************/

static int not_equal_Int8_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8  *tin1  = (Int8 *) buffers[0];
    Int8  *tin2  = (Int8 *) buffers[1];
    Bool *tout  = (Bool *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = *tin1 != *tin2;
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3(not_equal_Int8_vector_vector, sizeof(Int8), sizeof(Int8), sizeof(Bool));


static int not_equal_Int8_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8 *tin1     = (Int8 *) buffers[0];
    Int8 tscalar   = *(Int8 *) buffers[1];
    Bool *tout    = (Bool *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = *tin1 != tscalar;
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3vs(not_equal_Int8_vector_scalar, sizeof(Int8), sizeof(Int8), sizeof(Bool));


static int not_equal_Int8_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8 tscalar = *(Int8 *) buffers[0];
    Int8 *tin2   =  (Int8 *) buffers[1];
    Bool *tout  = (Bool *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = tscalar != *tin2;
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3sv(not_equal_Int8_scalar_vector, sizeof(Int8), sizeof(Int8), sizeof(Bool));

/*********************  greater  *********************/

static int greater_Int8_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8  *tin1  = (Int8 *) buffers[0];
    Int8  *tin2  = (Int8 *) buffers[1];
    Bool *tout  = (Bool *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = *tin1 > *tin2;
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3(greater_Int8_vector_vector, sizeof(Int8), sizeof(Int8), sizeof(Bool));


static int greater_Int8_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8 *tin1     = (Int8 *) buffers[0];
    Int8 tscalar   = *(Int8 *) buffers[1];
    Bool *tout    = (Bool *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = *tin1 > tscalar;
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3vs(greater_Int8_vector_scalar, sizeof(Int8), sizeof(Int8), sizeof(Bool));


static int greater_Int8_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8 tscalar = *(Int8 *) buffers[0];
    Int8 *tin2   =  (Int8 *) buffers[1];
    Bool *tout  = (Bool *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = tscalar > *tin2;
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3sv(greater_Int8_scalar_vector, sizeof(Int8), sizeof(Int8), sizeof(Bool));

/*********************  greater_equal  *********************/

static int greater_equal_Int8_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8  *tin1  = (Int8 *) buffers[0];
    Int8  *tin2  = (Int8 *) buffers[1];
    Bool *tout  = (Bool *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = *tin1 >= *tin2;
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3(greater_equal_Int8_vector_vector, sizeof(Int8), sizeof(Int8), sizeof(Bool));


static int greater_equal_Int8_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8 *tin1     = (Int8 *) buffers[0];
    Int8 tscalar   = *(Int8 *) buffers[1];
    Bool *tout    = (Bool *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = *tin1 >= tscalar;
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3vs(greater_equal_Int8_vector_scalar, sizeof(Int8), sizeof(Int8), sizeof(Bool));


static int greater_equal_Int8_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8 tscalar = *(Int8 *) buffers[0];
    Int8 *tin2   =  (Int8 *) buffers[1];
    Bool *tout  = (Bool *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = tscalar >= *tin2;
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3sv(greater_equal_Int8_scalar_vector, sizeof(Int8), sizeof(Int8), sizeof(Bool));

/*********************  less  *********************/

static int less_Int8_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8  *tin1  = (Int8 *) buffers[0];
    Int8  *tin2  = (Int8 *) buffers[1];
    Bool *tout  = (Bool *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = *tin1 < *tin2;
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3(less_Int8_vector_vector, sizeof(Int8), sizeof(Int8), sizeof(Bool));


static int less_Int8_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8 *tin1     = (Int8 *) buffers[0];
    Int8 tscalar   = *(Int8 *) buffers[1];
    Bool *tout    = (Bool *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = *tin1 < tscalar;
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3vs(less_Int8_vector_scalar, sizeof(Int8), sizeof(Int8), sizeof(Bool));


static int less_Int8_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8 tscalar = *(Int8 *) buffers[0];
    Int8 *tin2   =  (Int8 *) buffers[1];
    Bool *tout  = (Bool *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = tscalar < *tin2;
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3sv(less_Int8_scalar_vector, sizeof(Int8), sizeof(Int8), sizeof(Bool));

/*********************  less_equal  *********************/

static int less_equal_Int8_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8  *tin1  = (Int8 *) buffers[0];
    Int8  *tin2  = (Int8 *) buffers[1];
    Bool *tout  = (Bool *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = *tin1 <= *tin2;
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3(less_equal_Int8_vector_vector, sizeof(Int8), sizeof(Int8), sizeof(Bool));


static int less_equal_Int8_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8 *tin1     = (Int8 *) buffers[0];
    Int8 tscalar   = *(Int8 *) buffers[1];
    Bool *tout    = (Bool *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = *tin1 <= tscalar;
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3vs(less_equal_Int8_vector_scalar, sizeof(Int8), sizeof(Int8), sizeof(Bool));


static int less_equal_Int8_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8 tscalar = *(Int8 *) buffers[0];
    Int8 *tin2   =  (Int8 *) buffers[1];
    Bool *tout  = (Bool *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = tscalar <= *tin2;
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3sv(less_equal_Int8_scalar_vector, sizeof(Int8), sizeof(Int8), sizeof(Bool));

/*********************  logical_and  *********************/

static int logical_and_Int8_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8  *tin1  = (Int8 *) buffers[0];
    Int8  *tin2  = (Int8 *) buffers[1];
    Bool *tout  = (Bool *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = logical_and(*tin1, *tin2);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3(logical_and_Int8_vector_vector, sizeof(Int8), sizeof(Int8), sizeof(Bool));


static int logical_and_Int8_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8 *tin1     = (Int8 *) buffers[0];
    Int8 tscalar   = *(Int8 *) buffers[1];
    Bool *tout    = (Bool *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = logical_and(*tin1, tscalar);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3vs(logical_and_Int8_vector_scalar, sizeof(Int8), sizeof(Int8), sizeof(Bool));


static int logical_and_Int8_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8 tscalar = *(Int8 *) buffers[0];
    Int8 *tin2   =  (Int8 *) buffers[1];
    Bool *tout  = (Bool *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = logical_and(tscalar, *tin2);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3sv(logical_and_Int8_scalar_vector, sizeof(Int8), sizeof(Int8), sizeof(Bool));

/*********************  logical_or  *********************/

static int logical_or_Int8_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8  *tin1  = (Int8 *) buffers[0];
    Int8  *tin2  = (Int8 *) buffers[1];
    Bool *tout  = (Bool *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = logical_or(*tin1, *tin2);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3(logical_or_Int8_vector_vector, sizeof(Int8), sizeof(Int8), sizeof(Bool));


static int logical_or_Int8_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8 *tin1     = (Int8 *) buffers[0];
    Int8 tscalar   = *(Int8 *) buffers[1];
    Bool *tout    = (Bool *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = logical_or(*tin1, tscalar);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3vs(logical_or_Int8_vector_scalar, sizeof(Int8), sizeof(Int8), sizeof(Bool));


static int logical_or_Int8_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8 tscalar = *(Int8 *) buffers[0];
    Int8 *tin2   =  (Int8 *) buffers[1];
    Bool *tout  = (Bool *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = logical_or(tscalar, *tin2);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3sv(logical_or_Int8_scalar_vector, sizeof(Int8), sizeof(Int8), sizeof(Bool));

/*********************  logical_xor  *********************/

static int logical_xor_Int8_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8  *tin1  = (Int8 *) buffers[0];
    Int8  *tin2  = (Int8 *) buffers[1];
    Bool *tout  = (Bool *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = logical_xor(*tin1, *tin2);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3(logical_xor_Int8_vector_vector, sizeof(Int8), sizeof(Int8), sizeof(Bool));


static int logical_xor_Int8_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8 *tin1     = (Int8 *) buffers[0];
    Int8 tscalar   = *(Int8 *) buffers[1];
    Bool *tout    = (Bool *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = logical_xor(*tin1, tscalar);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3vs(logical_xor_Int8_vector_scalar, sizeof(Int8), sizeof(Int8), sizeof(Bool));


static int logical_xor_Int8_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8 tscalar = *(Int8 *) buffers[0];
    Int8 *tin2   =  (Int8 *) buffers[1];
    Bool *tout  = (Bool *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = logical_xor(tscalar, *tin2);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3sv(logical_xor_Int8_scalar_vector, sizeof(Int8), sizeof(Int8), sizeof(Bool));

/*********************  logical_not  *********************/

static int logical_not_Int8_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8 *tin = (Int8 *) buffers[0];
    Bool *tout = (Bool *) buffers[1];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = !*tin;
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR2(logical_not_Int8_vector, sizeof(Int8), sizeof(Bool));
/*********************  bitwise_and  *********************/

static int bitwise_and_Int8_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8  *tin1  = (Int8 *) buffers[0];
    Int8  *tin2  = (Int8 *) buffers[1];
    Int8 *tout  = (Int8 *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = *tin1 & *tin2;
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3(bitwise_and_Int8_vector_vector, sizeof(Int8), sizeof(Int8), sizeof(Int8));


static int bitwise_and_Int8_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8 *tin1     = (Int8 *) buffers[0];
    Int8 tscalar   = *(Int8 *) buffers[1];
    Int8 *tout    = (Int8 *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = *tin1 & tscalar;
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3vs(bitwise_and_Int8_vector_scalar, sizeof(Int8), sizeof(Int8), sizeof(Int8));


static int bitwise_and_Int8_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8 tscalar = *(Int8 *) buffers[0];
    Int8 *tin2   =  (Int8 *) buffers[1];
    Int8 *tout  = (Int8 *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = tscalar & *tin2;
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3sv(bitwise_and_Int8_scalar_vector, sizeof(Int8), sizeof(Int8), sizeof(Int8));


static void _bitwise_and_Int8_accumulate(
            long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    Int8 *tin   = (Int8 *) ((char *) input  + inboffset);
    Int8 *tout = (Int8 *) ((char *) output + outboffset);
    Int8 lastval;
    
    if (dim == 0) {
        for (i=1; i<niters[dim]; i++) {
            lastval = *tout;
            tin = (Int8 *) ((char *) tin + inbstrides[dim]);
            tout = (Int8 *) ((char *) tout + outbstrides[dim]);
            *tout = lastval & *tin;
        }
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            _bitwise_and_Int8_accumulate(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
}

static int bitwise_and_Int8_accumulate(
            long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
            BEGIN_THREADS
            _bitwise_and_Int8_accumulate(dim, dummy, niters,
                  input, inboffset, inbstrides, output, outboffset, outbstrides);
            END_THREADS
            return 0;
}

STRIDING_DESCR2(bitwise_and_Int8_accumulate, CHECK_ALIGN, sizeof(Int8), sizeof(Int8));

static void _bitwise_and_Int8_reduce(long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    Int8  *tin   = (Int8 *) ((char *) input  + inboffset);
    Int8 *tout  = (Int8 *) ((char *) output + outboffset);
    Int8 net;
    
    if (dim == 0) {
        net = *tout;
        for (i=1; i<niters[dim]; i++) {
            tin = (Int8 *) ((char *) tin + inbstrides[dim]);
            net = net & *tin;
        }
        *tout = net;
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            _bitwise_and_Int8_reduce(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
}

static int bitwise_and_Int8_reduce(long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
            BEGIN_THREADS
            _bitwise_and_Int8_reduce(dim, dummy, niters,
                             input, inboffset, inbstrides, output, outboffset, outbstrides );
            END_THREADS
            return 0;
}

STRIDING_DESCR2(bitwise_and_Int8_reduce, CHECK_ALIGN, sizeof(Int8), sizeof(Int8));
/*********************  bitwise_or  *********************/

static int bitwise_or_Int8_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8  *tin1  = (Int8 *) buffers[0];
    Int8  *tin2  = (Int8 *) buffers[1];
    Int8 *tout  = (Int8 *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = *tin1 | *tin2;
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3(bitwise_or_Int8_vector_vector, sizeof(Int8), sizeof(Int8), sizeof(Int8));


static int bitwise_or_Int8_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8 *tin1     = (Int8 *) buffers[0];
    Int8 tscalar   = *(Int8 *) buffers[1];
    Int8 *tout    = (Int8 *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = *tin1 | tscalar;
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3vs(bitwise_or_Int8_vector_scalar, sizeof(Int8), sizeof(Int8), sizeof(Int8));


static int bitwise_or_Int8_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8 tscalar = *(Int8 *) buffers[0];
    Int8 *tin2   =  (Int8 *) buffers[1];
    Int8 *tout  = (Int8 *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = tscalar | *tin2;
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3sv(bitwise_or_Int8_scalar_vector, sizeof(Int8), sizeof(Int8), sizeof(Int8));


static void _bitwise_or_Int8_accumulate(
            long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    Int8 *tin   = (Int8 *) ((char *) input  + inboffset);
    Int8 *tout = (Int8 *) ((char *) output + outboffset);
    Int8 lastval;
    
    if (dim == 0) {
        for (i=1; i<niters[dim]; i++) {
            lastval = *tout;
            tin = (Int8 *) ((char *) tin + inbstrides[dim]);
            tout = (Int8 *) ((char *) tout + outbstrides[dim]);
            *tout = lastval | *tin;
        }
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            _bitwise_or_Int8_accumulate(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
}

static int bitwise_or_Int8_accumulate(
            long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
            BEGIN_THREADS
            _bitwise_or_Int8_accumulate(dim, dummy, niters,
                  input, inboffset, inbstrides, output, outboffset, outbstrides);
            END_THREADS
            return 0;
}

STRIDING_DESCR2(bitwise_or_Int8_accumulate, CHECK_ALIGN, sizeof(Int8), sizeof(Int8));

static void _bitwise_or_Int8_reduce(long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    Int8  *tin   = (Int8 *) ((char *) input  + inboffset);
    Int8 *tout  = (Int8 *) ((char *) output + outboffset);
    Int8 net;
    
    if (dim == 0) {
        net = *tout;
        for (i=1; i<niters[dim]; i++) {
            tin = (Int8 *) ((char *) tin + inbstrides[dim]);
            net = net | *tin;
        }
        *tout = net;
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            _bitwise_or_Int8_reduce(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
}

static int bitwise_or_Int8_reduce(long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
            BEGIN_THREADS
            _bitwise_or_Int8_reduce(dim, dummy, niters,
                             input, inboffset, inbstrides, output, outboffset, outbstrides );
            END_THREADS
            return 0;
}

STRIDING_DESCR2(bitwise_or_Int8_reduce, CHECK_ALIGN, sizeof(Int8), sizeof(Int8));
/*********************  bitwise_xor  *********************/

static int bitwise_xor_Int8_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8  *tin1  = (Int8 *) buffers[0];
    Int8  *tin2  = (Int8 *) buffers[1];
    Int8 *tout  = (Int8 *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = *tin1 ^ *tin2;
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3(bitwise_xor_Int8_vector_vector, sizeof(Int8), sizeof(Int8), sizeof(Int8));


static int bitwise_xor_Int8_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8 *tin1     = (Int8 *) buffers[0];
    Int8 tscalar   = *(Int8 *) buffers[1];
    Int8 *tout    = (Int8 *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = *tin1 ^ tscalar;
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3vs(bitwise_xor_Int8_vector_scalar, sizeof(Int8), sizeof(Int8), sizeof(Int8));


static int bitwise_xor_Int8_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8 tscalar = *(Int8 *) buffers[0];
    Int8 *tin2   =  (Int8 *) buffers[1];
    Int8 *tout  = (Int8 *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = tscalar ^ *tin2;
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3sv(bitwise_xor_Int8_scalar_vector, sizeof(Int8), sizeof(Int8), sizeof(Int8));


static void _bitwise_xor_Int8_accumulate(
            long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    Int8 *tin   = (Int8 *) ((char *) input  + inboffset);
    Int8 *tout = (Int8 *) ((char *) output + outboffset);
    Int8 lastval;
    
    if (dim == 0) {
        for (i=1; i<niters[dim]; i++) {
            lastval = *tout;
            tin = (Int8 *) ((char *) tin + inbstrides[dim]);
            tout = (Int8 *) ((char *) tout + outbstrides[dim]);
            *tout = lastval ^ *tin;
        }
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            _bitwise_xor_Int8_accumulate(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
}

static int bitwise_xor_Int8_accumulate(
            long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
            BEGIN_THREADS
            _bitwise_xor_Int8_accumulate(dim, dummy, niters,
                  input, inboffset, inbstrides, output, outboffset, outbstrides);
            END_THREADS
            return 0;
}

STRIDING_DESCR2(bitwise_xor_Int8_accumulate, CHECK_ALIGN, sizeof(Int8), sizeof(Int8));

static void _bitwise_xor_Int8_reduce(long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    Int8  *tin   = (Int8 *) ((char *) input  + inboffset);
    Int8 *tout  = (Int8 *) ((char *) output + outboffset);
    Int8 net;
    
    if (dim == 0) {
        net = *tout;
        for (i=1; i<niters[dim]; i++) {
            tin = (Int8 *) ((char *) tin + inbstrides[dim]);
            net = net ^ *tin;
        }
        *tout = net;
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            _bitwise_xor_Int8_reduce(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
}

static int bitwise_xor_Int8_reduce(long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
            BEGIN_THREADS
            _bitwise_xor_Int8_reduce(dim, dummy, niters,
                             input, inboffset, inbstrides, output, outboffset, outbstrides );
            END_THREADS
            return 0;
}

STRIDING_DESCR2(bitwise_xor_Int8_reduce, CHECK_ALIGN, sizeof(Int8), sizeof(Int8));
/*********************  bitwise_not  *********************/

static int bitwise_not_Int8_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8 *tin = (Int8 *) buffers[0];
    Int8 *tout = (Int8 *) buffers[1];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = ~*tin;
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR2(bitwise_not_Int8_vector, sizeof(Int8), sizeof(Int8));
/*********************  bitwise_not  *********************/
/*********************  rshift  *********************/

static int rshift_Int8_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8  *tin1  = (Int8 *) buffers[0];
    Int8  *tin2  = (Int8 *) buffers[1];
    Int8 *tout  = (Int8 *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = *tin1 >> *tin2;
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3(rshift_Int8_vector_vector, sizeof(Int8), sizeof(Int8), sizeof(Int8));


static int rshift_Int8_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8 *tin1     = (Int8 *) buffers[0];
    Int8 tscalar   = *(Int8 *) buffers[1];
    Int8 *tout    = (Int8 *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = *tin1 >> tscalar;
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3vs(rshift_Int8_vector_scalar, sizeof(Int8), sizeof(Int8), sizeof(Int8));


static int rshift_Int8_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8 tscalar = *(Int8 *) buffers[0];
    Int8 *tin2   =  (Int8 *) buffers[1];
    Int8 *tout  = (Int8 *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = tscalar >> *tin2;
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3sv(rshift_Int8_scalar_vector, sizeof(Int8), sizeof(Int8), sizeof(Int8));


static void _rshift_Int8_accumulate(
            long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    Int8 *tin   = (Int8 *) ((char *) input  + inboffset);
    Int8 *tout = (Int8 *) ((char *) output + outboffset);
    Int8 lastval;
    
    if (dim == 0) {
        for (i=1; i<niters[dim]; i++) {
            lastval = *tout;
            tin = (Int8 *) ((char *) tin + inbstrides[dim]);
            tout = (Int8 *) ((char *) tout + outbstrides[dim]);
            *tout = lastval >> *tin;
        }
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            _rshift_Int8_accumulate(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
}

static int rshift_Int8_accumulate(
            long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
            BEGIN_THREADS
            _rshift_Int8_accumulate(dim, dummy, niters,
                  input, inboffset, inbstrides, output, outboffset, outbstrides);
            END_THREADS
            return 0;
}

STRIDING_DESCR2(rshift_Int8_accumulate, CHECK_ALIGN, sizeof(Int8), sizeof(Int8));

static void _rshift_Int8_reduce(long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    Int8  *tin   = (Int8 *) ((char *) input  + inboffset);
    Int8 *tout  = (Int8 *) ((char *) output + outboffset);
    Int8 net;
    
    if (dim == 0) {
        net = *tout;
        for (i=1; i<niters[dim]; i++) {
            tin = (Int8 *) ((char *) tin + inbstrides[dim]);
            net = net >> *tin;
        }
        *tout = net;
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            _rshift_Int8_reduce(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
}

static int rshift_Int8_reduce(long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
            BEGIN_THREADS
            _rshift_Int8_reduce(dim, dummy, niters,
                             input, inboffset, inbstrides, output, outboffset, outbstrides );
            END_THREADS
            return 0;
}

STRIDING_DESCR2(rshift_Int8_reduce, CHECK_ALIGN, sizeof(Int8), sizeof(Int8));
/*********************  lshift  *********************/

static int lshift_Int8_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8  *tin1  = (Int8 *) buffers[0];
    Int8  *tin2  = (Int8 *) buffers[1];
    Int8 *tout  = (Int8 *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = *tin1 << *tin2;
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3(lshift_Int8_vector_vector, sizeof(Int8), sizeof(Int8), sizeof(Int8));


static int lshift_Int8_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8 *tin1     = (Int8 *) buffers[0];
    Int8 tscalar   = *(Int8 *) buffers[1];
    Int8 *tout    = (Int8 *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = *tin1 << tscalar;
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3vs(lshift_Int8_vector_scalar, sizeof(Int8), sizeof(Int8), sizeof(Int8));


static int lshift_Int8_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8 tscalar = *(Int8 *) buffers[0];
    Int8 *tin2   =  (Int8 *) buffers[1];
    Int8 *tout  = (Int8 *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = tscalar << *tin2;
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3sv(lshift_Int8_scalar_vector, sizeof(Int8), sizeof(Int8), sizeof(Int8));


static void _lshift_Int8_accumulate(
            long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    Int8 *tin   = (Int8 *) ((char *) input  + inboffset);
    Int8 *tout = (Int8 *) ((char *) output + outboffset);
    Int8 lastval;
    
    if (dim == 0) {
        for (i=1; i<niters[dim]; i++) {
            lastval = *tout;
            tin = (Int8 *) ((char *) tin + inbstrides[dim]);
            tout = (Int8 *) ((char *) tout + outbstrides[dim]);
            *tout = lastval << *tin;
        }
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            _lshift_Int8_accumulate(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
}

static int lshift_Int8_accumulate(
            long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
            BEGIN_THREADS
            _lshift_Int8_accumulate(dim, dummy, niters,
                  input, inboffset, inbstrides, output, outboffset, outbstrides);
            END_THREADS
            return 0;
}

STRIDING_DESCR2(lshift_Int8_accumulate, CHECK_ALIGN, sizeof(Int8), sizeof(Int8));

static void _lshift_Int8_reduce(long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    Int8  *tin   = (Int8 *) ((char *) input  + inboffset);
    Int8 *tout  = (Int8 *) ((char *) output + outboffset);
    Int8 net;
    
    if (dim == 0) {
        net = *tout;
        for (i=1; i<niters[dim]; i++) {
            tin = (Int8 *) ((char *) tin + inbstrides[dim]);
            net = net << *tin;
        }
        *tout = net;
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            _lshift_Int8_reduce(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
}

static int lshift_Int8_reduce(long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
            BEGIN_THREADS
            _lshift_Int8_reduce(dim, dummy, niters,
                             input, inboffset, inbstrides, output, outboffset, outbstrides );
            END_THREADS
            return 0;
}

STRIDING_DESCR2(lshift_Int8_reduce, CHECK_ALIGN, sizeof(Int8), sizeof(Int8));
/*********************  floor  *********************/

static int floor_Int8_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8 *tin = (Int8 *) buffers[0];
    Int8 *tout = (Int8 *) buffers[1];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = (*tin);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR2(floor_Int8_vector, sizeof(Int8), sizeof(Int8));
/*********************  floor  *********************/
/*********************  ceil  *********************/

static int ceil_Int8_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8 *tin = (Int8 *) buffers[0];
    Int8 *tout = (Int8 *) buffers[1];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = (*tin);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR2(ceil_Int8_vector, sizeof(Int8), sizeof(Int8));
/*********************  ceil  *********************/
/*********************  maximum  *********************/

static int maximum_Int8_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8  *tin1  = (Int8 *) buffers[0];
    Int8  *tin2  = (Int8 *) buffers[1];
    Int8 *tout  = (Int8 *) buffers[2];
    Int8 temp1, temp2;
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = ufmaximum(*tin1, *tin2);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3(maximum_Int8_vector_vector, sizeof(Int8), sizeof(Int8), sizeof(Int8));


static int maximum_Int8_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8 *tin1     = (Int8 *) buffers[0];
    Int8 tscalar   = *(Int8 *) buffers[1];
    Int8 *tout    = (Int8 *) buffers[2];
    Int8 temp1, temp2;
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = ufmaximum(*tin1, tscalar);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3vs(maximum_Int8_vector_scalar, sizeof(Int8), sizeof(Int8), sizeof(Int8));


static int maximum_Int8_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8 tscalar = *(Int8 *) buffers[0];
    Int8 *tin2   =  (Int8 *) buffers[1];
    Int8 *tout  = (Int8 *) buffers[2];
    Int8 temp1, temp2;
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = ufmaximum(tscalar, *tin2);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3sv(maximum_Int8_scalar_vector, sizeof(Int8), sizeof(Int8), sizeof(Int8));


static void _maximum_Int8_accumulate(
            long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    Int8 *tin   = (Int8 *) ((char *) input  + inboffset);
    Int8 *tout = (Int8 *) ((char *) output + outboffset);
    Int8 lastval;
    Int8 temp1, temp2;
    if (dim == 0) {
        for (i=1; i<niters[dim]; i++) {
            lastval = *tout;
            tin = (Int8 *) ((char *) tin + inbstrides[dim]);
            tout = (Int8 *) ((char *) tout + outbstrides[dim]);
            *tout = ufmaximum(lastval ,*tin);
        }
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            _maximum_Int8_accumulate(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
}

static int maximum_Int8_accumulate(
            long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
            BEGIN_THREADS
            _maximum_Int8_accumulate(dim, dummy, niters,
                  input, inboffset, inbstrides, output, outboffset, outbstrides);
            END_THREADS
            return 0;
}

STRIDING_DESCR2(maximum_Int8_accumulate, CHECK_ALIGN, sizeof(Int8), sizeof(Int8));

static void _maximum_Int8_reduce(long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    Int8  *tin   = (Int8 *) ((char *) input  + inboffset);
    Int8 *tout  = (Int8 *) ((char *) output + outboffset);
    Int8 net;
    Int8 temp1, temp2;
    if (dim == 0) {
        net = *tout;
        for (i=1; i<niters[dim]; i++) {
            tin = (Int8 *) ((char *) tin + inbstrides[dim]);
            net = ufmaximum(net, *tin);
        }
        *tout = net;
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            _maximum_Int8_reduce(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
}

static int maximum_Int8_reduce(long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
            BEGIN_THREADS
            _maximum_Int8_reduce(dim, dummy, niters,
                             input, inboffset, inbstrides, output, outboffset, outbstrides );
            END_THREADS
            return 0;
}

STRIDING_DESCR2(maximum_Int8_reduce, CHECK_ALIGN, sizeof(Int8), sizeof(Int8));
/*********************  minimum  *********************/

static int minimum_Int8_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8  *tin1  = (Int8 *) buffers[0];
    Int8  *tin2  = (Int8 *) buffers[1];
    Int8 *tout  = (Int8 *) buffers[2];
    Int8 temp1, temp2;
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = ufminimum(*tin1, *tin2);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3(minimum_Int8_vector_vector, sizeof(Int8), sizeof(Int8), sizeof(Int8));


static int minimum_Int8_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8 *tin1     = (Int8 *) buffers[0];
    Int8 tscalar   = *(Int8 *) buffers[1];
    Int8 *tout    = (Int8 *) buffers[2];
    Int8 temp1, temp2;
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = ufminimum(*tin1, tscalar);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3vs(minimum_Int8_vector_scalar, sizeof(Int8), sizeof(Int8), sizeof(Int8));


static int minimum_Int8_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8 tscalar = *(Int8 *) buffers[0];
    Int8 *tin2   =  (Int8 *) buffers[1];
    Int8 *tout  = (Int8 *) buffers[2];
    Int8 temp1, temp2;
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = ufminimum(tscalar, *tin2);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3sv(minimum_Int8_scalar_vector, sizeof(Int8), sizeof(Int8), sizeof(Int8));


static void _minimum_Int8_accumulate(
            long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    Int8 *tin   = (Int8 *) ((char *) input  + inboffset);
    Int8 *tout = (Int8 *) ((char *) output + outboffset);
    Int8 lastval;
    Int8 temp1, temp2;
    if (dim == 0) {
        for (i=1; i<niters[dim]; i++) {
            lastval = *tout;
            tin = (Int8 *) ((char *) tin + inbstrides[dim]);
            tout = (Int8 *) ((char *) tout + outbstrides[dim]);
            *tout = ufminimum(lastval ,*tin);
        }
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            _minimum_Int8_accumulate(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
}

static int minimum_Int8_accumulate(
            long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
            BEGIN_THREADS
            _minimum_Int8_accumulate(dim, dummy, niters,
                  input, inboffset, inbstrides, output, outboffset, outbstrides);
            END_THREADS
            return 0;
}

STRIDING_DESCR2(minimum_Int8_accumulate, CHECK_ALIGN, sizeof(Int8), sizeof(Int8));

static void _minimum_Int8_reduce(long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    Int8  *tin   = (Int8 *) ((char *) input  + inboffset);
    Int8 *tout  = (Int8 *) ((char *) output + outboffset);
    Int8 net;
    Int8 temp1, temp2;
    if (dim == 0) {
        net = *tout;
        for (i=1; i<niters[dim]; i++) {
            tin = (Int8 *) ((char *) tin + inbstrides[dim]);
            net = ufminimum(net, *tin);
        }
        *tout = net;
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            _minimum_Int8_reduce(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
}

static int minimum_Int8_reduce(long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
            BEGIN_THREADS
            _minimum_Int8_reduce(dim, dummy, niters,
                             input, inboffset, inbstrides, output, outboffset, outbstrides );
            END_THREADS
            return 0;
}

STRIDING_DESCR2(minimum_Int8_reduce, CHECK_ALIGN, sizeof(Int8), sizeof(Int8));
/*********************  fabs  *********************/

static int fabs_Int8_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8 *tin = (Int8 *) buffers[0];
    Float64 *tout = (Float64 *) buffers[1];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = fabs(*tin);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR2(fabs_Int8_vector, sizeof(Int8), sizeof(Float64));
/*********************  _round  *********************/

static int _round_Int8_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8 *tin = (Int8 *) buffers[0];
    Float64 *tout = (Float64 *) buffers[1];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = num_round(*tin);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR2(_round_Int8_vector, sizeof(Int8), sizeof(Float64));
/*********************  hypot  *********************/

static int hypot_Int8_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8  *tin1  = (Int8 *) buffers[0];
    Int8  *tin2  = (Int8 *) buffers[1];
    Float64 *tout  = (Float64 *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = hypot(*tin1, *tin2);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3(hypot_Int8_vector_vector, sizeof(Int8), sizeof(Int8), sizeof(Float64));


static int hypot_Int8_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8 *tin1     = (Int8 *) buffers[0];
    Int8 tscalar   = *(Int8 *) buffers[1];
    Float64 *tout    = (Float64 *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = hypot(*tin1, tscalar);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3vs(hypot_Int8_vector_scalar, sizeof(Int8), sizeof(Int8), sizeof(Float64));


static int hypot_Int8_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Int8 tscalar = *(Int8 *) buffers[0];
    Int8 *tin2   =  (Int8 *) buffers[1];
    Float64 *tout  = (Float64 *) buffers[2];
    
    BEGIN_THREADS
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = hypot(tscalar, *tin2);
    }
    END_THREADS
    return 0;
}

UFUNC_DESCR3sv(hypot_Int8_scalar_vector, sizeof(Int8), sizeof(Int8), sizeof(Float64));

/*********************  minus  *********************/
/*********************  add  *********************/
/*********************  subtract  *********************/
/*********************  multiply  *********************/
/*********************  divide  *********************/
/*********************  true_divide  *********************/
/*********************  remainder  *********************/
/*********************  power  *********************/
/*********************  abs  *********************/
/*********************  sin  *********************/
/*********************  cos  *********************/
/*********************  tan  *********************/
/*********************  arcsin  *********************/
/*********************  arccos  *********************/
/*********************  arctan  *********************/
/*********************  arcsinh  *********************/
/*********************  arccosh  *********************/
/*********************  arctanh  *********************/
/*********************  log  *********************/
/*********************  log10  *********************/
/*********************  exp  *********************/
/*********************  sinh  *********************/
/*********************  cosh  *********************/
/*********************  tanh  *********************/
/*********************  sqrt  *********************/
/*********************  equal  *********************/
/*********************  not_equal  *********************/
/*********************  logical_and  *********************/
/*********************  logical_or  *********************/
/*********************  logical_xor  *********************/
/*********************  logical_not  *********************/
/*********************  floor  *********************/
/*********************  ceil  *********************/
/*********************  maximum  *********************/
/*********************  minimum  *********************/
/*********************  fabs  *********************/
/*********************  _round  *********************/
/*********************  hypot  *********************/

static PyMethodDef _ufuncInt8Methods[] = {

	{NULL,      NULL}        /* Sentinel */
};

static PyObject *init_funcDict(void) {
    PyObject *dict;
    dict = PyDict_New();
    NA_add_cfunc(dict, "('minus', 'vector', (('Int8',), ('Int8',)))", (void *) &minus_Int8_vector_descr);
    NA_add_cfunc(dict, "('add', 'vector_vector', (('Int8', 'Int8'), ('Int8',)))", (void *) &add_Int8_vector_vector_descr);
    NA_add_cfunc(dict, "('add', 'vector_scalar', (('Int8', 'Int8'), ('Int8',)))", (void *) &add_Int8_vector_scalar_descr);
    NA_add_cfunc(dict, "('add', 'scalar_vector', (('Int8', 'Int8'), ('Int8',)))", (void *) &add_Int8_scalar_vector_descr);
    NA_add_cfunc(dict, "('add', 'reduce', (('Int8',), ('Int8',)))", (void *) &add_Int8_reduce_descr);
    NA_add_cfunc(dict, "('add', 'accumulate', (('Int8',), ('Int8',)))", (void *) &add_Int8_accumulate_descr);
    NA_add_cfunc(dict, "('subtract', 'vector_vector', (('Int8', 'Int8'), ('Int8',)))", (void *) &subtract_Int8_vector_vector_descr);
    NA_add_cfunc(dict, "('subtract', 'vector_scalar', (('Int8', 'Int8'), ('Int8',)))", (void *) &subtract_Int8_vector_scalar_descr);
    NA_add_cfunc(dict, "('subtract', 'scalar_vector', (('Int8', 'Int8'), ('Int8',)))", (void *) &subtract_Int8_scalar_vector_descr);
    NA_add_cfunc(dict, "('subtract', 'reduce', (('Int8',), ('Int8',)))", (void *) &subtract_Int8_reduce_descr);
    NA_add_cfunc(dict, "('subtract', 'accumulate', (('Int8',), ('Int8',)))", (void *) &subtract_Int8_accumulate_descr);
    NA_add_cfunc(dict, "('multiply', 'vector_vector', (('Int8', 'Int8'), ('Int8',)))", (void *) &multiply_Int8_vector_vector_descr);
    NA_add_cfunc(dict, "('multiply', 'vector_scalar', (('Int8', 'Int8'), ('Int8',)))", (void *) &multiply_Int8_vector_scalar_descr);
    NA_add_cfunc(dict, "('multiply', 'scalar_vector', (('Int8', 'Int8'), ('Int8',)))", (void *) &multiply_Int8_scalar_vector_descr);
    NA_add_cfunc(dict, "('multiply', 'reduce', (('Int8',), ('Int8',)))", (void *) &multiply_Int8_reduce_descr);
    NA_add_cfunc(dict, "('multiply', 'accumulate', (('Int8',), ('Int8',)))", (void *) &multiply_Int8_accumulate_descr);
    NA_add_cfunc(dict, "('divide', 'vector_vector', (('Int8', 'Int8'), ('Int8',)))", (void *) &divide_Int8_vector_vector_descr);
    NA_add_cfunc(dict, "('divide', 'vector_scalar', (('Int8', 'Int8'), ('Int8',)))", (void *) &divide_Int8_vector_scalar_descr);
    NA_add_cfunc(dict, "('divide', 'scalar_vector', (('Int8', 'Int8'), ('Int8',)))", (void *) &divide_Int8_scalar_vector_descr);
    NA_add_cfunc(dict, "('divide', 'reduce', (('Int8',), ('Int8',)))", (void *) &divide_Int8_reduce_descr);
    NA_add_cfunc(dict, "('divide', 'accumulate', (('Int8',), ('Int8',)))", (void *) &divide_Int8_accumulate_descr);
    NA_add_cfunc(dict, "('floor_divide', 'vector_vector', (('Int8', 'Int8'), ('Int8',)))", (void *) &floor_divide_Int8_vector_vector_descr);
    NA_add_cfunc(dict, "('floor_divide', 'vector_scalar', (('Int8', 'Int8'), ('Int8',)))", (void *) &floor_divide_Int8_vector_scalar_descr);
    NA_add_cfunc(dict, "('floor_divide', 'scalar_vector', (('Int8', 'Int8'), ('Int8',)))", (void *) &floor_divide_Int8_scalar_vector_descr);
    NA_add_cfunc(dict, "('floor_divide', 'reduce', (('Int8',), ('Int8',)))", (void *) &floor_divide_Int8_reduce_descr);
    NA_add_cfunc(dict, "('floor_divide', 'accumulate', (('Int8',), ('Int8',)))", (void *) &floor_divide_Int8_accumulate_descr);
    NA_add_cfunc(dict, "('true_divide', 'vector_vector', (('Int8', 'Int8'), ('Float32',)))", (void *) &true_divide_Int8_vector_vector_descr);
    NA_add_cfunc(dict, "('true_divide', 'vector_scalar', (('Int8', 'Int8'), ('Float32',)))", (void *) &true_divide_Int8_vector_scalar_descr);
    NA_add_cfunc(dict, "('true_divide', 'scalar_vector', (('Int8', 'Int8'), ('Float32',)))", (void *) &true_divide_Int8_scalar_vector_descr);
    NA_add_cfunc(dict, "('remainder', 'vector_vector', (('Int8', 'Int8'), ('Int8',)))", (void *) &remainder_Int8_vector_vector_descr);
    NA_add_cfunc(dict, "('remainder', 'vector_scalar', (('Int8', 'Int8'), ('Int8',)))", (void *) &remainder_Int8_vector_scalar_descr);
    NA_add_cfunc(dict, "('remainder', 'scalar_vector', (('Int8', 'Int8'), ('Int8',)))", (void *) &remainder_Int8_scalar_vector_descr);
    NA_add_cfunc(dict, "('remainder', 'reduce', (('Int8',), ('Int8',)))", (void *) &remainder_Int8_reduce_descr);
    NA_add_cfunc(dict, "('remainder', 'accumulate', (('Int8',), ('Int8',)))", (void *) &remainder_Int8_accumulate_descr);
    NA_add_cfunc(dict, "('power', 'vector_vector', (('Int8', 'Int8'), ('Int8',)))", (void *) &power_Int8_vector_vector_descr);
    NA_add_cfunc(dict, "('power', 'vector_scalar', (('Int8', 'Int8'), ('Int8',)))", (void *) &power_Int8_vector_scalar_descr);
    NA_add_cfunc(dict, "('power', 'scalar_vector', (('Int8', 'Int8'), ('Int8',)))", (void *) &power_Int8_scalar_vector_descr);
    NA_add_cfunc(dict, "('power', 'reduce', (('Int8',), ('Int8',)))", (void *) &power_Int8_reduce_descr);
    NA_add_cfunc(dict, "('power', 'accumulate', (('Int8',), ('Int8',)))", (void *) &power_Int8_accumulate_descr);
    NA_add_cfunc(dict, "('abs', 'vector', (('Int8',), ('Int8',)))", (void *) &abs_Int8_vector_descr);
    NA_add_cfunc(dict, "('sin', 'vector', (('Int8',), ('Float64',)))", (void *) &sin_Int8_vector_descr);
    NA_add_cfunc(dict, "('cos', 'vector', (('Int8',), ('Float64',)))", (void *) &cos_Int8_vector_descr);
    NA_add_cfunc(dict, "('tan', 'vector', (('Int8',), ('Float64',)))", (void *) &tan_Int8_vector_descr);
    NA_add_cfunc(dict, "('arcsin', 'vector', (('Int8',), ('Float64',)))", (void *) &arcsin_Int8_vector_descr);
    NA_add_cfunc(dict, "('arccos', 'vector', (('Int8',), ('Float64',)))", (void *) &arccos_Int8_vector_descr);
    NA_add_cfunc(dict, "('arctan', 'vector', (('Int8',), ('Float64',)))", (void *) &arctan_Int8_vector_descr);
    NA_add_cfunc(dict, "('arctan2', 'vector_vector', (('Int8', 'Int8'), ('Float64',)))", (void *) &arctan2_Int8_vector_vector_descr);
    NA_add_cfunc(dict, "('arctan2', 'vector_scalar', (('Int8', 'Int8'), ('Float64',)))", (void *) &arctan2_Int8_vector_scalar_descr);
    NA_add_cfunc(dict, "('arctan2', 'scalar_vector', (('Int8', 'Int8'), ('Float64',)))", (void *) &arctan2_Int8_scalar_vector_descr);
    NA_add_cfunc(dict, "('log', 'vector', (('Int8',), ('Float64',)))", (void *) &log_Int8_vector_descr);
    NA_add_cfunc(dict, "('log10', 'vector', (('Int8',), ('Float64',)))", (void *) &log10_Int8_vector_descr);
    NA_add_cfunc(dict, "('exp', 'vector', (('Int8',), ('Float64',)))", (void *) &exp_Int8_vector_descr);
    NA_add_cfunc(dict, "('sinh', 'vector', (('Int8',), ('Float64',)))", (void *) &sinh_Int8_vector_descr);
    NA_add_cfunc(dict, "('cosh', 'vector', (('Int8',), ('Float64',)))", (void *) &cosh_Int8_vector_descr);
    NA_add_cfunc(dict, "('tanh', 'vector', (('Int8',), ('Float64',)))", (void *) &tanh_Int8_vector_descr);
    NA_add_cfunc(dict, "('arcsinh', 'vector', (('Int8',), ('Float64',)))", (void *) &arcsinh_Int8_vector_descr);
    NA_add_cfunc(dict, "('arccosh', 'vector', (('Int8',), ('Float64',)))", (void *) &arccosh_Int8_vector_descr);
    NA_add_cfunc(dict, "('arctanh', 'vector', (('Int8',), ('Float64',)))", (void *) &arctanh_Int8_vector_descr);
    NA_add_cfunc(dict, "('sqrt', 'vector', (('Int8',), ('Float64',)))", (void *) &sqrt_Int8_vector_descr);
    NA_add_cfunc(dict, "('equal', 'vector_vector', (('Int8', 'Int8'), ('Bool',)))", (void *) &equal_Int8_vector_vector_descr);
    NA_add_cfunc(dict, "('equal', 'vector_scalar', (('Int8', 'Int8'), ('Bool',)))", (void *) &equal_Int8_vector_scalar_descr);
    NA_add_cfunc(dict, "('equal', 'scalar_vector', (('Int8', 'Int8'), ('Bool',)))", (void *) &equal_Int8_scalar_vector_descr);
    NA_add_cfunc(dict, "('not_equal', 'vector_vector', (('Int8', 'Int8'), ('Bool',)))", (void *) &not_equal_Int8_vector_vector_descr);
    NA_add_cfunc(dict, "('not_equal', 'vector_scalar', (('Int8', 'Int8'), ('Bool',)))", (void *) &not_equal_Int8_vector_scalar_descr);
    NA_add_cfunc(dict, "('not_equal', 'scalar_vector', (('Int8', 'Int8'), ('Bool',)))", (void *) &not_equal_Int8_scalar_vector_descr);
    NA_add_cfunc(dict, "('greater', 'vector_vector', (('Int8', 'Int8'), ('Bool',)))", (void *) &greater_Int8_vector_vector_descr);
    NA_add_cfunc(dict, "('greater', 'vector_scalar', (('Int8', 'Int8'), ('Bool',)))", (void *) &greater_Int8_vector_scalar_descr);
    NA_add_cfunc(dict, "('greater', 'scalar_vector', (('Int8', 'Int8'), ('Bool',)))", (void *) &greater_Int8_scalar_vector_descr);
    NA_add_cfunc(dict, "('greater_equal', 'vector_vector', (('Int8', 'Int8'), ('Bool',)))", (void *) &greater_equal_Int8_vector_vector_descr);
    NA_add_cfunc(dict, "('greater_equal', 'vector_scalar', (('Int8', 'Int8'), ('Bool',)))", (void *) &greater_equal_Int8_vector_scalar_descr);
    NA_add_cfunc(dict, "('greater_equal', 'scalar_vector', (('Int8', 'Int8'), ('Bool',)))", (void *) &greater_equal_Int8_scalar_vector_descr);
    NA_add_cfunc(dict, "('less', 'vector_vector', (('Int8', 'Int8'), ('Bool',)))", (void *) &less_Int8_vector_vector_descr);
    NA_add_cfunc(dict, "('less', 'vector_scalar', (('Int8', 'Int8'), ('Bool',)))", (void *) &less_Int8_vector_scalar_descr);
    NA_add_cfunc(dict, "('less', 'scalar_vector', (('Int8', 'Int8'), ('Bool',)))", (void *) &less_Int8_scalar_vector_descr);
    NA_add_cfunc(dict, "('less_equal', 'vector_vector', (('Int8', 'Int8'), ('Bool',)))", (void *) &less_equal_Int8_vector_vector_descr);
    NA_add_cfunc(dict, "('less_equal', 'vector_scalar', (('Int8', 'Int8'), ('Bool',)))", (void *) &less_equal_Int8_vector_scalar_descr);
    NA_add_cfunc(dict, "('less_equal', 'scalar_vector', (('Int8', 'Int8'), ('Bool',)))", (void *) &less_equal_Int8_scalar_vector_descr);
    NA_add_cfunc(dict, "('logical_and', 'vector_vector', (('Int8', 'Int8'), ('Bool',)))", (void *) &logical_and_Int8_vector_vector_descr);
    NA_add_cfunc(dict, "('logical_and', 'vector_scalar', (('Int8', 'Int8'), ('Bool',)))", (void *) &logical_and_Int8_vector_scalar_descr);
    NA_add_cfunc(dict, "('logical_and', 'scalar_vector', (('Int8', 'Int8'), ('Bool',)))", (void *) &logical_and_Int8_scalar_vector_descr);
    NA_add_cfunc(dict, "('logical_or', 'vector_vector', (('Int8', 'Int8'), ('Bool',)))", (void *) &logical_or_Int8_vector_vector_descr);
    NA_add_cfunc(dict, "('logical_or', 'vector_scalar', (('Int8', 'Int8'), ('Bool',)))", (void *) &logical_or_Int8_vector_scalar_descr);
    NA_add_cfunc(dict, "('logical_or', 'scalar_vector', (('Int8', 'Int8'), ('Bool',)))", (void *) &logical_or_Int8_scalar_vector_descr);
    NA_add_cfunc(dict, "('logical_xor', 'vector_vector', (('Int8', 'Int8'), ('Bool',)))", (void *) &logical_xor_Int8_vector_vector_descr);
    NA_add_cfunc(dict, "('logical_xor', 'vector_scalar', (('Int8', 'Int8'), ('Bool',)))", (void *) &logical_xor_Int8_vector_scalar_descr);
    NA_add_cfunc(dict, "('logical_xor', 'scalar_vector', (('Int8', 'Int8'), ('Bool',)))", (void *) &logical_xor_Int8_scalar_vector_descr);
    NA_add_cfunc(dict, "('logical_not', 'vector', (('Int8',), ('Bool',)))", (void *) &logical_not_Int8_vector_descr);
    NA_add_cfunc(dict, "('bitwise_and', 'vector_vector', (('Int8', 'Int8'), ('Int8',)))", (void *) &bitwise_and_Int8_vector_vector_descr);
    NA_add_cfunc(dict, "('bitwise_and', 'vector_scalar', (('Int8', 'Int8'), ('Int8',)))", (void *) &bitwise_and_Int8_vector_scalar_descr);
    NA_add_cfunc(dict, "('bitwise_and', 'scalar_vector', (('Int8', 'Int8'), ('Int8',)))", (void *) &bitwise_and_Int8_scalar_vector_descr);
    NA_add_cfunc(dict, "('bitwise_and', 'reduce', (('Int8',), ('Int8',)))", (void *) &bitwise_and_Int8_reduce_descr);
    NA_add_cfunc(dict, "('bitwise_and', 'accumulate', (('Int8',), ('Int8',)))", (void *) &bitwise_and_Int8_accumulate_descr);
    NA_add_cfunc(dict, "('bitwise_or', 'vector_vector', (('Int8', 'Int8'), ('Int8',)))", (void *) &bitwise_or_Int8_vector_vector_descr);
    NA_add_cfunc(dict, "('bitwise_or', 'vector_scalar', (('Int8', 'Int8'), ('Int8',)))", (void *) &bitwise_or_Int8_vector_scalar_descr);
    NA_add_cfunc(dict, "('bitwise_or', 'scalar_vector', (('Int8', 'Int8'), ('Int8',)))", (void *) &bitwise_or_Int8_scalar_vector_descr);
    NA_add_cfunc(dict, "('bitwise_or', 'reduce', (('Int8',), ('Int8',)))", (void *) &bitwise_or_Int8_reduce_descr);
    NA_add_cfunc(dict, "('bitwise_or', 'accumulate', (('Int8',), ('Int8',)))", (void *) &bitwise_or_Int8_accumulate_descr);
    NA_add_cfunc(dict, "('bitwise_xor', 'vector_vector', (('Int8', 'Int8'), ('Int8',)))", (void *) &bitwise_xor_Int8_vector_vector_descr);
    NA_add_cfunc(dict, "('bitwise_xor', 'vector_scalar', (('Int8', 'Int8'), ('Int8',)))", (void *) &bitwise_xor_Int8_vector_scalar_descr);
    NA_add_cfunc(dict, "('bitwise_xor', 'scalar_vector', (('Int8', 'Int8'), ('Int8',)))", (void *) &bitwise_xor_Int8_scalar_vector_descr);
    NA_add_cfunc(dict, "('bitwise_xor', 'reduce', (('Int8',), ('Int8',)))", (void *) &bitwise_xor_Int8_reduce_descr);
    NA_add_cfunc(dict, "('bitwise_xor', 'accumulate', (('Int8',), ('Int8',)))", (void *) &bitwise_xor_Int8_accumulate_descr);
    NA_add_cfunc(dict, "('bitwise_not', 'vector', (('Int8',), ('Int8',)))", (void *) &bitwise_not_Int8_vector_descr);
    NA_add_cfunc(dict, "('rshift', 'vector_vector', (('Int8', 'Int8'), ('Int8',)))", (void *) &rshift_Int8_vector_vector_descr);
    NA_add_cfunc(dict, "('rshift', 'vector_scalar', (('Int8', 'Int8'), ('Int8',)))", (void *) &rshift_Int8_vector_scalar_descr);
    NA_add_cfunc(dict, "('rshift', 'scalar_vector', (('Int8', 'Int8'), ('Int8',)))", (void *) &rshift_Int8_scalar_vector_descr);
    NA_add_cfunc(dict, "('rshift', 'reduce', (('Int8',), ('Int8',)))", (void *) &rshift_Int8_reduce_descr);
    NA_add_cfunc(dict, "('rshift', 'accumulate', (('Int8',), ('Int8',)))", (void *) &rshift_Int8_accumulate_descr);
    NA_add_cfunc(dict, "('lshift', 'vector_vector', (('Int8', 'Int8'), ('Int8',)))", (void *) &lshift_Int8_vector_vector_descr);
    NA_add_cfunc(dict, "('lshift', 'vector_scalar', (('Int8', 'Int8'), ('Int8',)))", (void *) &lshift_Int8_vector_scalar_descr);
    NA_add_cfunc(dict, "('lshift', 'scalar_vector', (('Int8', 'Int8'), ('Int8',)))", (void *) &lshift_Int8_scalar_vector_descr);
    NA_add_cfunc(dict, "('lshift', 'reduce', (('Int8',), ('Int8',)))", (void *) &lshift_Int8_reduce_descr);
    NA_add_cfunc(dict, "('lshift', 'accumulate', (('Int8',), ('Int8',)))", (void *) &lshift_Int8_accumulate_descr);
    NA_add_cfunc(dict, "('floor', 'vector', (('Int8',), ('Int8',)))", (void *) &floor_Int8_vector_descr);
    NA_add_cfunc(dict, "('ceil', 'vector', (('Int8',), ('Int8',)))", (void *) &ceil_Int8_vector_descr);
    NA_add_cfunc(dict, "('maximum', 'vector_vector', (('Int8', 'Int8'), ('Int8',)))", (void *) &maximum_Int8_vector_vector_descr);
    NA_add_cfunc(dict, "('maximum', 'vector_scalar', (('Int8', 'Int8'), ('Int8',)))", (void *) &maximum_Int8_vector_scalar_descr);
    NA_add_cfunc(dict, "('maximum', 'scalar_vector', (('Int8', 'Int8'), ('Int8',)))", (void *) &maximum_Int8_scalar_vector_descr);
    NA_add_cfunc(dict, "('maximum', 'reduce', (('Int8',), ('Int8',)))", (void *) &maximum_Int8_reduce_descr);
    NA_add_cfunc(dict, "('maximum', 'accumulate', (('Int8',), ('Int8',)))", (void *) &maximum_Int8_accumulate_descr);
    NA_add_cfunc(dict, "('minimum', 'vector_vector', (('Int8', 'Int8'), ('Int8',)))", (void *) &minimum_Int8_vector_vector_descr);
    NA_add_cfunc(dict, "('minimum', 'vector_scalar', (('Int8', 'Int8'), ('Int8',)))", (void *) &minimum_Int8_vector_scalar_descr);
    NA_add_cfunc(dict, "('minimum', 'scalar_vector', (('Int8', 'Int8'), ('Int8',)))", (void *) &minimum_Int8_scalar_vector_descr);
    NA_add_cfunc(dict, "('minimum', 'reduce', (('Int8',), ('Int8',)))", (void *) &minimum_Int8_reduce_descr);
    NA_add_cfunc(dict, "('minimum', 'accumulate', (('Int8',), ('Int8',)))", (void *) &minimum_Int8_accumulate_descr);
    NA_add_cfunc(dict, "('fabs', 'vector', (('Int8',), ('Float64',)))", (void *) &fabs_Int8_vector_descr);
    NA_add_cfunc(dict, "('_round', 'vector', (('Int8',), ('Float64',)))", (void *) &_round_Int8_vector_descr);
    NA_add_cfunc(dict, "('hypot', 'vector_vector', (('Int8', 'Int8'), ('Float64',)))", (void *) &hypot_Int8_vector_vector_descr);
    NA_add_cfunc(dict, "('hypot', 'vector_scalar', (('Int8', 'Int8'), ('Float64',)))", (void *) &hypot_Int8_vector_scalar_descr);
    NA_add_cfunc(dict, "('hypot', 'scalar_vector', (('Int8', 'Int8'), ('Float64',)))", (void *) &hypot_Int8_scalar_vector_descr);
    return dict;
}

/* platform independent*/
#ifdef MS_WIN32
__declspec(dllexport)
#endif
void init_ufuncInt8(void) {
    PyObject *m, *d;
    m = Py_InitModule("_ufuncInt8", _ufuncInt8Methods);
    d = PyModule_GetDict(m);
    _Error = PyErr_NewException("_ufuncInt8.error", NULL, NULL);
    PyDict_SetItemString(d, "error", _Error);
    import_libnumarray();
    PyDict_SetItemString(d, "functionDict", init_funcDict());
}
