/******************************************************************************
*		       							      *
* console/main.c (part of rcalc)				       	      *
* Copyright (C) 2001 Free Software Foundation, Inc.			      *
*								       	      *
* This program is free software; you can redistribute it and/or modify 	      *
* it under the terms of the GNU General Public License as published by 	      *
* the Free Software Foundation; either version 2 of the License, or    	      *
* (at your option) any later version.				       	      *
*								       	      *
* This program is distributed in the hope that it will be useful,      	      *
* but WITHOUT ANY WARRANTY; without even the implied warranty of       	      *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the	       	      *
* GNU General Public License for more details.			       	      *
*								       	      *
* You should have received a copy of the GNU General Public License    	      *
* along with this program; if not, write to the Free Software	       	      *
* Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.	       	      *
*								       	      *
******************************************************************************/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include "main.h"		/* This file's header	     		*/
#include "readline.h"		/* Wrapper around readline		*/
#include <engine/engine.h>	/* The calculation engine		*/
#include <engine/debug.h>	/* Debugging functions			*/

#include <errno.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <stdio.h>

/*****************************************************************************/

/* Function to display help for the engine. Has exactly the same
** behaviour as gnome_help_display(), except that lynx is used to
** display the file.
*/
static void lynx_help_display( const char *file_name, const char *link_id,  GError **error )
{
	gchar *file, *url;
	pid_t child_pid;

#ifndef RCALC_TEXT_ONLY	
	/*GList *lang_list, *first;*/
	gchar *tempbuff, *tempfile;
#endif

	g_assert( file_name );
	g_assert( link_id );

#ifndef RCALC_TEXT_ONLY
   
	/*lang_list = gnome_i18n_get_language_list();
	first = g_list_first(lang_list);
	rc_printf("locale: %s\n",(gchar *)first->data);*/
	tempbuff = g_strdup_printf("C/%s",file_name);
   tempfile = gnome_program_locate_file(NULL,GNOME_FILE_DOMAIN_APP_HELP, 
	tempbuff, TRUE, NULL);
	file = g_strdup_printf("%s%s",tempfile,link_id);
	g_free( tempbuff );
	g_free( tempfile );
		
	/* the following function no longer exists... */
	/*file = gnome_help_file_path("rcalc","index.html");*/
#else
	/* if we don't use gnome */
	/* This need to be fixed!! this path won't stay that way */
	file = g_strdup_printf("%s/gnome/help/rcalc/C/%s%s",PACKAGE_DATA_DIR,file_name,link_id);
#endif	/* ifndef RCALC_TEXT_ONLY */

	if( !file ) { rc_error("Cannot find help file: %s\n", file); return; }
	else rc_printf("opening help file: %s\n",file);
	
	url = alloca( strlen(file)+6 );
	strcpy( url, "file:" );
	strcat( url, file );
	g_free( file );

	/* Run Lynx on it. Mostly copied from the system(3) manpage.
	*/
	child_pid = fork();
	if( child_pid == -1 )
	{
		g_error( "fork(): failed due to: %s.", g_strerror( errno ) );
	}
	if( child_pid==0 )
	{
		char *argv[3];

		argv[0] = "lynx";
		argv[1] = url;
		argv[2] = 0;

		execv( "/usr/bin/lynx", argv );

		exit( 127 );
	}
	do
	{
		if( waitpid( child_pid, NULL, 0 ) == -1 )
		{
			if( errno != EINTR ) return;
		}
		else return;
	}
	while(1);
}

/*****************************************************************************/

/* Define the prompt to use.
** Nonprinting characters must be protected by `\[' and `\]' as in bash.
*/
#define DEFAULTCOLOUR	"\x1B[0m"
#define PROMPTCOLOUR	"\x1B[0;36m"

#define PROMPT		"\\[" PROMPTCOLOUR "\\]"	\
			PACKAGE ">"			\
			"\\[" DEFAULTCOLOUR "\\]"	\
			" "

/* The main process for the text-based terminal UI.
*/
void rc_console_main( void )
{
	RcReadline *readline;
	RcEngine *engine;
	
	rc_debug( RC_DBG_TOPLEVEL, "executing console user interface." );

	/* Create a readline to read the user's input.
	*/
	readline = rcc_readline_new( PROMPT );
	
	/* Create and initialise the engine.
	*/
	
	engine = rc_engine_new();
	rc_engine_set_streams( engine, stdout, stdout );
	rc_engine_set_help_func( engine, lynx_help_display );

	rc_engine_load_state( engine );	
	  
	/* The main loop.
	*/
	while(1)
	{
		const gchar *user_input;

		user_input = rcc_readline_read_line( readline );
		if( !user_input )
		{
			fputc( '\n', stdout );
			break;
		}
		if( rc_engine_execute( engine, user_input ) )
		{
			break;
		}
	}

	/* Save the state and free the engine.
	*/
	
	rc_engine_save_state ( engine );
	rc_engine_delete( engine );

	/* Free the readline.
	*/
	rcc_readline_delete( readline );
}

/*** end of console/main.c ***************************************************/
