/*
 * Handle device ACLs
 *
 * Copyright (C) 2001-2002, Olaf Kirch <okir@lst.de>
 */

#include "resmgrd.h"
#include <stdlib.h>
#include <stdio.h>
#include <errno.h>
#include <fcntl.h>
#include <assert.h>
#include <ctype.h>


enum { AND, OR, NOT, USERMATCH, GROUPMATCH, TTYMATCH, ALWAYS };
enum {
	LITERAL,
	OPERATOR,
};
#define LEFT_BRACE	'('
#define RIGHT_BRACE	')'

struct res_acl {
	int		r_op;
	int		r_retval;
	union {
	    struct {
		res_acl_t *r1, *r2;
	    } _sub;
	    char *	_lit;
	} u;
};
#define r_expr1		u._sub.r1
#define r_expr2		u._sub.r2
#define r_literal	u._lit

struct acl_raw {
	char *		cur;
	const char *	error;
};

static struct match_op {
	const char *	name;
	unsigned int	len;
	int		op;
} match_ops[] = {
	{ "user",	4,	USERMATCH },
	{ "group",	5,	GROUPMATCH },
	{ "tty",	3,	TTYMATCH },
	{ NULL, 0, -1 }
};

static res_acl_t *	rule_parse(struct acl_raw *);
static res_acl_t *	rule_new_literal(int, const char *);
static res_acl_t *	rule_new_expr(int, res_acl_t *, res_acl_t *);


/*
 * Parse a resource ACL
 */
res_acl_t *
res_acl_parse(unsigned int argc, char **argv)
{
	struct acl_raw	data;
	res_acl_t	*res;
	char		*line;

	line = merge_list(argc, argv, " ");

	data.cur   = line;
	data.error = NULL;
	
	res = rule_parse(&data);
	if (!data.error && *data.cur)
		data.error = "unexpected closing bracket in ACL";

	if (data.error) {
		log("error when parsing ACL: %s", data.error);
		log("> %s\n", line);
		log("> %*.*s^--- here\n",
			data.cur - line, data.cur - line, "");
		if (res)
			res_acl_free(res);
		res = NULL;
	}

	return res;
}

void
res_acl_print(res_acl_t *r, FILE *fp)
{
	switch (r->r_op) {
	case ALWAYS:
		fprintf(fp, "any");
		break;
	case USERMATCH:
		fprintf(fp, "user=%s", r->r_literal);
		break;
	case GROUPMATCH:
		fprintf(fp, "group=%s", r->r_literal);
		break;
	case TTYMATCH:
		fprintf(fp, "tty=%s", r->r_literal);
		break;
	case NOT:
		fprintf(fp, "!");
		res_acl_print(r->r_expr1, fp);
		break;
	case AND:
		fprintf(fp, "(");
		res_acl_print(r->r_expr1, fp);
		fprintf(fp, " && ");
		res_acl_print(r->r_expr2, fp);
		fprintf(fp, ")");
		break;
	case OR:
		fprintf(fp, "(");
		res_acl_print(r->r_expr1, fp);
		fprintf(fp, " || ");
		res_acl_print(r->r_expr2, fp);
		fprintf(fp, ")");
		break;
	default:
		fatal("bad rule type %d in res_acl_print", r->r_op);
	}
}

void
res_acl_free(res_acl_t *r)
{
	if (r == NULL)
		return;

	switch (r->r_op) {
	case ALWAYS:
	case USERMATCH:
	case GROUPMATCH:
	case TTYMATCH:
		if (r->r_literal)
			free(r->r_literal);
		break;
	
	case AND:
	case OR:
	case NOT:
		res_acl_free(r->r_expr1);
		res_acl_free(r->r_expr2);
		break;

	default:
		fatal("bad rule type %d in res_acl_free", r->r_op);
	}

	free(r);
}

void
res_acl_add(res_acl_t **p, res_acl_t *rule, int retval)
{
	rule->r_retval = retval;
	if (*p != NULL)
		rule = rule_new_expr(OR, *p, rule);
	*p = rule;
}

int
res_acl_match(res_acl_t *r, const char *tty,
		const char *user, const char **groups)
{
	int	match = 0;

	if (r == NULL)
		return 0;

	switch (r->r_op) {
	case USERMATCH:
		if (rsm_glob(r->r_literal, user, NULL))
			match = r->r_retval;
		break;
	case GROUPMATCH:
		while (*groups && !match) {
			if (rsm_glob(r->r_literal, *groups++, NULL))
				match = r->r_retval;
		}
		break;
	case TTYMATCH:
		if (rsm_glob(r->r_literal, tty, NULL))
			match = r->r_retval;
		break;
	case AND:
		match = res_acl_match(r->r_expr1, tty, user, groups)
		     && res_acl_match(r->r_expr2, tty, user, groups);
		break;
	case OR:
		/* OR rules are special because they may return negative
		 * match values */
		match = res_acl_match(r->r_expr1, tty, user, groups);
		if (match == 0)
			match = res_acl_match(r->r_expr2, tty, user, groups);
		break;
	case NOT:
		match = res_acl_match(r->r_expr1, tty, user, groups);
		match = (match <= 0)? 1 : r->r_retval;
		break;
	case ALWAYS:
		match = 1;
		break;
	default:
		fatal("bad rule type %d in res_acl_match", r->r_op);
	}

	if (match > 0)
		match = r->r_retval;

	return match;
}

res_acl_t *
rule_parse(struct acl_raw *data)
{
	res_acl_t	*result = NULL, *term = NULL, *lit;
	char		*word;
	int		not, expect;

	expect = LITERAL;
	not = 0;
	while (*data->cur) {
		word = data->cur;

#ifdef TEST
		printf("expect=%s not=%d word=%s\n",
			(expect == LITERAL)? "literal " : "operator",
			not, word);
#endif

		while (*word == ' ') {
			data->cur++;
			word++;
		}
		if (*word == '\0')
			break;

		if (expect == OPERATOR) {
			if (*word == RIGHT_BRACE)
				break;

			if (!strncmp(word, "||", 2)) {
				if (term) {
					result = rule_new_expr(OR,
							result,
							term);
					term = NULL;
				}
			} else if (strncmp(word, "&&", 2)) {
				data->error = "invalid operator";
				goto error;
			}
			expect = LITERAL;
			data->cur = word + 2;
			continue;
		}

		/* expect == LITERAL: */

		if (*word == '!') {
			not = !not;
			data->cur = word + 1;
			continue;
		}

		if (*word == LEFT_BRACE) {
			data->cur = word + 1;
			if (!(lit = rule_parse(data)))
				goto error;
			word = data->cur;
			if (*word++ != RIGHT_BRACE) {
				data->error = "Missing closing brace";
				goto error;
			}
		} else
		if (!strncmp(word, "any", 3)) {
			lit = rule_new_literal(ALWAYS, NULL);
			word += 3;
		} else {
			struct match_op	*m;
			char	*wend, save;
			int	op;

			for (op = -1, m = match_ops; m->name; m++) {
				if (!strncmp(word, m->name, m->len)
				 && !isalpha(word[m->len])) {
					word += m->len;
					op = m->op;
				}
			}
			if (op < 0) {
				data->error = "bad literal";
				goto error;
			}

			data->cur = word;
			if (!strncmp(word, "!=", 2)) {
				not = !not;
				word++;
			} else if (*word != '=') {
				data->error = "expected '=' or '!='";
				goto error;
			}
			word++;

			wend = word + strcspn(word, " &|()!");
			save = *wend;
			*wend = '\0';

			lit = rule_new_literal(op, word);

			*wend = save;
			word = wend;
		}
		data->cur = word;

		if (not)
			lit = rule_new_expr(NOT, lit, NULL);

		if (term == NULL) {
			term = lit;
		} else {
			term = rule_new_expr(AND, term, lit);
		}

		not = 0;
		expect = OPERATOR;
	}

	if (expect == LITERAL) {
		data->error = "unexpected end of ACL";
		goto error;
	}

	if (term) {
		result = rule_new_expr(OR, result, term);
		term = NULL;
	}

	if (result != NULL)
		return result;

	data->error = "empty access control list";

error:	if (term)
		res_acl_free(term);
	if (result)
		res_acl_free(result);
	return NULL;
}


static res_acl_t *
rule_new_literal(int op, const char *value)
{
	res_acl_t	*r;

	if ((r = (res_acl_t *) malloc(sizeof(*r))) == NULL)
		fatal("rule_new: out of memory");
	memset(r, 0, sizeof(*r));

	r->r_op = op;
	r->r_retval = 1;
	if (value && (r->r_literal = strdup(value)) == NULL)
		fatal("rule_new: out of memory");

	return r;
}

static res_acl_t *
rule_new_expr(int op, res_acl_t *r1, res_acl_t *r2)
{
	res_acl_t	*r;

	assert(r1 || r2);
	if (op == AND || op == OR) {
		if (r1 == NULL)
			return r2;
		if (r2 == NULL)
			return r1;
	}

	if ((r = (res_acl_t *) malloc(sizeof(*r))) == NULL)
		fatal("rule_new: out of memory");
	memset(r, 0, sizeof(*r));

	r->r_op = op;
	r->r_retval = 1;
	r->r_expr1 = r1;
	r->r_expr2 = r2;
	return r;
}

#ifdef TEST
int
main(int argc, char **argv)
{
	res_acl_t *acl;
	char	*args[128];
	int	n, nargs = 0;

	for (n = 1; n < argc; n++)
		nargs += line_split(argv[n], args+nargs, 128-nargs);

	acl = res_acl_parse(nargs, args);
	if (acl) {
		res_acl_print(acl, stdout);
		printf("\n");
	} else {
		fprintf(stderr, "Failed to parse ACL\n");
	}
	return 0;
}
#endif
