/*
 * Session handling
 *
 * Copyright (C) 2001-2002, Olaf Kirch <okir@lst.de>
 */

#include <stdlib.h>
#include <stdio.h>
#include <errno.h>

#include "resmgrd.h"

#define MAXGROUPS		32

res_session_t *	res_sessions;
res_user_t *	res_users;

/*
 * Log in the user
 */
int
res_user_login(res_user_t *user, const char *id)
{
	res_session_t	*sess;

	/* Force logout of any stale session */
	if (!res_user_logout(user, id))
		return 0;

	sess = (res_session_t *) calloc(1, sizeof(*sess));
	sess->user = user;
	sess->id = strdup(id);

	sess->next = res_sessions;
	res_sessions = sess;

	user->nsessions++;
	user->refcnt++;

	return 1;
}

int
res_user_logout(res_user_t *user, const char *id)
{
	res_session_t	*sess, **ptr;
	res_user_t	*u;

	ptr = &res_sessions;
	while ((sess = *ptr) != NULL) {
		if (!strcmp(sess->id, id)) {
			if (user && user != sess->user) {
				log("user %s tries to close session "
					"owned by %s",
					user->name, sess->user->name);
				return 0;
			}
			u = sess->user;
			*ptr = sess->next;
			free(sess->id);
			free(sess);

			u->nsessions--;
			res_user_free(u);
		} else {
			ptr = &sess->next;
		}
	}

	return 1;
}

/*
 * Get the user object for the named user
 */
res_user_t *
res_user_get(const char *name)
{
	res_user_t	*user;

	for (user = res_users; user; user = user->next) {
		if (!strcmp(user->name, name))
			return user;
	}

	return NULL;
}

res_user_t *
res_user_create(const char *name)
{
	res_user_t	*user;

	user = (res_user_t *) calloc(1, sizeof(*user));
	user->name = strdup(name);

	user->next = res_users;
	res_users = user;
	return user;
}

void
res_user_free(res_user_t *user)
{
	res_user_t	**p;

	user->refcnt--;
	if (user->refcnt != 0)
		return;

	for (p = &res_users; *p; ) {
		if (*p == user)
			*p = user->next;
		else
			p = &(*p)->next;
	}
	res_user_revoke_all(user);
	free(user->classes);
	free(user->name);
	memset(user, 0, sizeof(*user));
	free(user);
}

/*
 * Grant a user access to a resource class
 */
void
res_user_grant(res_user_t *user, res_class_t *cls)
{
	unsigned int	n = user->nclasses;

	for (n = 0; n < user->nclasses; n++) {
		if (user->classes[n] == cls)
			return;
	}

	user->classes = (res_class_t **) realloc(user->classes, 
				(n + 1) * sizeof(res_class_t *));
	user->classes[n] = cls;
	user->nclasses += 1;
	cls->refcnt++;
}

/*
 * Check default ACLs when creating a new session
 */
void
res_user_resolve_acls(res_user_t *user, const char *tty)
{
	res_class_t	*cl;
	char		*groups[MAXGROUPS+1];
	unsigned int	n;

	get_groups(user->name, groups, MAXGROUPS);
	for (cl = res_classes; cl; cl = cl->next) {
		int	permission;

		permission = res_acl_match(cl->acl, tty, user->name,
					(const char **) groups);
		if (permission > 0) {
			if (opt_debug)
				log("granted user %s access to %s\n",
					user->name, cl->name);
			res_user_grant(user, cl);
		} else if (permission < 0) {
			if (opt_debug)
				log("denied user %s access to %s\n",
					user->name, cl->name);
			continue;
		}
	}

	for (n = 0; groups[n]; n++)
		free(groups[n]);
}

/*
 * Revoke a user's right to a given resource class
 */
void
res_user_revoke(res_user_t *user, res_class_t *cls)
{
	unsigned int	m, n;

	for (m = n = 0; n < user->nclasses; n++) {
		if (user->classes[n] == cls) {
			res_class_revoke(cls, user);
			res_class_free(cls);
		} else {
			user->classes[m++] = user->classes[n];
		}
	}
	user->nclasses--;
}

/*
 * Revoke all leases a user has
 */
void
res_user_revoke_all(res_user_t *user)
{
	unsigned int	n;
	res_class_t	*cls;

	for (n = 0; n < user->nclasses; n++) {
		cls = user->classes[n];

		res_class_revoke(cls, user);
		res_class_free(cls);
	}
	user->nclasses = 0;
}

/*
 * Get the given device
 */
res_device_t *
res_user_get_device(res_user_t *user, res_name_t *name, int readonly)
{
	res_class_t	*cls;
	res_device_t	*dev;
	unsigned int	n;

	for (n = 0; n < user->nclasses; n++) {
		cls = user->classes[n];
		if (!(dev = res_class_get_device(cls, name)))
			continue;

		/* Check read-only flag */
		if (!readonly && (dev->flags & DEV_FLAGS_RO))
			continue;

		return dev;
	}

	return NULL;
}

/*
 * Open the given device
 */
int
res_user_open(res_user_t *user, res_name_t *name, int flags)
{
	res_device_t	*dev;

	dev = res_user_get_device(user, name, flags & DEV_FLAGS_RO);
	if (dev == NULL) {
		errno = EACCES;
		return -1;
	}

	return res_name_open(name, flags);
}
