/*
 * Entry point and user interface for the applet.
 *
 * Rhythmbox Applet
 * Copyright (C) 2004-2005 Paul Kuliniewicz <kuliniew@purdue.edu>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */

#include <config.h>

#include "rb-proxy.h"
#include "rb-rating.h"
#include "rb-stock-icons.h"

#include <panel-applet.h>
#include <panel-applet-gconf.h>
#include <libgnome/gnome-i18n.h>
#include <libgnomeui/gnome-about.h>
#include <libgnomeui/gnome-window-icon.h>
#include <gtk/gtkbutton.h>
#include <gtk/gtkhbox.h>
#include <gtk/gtkimage.h>
#include <gtk/gtklabel.h>
#include <gtk/gtktogglebutton.h>
#include <gtk/gtktooltips.h>
#include <gtk/gtkvbox.h>
#include <glade/glade-xml.h>

#include <string.h>

/* Avoid compiler warnings in BONOBO_UI_VERB and friends. */
#define MY_BONOBO_UI_UNSAFE_VERB(name,cb) \
	{ (name), ((BonoboUIVerbFn)(cb)), NULL, NULL }
#define MY_BONOBO_UI_VERB_END             \
	{ NULL, NULL, NULL, NULL }

typedef struct {
	GtkWidget *applet;

	RbProxy *proxy;
	gchar *song_tip;

	GtkTooltips *tooltips;

	GtkWidget *preferences_dialog;
	GtkWidget *about_dialog;

	GtkWidget *main_box;
	GtkWidget *button_box;
	GtkWidget *rating;
	GtkWidget *time;
	GtkWidget *time_event;
	GtkWidget *previous;
	GtkWidget *play_pause_stop;
	GtkWidget *next;

	GtkWidget *play_image;
	GtkWidget *pause_image;
	GtkWidget *stop_image;
	GtkWidget *previous_image;
	GtkWidget *next_image;
} AppletData;

static gboolean rb_applet_fill (PanelApplet *applet,
			    const gchar *iid,
			    gpointer unused);

static void set_play_pause_stop_appearance (AppletData *data,
					    GtkWidget *image,
					    const gchar *tip);
static void set_initial_state (AppletData *data);

static void display_rating (AppletData *data, int rating);
static void enable_rating (AppletData *data, gboolean enable);

static void preferences_cb (BonoboUIComponent *uic,
			    AppletData *data,
			    const gchar *verbname);
static void about_cb (BonoboUIComponent *uic,
		      AppletData *data,
		      const gchar *verbname);
static void rate_cb (RBRating *rating,
		     double score,
		     AppletData *data);
static void show_rating_toggled_cb (GtkToggleButton *button,
				    AppletData *data);
static void response_cb (GtkDialog *dialog,
			 gint response,
			 gpointer unused); 

static void popup_component_event_cb (BonoboUIComponent *uic,
				      const gchar *path,
				      Bonobo_UIComponent_EventType type,
				      const gchar *state_string,
				      AppletData *data);

static void play_pause_stop_cb (GtkButton *button,
				AppletData *data);
static void previous_cb (GtkButton *button,
			 AppletData *data);
static void next_cb (GtkButton *button,
		     AppletData *data);

static gboolean button_press_cb (GtkWidget *widget,
				 GdkEventButton *event,
				 AppletData *data);
static void change_orient_cb (PanelApplet *applet,
			      PanelAppletOrient orient,
			      AppletData *data);
static void destroy_cb (GtkWidget *widget,
			AppletData *data);

static void time_updated_cb (RbProxy *proxy,
			     glong play_time,
			     AppletData *data);
static void song_changed_cb (RbProxy *proxy,
			     const RbProxySongInfo *song,
			     AppletData *data);
static void playing_changed_cb (RbProxy *proxy,
				gboolean playing,
				AppletData *data);

static const BonoboUIVerb rb_applet_menu_verbs[] = {
	MY_BONOBO_UI_UNSAFE_VERB ("Preferences", preferences_cb),
	MY_BONOBO_UI_UNSAFE_VERB ("About", about_cb),
	MY_BONOBO_UI_VERB_END
};

/* The applet itself. */

static gboolean
rb_applet_fill (PanelApplet *applet,
		const gchar *iid,
		gpointer unused)
{
	AppletData *data;
	GtkWidget *label;
	BonoboUIComponent *component;

	if (strcmp (iid, "OAFIID:GNOME_Rhythmbox_Applet") != 0)
		return FALSE;

	gnome_window_icon_set_default_from_file (
		PKG_DATA_DIR "rhythmbox.png");

	data = g_new0 (AppletData, 1);
	data->applet = GTK_WIDGET (applet);
	panel_applet_set_flags (applet, PANEL_APPLET_EXPAND_MINOR);

	rb_stock_icons_init ();

	data->proxy = rb_proxy_new ();
	data->tooltips = gtk_tooltips_new ();
	data->song_tip = NULL;
	data->main_box = NULL;
	data->preferences_dialog = NULL;
	data->about_dialog = NULL;

	/* Images */

	data->play_image = gtk_image_new_from_stock
		(RB_STOCK_PLAY, GTK_ICON_SIZE_SMALL_TOOLBAR);
	g_object_ref (data->play_image);
	data->pause_image = gtk_image_new_from_stock
		(RB_STOCK_PAUSE, GTK_ICON_SIZE_SMALL_TOOLBAR);
	g_object_ref (data->pause_image);
	data->stop_image = gtk_image_new_from_stock
		(RB_STOCK_STOP, GTK_ICON_SIZE_SMALL_TOOLBAR);
	g_object_ref (data->stop_image);
	data->previous_image = gtk_image_new_from_stock
		(RB_STOCK_PREVIOUS, GTK_ICON_SIZE_SMALL_TOOLBAR);
	g_object_ref (data->previous_image);
	data->next_image = gtk_image_new_from_stock
		(RB_STOCK_NEXT, GTK_ICON_SIZE_SMALL_TOOLBAR);
	g_object_ref (data->next_image);

	/* Applet widgets */

	data->rating = GTK_WIDGET (rb_rating_new ());
	gtk_tooltips_set_tip (data->tooltips, data->rating,
			      _("Rate current song"), NULL);
	g_signal_connect (data->rating, "rated",
			  G_CALLBACK (rate_cb), data);

	data->time_event = gtk_event_box_new ();
	data->time = gtk_label_new ("--:--");
	gtk_container_add (GTK_CONTAINER (data->time_event), data->time);

	data->button_box = gtk_hbox_new (FALSE, 0);

	data->previous = gtk_button_new ();
	gtk_container_add
		(GTK_CONTAINER (data->previous), data->previous_image);
	gtk_tooltips_set_tip (data->tooltips, data->previous,
			      _("Play previous song"), NULL);
	g_signal_connect (data->previous, "clicked",
			  G_CALLBACK (previous_cb), data);
	g_signal_connect (data->previous, "button_press_event",
			  G_CALLBACK (button_press_cb), data);
	gtk_box_pack_start_defaults
		(GTK_BOX (data->button_box), data->previous);

	data->play_pause_stop = gtk_button_new ();
	gtk_container_add
		(GTK_CONTAINER (data->play_pause_stop), data->play_image);
	gtk_tooltips_set_tip (data->tooltips, data->play_pause_stop,
			      _("Start playing"), NULL);
	g_signal_connect (data->play_pause_stop, "clicked",
			  G_CALLBACK (play_pause_stop_cb), data);
	g_signal_connect (data->play_pause_stop, "button_press_event",
			  G_CALLBACK (button_press_cb), data);
	gtk_box_pack_start_defaults
		(GTK_BOX (data->button_box), data->play_pause_stop);

	data->next = gtk_button_new ();
	gtk_container_add
		(GTK_CONTAINER (data->next), data->next_image);
	gtk_tooltips_set_tip (data->tooltips, data->next,
			      _("Play next song"), NULL);
	g_signal_connect (data->next, "clicked",
			  G_CALLBACK (next_cb), data);
	g_signal_connect (data->next, "button_press_event",
			  G_CALLBACK (button_press_cb), data);
	gtk_box_pack_start_defaults
		(GTK_BOX (data->button_box), data->next);

	panel_applet_add_preferences (applet,
				      "/schemas/apps/rhythmbox-applet/prefs",
				      NULL);

	panel_applet_setup_menu_from_file (applet,
					   DATA_DIR,
					   "GNOME_Rhythmbox_Applet.xml",
					   NULL,
					   rb_applet_menu_verbs,
					   data);

	/* Misc. signals */

	g_signal_connect (data->proxy, "song_changed",
			  G_CALLBACK (song_changed_cb), data);

	g_signal_connect (data->proxy, "playing_changed",
			  G_CALLBACK (playing_changed_cb), data);

	g_signal_connect (data->proxy, "time_updated",
			  G_CALLBACK (time_updated_cb), data);

	g_signal_connect (applet, "change_orient",
			  G_CALLBACK (change_orient_cb), data);

	g_signal_connect (applet, "destroy",
			  G_CALLBACK (destroy_cb), data);

	component = panel_applet_get_popup_component (PANEL_APPLET (applet));
	g_signal_connect (component, "ui-event",
			  G_CALLBACK (popup_component_event_cb), data);

	/* Initialize with current state. */

	set_initial_state (data);

	/* Finish up. */

	gtk_widget_show (GTK_WIDGET (applet));

	return TRUE;
}

static void
set_initial_state (AppletData *data)
{
	/* eew, hackish */

	RbProxySongInfo *song = rb_proxy_get_song_info (data->proxy);
	gboolean connected = rb_proxy_is_connected (data->proxy);
	gboolean playing = rb_proxy_is_playing (data->proxy);
	PanelAppletOrient orient = panel_applet_get_orient (
		PANEL_APPLET (data->applet));

	change_orient_cb (PANEL_APPLET (data->applet), orient, data);
	song_changed_cb (data->proxy, song, data);
	playing_changed_cb (data->proxy, playing, data);

	rb_proxy_song_info_free (song);
}

static void
set_play_pause_stop_appearance (AppletData *data,
				GtkWidget *image,
				const gchar *tip)
{
	gtk_container_remove (
		GTK_CONTAINER (data->play_pause_stop),
		gtk_bin_get_child (GTK_BIN (data->play_pause_stop)));
	gtk_container_add (GTK_CONTAINER (data->play_pause_stop), image);
	gtk_widget_show (image);
	gtk_tooltips_set_tip (data->tooltips, data->play_pause_stop,
			      tip, NULL);
}

static void
display_rating (AppletData *data, int rating)
{
	BonoboUIComponent *component;
	char path[] = "/commands/RateX";

	/* Set the rating displayed by the rating widget. */

	g_object_set (data->rating,
		      "score", (double) rating,
		      NULL);

	/* Set the rating displayed by the context menu. */

	component = panel_applet_get_popup_component (
		PANEL_APPLET (data->applet));

	path[14] = (char) ('0' + rating);
	bonobo_ui_component_set_prop (component,
				      path,
				      "state",
				      "1",
				      NULL);
}

static void
enable_rating (AppletData *data, gboolean enable)
{
	BonoboUIComponent *component;
	char path[] = "/commands/RateX";
	int i;

	component = panel_applet_get_popup_component (
		PANEL_APPLET (data->applet));

	if (!enable)
	{
		g_object_set (data->rating, "score", 0.0, NULL);
		bonobo_ui_component_set_prop (component,
					      "/commands/Rate0",
					      "state",
					      "1",
					      NULL);
	}

	gtk_widget_set_sensitive (data->rating, enable);

	for (i = 0; i <= 5; i++)
	{
		path[14] = (char) ('0' + i);
		bonobo_ui_component_set_prop (component,
					      path,
					      "sensitive",
					      enable ? "1" : "0",
					      NULL);
	}
}

/* Callbacks for user actions. */

static void
play_pause_stop_cb (GtkButton *button, AppletData *data)
{
	rb_proxy_play_pause_stop (data->proxy);
}

static void
previous_cb (GtkButton *button, AppletData *data)
{
	rb_proxy_previous (data->proxy);
}

static void
next_cb (GtkButton *button, AppletData *data)
{
	rb_proxy_next (data->proxy);
}

static void
rate_cb (RBRating *rating, double score, AppletData *data)
{
	double current;
	double delta;

	g_return_if_fail (score >= 0.0);
	g_return_if_fail (score <= 5.0);

	/*
	 * Only rate if it's an actual change, mainly because the
	 * context menu will create spurious events indistinguishable
	 * from actually rating, every time the menu is displayed.
	 *
	 * Testing score and current for equality *ought* to work,
	 * since the Bonobo bindings currently round off anyway, but
	 * I'll be paranoid about it.
	 */

	g_object_get (data->rating, "score", &current, NULL);
	delta = current - score;
	if (delta > 0.1 || delta < -0.1)
	{
		rb_proxy_rate_current (data->proxy, score);
		display_rating (data, (int) score);
	}
}

static void
preferences_cb (BonoboUIComponent *uic,
		AppletData *data,
		const gchar *verbname)
{
	static GladeXML *xml = NULL;
	GtkWidget *show_rating;
	gboolean rating_visible;

	if (data->preferences_dialog == NULL)
	{
		if (xml != NULL)
			g_object_unref (xml);

		xml = glade_xml_new (PKG_DATA_DIR "rhythmbox-applet.glade",
				     NULL, NULL);

		data->preferences_dialog = glade_xml_get_widget (
			xml, "preferences-dialog");

		show_rating = glade_xml_get_widget (xml, "show-rating");

		g_object_get (data->rating, "visible", &rating_visible, NULL);
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (show_rating),
					      rating_visible);

		g_signal_connect (show_rating,
				  "toggled",
				  G_CALLBACK (show_rating_toggled_cb),
				  data);

		g_signal_connect (data->preferences_dialog,
				  "response",
				  G_CALLBACK (response_cb),
				  NULL);

		g_signal_connect (data->preferences_dialog,
				  "destroy",
				  G_CALLBACK (gtk_widget_destroyed),
				  &data->preferences_dialog);

		gtk_widget_show (data->preferences_dialog);
	}
	else
	{
		gtk_window_set_screen (GTK_WINDOW (data->preferences_dialog),
				       gtk_widget_get_screen (
					       GTK_WIDGET (data->applet)));
		gtk_window_present (GTK_WINDOW (data->preferences_dialog));
	}
}

static void
show_rating_toggled_cb (GtkToggleButton *button,
			AppletData *data)
{
	gboolean show_it = gtk_toggle_button_get_active (button);

	g_object_set (data->rating, "visible", show_it, FALSE);
	panel_applet_gconf_set_bool (PANEL_APPLET (data->applet),
				     "show_rating", show_it,
				     NULL);
}

static void
response_cb (GtkDialog *dialog,
	     gint response,
	     gpointer unused)
{
	gtk_widget_hide (GTK_WIDGET (dialog));
}

static void
about_cb (BonoboUIComponent *uic,
	  AppletData *data,
	  const gchar *verbname)
{
	static const gchar *authors[] = {
		"Paul Kuliniewicz <kuliniew@purdue.edu>",
		NULL
	};

	static const gchar *documenters[] = {
		NULL
	};

	const gchar *translator_credits = _("translator-credits");

	if (data->about_dialog == NULL)
	{
		data->about_dialog = gnome_about_new (
			_("Rhythmbox Applet"), PACKAGE_VERSION,
			_("(C) 2004-2005 Paul Kuliniewicz"),
			_("Control Rhythmbox from a GNOME panel."),
			authors,
			documenters,
			strcmp (translator_credits, "translator-credits") != 0 ?
				translator_credits : NULL,
			NULL);

		g_signal_connect (data->about_dialog,
				  "destroy",
				  G_CALLBACK (gtk_widget_destroyed),
				  &data->about_dialog);

		gtk_widget_show (data->about_dialog);
	}
	else
	{
		gtk_window_set_screen (GTK_WINDOW (data->about_dialog),
				       gtk_widget_get_screen (
					       GTK_WIDGET (data->applet)));
		gtk_window_present (GTK_WINDOW (data->about_dialog));
	}
}

static void
popup_component_event_cb (BonoboUIComponent *uic,
			  const gchar *path,
			  Bonobo_UIComponent_EventType type,
			  const gchar *state_string,
			  AppletData *data)
{
	/* Handle when "RateX" is set to true, for some X. */

	if (strncmp (path, "Rate", 4) == 0 &&
	    path[4] >= '0' &&
	    path[4] <= '5' &&
	    path[5] == '\0' &&
	    strcmp (state_string, "1") == 0)
	{
		/* Fake the user manipulating the rating widget. */

		rate_cb (RB_RATING (data->rating),
			 (double) (path[4] - '0'),
			 data);
	}
}

/* Callbacks for other user interface events. */

static gboolean
button_press_cb (GtkWidget *widget, GdkEventButton *event, AppletData *data)
{
	/* Make sure non-left-clicks propagate up to the applet. */
	if (event->button != 1)
		g_signal_stop_emission_by_name (widget, "button_press_event");
	return FALSE;
}

static void
change_orient_cb (PanelApplet *applet,
		  PanelAppletOrient orient,
		  AppletData *data)
{
	if (data->main_box != NULL)
	{
		g_object_ref (G_OBJECT (data->rating));
		gtk_container_remove (GTK_CONTAINER (data->rating),
				      data->rating);

		g_object_ref (G_OBJECT (data->time_event));
		gtk_container_remove (GTK_CONTAINER (data->main_box),
				      data->time_event);

		g_object_ref (G_OBJECT (data->button_box));
		gtk_container_remove (GTK_CONTAINER (data->main_box),
				      data->button_box);

		gtk_container_remove (GTK_CONTAINER (applet),
				      data->main_box);
		gtk_widget_destroy (data->main_box);
	}

	switch (orient)
	{
	case PANEL_APPLET_ORIENT_UP:
	case PANEL_APPLET_ORIENT_DOWN:
		data->main_box = gtk_hbox_new (FALSE, 5);
		break;
	case PANEL_APPLET_ORIENT_LEFT:
	case PANEL_APPLET_ORIENT_RIGHT:
		data->main_box = gtk_vbox_new (FALSE, 0);
		break;
	default:
		g_return_if_reached ();
	}

	gtk_box_pack_start_defaults (GTK_BOX (data->main_box),
				     data->rating);
	g_object_set (data->rating,
		      "visible",
		      panel_applet_gconf_get_bool (PANEL_APPLET (data->applet),
			      			   "show_rating",
						   NULL),
		      NULL);

	gtk_box_pack_start_defaults (GTK_BOX (data->main_box),
				     data->time_event);
	gtk_widget_show_all (data->time_event);

	gtk_box_pack_start_defaults (GTK_BOX (data->main_box),
				     data->button_box);
	gtk_widget_show_all (data->button_box);

	gtk_container_add (GTK_CONTAINER (applet), data->main_box);
	gtk_widget_show (data->main_box);
}

static void
destroy_cb (GtkWidget *widget, AppletData *data)
{
	g_object_unref (data->play_image);
	g_object_unref (data->pause_image);
	g_object_unref (data->stop_image);
	g_object_unref (data->previous_image);
	g_object_unref (data->next_image);

	gtk_object_destroy (GTK_OBJECT (data->tooltips));

	g_object_unref (G_OBJECT (data->proxy));

	rb_stock_icons_shutdown ();

	g_free (data);
}

/* Callbacks for Rhythmbox events. */

static void
time_updated_cb (RbProxy *proxy, glong play_time, AppletData *data)
{
	if (play_time >= 0)
	{
		glong minutes = play_time / 60;
		glong seconds = play_time % 60;
		gchar *text = g_strdup_printf ("%ld:%.2ld", minutes, seconds);
		gtk_label_set_text (GTK_LABEL (data->time), text);
		g_free (text);
		text = g_strdup_printf (data->song_tip, minutes, seconds);
		gtk_tooltips_set_tip (data->tooltips, data->time_event,
				      text, NULL);
		g_free (text);
	}
	else
	{
		gtk_label_set_text (GTK_LABEL (data->time), "--:--");
		gtk_tooltips_set_tip (data->tooltips, data->time_event,
				      _("Not playing"), NULL);
	}
}

static void
song_changed_cb (RbProxy *proxy, const RbProxySongInfo *song, AppletData *data)
{
	g_free (data->song_tip);

	if (song != NULL)
	{
		if (song->duration > 0)
		{
			/* playing local file */

			gchar *whence = g_strdup_printf (
				/* To translator: from ALBUM by ARTIST */
				_("from %s by %s"),
				song->album,
				song->artist);

			gchar *when = g_strdup_printf (
				/* To translator: CURRENT-TIME of TOTAL-TIME */
				_("%%ld:%%.2ld of %d:%.2d"),
				song->duration / 60,
				song->duration % 60);

			data->song_tip = g_strdup_printf (
				"%s\n%s\n%s",
				song->title,
				whence,
				when);

			g_free (whence);
			g_free (when);
		}
		else
		{
			/* playing Internet radio */

			data->song_tip = g_strdup_printf (
				"%s\n%%ld:%%.2ld",
				song->title);
		}

		display_rating (data, song->rating);
	}
	else
	{
		data->song_tip = g_strdup (_("Not playing"));
		display_rating (data, 0);
	}

	time_updated_cb (proxy, rb_proxy_get_play_time (proxy), data);
}

static void
playing_changed_cb (RbProxy *proxy, gboolean playing, AppletData *data)
{
	GtkWidget *image;
	gboolean has_song;
	const gchar *tip;

	if (!playing)
	{
		image = data->play_image;
		tip = _("Start playing");
	}
	else if (rb_proxy_is_pausable (proxy))
	{
		image = data->pause_image;
		tip = _("Pause playback");
	}
	else
	{
		image = data->stop_image;
		tip = _("Stop playback");
	}

	has_song = playing || rb_proxy_is_pausable (proxy);

	enable_rating (data, has_song);
	gtk_widget_set_sensitive (data->previous, has_song);
	gtk_widget_set_sensitive (data->next, has_song);
	set_play_pause_stop_appearance (data, image, tip);
}

/* Entry point. */

PANEL_APPLET_BONOBO_FACTORY ("OAFIID:GNOME_Rhythmbox_Applet_Factory",
			     PANEL_TYPE_APPLET,
			     "Rhythmbox Applet Factory",
			     "0",
			     rb_applet_fill,
			     NULL);
