/*
 *  contact-window.c     - 
 *
 *  Created: 20011216
 *
 *  Copyright (c) 2001 by Tomasz Trojanwoski
 *  All rights reserved
 *
 *  $Id: contact-window.c,v 1.116 2002/03/23 20:20:59 tomek Exp $
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <contact-window.h>

#include <intl.h>
#include <about-dialog.h>
#include <confirm-dialog.h>
#include <windows-list.h>
#include <database.h>
#include <fields-list.h>
#include <name-dialog.h>
#include <phone-dialog.h>
#include <address-dialog.h>
#include <appointment-window.h>

/* Funkcja przepisujca dane z pola edycji nazwy firmy okienka kontaktu do struktury danych
   kontaktu. */
static void copy_data_firm_name(contact_data *data)
{
    gchar *text;
    GtkEntry *entry;

    g_assert(data != NULL);

    /* Przepisywanie danych z pola edycji nazwy firmy okienka kontaktu do struktury danych 
       kontaktu i sprawdzanie czy dane zostay zmodyfikowane, a jeeli tak - ustawianie flagi
       modified struktury danych kontaktu. */
    entry = gtk_object_get_data(GTK_OBJECT(data->contact_window), "firm_name");
    text = gtk_entry_get_text(GTK_ENTRY(entry));
    if (strcmp(text, data->firm_name) != 0)
    {
        /* Pobranie zawartoci pola edycji i skopiowanie do struktury contact_data. */
        strcpy(data->firm_name, text);
    
        /* Utworzenie nowej listy wyboru w polu "Zapisz jako". */
        create_save_as_list(data);
    
        /* Ustawianie flagi modified struktury danych kontaktu. */
        data->modified = TRUE;
        data->global_modified = TRUE;
    }
}

/* Funkcja przepisujca dane z pola edycji adresu e-mail okienka kontaktu do struktury danych
   kontaktu. */
static void copy_data_email(contact_data *data)
{
    gchar *text;
    GtkEntry *entry;
    
    g_assert(data != NULL);
    
    /* Przepisywanie danych z pola edycji adresu e-mail okienka kontaktu do struktury danych 
       kontaktu i sprawdzanie czy dane zostay zmodyfikowane, a jeeli tak - ustawianie flagi
       modified struktury danych kontaktu. */
    entry = gtk_object_get_data(GTK_OBJECT(data->contact_window), "email");
    text = gtk_entry_get_text(GTK_ENTRY(entry));
    if (strcmp(text, data->email[data->email_type].value) != 0)
    {
        strcpy(data->email[data->email_type].value, text);
	data->email[data->email_type].modified = TRUE;
	data->global_modified = TRUE;
    }
}

/* Funkcja prepisujca dane z okienka kontaktu do struktury danych kontaktu. */
static void copy_data(contact_data *data)
{
    gpointer entry;
    gchar *text;
    
    g_assert(data != NULL);

    /* Przepisywanie danych elementw okna dialogowego do struktury danych 
       kontaktu i sprawdzanie czy dane zostay zmodyfikowane, a jeeli tak
       - ustawianie flagi modified struktury danych kontaktu. */
    copy_data_firm_name(data);
    
    entry = gtk_object_get_data(GTK_OBJECT(data->contact_window), "position");
    text = gtk_entry_get_text(GTK_ENTRY(entry));
    if (strcmp(text, data->position) != 0)
    {
        strcpy(data->position, text);
	data->modified = TRUE;
	data->global_modified = TRUE;
    }

    entry = gtk_object_get_data(GTK_OBJECT(data->contact_window), "save_as");
    text = gtk_entry_get_text(GTK_ENTRY(entry));
    if (strcmp(text, data->save_as) != 0)
    {
        strcpy(data->save_as, text);
	data->modified = TRUE;
	data->global_modified = TRUE;
    }
    
    copy_data_email(data);
    
    entry = gtk_object_get_data(GTK_OBJECT(data->contact_window), "www");
    text = gtk_entry_get_text(GTK_ENTRY(entry));
    if (strcmp(text, data->www) != 0)
    {
        strcpy(data->www, text);
	data->modified = TRUE;
	data->global_modified = TRUE;
    }
}

/* Funkcja wstawiajca dane ze zmiennej zawierajcej dane kontaktu do bazy. */
static void insert_data(contact_data *data)
{
    int i;

    g_assert(data != NULL);

    /* Jeeli identyfikator kontaktu jest rwny 0, oznacza to e jest tworzony nowy kontakt,
       w przeciwnym wypadku - uaktualniany istniejcy */
    if (data->id == 0)
        /* Zapisanie danych kontaktu do bazy danych. */
        db_contact_add(data);
    else
        /* Uaktualnienie danych kontaktu w bazie danych. */
	db_contact_update(data);
    
    /* Wywoywanie funkcji uaktualniajcej list kontaktw w gwnym
       oknie programu. */
    show_fields_list(LT_CONTACT, FALSE);
	
    /* Zerowanie wszystkich flag modified struktury okna kontaktu. */
    data->modified = FALSE;
    data->global_modified = FALSE;
    for (i = 0; i < 3; i++)
        data->address[i].modified = FALSE;
    for (i = 0; i < 19; i++)
        data->phone[i].modified = FALSE;
    for (i = 0; i < 3; i++)
        data->email[i].modified = FALSE;
}

/* Funkcja obsugujca menu Plik-Nowy-Kontakt otwierajca nowe okno kontaktw. */
void new_contact_menu(GtkWidget *widget, gpointer data)
{
    create_contact_window(0);
}

/* Funkcja obsugujca menu Plik-Zapisz okienka kontaktw. */
static void save_menu(GtkWidget *widget, contact_data *data)
{
    g_assert(data != NULL);

    /* Kopiowanie danych kontaktu do struktury. */
    copy_data(data);
    
    if (data->global_modified)
        /* Wstawienie danych do bazy. */
        insert_data(data);
}

/* Funkcja obsugujca menu Plik-Zapisz i nowy, zapisujca dane kontaktu do bazy i czyszczca
   zawarto okna kontaktw. */
static void save_and_new_menu(GtkWidget *widget, contact_data *data)
{
    GtkWidget *contact_window;
    gpointer entry;
    
    g_assert(data != NULL);

    /* Kopiowanie danych kontaktu do struktury. */
    copy_data(data);

    if (data->global_modified)
        /* Wstawianie danych do bazy. */
        insert_data(data);
    
    /* Jeeli pole id struktury danych kontaktu jest rwne 0, znaczy to e dane nowego kontaktu
       nie byy modyfikowane, wic nie jest konieczne zerowanie struktury i elementw okienka
       kontaktw. */
    if (data->id != 0)
    {
        int i;

        /* Zerowanie elementw okienka kontaktw. */
        entry = gtk_object_get_data(GTK_OBJECT(data->contact_window), "full_name");
        gtk_entry_set_text(GTK_ENTRY(entry), "");

        entry = gtk_object_get_data(GTK_OBJECT(data->contact_window), "firm_name");
        gtk_entry_set_text(GTK_ENTRY(entry), "");

        entry = gtk_object_get_data(GTK_OBJECT(data->contact_window), "position");
        gtk_entry_set_text(GTK_ENTRY(entry), "");

        entry = gtk_object_get_data(GTK_OBJECT(data->contact_window), "save_as");
        gtk_entry_set_text(GTK_ENTRY(entry), "");
	
	entry = gtk_object_get_data(GTK_OBJECT(data->contact_window), "address_menu");
	gtk_option_menu_set_history(GTK_OPTION_MENU(entry), 0);
	
	entry = gtk_object_get_data(GTK_OBJECT(data->contact_window), "send_to");
	GTK_TOGGLE_BUTTON(entry)->active = 0;
	gtk_widget_draw(GTK_WIDGET(entry), NULL);
	
	entry = gtk_object_get_data(GTK_OBJECT(data->contact_window), "phone0_menu");
	gtk_option_menu_set_history(GTK_OPTION_MENU(entry), PHONE_TYPE_HOME);

	entry = gtk_object_get_data(GTK_OBJECT(data->contact_window), "phone1_menu");
	gtk_option_menu_set_history(GTK_OPTION_MENU(entry), PHONE_TYPE_WORK);

	entry = gtk_object_get_data(GTK_OBJECT(data->contact_window), "phone2_menu");
	gtk_option_menu_set_history(GTK_OPTION_MENU(entry), PHONE_TYPE_FAX_WORK);

	entry = gtk_object_get_data(GTK_OBJECT(data->contact_window), "phone3_menu");
	gtk_option_menu_set_history(GTK_OPTION_MENU(entry), PHONE_TYPE_MOBILE);
	
	entry = gtk_object_get_data(GTK_OBJECT(data->contact_window), "email_menu");
	gtk_option_menu_set_history(GTK_OPTION_MENU(entry), EMAIL_TYPE_1);
	
	entry = gtk_object_get_data(GTK_OBJECT(data->contact_window), "email");
	gtk_entry_set_text(GTK_ENTRY(entry), "");

	entry = gtk_object_get_data(GTK_OBJECT(data->contact_window), "www");
	gtk_entry_set_text(GTK_ENTRY(entry), "");

        /* Zerowanie struktury danych kontaktu. */
        contact_window = data->contact_window;
	clear_contact_data(data);
        data->contact_window = contact_window;
	
	create_save_as_list(data);
	create_full_address(data);
	for (i = 0; i < 4; i++)
	    create_full_phone(data, i);
    }
    
    /* Inicjalizacja pierwszej zakadki. */
    entry = gtk_object_get_data(GTK_OBJECT(data->contact_window), "notebook");
    gtk_notebook_set_page(GTK_NOTEBOOK(entry), 0);
}

/* Funkcja obsugujca menu Plik-Usun. */
static void delete_menu(GtkWidget *widget, contact_data *data)
{
    gboolean may_delete = TRUE;

    g_assert(data != NULL);

    /* Sprawdzenie czy jest konieczne wywoywanie okienka potwierdzajacego. */
    if (data->global_modified || data->id != 0)
    {
        /* Wywoanie okienka potwierdzajcego. */
	if (G_CONFIRM_YES != confirm_dialog(data->contact_window,
	                                    _("Are you sure you want to delete this contact?"),
					    G_CONFIRM_YES | G_CONFIRM_NO))
	    may_delete = FALSE;
    }
    
    if (may_delete)
    {
	if (data->id != 0)
	{
	    /* Usuwanie kontaktu z bazy. */
            db_contact_delete(data->id);
	    
	    /* Uaktualnianie listy kontaktw. */
	    show_fields_list(LT_CONTACT, FALSE);
	}

	/* Usunicie okienka z listy okienek kontaktw. */
	remove_contact_window(data->contact_window);

        /* Zamknicie okienka kontaktw. */
        gtk_widget_destroy(data->contact_window);

        /* Zwolnienie pamieci zajmowanej przez dane kontaktw. */
        g_free(data);
    }
}

/* Funkcja obsugujca menu Plik-Zamknij okienka kontaktw. */
static void close_menu(GtkWidget *widget, contact_data *data)
{
    close_contact_window(data, TRUE);
}

/* Funkcja obsugujca sygna "delete_event" okienka kontaktw. */
static gint wnd_delete_event(GtkWidget *widget, GdkEvent *event, contact_data *data)
{
    return (!close_contact_window(data, TRUE));
}

/* Funkcja obsugujca sygna "clicked" przycisku "Imi i nazwisko". */
static void name_button_clicked(GtkWidget *widget, contact_data *data)
{
    g_assert(data != NULL);

    create_name_dialog(data);
}

/* Funkcja obsugujaca sygna "clicked" przycisku "Adres". */
static void address_button_clicked(GtkWidget *widget, contact_data *data)
{
    g_assert(data != NULL);

    create_address_dialog(data);
}

/* Funkcja obsugujca sygna "clicked" przycisku edycji telefonu. */
static void phone_button_clicked(GtkWidget *widget, guint phone_pos)
{
    contact_data *data;
    
    g_assert(widget != NULL);

    /* Pobieranie wskanika do struktury przechowujcej dane kontaktu. */
    data = (contact_data *)gtk_object_get_data(GTK_OBJECT(widget), "data");
    g_assert(data != NULL);
    
    create_phone_dialog(data, phone_pos);
}

/* Funkcja obsugujca sygna "changed" pola edycji "Organizacja". */
static void firm_entry_focus_out(GtkWidget *widget, GdkEventFocus *event, contact_data *data)
{
    copy_data_firm_name(data);
}

/* Funkcja obsugujca sygna "activate" menu listy wyboru typu adresu. */
static void address_type_select(GtkWidget *widget, address_type_enum address_type)
{
    contact_data *data;
    GtkWidget *toggle_button;
    
    g_assert(widget != NULL);

    /* Wstawienie wybranej wartoci okrelajcej typ adresu do struktury danych kontaktu */
    data = (contact_data *)gtk_object_get_data(GTK_OBJECT(widget), "data");
    g_assert(data != NULL);
    data->address_type = address_type;
    
    /* Przepisanie penego adresu do pola tekstowego adresu. */
    create_full_address(data);
    
    /* Ustawianie przycisku "Wysykowy" */
    toggle_button = gtk_object_get_data(GTK_OBJECT(data->contact_window),  "send_to");
    g_assert(toggle_button != NULL);
    if (address_type == data->send_to)
        GTK_TOGGLE_BUTTON(toggle_button)->active = 1;
    else
        GTK_TOGGLE_BUTTON(toggle_button)->active = 0;
    gtk_widget_draw(toggle_button, NULL);
}

/* Funkcja obsugujca sygna "toggled" przycisku "Wysykowy". */
static void send_to_toggled(GtkWidget *widget, contact_data *data)
{
    g_assert(data != NULL);
    g_assert(widget != NULL);
    
    /* Wstawianie identyfikatora wybranego adresu do pola struktury danych kontaktu. */
    if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widget)))
        data->send_to = data->address_type;
    else
        data->send_to = ADDRESS_TYPE_NONE;

    data->modified = TRUE;
    data->global_modified = TRUE;
}

/* Funkcja obsugujca sygna "activate" menu listy wyboru typu telefonu. */
static void phone_type_select(GtkWidget *widget, int phone_type_pos)
{
    contact_data *data;
    phone_type_enum phone_type = phone_type_pos & 0xff;
    phone_pos_enum  phone_pos = (phone_type_pos & ~0xff) >> 8;

    g_assert(widget != NULL);
    
    /* Wstawienie wybranej wartoci okrelajcej typ telefonu do struktury danych kontaktu. */
    data = (contact_data *)gtk_object_get_data(GTK_OBJECT(widget), "data");
    g_assert(data != NULL);
    data->phone_type[phone_pos] = phone_type;
    
    /* Przepisanie penego numeru telefonu do pola edycji numeru telefonu. */
    create_full_phone(data, phone_pos);
}

/* Funkcja obsugujca sygna "activate" menu listy wyboru typu adresu email. */
static void email_type_select(GtkWidget *widget, email_type_enum email_type)
{
    contact_data *data;
    
    g_assert(widget != NULL);

    /* Pobranie wskanika do struktury danych kontaktu. */
    data = (contact_data *)gtk_object_get_data(GTK_OBJECT(widget), "data");
    g_assert(data != NULL);

    /* Kopiowanie danych z pola edycji adresu e-mail do struktury danych kontaktu. */
    copy_data_email(data);
    
    /* Wstawianie wybranej wartoci okrelajcej typ adresu e-mail do struktury danych kontaktu. */
    data->email_type = email_type;
    
    /* Kopiowanie adresu e-mail do pola edycji adresu e-mail. */
    gtk_entry_set_text(gtk_object_get_data(GTK_OBJECT(data->contact_window), "email"),
                       data->email[data->email_type].value);
}

/* Funkcja obsugujca sygna "focus_out_event" pola edycji adresu e-mail. */
static void email_entry_focus_out(GtkWidget *widget, GdkEventFocus *event, contact_data *data)
{
    /* Kopiowanie danych z pola edycji adresu e-mail do struktury danych kontaktu. */
    copy_data_email(data);
}

/* Tworzenie elementu menu listy wyboru typu adresu. */
static GtkWidget *make_address_menu_item(char              *name,
                                         contact_data      *data,
					 address_type_enum address_type)
{
    GtkWidget *item;

    item = gtk_menu_item_new_with_label(name);
    gtk_widget_show(item);
			    
    gtk_object_set_data(GTK_OBJECT(item), "data", data);
			    
    gtk_signal_connect(GTK_OBJECT(item), "activate", GTK_SIGNAL_FUNC(address_type_select),
		       GUINT_TO_POINTER(address_type));
    
    return item;
}

/* Tworzenie elementu menu listy typu telefonu. */
static GtkWidget *make_phone_menu_item(char            *name,
                                       contact_data    *data,
				       phone_type_enum phone_type)
{
    GtkWidget *item;
    
    item = gtk_menu_item_new_with_label(name);
    gtk_widget_show(item);

    gtk_object_set_data(GTK_OBJECT(item), "data", data);
			    
    gtk_signal_connect(GTK_OBJECT(item), "activate", GTK_SIGNAL_FUNC(phone_type_select),
		       GUINT_TO_POINTER(phone_type));

    return item;
}

static char *phone_menu_items[] =
{
    N_("Assistant"),
    N_("Home"),
    N_("Home 2"),
    N_("Home Fax"),
    N_("Business Fax"),
    N_("Company"),
    N_("Other"),
    N_("Other Fax"),
    N_("ISDN"),
    N_("Pager"),
    N_("Primary"),
    N_("Business"),
    N_("Business 2"),
    N_("Radio"),
    N_("Car"),
    N_("Mobile"),
    N_("Telex"),
    N_("TTY/TDD"),
    N_("Callback")
};

/* Tworzenie menu listy wyboru typu telefonu. */
static GtkWidget *make_phone_menu(contact_data *data, phone_pos_enum phone_pos)
{
    GtkWidget *menu;
    int i;
    
    g_assert(data != NULL);
    
    menu = gtk_menu_new();
    
    for (i = 0; i < 19; i++)
        gtk_menu_append(GTK_MENU(menu), 
	                make_phone_menu_item(gettext(phone_menu_items[i]), data, phone_pos + i));

    return menu;
}

/* Tworzenie poszczeglnych zespow edycji telefonw. */
static void make_phone_lines(GtkWidget *fixed, contact_data *data)
{
    GtkWidget *optionmenu;
    GtkWidget *entry;
    GtkWidget *button;

    int i;
    
    for (i = 0; i < 4; i++)
    {
        /* Lista wyboru typu telefonu. */
	optionmenu = gtk_option_menu_new();
	gtk_widget_show(optionmenu);

	switch (i)
	{
	    case 0:
	        gtk_object_set_data(GTK_OBJECT(data->contact_window), "phone0_menu", optionmenu);
		break;
	    case 1:
	        gtk_object_set_data(GTK_OBJECT(data->contact_window), "phone1_menu", optionmenu);
		break;
	    case 2:
	        gtk_object_set_data(GTK_OBJECT(data->contact_window), "phone2_menu", optionmenu);
		break;
	    case 3:
	        gtk_object_set_data(GTK_OBJECT(data->contact_window), "phone3_menu", optionmenu);
		break;
	}
	
	gtk_fixed_put(GTK_FIXED(fixed), optionmenu, 8, i * 32 + 8);
	gtk_widget_set_uposition(optionmenu, 8, i * 32 + 8);
	gtk_widget_set_usize(optionmenu, 124, 24);
	
	gtk_option_menu_set_menu(GTK_OPTION_MENU(optionmenu), make_phone_menu(data, i << 8));
	gtk_option_menu_set_history(GTK_OPTION_MENU(optionmenu), data->phone_type[i]);
	
	/* Tworzenie pola edycji pierwszego telefonu. */
	entry = gtk_entry_new();
	gtk_widget_show(entry);
	
	switch (i)
	{
	    case 0:
	        gtk_object_set_data(GTK_OBJECT(data->contact_window), "phone0", entry);
		break;
	    case 1:
	        gtk_object_set_data(GTK_OBJECT(data->contact_window), "phone1", entry);
		break;
	    case 2:
	        gtk_object_set_data(GTK_OBJECT(data->contact_window), "phone2", entry);
		break;
	    case 3:
	        gtk_object_set_data(GTK_OBJECT(data->contact_window), "phone3", entry);
		break;
	}
	
	gtk_fixed_put(GTK_FIXED(fixed), entry, 136, i * 32 + 8);
	gtk_widget_set_uposition(entry, 136, i * 32 + 8);
	gtk_widget_set_usize(entry, 168, 24);
	gtk_entry_set_editable(GTK_ENTRY(entry), FALSE);

	create_full_phone(data, i);
		    
	/* Tworzenie przycisku edycji pierwszego telefonu. */
	button = gtk_button_new_with_label("...");
	gtk_widget_show(button);
		    
	gtk_object_set_data(GTK_OBJECT(button), "data", data);
		    
	gtk_signal_connect(GTK_OBJECT(button), "clicked", GTK_SIGNAL_FUNC(phone_button_clicked),
			   GUINT_TO_POINTER(i));
		    
	gtk_fixed_put(GTK_FIXED(fixed), button, 312, i * 32 + 8);
	gtk_widget_set_uposition(button, 312, i * 32 + 8);
	gtk_widget_set_usize(button, 24, 24);
    }
}

/* Tworzenie elementu menu listy wyboru typu adresu e-mail. */
static GtkWidget *make_email_menu_item(char            *name,
                                       contact_data    *data,
				       email_type_enum email_type)
{
    GtkWidget *item;
    
    item = gtk_menu_item_new_with_label(name);
    gtk_widget_show(item);
    
    gtk_object_set_data(GTK_OBJECT(item), "data", data);
    
    gtk_signal_connect(GTK_OBJECT(item), "activate", GTK_SIGNAL_FUNC(email_type_select),
		       GUINT_TO_POINTER(email_type));
    
    return item;
};

/* Struktura zawierajca menu gwne programu. */
static GtkItemFactoryEntry contact_menu[] = 
{
    { NULL, NULL,         NULL,                 0, "<Branch>" },       /*  0 */
    { NULL, NULL,         NULL,                 0, "<Branch>" },       /*  1 */
    { NULL, "<control>N", new_contact_menu,     0, NULL },             /*  2 */
    { NULL, NULL,         NULL,                 0, "<Separator>" },    /*  3 */
    { NULL, NULL,         new_appointment_menu, 0, NULL },             /*  4 */
    { NULL, NULL,         NULL,                 0, "<Separator>" },    /*  5 */
    { NULL, "<control>S", save_menu,            0, NULL },             /*  4 */
    { NULL, NULL,         save_and_new_menu,    0, NULL },             /*  7 */
    { NULL, NULL,         NULL,                 0, "<Separator>" },    /*  8 */
    { NULL, NULL,         delete_menu,          0, NULL },             /*  9 */
    { NULL, NULL,         NULL,                 0, "<Separator>" },    /* 10 */
    { NULL, NULL,         close_menu,           0, NULL },             /* 11 */
    { NULL, NULL,         NULL,                 0, "<Branch>" },       /* 12 */
    { NULL, NULL,         new_contact_menu,     0, NULL },             /* 13 */
    { NULL, NULL,         NULL,                 0, "<LastBranch>" },   /* 14 */
    { NULL, NULL,         about_menu,           0, NULL },             /* 15 */
};

/* Funkcja tworzaca okno kontaktw. 

   Struktura okna:
   
     GtkWindow (comtact_window)
      +- GtkVBox (main_vbox)
          +- GtkMenuBar (menubar)
	  +- GtkNotebook (notebook, "notebook")
	  |   +- GtkFixed                                            "Oglne"
	  |   |   +- GtkButton                              "Imi i nazwisko"
	  |   |   +- GtkEntry ("full_name")
	  |   |   +- GtkLabel                                   "Organizacja"
	  |   |   +- GtkEntry ("firm_name")
	  |   |   +- GtkLabel                                    "Stanowisko"
	  |   |   +- GtkEntry ("position")
	  |   |   +- GtkLabel                                   "Zapisz jako"
	  |   |   +- GtkCombo ("save_as_combo")
	  |   |       +- GtkEntry ("save_as")
	  |   +- GtkLabel                                            "Oglne"
	  |   +- GtkFixed                                             "Adres"
	  |   |   +- GtkButton                                        "Adres"
	  |   |   +- GtkOptionMenu ("address_menu")
	  |   |   |   +- GtkMenu
	  |   |   |       +- GtkMenuItem                               "Dom"
	  |   |   |       +- GtkMenuItem                             "Praca"
	  |   |   |       +- GtkMenuItem                              "Inny"
	  |   |   +- GtkCheckButton ("send_to")
	  |   |   +- GtkText ("full_address")
	  |   +- GtkLabel                                           "Adresy"
	  |   +- GtkFixed
	  |   |   +- GtkOptionMenu (1) ("phone0_menu")
	  |   |   |   +- GtkMenu
	  |   |   |       +- GtkMenuItem...
	  |   |   +- GtkEntry ("phone0")
	  |   |   +- GtkButton                                         "..."
	  |   |   +- GtkOptionMenu (2) ("phone1_menu")
	  |   |   |   +- GtkMenu
	  |   |   |       +- GtkMenuItem...
	  |   |   +- GtkEntry ("phone1")
	  |   |   +- GtkButton                                         "..."
	  |   |   +- GtkOptionMenu (3) ("phone2_menu")
	  |   |   |   +- GtkMenu
	  |   |   |       +- GtkMenuItem...
	  |   |   +- GtkEntry ("phone2")
	  |   |   +- GtkButton                                         "..."
	  |   |   +- GtkOptionMenu (4) ("phone3_menu")
	  |   |   |   +- GtkMenu
	  |   |   |       +- GtkMenuItem...
	  |   |   +- GtkEntry ("phone3")
	  |   |   +- GtkButton                                         "..."
	  |   +- GtkLabel                                         "Telefony"
	  |   +- GtkFixed
	  |   |   +- GtkOptionMenu ("email_menu")
	  |   |   |   +- GtkMenu
	  |   |   |       +- GtkMenuItem...
	  |   |   +- GtkEntry ("email")
	  |   |   +- GtkLabel                             "Strona sieci WWW"
	  |   |   +- GtkEntry ("www")
	  |   +- GtkLabel                                         "Internet"
	  +- GtkStatusBar (statusbar)

   Klucze zwizane z okienkiem kontaktw:
   
     data          - wskanik do struktury danych kontaktu.
     notebook      - wskanik do zakadek
     full_name     - wskanik do pola edycji "Imi i nazwisko"
     firm_name     - wskanik do pola edycji "Organizacja"
     position      - wskanik do pola edycji "Stanowisko"
     save_as_combo - wskanik do pola wyboru "Zapisz jako"
     save_as       - wskanik do pola edycji pola wyboru "Zapisz jako"
     address_menu  - wskanik do listy wyboru typu adresu
     full_address  - wskanik do pola tekstowego "Adres"
     send_to       - wskanik do przycisku "Wysykowy"
     phone0_menu   - wskanik do listy wyboru pierwszego telefonu
     phone0        - wskanik do pola edycji pierwszego telefonu
     phone1_menu   - wskanik do listy wyboru drugiego telefonu
     phone1        - wskanik do pola edycji drugiego telefonu
     phone2_menu   - wskanik do listy wyboru trzeciego telefonu
     phone2        - wskanik do pola edycji trzeciego telefonu
     phone3_menu   - wskanik do listy wyboru czwartego telefonu
     phone3        - wskanik do pola edycji czwartego telefonu
     email_menu    - wskanik do pola wyboru typu adresu e-mail
     email         - wskanik do pola edycji "Adres e-mail"
     www           - wskanik do pola edycji "Strona sieci WWW"

*/
void create_contact_window(gint id)
{
    GtkWidget *contact_window;
    GtkWidget *main_vbox;
    GtkWidget *menubar;
    GtkWidget *statusbar;
    GtkWidget *notebook;
    GtkWidget *fixed;
    GtkWidget *button;
    GtkWidget *entry;
    GtkWidget *label;
    GtkWidget *combo;
    GtkWidget *optionmenu;
    GtkWidget *menu;
    GtkWidget *checkbutton;
    GtkWidget *text;
    
    GtkItemFactory *item_factory;
    GtkAccelGroup *accel_group;
    
    contact_data *data = NULL;

    /* Sprawdzenie czy istnieje ju okienko przechowujce kontakt o zadanym identyfikatorze,
       i jeeli tak - uaktywnienie go. */
    if (id != 0)
    {
        contact_window = find_contact_window_id(id);
        if (contact_window != NULL)
        {
	    gtk_widget_hide(contact_window);
	    gtk_widget_show(contact_window);
            return;
        }
    }

    /* Utworzenie okienka kontakw. */
    contact_window = gtk_window_new(GTK_WINDOW_TOPLEVEL);

    /* Dodanie okienka do listy okienek kontaktw. */
    add_contact_window(contact_window);
    
    /* Utworzenie zmiennej zawierajcej dane okna kontaktw. */
    data = (contact_data *)g_malloc0(sizeof(contact_data));
    clear_contact_data(data);

    /* Wypenianie zmiennej zawierajacej dane okna kontaktw. */
    data->contact_window = contact_window;
    
    if (id != 0)
        db_contact_get(data, id);
    
    /* Tworzenie klucza "data" przechowujcego adres struktury danych kontaktu. */
    gtk_object_set_data(GTK_OBJECT(contact_window), "data", data);
    
    /* Przypisanie funkcji obsugi sygnaw do okienka kontaktw. */
    gtk_signal_connect(GTK_OBJECT(contact_window), "delete_event",
                       GTK_SIGNAL_FUNC(wnd_delete_event), data);
    
    /* Ustalenie tytuu i rozmiarw okna. */
    gtk_window_set_title(GTK_WINDOW(contact_window), _("Contact"));
    gtk_widget_set_usize(GTK_WIDGET(contact_window), 352, 216);
    
        /* Obszar zawartoci okna gwnego. */
	main_vbox = gtk_vbox_new(FALSE, 1);
	gtk_widget_show(main_vbox);
	
	gtk_container_border_width(GTK_CONTAINER(main_vbox), 1);
	gtk_container_add(GTK_CONTAINER(contact_window), main_vbox);
	
	    /* Uzupenienie struktury zawierajacej menu okienka kontaktw. */
	    contact_menu[6].callback_action = (guint)data;
	    contact_menu[7].callback_action = (guint)data;
	    contact_menu[9].callback_action = (guint)data;
	    contact_menu[11].callback_action = (guint)data;
	    contact_menu[15].callback_action = (guint)contact_window;
	
	    /* Tworzenie menu gwnego programu. */
	    accel_group = gtk_accel_group_new();
	    
	    item_factory = gtk_item_factory_new(GTK_TYPE_MENU_BAR, "<main>",
	                                        accel_group);
	    gtk_item_factory_create_items(item_factory, sizeof(contact_menu) /
	                                  sizeof(contact_menu[0]),
					  contact_menu, NULL);
	    gtk_window_add_accel_group(GTK_WINDOW(contact_window), accel_group);
	    menubar = gtk_item_factory_get_widget(item_factory, "<main>");
	    
	    gtk_widget_show(menubar);
	    
	gtk_box_pack_start(GTK_BOX(main_vbox), menubar, FALSE, FALSE, 0);
	
	    /* Tworzenie gwnej czci okna. */
	    notebook = gtk_notebook_new();
	    gtk_widget_show(notebook);
	    
	    gtk_object_set_data(GTK_OBJECT(contact_window), "notebook", notebook);
	    
	gtk_box_pack_start(GTK_BOX(main_vbox), notebook, TRUE, TRUE, 0);

	        /* Tworzenie zawartoci pierwszej zakadki. */
		fixed = gtk_fixed_new();
		gtk_widget_show(fixed);
		
		    /* Przycisk "Imi i nazwisko". */
		    button = gtk_button_new_with_label(_("Full Name..."));
		    gtk_widget_show(button);
		    
		    gtk_signal_connect(GTK_OBJECT(button), "clicked",
		                       GTK_SIGNAL_FUNC(name_button_clicked),
				       data);
		    
		    gtk_fixed_put(GTK_FIXED(fixed), button, 8, 8);
		    gtk_widget_set_uposition(button, 8, 8);
		    gtk_widget_set_usize(button, 120, 24);
		    
		    /* Pole edycji "Imi i nazwisko". */
		    entry = gtk_entry_new();
		    gtk_widget_show(entry);
		    
		    gtk_object_set_data(GTK_OBJECT(contact_window),
		                        "full_name", entry);
		    
		    gtk_fixed_put(GTK_FIXED(fixed), entry, 136, 8);
		    gtk_widget_set_uposition(entry, 136, 8);
		    gtk_widget_set_usize(entry, 200, 24);
		    gtk_entry_set_editable(GTK_ENTRY(entry), FALSE);
		    
		    create_full_name(data);
		    
		    /* Etykieta "Organizacja" */
		    label = gtk_label_new(_("Organization:"));
		    gtk_widget_show(label);
		    
		    gtk_fixed_put(GTK_FIXED(fixed), label, 8, 40);
		    gtk_widget_set_uposition(label, 8, 40);
		    gtk_widget_set_usize(label, 120, 24);
		    gtk_label_set_line_wrap(GTK_LABEL(label), TRUE);
		    
		    /* Pole edycji "Organizacja" */
		    entry = gtk_entry_new_with_max_length(MAX_FIRM_NAME);
		    gtk_widget_show(entry);
		    
		    gtk_object_set_data(GTK_OBJECT(contact_window),
		                        "firm_name", entry);
		    
		    gtk_fixed_put(GTK_FIXED(fixed), entry, 136, 40);
		    gtk_widget_set_uposition(entry, 136, 40);
		    gtk_widget_set_usize(entry, 200, 24);
		    gtk_entry_set_text(GTK_ENTRY(entry), data->firm_name);
		    
		    gtk_signal_connect(GTK_OBJECT(entry), "focus_out_event",
		                       GTK_SIGNAL_FUNC(firm_entry_focus_out), 
				       data);
		    
		    /* Etykieta "Stanowisko". */
		    label = gtk_label_new(_("Job title:"));
		    gtk_widget_show(label);
		    
		    gtk_fixed_put(GTK_FIXED(fixed), label, 8, 72);
		    gtk_widget_set_uposition(label, 8, 72);
		    gtk_widget_set_usize(label, 120, 24);
		    gtk_label_set_line_wrap(GTK_LABEL(label), TRUE);
		    
		    /* Pole edycji "Stanowisko". */
		    entry = gtk_entry_new_with_max_length(MAX_POSITION_NAME);
		    gtk_widget_show(entry);
		    
		    gtk_object_set_data(GTK_OBJECT(contact_window),
		                        "position", entry);
		    
		    gtk_fixed_put(GTK_FIXED(fixed), entry, 136, 72);
		    gtk_widget_set_uposition(entry, 136, 72);
		    gtk_widget_set_usize(entry, 200, 24);
		    gtk_entry_set_text(GTK_ENTRY(entry), data->position);
		    
		    /* Etykieta "Zapisz jako". */
		    label = gtk_label_new(_("Save as:"));
		    gtk_widget_show(label);
		    
		    gtk_fixed_put(GTK_FIXED(fixed), label, 8, 104);
		    gtk_widget_set_uposition(label, 8, 104);
		    gtk_widget_set_usize(label, 120, 24);
		    gtk_label_set_line_wrap(GTK_LABEL(label), TRUE);
		    
		    /* Pole wyboru "Zapisz jako". */
		    combo = gtk_combo_new();
		    gtk_widget_show(combo);
		    
		    gtk_object_set_data(GTK_OBJECT(contact_window),
		                        "save_as_combo", combo);
		    
		    gtk_fixed_put(GTK_FIXED(fixed), combo, 136, 104);
		    gtk_widget_set_uposition(combo, 136, 104);
		    gtk_widget_set_usize(combo, 200, 24);

		    create_save_as_list(data);
			
			/* Pole edycji pola wyboru "Zapisz jako". */
			entry = GTK_COMBO(combo)->entry;
			gtk_widget_show(entry);
			
			gtk_object_set_data(GTK_OBJECT(contact_window),
			                    "save_as", entry);
			
			gtk_entry_set_editable(GTK_ENTRY(entry), FALSE);
			gtk_entry_set_text(GTK_ENTRY(entry), data->save_as);
		
	        /* Tworzenie tytuu pierwszej zakadki. */
		label = gtk_label_new(_("General"));
		gtk_widget_show(label);
		
	    gtk_notebook_append_page(GTK_NOTEBOOK(notebook), fixed, label);
	    
	        /* Tworzenie zawartoci zakadki "Adresy". */
		fixed = gtk_fixed_new();
		gtk_widget_show(fixed);
		
		    /* Przycisk "Adres". */
		    button = gtk_button_new_with_label(_("Address..."));
		    gtk_widget_show(button);
		    
		    gtk_signal_connect(GTK_OBJECT(button), "clicked",
		                       GTK_SIGNAL_FUNC(address_button_clicked),
				       data);
		    
		    gtk_fixed_put(GTK_FIXED(fixed), button, 8, 8);
		    gtk_widget_set_uposition(button, 8, 8);
		    gtk_widget_set_usize(button, 120, 24);
		    
		    /* Lista wyboru "Adres". */
		    optionmenu = gtk_option_menu_new();
		    gtk_widget_show(optionmenu);
		    
		    gtk_object_set_data(GTK_OBJECT(contact_window), "address_menu", optionmenu);
		    
		    gtk_fixed_put(GTK_FIXED(fixed), optionmenu, 8, 40);
		    gtk_widget_set_uposition(optionmenu, 8, 40);
		    gtk_widget_set_usize(optionmenu, 120, 24);
		    
		        /* Tworzenie menu listy wyboru. */
		        menu = gtk_menu_new();
			
			gtk_menu_append(GTK_MENU(menu), 
			    make_address_menu_item(_("Home"), data, 
			                           ADDRESS_TYPE_HOME));
			gtk_menu_append(GTK_MENU(menu), 
			    make_address_menu_item(_("Work"), data, 
			                           ADDRESS_TYPE_WORK));
			gtk_menu_append(GTK_MENU(menu), 
			    make_address_menu_item(_("Other"), data, 
			                           ADDRESS_TYPE_OTHER));
			
		    gtk_option_menu_set_menu(GTK_OPTION_MENU(optionmenu), menu);
		    if (data->send_to != ADDRESS_TYPE_NONE)
		        gtk_option_menu_set_history(GTK_OPTION_MENU(optionmenu),
			                            data->send_to);
			
		    /* Pole wywietlania adresu. */
		    text = gtk_text_new(NULL, NULL);
		    gtk_widget_show(text);
		    
		    gtk_object_set_data(GTK_OBJECT(contact_window),
		                        "full_address", text);
		    
		    gtk_fixed_put(GTK_FIXED(fixed), text, 136, 8);
		    gtk_widget_set_uposition(text, 136, 9);
		    gtk_widget_set_usize(text, 200, 120);

                    create_full_address(data);
		    
		    /* Przycisk "Wysylkowy".  */
		    checkbutton = gtk_check_button_new_with_label(_("Send to"));
		    gtk_widget_show(checkbutton);

		    gtk_object_set_data(GTK_OBJECT(contact_window),
		                        "send_to", checkbutton);
		    
		    gtk_fixed_put(GTK_FIXED(fixed), checkbutton, 8, 72);
		    gtk_widget_set_uposition(checkbutton, 8, 72);
		    gtk_widget_set_usize(checkbutton, 120, 24);
		    
		    if (data->send_to != ADDRESS_TYPE_NONE)
		        gtk_toggle_button_set_active(
			                    GTK_TOGGLE_BUTTON(checkbutton),
		                            TRUE);

		    gtk_signal_connect(GTK_OBJECT(checkbutton), "toggled",
		                       GTK_SIGNAL_FUNC(send_to_toggled),
				       data);
		
		/* Tworzenie tytuu zakadki "Adresy". */
		label = gtk_label_new(_("Addresses"));
		gtk_widget_show(label);
		
	    gtk_notebook_append_page(GTK_NOTEBOOK(notebook), fixed, label);
	    
	        /* Tworzenie zawartoci zakadki "Telefony". */
		fixed = gtk_fixed_new();
		gtk_widget_show(fixed);
		
		make_phone_lines(fixed, data);
		
		/* Tworzenie tytuu zakadki "Telefony". */
		label = gtk_label_new(_("Phones"));
		gtk_widget_show(label);

	    gtk_notebook_append_page(GTK_NOTEBOOK(notebook), fixed, label);
		
		/* Tworzenie zawartoci zakadki "Internet". */
		fixed = gtk_fixed_new();
		gtk_widget_show(fixed);
		
		    /* Lista wyboru "Adres e-mail". */
		    optionmenu = gtk_option_menu_new();
		    gtk_widget_show(optionmenu);
		    
		    gtk_object_set_data(GTK_OBJECT(data->contact_window), "email_menu", optionmenu);
		    
		    gtk_fixed_put(GTK_FIXED(fixed), optionmenu, 8, 8);
		    gtk_widget_set_uposition(optionmenu, 8, 8);
		    gtk_widget_set_usize(optionmenu, 120, 24);
		    
		        /* Tworzenie menu listy wyboru. */
			menu = gtk_menu_new();
			
			gtk_menu_append(GTK_MENU(menu),
			    make_email_menu_item(_("Primary Email"), data, EMAIL_TYPE_1));
			gtk_menu_append(GTK_MENU(menu),
			    make_email_menu_item(_("Email 2"), data, EMAIL_TYPE_2));
			gtk_menu_append(GTK_MENU(menu),
			    make_email_menu_item(_("Email 3"), data, EMAIL_TYPE_3));
		    
		    gtk_option_menu_set_menu(GTK_OPTION_MENU(optionmenu), menu);
		    
		    /* Pole edycji "Adres e-mail". */
		    entry = gtk_entry_new_with_max_length(MAX_EMAIL);
		    gtk_widget_show(entry);
		    
		    gtk_object_set_data(GTK_OBJECT(contact_window),
		                        "email", entry);
		    
		    gtk_fixed_put(GTK_FIXED(fixed), entry, 136, 8);
		    gtk_widget_set_uposition(entry, 136, 8);
		    gtk_widget_set_usize(entry, 200, 24);
		    gtk_entry_set_text(GTK_ENTRY(entry), data->email[0].value);

		    gtk_signal_connect(GTK_OBJECT(entry), "focus_out_event",
		                       GTK_SIGNAL_FUNC(email_entry_focus_out),
				       data);
		    
		    /* Etykieta "Strona sieci WWW". */
		    label = gtk_label_new(_("Web page address:"));
		    gtk_widget_show(label);
		    
		    gtk_fixed_put(GTK_FIXED(fixed), label, 8, 40);
		    gtk_widget_set_uposition(label, 8, 40);
		    gtk_widget_set_usize(label, 120, 24);
		    
		    /* Pole edycji "Strona sieci WWW". */
		    entry = gtk_entry_new_with_max_length(MAX_WWW);
		    gtk_widget_show(entry);
		    
		    gtk_object_set_data(GTK_OBJECT(contact_window), "www", entry);
		    
		    gtk_fixed_put(GTK_FIXED(fixed), entry, 136, 40);
		    gtk_widget_set_uposition(entry, 136, 40);
		    gtk_widget_set_usize(entry, 200, 24);
		    gtk_entry_set_text(GTK_ENTRY(entry), data->www);
		    
		/* Tworzenie tytuu zakadki "Internet". */
		label = gtk_label_new(_("Internet"));
		gtk_widget_show(label);
		
	    gtk_notebook_append_page(GTK_NOTEBOOK(notebook), fixed, label);
	
	    /* Tworzenie linii statusu. */
	    statusbar = gtk_statusbar_new();
	    gtk_widget_show(statusbar);
	    
	gtk_box_pack_start(GTK_BOX(main_vbox), statusbar, FALSE, FALSE, 0);
    
    /* Wywietlenie okienka kontaktw. */
    gtk_widget_show(contact_window);
}

/* Funkcja kopiujaca dane kontaktu do struktury i wstawiajca je, o ile jest to konieczne, do
   bazy danych, a nastepnie zamykajca okienko kontaktw. Zwraca TRUE jeeli okienko zostao
   zamknite. */
gint close_contact_window(contact_data *data, gboolean remove_from_list)
{
    g_assert(data != NULL);

    /* Kopiowanie danych kontaktu do struktury. */
    copy_data(data);

    /* Sprawdzanie czy jest konieczne wywoywanie okienka potwierdzajacego. */
    if (data->global_modified)
    {
        /* Wywoanie okienka potwierdzajcego. */
        switch (confirm_dialog(data->contact_window, _("Do you want to save changes?"),
	        G_CONFIRM_YES | G_CONFIRM_NO | G_CONFIRM_CANCEL))
        {
            case G_CONFIRM_YES:
	        /* Wstawianie danych do bazy. */
                insert_data(data);

	    case G_CONFIRM_NO:
	        /* Usunicie okienka z listy okienek kontaktw. */
		if (remove_from_list)
		    remove_contact_window(data->contact_window);
	    
                /* Zamknicie okienka kontaktw. */
                gtk_widget_destroy(data->contact_window);

                /* Zwolnienie pamieci zajmowanej przez dane kontaktw. */
                g_free(data);
                return TRUE;
	    default:
	        g_assert_not_reached();
        }
    }
    else
    {
	/* Usunicie okienka z listy okienek kontaktw. */
	if (remove_from_list)
	    remove_contact_window(data->contact_window);

        /* Zamknicie okienka kontaktw. */
        gtk_widget_destroy(data->contact_window);

        /* Zwolnienie pamieci zajmowanej przez dane kontaktw. */
        g_free(data);
	return TRUE;
    }

    return FALSE;
}

/* Funkcja inicjalizujca menu okienka kontaktw. */
void init_gettext_contact_window()
{
    contact_menu[0].path = _("/_File");
    contact_menu[1].path = _("/File/_New");
    contact_menu[2].path = _("/File/New/_Contact");
    contact_menu[3].path = _("/File/New/sep1");
    contact_menu[4].path = _("/File/New/_Appointment");
    contact_menu[5].path = _("/File/sep2");
    contact_menu[6].path = _("/File/_Save");
    contact_menu[7].path = _("/File/Save _and new");
    contact_menu[8].path = _("/File/sep3");
    contact_menu[9].path = _("/File/_Delete");
    contact_menu[10].path = _("/File/sep4");
    contact_menu[11].path = _("/File/_Close");
    contact_menu[12].path = _("/_Contact");
    contact_menu[13].path = _("/Contact/_New contact");
    contact_menu[14].path = _("/_Help");
    contact_menu[15].path = _("/Help/_About...");
}

