/*
 * robwxvis.cpp
 * 
 * Copyright (c) 2000-2004 by Florian Fischer (florianfischer@gmx.de)
 * and Martin Trautmann (martintrautmann@gmx.de) 
 * 
 * This file may be distributed and/or modified under the terms of the 
 * GNU General Public License version 2 as published by the Free Software 
 * Foundation. 
 * 
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
 * WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 * 
 */

// Implements the main visualization frame

#include <wx/wx.h>
#include <wx/config.h>
#include <wx/splitter.h>
#include <wx/imaglist.h>
#include <wx/mimetype.h>

#include "robpackage.h"
#include "robwxvis.h"
#include "robwxutil.h"
#include "robwxdlgs.h"
#include "robwxhist.h"
#include <rtsystem.h>
#include <rtfile.h>
#ifdef __RT_OPENGL__
#  include "robwxgl.h"
#endif

#include <rtmath.h>
using namespace lrt;
using namespace rt;

// images
#include "bitmaps/abort.xpm"
#include "bitmaps/start.xpm"
#include "bitmaps/slow.xpm"
#include "bitmaps/pause.xpm"
#include "bitmaps/startlock.xpm"
#include "bitmaps/fieldno.xpm"
#include "bitmaps/field2d.xpm"
#include "bitmaps/field3d.xpm"
#ifndef __WXMSW__
#  include "bitmaps/robotour.xpm"
#endif

/////////////////////////// my event system
DEFINE_EVENT_TYPE(EVT_RT_SHOW_TYPE)
DEFINE_EVENT_TYPE(EVT_RT_INITSIM_TYPE)
DEFINE_EVENT_TYPE(EVT_RT_STEPSIM_TYPE)
DEFINE_EVENT_TYPE(EVT_RT_EXITSIM_TYPE)
DEFINE_EVENT_TYPE(EVT_RT_STEPTOUR_TYPE)
DEFINE_EVENT_TYPE(EVT_RT_PLAYER_TYPE)
DEFINE_EVENT_TYPE(EVT_RT_SETUPDLG_TYPE)
DEFINE_EVENT_TYPE(EVT_RT_MSGINFO_TYPE)
DEFINE_EVENT_TYPE(EVT_RT_MSGWARN_TYPE)
DEFINE_EVENT_TYPE(EVT_RT_MSGERROR_TYPE)
DEFINE_EVENT_TYPE(EVT_RT_MSGROBERR_TYPE)

DEFINE_EVENT_TYPE(EVT_DEBUG_UPDATE_TYPE)
DEFINE_EVENT_TYPE(EVT_DEBUG_CLOSE_TYPE)

/************* my event system **************/
IMPLEMENT_DYNAMIC_CLASS(RoboTourEvent, wxEvent)
RoboTourEvent::RoboTourEvent(int eventType)
	: simData(), retData(0)
{ 
	SetId(-1); 
	SetEventType(eventType);
}

RoboTourEvent::RoboTourEvent(const RoboTourEvent& evt)
    : wxEvent(evt), botData(evt.botData), simData(evt.simData), 
	  tourData(evt.tourData), tourInitData(evt.tourInitData), playerData(evt.playerData), 
	  retData(evt.retData), msgData(evt.msgData)
{
}

void RoboTourEvent::LoadBotData(Simulation* sim)
{
	botData.clear();
	int end = (sim->fieldHeight * sim->fieldWidth) - 1;
	botData.check(end);
	for(int i = 0; i <= end; i++)
		botData[i] = BotInfo(sim->field[i]);
}

void RoboTourEvent::LoadSimData(Simulation* sim, int simNum)
{
	simData.height = sim->fieldHeight;
	simData.width = sim->fieldWidth;
	simData.cycle = sim->curCycle;
	simData.timeout = sim->glob.timeout;
	simData.simNum = simNum; 
	simData.maxBanks = sim->glob.maxBanks;
	simData.maxBots = Math::min(sim->fieldHeight * sim->fieldWidth, sim->glob.maxMybots);
}

void RoboTourEvent::LoadTourData(Simulation* sim, const Array<TourResult>& tr, const TourInitInfo& ti)
{
	tourInitData = ti;

	tourData.clear();
	if(tr.length())
		tourData.check(tr.length() - 1); 

	for(int i = 0; i < tr.length(); i++)
	{
		TourInfo& tref = tourData[i]; 
		tref = tr[i];
		if(sim) 
		{
			for(int sp = 0; sp < sim->getNumPrograms(); sp++)
				if(sim->getProgram(sp)->globalNum == i)
				{
					tref.playerNum = sp;
					tref.numBots = sim->getProgram(sp)->vars[progMybots]; 
				}
		}
	}
}

void RoboTourEvent::LoadPlayerData(int playerNum, Simulation* sim, const Array<TourResult>& tra)
{
	if((playerNum < 0) || (playerNum >= tra.length()))
		return; // invalid player number

	const TourResult& tr = tra[playerNum];

	playerData += Pair<String, String>("File name", tr.fileName);

	for(StringMap<Program::HeaderField>::Iterator iter = tr.headers.begin(); iter.hasElement(); ++iter)
	{
		Program::HeaderField& hf = (*iter).getValue(); 
		if(hf.type == headerPublished)
			playerData += Pair<String, String>(hf.name, hf.value);
	}
}

wxEvent* RoboTourEvent::Clone() const
{
    return new RoboTourEvent(*this);
}

//////////////////////////////// DebugEvent
IMPLEMENT_DYNAMIC_CLASS(DebugEvent, wxEvent)
DebugEvent::DebugEvent(int eventType)
{ 
	SetId(-1); 
	SetEventType(eventType);
}

wxEvent* DebugEvent::Clone() const
{
    return new DebugEvent(*this);
}



//////////////////////////// RobVIsFrame
RobVisFrame* RobVisFrame::theVisFrame = 0;
GuiInfo RobVisFrame::guiInfo;

BEGIN_EVENT_TABLE(RobVisFrame, wxFrame)
//  EVT_PAINT(RobVisFrame::OnPaint)
//	EVT_SIZE(RobVisFrame::OnSize)
	EVT_RT_SHOW(RobVisFrame::OnShow)
	EVT_RT_INITSIM(RobVisFrame::OnInitSim)
	EVT_RT_STEPSIM(RobVisFrame::OnStepSim)
	EVT_RT_EXITSIM(RobVisFrame::OnExitSim)
	EVT_RT_STEPTOUR(RobVisFrame::OnStepTour)
	EVT_RT_PLAYER(RobVisFrame::OnPlayerInfo)
	EVT_RT_SETUPDLG(RobVisFrame::OnSetupDlg)
	EVT_RT_MSGINFO(RobVisFrame::OnMsgInfo)
	EVT_RT_MSGWARN(RobVisFrame::OnMsgWarn)
	EVT_RT_MSGERROR(RobVisFrame::OnMsgError)
	EVT_RT_MSGROBERR(RobVisFrame::OnMsgRobErr)
	EVT_DEBUG_UPDATE(RobVisFrame::OnDebugUpdate)
	EVT_DEBUG_CLOSE(RobVisFrame::OnDebugClose)
	EVT_MENU(RVF_NewTour, RobVisFrame::OnNewTourMenu)
	EVT_MENU(RVF_AbortTour, RobVisFrame::OnAbortTourMenu)
	EVT_MENU(wxID_EXIT, RobVisFrame::OnExitMenu)
	EVT_MENU(RVF_HideWin, RobVisFrame::OnHideMenu)
	EVT_MENU(RVF_Colours, RobVisFrame::OnProgramColourMenu)
	EVT_MENU(RVF_Filter, RobVisFrame::OnBotErrFilterMenu)
	EVT_MENU(RVF_UseHQ, RobVisFrame::OnUseHQMenu)
	EVT_MENU(RVF_ShowMsg, RobVisFrame::OnShowMessageList)
	EVT_MENU(RVF_Start, RobVisFrame::OnStart)
	EVT_MENU(RVF_StartLock, RobVisFrame::OnStartLock)
	EVT_MENU(RVF_Slow, RobVisFrame::OnSlowMode)
	EVT_MENU(RVF_Pause, RobVisFrame::OnPause)
	EVT_MENU(RVF_Abort, RobVisFrame::OnAbort)
	EVT_MENU(RVF_NoField, RobVisFrame::OnNoField)
	EVT_MENU(RVF_2DField, RobVisFrame::On2DField)
	EVT_MENU(RVF_3DField, RobVisFrame::On3DField)
	EVT_MENU(RVF_HelpIndex, RobVisFrame::OnHelp)
	EVT_MENU(RVF_About, RobVisFrame::OnAbout)
	EVT_CLOSE(RobVisFrame::OnClose)
	EVT_COMMAND_SCROLL(RVF_Speed, RobVisFrame::OnSpeed)
END_EVENT_TABLE()

RobVisFrame::RobVisFrame() 
	: wxFrame(0, -1, "RoboTour Visualization", wxDefaultPosition, wxSize(640, 480)),
	  simInfo(), errFilter(0), activeField(0), activeFieldNum(0),
	  framesStart(lrt::Time::getCurrentTime()), numFramesSince(0), haveSim(false)
{
	SetIcon(wxICON(robotour));

	wxMenu* mf = new wxMenu; 
	mf->Append(RVF_NewTour, "&New...\tCtrl-N", "Start a new tournament instead of the currently running one (if any).");
	mf->Append(RVF_AbortTour, "&Abort\tCtrl-A", "Abort the currently running tournament.");
	mf->AppendSeparator();
	mf->Append(RVF_HideWin, "&Hide\tAlt-H", "Hide the visualization. RoboTour will continue running.");
	mf->Append(wxID_EXIT, "&Exit\tAlt-X", "Exit RoboTour.");
	wxMenu* ms = new wxMenu; 
	ms->Append(RVF_Colours, "&Program colours...", "Select the robot colours for the different programs.");
	ms->Append(RVF_Filter, "&Robot errors...", "Select the robot errors which are displayed.");
#ifdef __RT_OPENGL__
	ms->Append(RVF_UseHQ, "&High 3D Quality", "Draw the 3D board in high quality (but slower).", wxITEM_CHECK);
#endif
	ms->Append(RVF_ShowMsg, "&Show messages", "Redisplay the message list if it has been closed.");
	wxMenu* mh = new wxMenu;
	mh->Append(RVF_HelpIndex, "&Index\tF1", "Lists online help documents for RoboTour."); 
	mh->Append(RVF_About, "&About RoboTour...", "Show information about RoboTour."); 
	wxMenuBar* mb = new wxMenuBar();
	mb->Append(mf, "&Tournament");
	mb->Append(ms, "&Options");
	mb->Append(mh, "&Help"); 
	SetMenuBar(mb); 


	wxBitmap startXPM(start_xpm);
	wxBitmap startLockXPM(startlock_xpm);
	wxBitmap slowXPM(slow_xpm);
	wxBitmap pauseXPM(pause_xpm);
	wxBitmap abortXPM(abort_xpm);
	wxBitmap fieldNoXPM(fieldno_xpm);
	wxBitmap field2DXPM(field2d_xpm);
	wxBitmap field3DXPM(field3d_xpm);

	wxToolBarBase* tb = CreateToolBar();
	tb->AddTool(RVF_Start, "Start", startXPM, "Start next simulation");
	tb->AddTool(RVF_StartLock, "StartLock", startLockXPM, "Start simulations automatically", wxITEM_CHECK);
	tb->AddTool(RVF_Slow, "Slow", slowXPM, "Slow simulation mode", wxITEM_CHECK);
	wxSlider* speedSlider = new wxSlider(tb, RVF_Speed, 10, 1, 100, wxDefaultPosition, wxSize(80, 16));
	speedSlider->SetToolTip("Simulation speed");
	tb->AddControl(speedSlider);
	tb->AddTool(RVF_Pause, "Pause", pauseXPM, "Pause simulation", wxITEM_CHECK);
	tb->AddTool(RVF_Abort, "Abort", abortXPM, "Abort current simulation");
	tb->AddSeparator();
	tb->AddRadioTool(RVF_NoField, "No field", fieldNoXPM, wxNullBitmap, "Don't display the field");
	tb->AddRadioTool(RVF_2DField, "2D field", field2DXPM, wxNullBitmap, "Display field from the top (2D)");
#ifdef __RT_OPENGL__
	tb->AddRadioTool(RVF_3DField, "3D field", field3DXPM, wxNullBitmap, "Display field three-dimensionally"); 
#endif
	tb->Realize();

	CreateStatusBar();

	vSplitter = new NoFlickerWindow<wxSplitterWindow>;
	vSplitter->Create(this, -1, wxDefaultPosition, wxDefaultSize, wxSP_3DSASH | wxSP_PERMIT_UNSPLIT);

	  hSplitter = new NoFlickerWindow<wxSplitterWindow>;
	  hSplitter->Create(vSplitter, -1);

	    infoPanel = new NoFlickerWindow<wxPanel>;
		infoPanel->Create(hSplitter, -1);
	    wxBoxSizer* infoSizer = new wxBoxSizer(wxVERTICAL);

		wxPanel* textPanel = new wxPanel(infoPanel, -1);
	      wxFlexGridSizer* infoGrid = new wxFlexGridSizer(2, 2, 10);
	        infoGrid->AddGrowableCol(0);
	        infoGrid->AddGrowableCol(1);
	        infoGrid->Add(new wxStaticText(textPanel, -1, "Simulation:"), 0, wxALIGN_RIGHT, 0); 
	        tSimCount = new wxStaticText(textPanel, -1, "0/0       ");
	        infoGrid->Add(tSimCount, 0, wxALIGN_LEFT, 0);
	        infoGrid->Add(new wxStaticText(textPanel, -1, "Cycle:"), 0, wxALIGN_RIGHT, 0); 
	        tCycleCount = new wxStaticText(textPanel, -1, "0/0       ");
	        infoGrid->Add(tCycleCount, 0, wxALIGN_LEFT, 0);
	      textPanel->SetSizerAndFit(infoGrid);
		infoSizer->Add(textPanel, 0, wxEXPAND, 0);

	    players = new PlayerListCtrl(infoPanel, this, -1);
	    infoSizer->Add(players, 1, wxEXPAND, 0);
		history = new HistoryCtrl(infoPanel, -1, wxDefaultPosition, wxSize(200, 150));
		infoSizer->Add(history, 0, wxEXPAND, 0);
	    infoPanel->SetSizer(infoSizer);

	    field += new NoFieldPanel(hSplitter, this, -1); 
		field[0]->Show(false);
		field += new FieldPanel(hSplitter, this, -1);
		field[1]->Show(false);
#ifdef __RT_OPENGL__
		field += new GLFieldPanel(hSplitter, this, -1);
		field[2]->Show(false);
#endif

	  SetActiveFieldDisplay(1, true /* toggle tools too */);
	  hSplitter->SetMinimumPaneSize(100);

	  messages = new ImprovedListCtrl(vSplitter, -1, wxDefaultPosition, wxDefaultSize, wxLC_REPORT | wxLC_SINGLE_SEL);
		messages->InsertColumn(0, "Message");
		messages->SetGrowableCol(0);
	    messages->InsertColumn(1, "Sim", wxLIST_FORMAT_RIGHT, 40);
		messages->InsertColumn(2, "Cycle", wxLIST_FORMAT_RIGHT, 60);
   
	vSplitter->SplitHorizontally(hSplitter, messages, -150);
	vSplitter->SetMinimumPaneSize(100);


	SetSizeHints(500, 300);

	LoadSettings();
	HandleProgramColourChange();
}

void RobVisFrame::HandleProgramColourChange()
{
	painter.LoadSettings();
	wxImageList* miList = painter.CreateImageList(16);
	numBotColours = miList->GetImageCount() - 1; 
	AddStdIcon(miList, wxART_INFORMATION, wxSize(16,16));
	AddStdIcon(miList, wxART_WARNING, wxSize(16,16));
	AddStdIcon(miList, wxART_ERROR, wxSize(16,16));
	messages->AssignImageList(miList, wxIMAGE_LIST_SMALL);
	players->HandleProgramColourChange();
	activeField->HandleProgramColourChange();

	for(lrt::Map<int, DebugDialog*>::Iterator it = debugDialogs.begin(); it.hasElement(); ++it)
		(*it).getValue()->HandleProgramColourChange();
}

void RobVisFrame::HandleFieldClick(int fieldX, int fieldY)
{
	if(!haveSim)
	{
		wxMessageBox("Cannot debug robots because the simulation has ended.", "Cannot debug", wxOK | wxICON_INFORMATION);
		return;
	}
	// REQUEST: send a debug request to the simulation thread here...
	DebugRequest req;
	req.debugX = fieldX;
	req.debugY = fieldY;
	RobVisFrame::guiInfo.reqDebug.append(req);
}

bool RobVisFrame::HandleDebugDialogClose(int debugID)
{
	// REQUEST: remove this debug id from updates
	DebugRequest req;
	req.debugCloseID = debugID;
	RobVisFrame::guiInfo.reqDebug.append(req);

	if(debugDialogs.isSet(debugID))
	{
		DebugDialog* ddlg = debugDialogs[debugID];
		ddlg->Destroy();
		debugDialogs.remove(debugID);
		ignoreDebugIDs += debugID;
		return true;
	}
	else
		return false;
}

bool RobVisFrame::AddStdIcon(wxImageList* iList, wxArtID id, const wxSize& sz)
{
	wxIcon icon = wxArtProvider::GetIcon(id, wxART_OTHER, sz);
	if(icon.Ok()) {
		iList->Add(icon); return true; 
	}
	else
		return false;
}

bool RobVisFrame::Filter(int errid)
{
	if(!errFilter)
		errFilter = new BotErrFilterDialog;
	return errFilter->Filter(errid);
}

void RobVisFrame::LoadSettings()
{
	wxConfigBase* cfg = wxConfig::Get(); long iTmp;
	Wx::RestoreRect(this, "/GUI/VisWindowRect");
	Wx::RestoreColumns(players, "/GUI/VisPlayerColumns");
	Wx::RestoreColumns(messages, "/GUI/VisMsgColumns");
	if(cfg->Read("/GUI/VisWhichField", &iTmp)) SetActiveFieldDisplay(iTmp, true /* toggleTools */);
	if(cfg->Read("/GUI/VisHSplitPoint", &iTmp)) hSplitter->SetSashPosition(iTmp);
	if(cfg->Read("/GUI/VisVSplitPoint", &iTmp)) vSplitter->SetSashPosition(iTmp);
#ifdef __RT_OPENGL__
	if(cfg->Read("/GUI/High3DQuality", &iTmp)) {
		GetMenuBar()->FindItem(RVF_UseHQ)->Check(iTmp);
		((GLFieldPanel*) field[2])->SetUseHQ(iTmp);
	}
#endif
}

void RobVisFrame::StoreSettings()
{
	painter.StoreSettings();
	wxConfigBase* cfg = wxConfig::Get();
	Wx::StoreRect(this, "/GUI/VisWindowRect");
	Wx::StoreColumns(players, "/GUI/VisPlayerColumns");
	Wx::StoreColumns(messages, "/GUI/VisMsgColumns");
	cfg->Write("/GUI/VisHSplitPoint", hSplitter->GetSashPosition());
	cfg->Write("/GUI/VisVSplitPoint", vSplitter->GetSashPosition());
	cfg->Write("/GUI/VisWhichField", activeFieldNum);
#ifdef __RT_OPENGL__
	cfg->Write("/GUI/High3DQuality", ((GLFieldPanel*) field[2])->GetUseHQ());
#endif
	cfg->Flush();
}

void RobVisFrame::SetActiveFieldDisplay(int num, bool toggleTools)
{
	//// ATTENTION! here are field tool ids! they have to be changed if more field displas are invented!
	static int toolIDs[] = { RVF_NoField, RVF_2DField, RVF_3DField }; 

	if(num < 0 || num >= field.length())
	{
		wxLogDebug("error: SetActiveFieldDisplay(invalid field num %d)", num);
		return;
	}

	if(toggleTools && GetToolBar())
	{
		wxToolBarBase* toolBar = GetToolBar();

		for(int t = 0; t < field.length(); t++)
			toolBar->ToggleTool(toolIDs[t], false);
		toolBar->ToggleTool(toolIDs[num], true);
	}

	wxWindow* oldFD = (activeField ? activeField->GetAsWindow() : 0);

	activeField = field[num];
	activeFieldNum = num;
	if(oldFD)
		hSplitter->ReplaceWindow(oldFD, activeField->GetAsWindow());
	else
		hSplitter->SplitVertically(infoPanel, activeField->GetAsWindow(), 300);

	for(int f = 0; f < field.length(); f++)
		if(f != num)
			field[f]->Show(false);
	activeField->Show(true);
}

bool RobVisFrame::Destroy()
{
	StoreSettings();

	if(errFilter)
	{
		errFilter->Destroy();
		errFilter = 0;
	}

	guiInfo.isDestroyed = true; 
	return wxFrame::Destroy();
}

RobVisFrame::~RobVisFrame()
{
	if(this == theVisFrame)
		theVisFrame = 0; 
}

void RobVisFrame::OnProgramColourMenu(wxCommandEvent&)
{
	ProgramColourDialog pcd(painter);
	if(pcd.ShowModal() == wxID_OK)
	{
		pcd.GetPainter().StoreSettings();
		painter.LoadSettings();
		HandleProgramColourChange();
	}
//	else wxLogWarning("not ok");
}

void RobVisFrame::OnBotErrFilterMenu(wxCommandEvent&)
{
	if(!errFilter)
		errFilter = new BotErrFilterDialog();
	errFilter->Show();
}

void RobVisFrame::OnUseHQMenu(wxCommandEvent& evt)
{
#ifdef __RT_OPENGL__
	((GLFieldPanel*) field[2])->SetUseHQ(evt.IsChecked());
#endif
}

void RobVisFrame::OnNewTourMenu(wxCommandEvent&)
{
	guiInfo.abortPressed = true; // abort current sim
	guiInfo.abortTourPressed = true; // and current tournament
	guiInfo.restart = gtourNew; // and start a new one immediately
}

void RobVisFrame::OnAbortTourMenu(wxCommandEvent&)
{
	guiInfo.abortPressed = true; // abort current sim
	guiInfo.abortTourPressed = true; // and current tournament
	guiInfo.restart = gtourNotYet; // and don't start a new one right now
}

void RobVisFrame::OnHideMenu(wxCommandEvent&)
{
	Show(false);
	Destroy();
}

void RobVisFrame::OnExitMenu(wxCommandEvent&)
{
	guiInfo.abortPressed = true; // abort current sim
	guiInfo.abortTourPressed = true; // and current tournament
	guiInfo.restart = gtourExit; // and exit simulation thread

	// wait until sim thread finished
	while(RobVisFrame::guiInfo.haveSimThread)
	{
		wxThread::Sleep(100);
		wxTheApp->Yield();
	}
	Destroy(); // exit myself
}

void RobVisFrame::OnClose(wxCloseEvent&)
{
	wxCommandEvent dummy;
	OnExitMenu(dummy);
}

void RobVisFrame::OnShowMessageList(wxCommandEvent&)
{
	if(!vSplitter->IsSplit()) {
		vSplitter->SplitHorizontally(hSplitter, messages, -150);
		messages->Show();
	}
}


void RobVisFrame::OnStart(wxCommandEvent&)
{
	guiInfo.startPressed = true; 
}

void RobVisFrame::OnStartLock(wxCommandEvent&)
{
	guiInfo.startLocked = GetToolBar()->GetToolState(RVF_StartLock);
}

void RobVisFrame::OnSlowMode(wxCommandEvent& evt)
{
	guiInfo.slowMode = GetToolBar()->GetToolState(RVF_Slow);
	// update speed setting
	OnSpeed(evt);
}

void RobVisFrame::OnPause(wxCommandEvent&)
{
	guiInfo.pausePressed = GetToolBar()->GetToolState(RVF_Pause); 
}

void RobVisFrame::OnAbort(wxCommandEvent&)
{
	GetToolBar()->ToggleTool(RVF_Pause, false);
	guiInfo.pausePressed = false; 
	guiInfo.abortPressed = true; 
}

void RobVisFrame::OnNoField(wxCommandEvent&)
{
	SetActiveFieldDisplay(0);
}

void RobVisFrame::On2DField(wxCommandEvent&)
{
	SetActiveFieldDisplay(1);
}

void RobVisFrame::On3DField(wxCommandEvent&)
{
#ifdef __RT_OPENGL__
	SetActiveFieldDisplay(2);
#endif
}

void RobVisFrame::OnAbout(wxCommandEvent&)
{
	wxMessageBox(wxString("RoboTour, the fast and portable RoboCom interpreter.\n") + rt::aboutText, 
		"About RoboTour", wxICON_INFORMATION | wxOK); 
}

void RobVisFrame::OnHelp(wxCommandEvent&)
{
	lrt::String helpFile = findHelpFile("index.html"); 
	if(!helpFile.length()) // NOT FOUND
		wxMessageBox("The RoboTour help was not found!", "RoboTour Help", wxICON_ERROR); 
	else
		Wx::ShowInBrowser(Wx::Conv(helpFile)); 
}

void RobVisFrame::OnSpeed(wxCommandEvent&)
{
	wxSlider* slider = (wxSlider*)FindWindowById(RVF_Speed);
	int sliderSpeed = slider->GetValue(); // range 0..100
	if(guiInfo.slowMode) {
		guiInfo.speed = (100-sliderSpeed) * (100-sliderSpeed) / 10 + 10;
		slider->SetToolTip(wxString::Format("Every cycle takes %d ms", guiInfo.speed));
	}
	else {
		guiInfo.speed = (sliderSpeed * sliderSpeed) / 30 + 1;
		slider->SetToolTip(wxString::Format("Display every %d'th cycle", guiInfo.speed));
	}
}

void RobVisFrame::OnShow(RoboTourEvent&)
{
	Show();

	guiInfo.numEvents++;
}

void RobVisFrame::OnInitSim(RoboTourEvent& event)
{
/*DBG*/
/*	wxSize sz = field->GetSize();
	System::println(String("fieldsz: ") + sz.x + "," + sz.y);
*/
	haveSim = true;

	simInfo = event.simData;
	for(int f = 0; f < field.length(); f++)
		field[f]->EnterSim(simInfo);
	if(event.tourData.length()) players->Update(event.tourData);
	curCycle = 0; 
	tSimCount->SetLabel(wxString::Format("%d/%d", simInfo.simNum, event.tourInitData.numSimulations));
	tCycleCount->SetLabel(wxString::Format("%d/%d", 0, event.simData.timeout));

	history->SetMaximum(simInfo.maxBots);
	history->ClearHistory();

	ignoreDebugIDs.clear();

	guiInfo.numEvents++;
}

void RobVisFrame::OnStepSim(RoboTourEvent& event)
{
	activeField->StepSim(event.botData);
	if(event.tourData.length()) {
		players->Update(event.tourData);
		history->UpdateFrom(event);
	}
	curCycle = event.simData.cycle;
	tCycleCount->SetLabel(wxString::Format("%d/%d", curCycle, event.simData.timeout));

	numFramesSince++;
	if(numFramesSince >= 20)
	{
		lrt::Time cur = lrt::Time::getCurrentTime();
		lrt::Time diff = cur - framesStart;
		float msPerFrame = ((float) diff.toInt()) / numFramesSince;
		float framesPerSecond = 1000.f / msPerFrame;
		SetStatusText(wxString::Format("%.2f Updates per second", framesPerSecond));
		numFramesSince = 0;
		framesStart = cur;
	}

	guiInfo.numEvents++;
}

void RobVisFrame::OnStepTour(RoboTourEvent& event)
{
	players->Update(event.tourData);

	guiInfo.numEvents++;
}

void RobVisFrame::OnExitSim(RoboTourEvent& event)
{
	for(int f = 0; f < field.length(); f++)
		field[f]->ExitSim();

	if(event.simData.winner != -1)
	{
		if(event.simData.winner == -2) // tie
		{
			AddMessage("The bots tied.", 0);
		}
		else 
		{
			AddMessage(wxString::Format("The winner is '%s'.", event.simData.winnerName.cStr()), event.simData.winner+1);
		}
	}

	haveSim = false; 

	// reset some guiInfo stuff for the next simulation
	guiInfo.startPressed = false; 

	guiInfo.numEvents++;
}

void RobVisFrame::OnPlayerInfo(RoboTourEvent& event)
{
	PlayerInfoDialog* pid = new PlayerInfoDialog(this, event.playerData);
	pid->Show();

	guiInfo.numEvents++;
}

void RobVisFrame::OnSetupDlg(RoboTourEvent& event)
{
	if(!event.retData) {
		wxLogError("RobVisFrame::OnSetupDlg: event.retData is 0, cannot popup dialog");
		return;
	}
	TourSetupDialog tsd((TourSetupInfo*) event.retData);
	tsd.ShowModal();
	guiInfo.startPressed = false; // but don't start the first sim immediately

	guiInfo.numEvents++;
}

void RobVisFrame::AddMessage(const wxString& msg, int icon)
{
	long msgPos = messages->GetItemCount();
	if(msgPos > 500)
	{
		messages->DeleteItem(0);
		msgPos--;
	}
	messages->InsertItem(msgPos, msg, icon);
	messages->SetItem(msgPos, 1, wxString::Format("%d", simInfo.simNum));
	messages->SetItem(msgPos, 2, wxString::Format("%d", curCycle));
	if(!vSplitter->IsSplit()) {
		vSplitter->SplitHorizontally(hSplitter, messages, -150);
		messages->Show();
	}
	messages->EnsureVisible(msgPos);
	messages->SetItemState(msgPos, wxLIST_STATE_SELECTED, wxLIST_STATE_SELECTED);
}

void RobVisFrame::OnMsgInfo(RoboTourEvent& event)
{
	AddMessage(Wx::Conv(event.msgData.msg), numBotColours + 1 /*INFO*/);
	guiInfo.numEvents++;
}

void RobVisFrame::OnMsgWarn(RoboTourEvent& event)
{
	AddMessage(Wx::Conv(event.msgData.msg), numBotColours + 2 /*WARN*/);
	guiInfo.numEvents++;
}

void RobVisFrame::OnMsgError(RoboTourEvent& event)
{
	AddMessage(Wx::Conv(event.msgData.msg), numBotColours + 3 /*ERRO*/);
	guiInfo.numEvents++;
}

void RobVisFrame::OnMsgRobErr(RoboTourEvent& event)
{
	curCycle = event.simData.cycle;
	if(Filter(event.msgData.id))
		AddMessage(Wx::Conv(event.msgData.msg), (event.msgData.program % numBotColours) + 1 /*RERR*/);
	guiInfo.numEvents++;
}

void RobVisFrame::OnDebugUpdate(DebugEvent& event)
{
	if(ignoreDebugIDs.indexOf(event.data.debugID) >= 0) // is ignored
	{
		guiInfo.numEvents++;
		return; 
	}

	if(!debugDialogs.isSet(event.data.debugID))
		debugDialogs[event.data.debugID] = new DebugDialog(event.data);
	else
		debugDialogs[event.data.debugID]->HandleUpdate(event.data);

	guiInfo.numEvents++;
}

void RobVisFrame::OnDebugClose(DebugEvent& event)
{
	if(debugDialogs.isSet(event.data.debugID))
	{
		DebugDialog* ddlg = debugDialogs[event.data.debugID];
		ddlg->Destroy();
		debugDialogs.remove(event.data.debugID);
		ignoreDebugIDs += event.data.debugID;
	}

	guiInfo.numEvents++;
}



///////////////////////////////// NoFieldPanel
BEGIN_EVENT_TABLE(NoFieldPanel, wxControl)
  EVT_PAINT(NoFieldPanel::OnPaint)
END_EVENT_TABLE()

NoFieldPanel::NoFieldPanel(wxWindow* parent, RobVisFrame* vframe, wxWindowID id)
	: wxControl(parent, id, wxDefaultPosition, wxSize(640, 480)), FieldDisplay(vframe)
{
}

NoFieldPanel::~NoFieldPanel()
{
}

void NoFieldPanel::EnterSim(const SimInfo& info)
{
}

void NoFieldPanel::ExitSim()
{
}

void NoFieldPanel::StepSim(const Vector<BotInfo>& bots)
{
}

void NoFieldPanel::Update(bool force)
{
}

void NoFieldPanel::HandleProgramColourChange()
{
}

void NoFieldPanel::OnPaint(wxPaintEvent& event)
{
	wxPaintDC dc(this);
	wxSize center = GetSize();
	int tx, ty; 
	wxString text = "Visualization disabled";
	GetTextExtent(text, &tx, &ty);
	center.x -= tx; center.y -= ty; 
	center.x /= 2; center.y /= 2;
	dc.DrawText(text, center.x, center.y);
}

bool NoFieldPanel::Show(bool doShow)
{
	return wxControl::Show(doShow);
}

wxWindow* NoFieldPanel::GetAsWindow()
{
	return (wxWindow*) this; 
}

//////////////////////////////// FieldPanel

BEGIN_EVENT_TABLE(FieldPanel, wxControl)
  EVT_PAINT(FieldPanel::OnPaint)
  EVT_LEFT_DOWN(FieldPanel::OnClick)
END_EVENT_TABLE()

FieldPanel::FieldPanel(wxWindow* parent, RobVisFrame* vframe, wxWindowID id)
	: wxControl(parent, id, wxDefaultPosition, wxSize(640, 480)), FieldDisplay(vframe), inSim(false)
{
	// init pixel sizes (all dummy values!) 
	fieldSize = 16; xOffset = 0; yOffset = 0; 
}

FieldPanel::~FieldPanel()
{
}

void FieldPanel::EnterSim(const SimInfo& info)
{
	simInfo = info; 
	status.clear();
	prevStatus.clear();
}

void FieldPanel::ExitSim()
{
	// ?!
}

void FieldPanel::StepSim(const Vector<BotInfo>& bots)
{
	status = bots;
	Update();
}

void FieldPanel::Update(bool force)
{
	wxClientDC dc(this);
	Paint(dc, force);
}

void FieldPanel::HandleProgramColourChange()
{
	Update(true);
}

void FieldPanel::OnClick(wxMouseEvent& event)
{
	int x = event.GetX(), y = event.GetY();
	x -= xOffset; y -= yOffset; 
	x /= fieldSize; y /= fieldSize; 
	if(x < 0 || x >= simInfo.width || y < 0 || y >= simInfo.width)
		lrt::System::println("click outside field!"); 
	else
		lrt::System::println(Wx::Conv(wxString::Format("click on field: %d,%d", x, y)));
	vframe->HandleFieldClick(x, y);
}

void FieldPanel::OnPaint(wxPaintEvent& event)
{
	wxPaintDC dc(this);
	Paint(dc, true);
}

wxSize FieldPanel::DoGetBestSize() const
{
	return wxSize(640, 500);
}

void FieldPanel::Paint(wxDC& device, bool force)
{
//	if(force)
//		System::println("paint field panel with force!!");

	// compute field size
	wxSize size = GetSize();
	// size of one field onscreen (px)
	fieldSize = Math::min(size.x / simInfo.width, size.y / simInfo.height);
	xOffset = (size.x - (fieldSize * simInfo.width)) / 2;
	yOffset = (size.y - (fieldSize * simInfo.height)) / 2;

//	device.SetFont(*wxSWISS_FONT);
//	wxCoord fontHeight = device.GetCharHeight();

	device.BeginDrawing();
		//erase_bkgnd
//		device.GetBrush().SetColour(GetBackgroundColour());
//		device.DrawRectangle(0, 0, size.GetWidth(), size.GetHeight());
		
		for(int x = 0; x < simInfo.width; x++)
		{
			for(int y = 0; y < simInfo.height; y++)
			{
				int px = xOffset + x * fieldSize;
				int py = yOffset + y * fieldSize;

				BotInfo& curInfo = status[x + y * simInfo.width];
				// optimize: skip drawing if equal to previous status
				if(!force && (curInfo == prevStatus[x + y * simInfo.width]))
					continue;
				prevStatus[x + y * simInfo.width] = curInfo; // is now current status

				vframe->GetBotPainter().DrawBot(curInfo, device, wxPoint(px, py), fieldSize, true /*withBG*/);
			}
		}
	device.EndDrawing();
}

bool FieldPanel::Show(bool doShow)
{
	return wxControl::Show(doShow);
}

wxWindow* FieldPanel::GetAsWindow()
{
	return (wxWindow*) this; 
}


// PlayerListCtrl  /////////////////////////////////

BEGIN_EVENT_TABLE(PlayerListCtrl, ImprovedListCtrl)
  EVT_LIST_ITEM_ACTIVATED(-1, PlayerListCtrl::OnClick)
END_EVENT_TABLE()

PlayerListCtrl::PlayerListCtrl(wxWindow* parent, RobVisFrame* vframe, wxWindowID id)
	: ImprovedListCtrl(parent, id, wxDefaultPosition, wxSize(200, -1), wxLC_REPORT),
	  vframe(vframe), numFieldSize(-1), haveImgList(false), imgListSize(1)
{
	int y;
	GetTextExtent("9999", &numFieldSize, &y); // text which must fit inside the columns

	InsertColumn(0, "Program");
	SetGrowableCol(0);
	InsertColumn(1, "W", wxLIST_FORMAT_RIGHT, numFieldSize);
	InsertColumn(2, "L", wxLIST_FORMAT_RIGHT, numFieldSize);
	InsertColumn(3, "T", wxLIST_FORMAT_RIGHT, numFieldSize);
	InsertColumn(4, "P", wxLIST_FORMAT_RIGHT, numFieldSize);
	InsertColumn(5, "Bots", wxLIST_FORMAT_RIGHT, numFieldSize);
}

PlayerListCtrl::~PlayerListCtrl()
{
}

void PlayerListCtrl::Update(const Vector<TourInfo>& tourData)
{
	// clear (too many items)
	if(GetItemCount() > tourData.length())
		for(int di = GetItemCount() -1; di >= tourData.length(); di--)
			DeleteItem(di);
	// insert (too few items)
	else if(GetItemCount() < tourData.length())
		for(int ii = GetItemCount(); ii < tourData.length(); ii++)
			InsertItem(ii, "");


	for(int i = 0; i < tourData.length(); i++)
	{
		const TourInfo& dat = tourData[i];
		SetImageIfNeeded(i, (dat.playerNum >= 0) ? (dat.playerNum % imgListSize) + 1 : 0);
		SetItemIfNeeded(i, 0, Wx::Conv(dat.headers["name"].value));
		SetItemIfNeeded(i, 1, dat.wins);
		SetItemIfNeeded(i, 2, dat.looses);
		SetItemIfNeeded(i, 3, dat.ties);
		SetItemIfNeeded(i, 4, dat.points);
		if(dat.numBots >= 0)
			SetItemIfNeeded(i, 5, dat.numBots);
		else
			SetItemIfNeeded(i, 5, "");
	}
}

void PlayerListCtrl::HandleProgramColourChange()
{
	AssignImageList(vframe->GetBotPainter().CreateImageList(16), wxIMAGE_LIST_SMALL);
	imgListSize = vframe->GetBotPainter().GetProgramColourCount();
	haveImgList = true; 
}

void PlayerListCtrl::SetImageIfNeeded(long row, int image)
{
	if(row >= images.length() || images[row] != image)
	{
		if(!haveImgList) {
			AssignImageList(vframe->GetBotPainter().CreateImageList(16), wxIMAGE_LIST_SMALL);
			haveImgList = true; 
		}

		images[row] = image; 
		SetItemImage(row, image, image);
	}
}

void PlayerListCtrl::SetItemIfNeeded(long row, int col, const wxString& str)
{
	wxString& lastText = strings[row][col];
	if(lastText != str)
	{
		lastText = str;
		SetItem(row, col, str);
	}
}

void PlayerListCtrl::OnClick(wxListEvent& le)
{
	RobVisFrame::guiInfo.reqPlayerInfo = le.GetIndex();
}


///////////////// PlayerInfoDialog /////////////////////

PlayerInfoDialog::PlayerInfoDialog(wxFrame* parent, const Vector< Pair<String, String> >& info)
	:wxDialog(parent, -1, wxString("Robot Info"), wxDefaultPosition, wxDefaultSize, wxDEFAULT_DIALOG_STYLE | wxRESIZE_BORDER)
{
	wxSizer* sz = new wxBoxSizer(wxVERTICAL);
	wxListCtrl* list = new wxListCtrl(this, -1, wxDefaultPosition, wxDefaultSize, wxLC_REPORT);
	list->InsertColumn(0, "Field", wxLIST_FORMAT_LEFT, -2);
	list->InsertColumn(1, "Value", wxLIST_FORMAT_LEFT, 250);

	for(int i = 0; i < info.length(); i++)
	{
		if(!info[i].getKey().compareIgnoreCase("name")) // we're just loading program name
			SetTitle("Robot Info - " + Wx::Conv(info[i].getValue())); // display it inside title

		list->InsertItem(i, Wx::Conv(info[i].getKey()));
		list->SetItem(i, 1, Wx::Conv(info[i].getValue()));
	}

	sz->Add(list, 1, wxEXPAND | wxALL, 2);

	wxButton* ok = new wxButton(this, wxID_OK, "OK");
	sz->Add(ok, 0, wxALIGN_CENTER_HORIZONTAL | wxALL, 2);

	SetSizer(sz);
	SetSizeHints(200, 300);
	SetSize(400, 400);
	Centre();
}


///////////////////////////// BotPainter /*////////////////////////

BotPainter::BotPainter()
{
	LoadSettings();
	backBrush = new wxBrush(wxSystemSettings::GetColour(wxSYS_COLOUR_BTNFACE), wxSOLID);
	backPen = new wxPen(wxSystemSettings::GetColour(wxSYS_COLOUR_3DSHADOW), 1, wxSOLID);
	botBrush = new wxBrush(programColours[0], wxSOLID);
	botPen = new wxPen(*wxBLACK, 1, wxSOLID);
	activePen = new wxPen(wxColour(180, 140, 0), 2, wxSOLID);
}

BotPainter::BotPainter(const BotPainter& bp) // need just copy the colours
	: programColours(bp.programColours)
{
	backBrush = new wxBrush(wxSystemSettings::GetColour(wxSYS_COLOUR_BTNFACE), wxSOLID);
	backPen = new wxPen(wxSystemSettings::GetColour(wxSYS_COLOUR_3DSHADOW), 1, wxSOLID);
	botBrush = new wxBrush(programColours[0], wxSOLID);
	botPen = new wxPen(*wxBLACK, 1, wxSOLID);
	activePen = new wxPen(wxColour(180, 140, 0), 2, wxSOLID);
}

BotPainter::~BotPainter()
{
	delete backBrush; 
	backBrush = 0;
	delete backPen;
	backPen = 0; 
	delete botBrush;
	botBrush = 0;
	delete botPen;
	botPen = 0;
	delete activePen;
	activePen = 0; 
}

void BotPainter::LoadSettings()
{
	// initialize defaults
	programColours.clear();
	programColours += wxColour(240, 0, 0);
	programColours += wxColour(0, 200, 0);
	programColours += wxColour(40, 40, 255);
	programColours += wxColour(200, 200, 0);
	programColours += wxColour(200, 0, 255);
	programColours += wxColour(0, 200, 255);
	programColours += wxColour(150, 150, 150);
	programColours += wxColour(160, 70, 10);

	programColours += wxColour(49, 89, 132);
	programColours += wxColour(49, 89, 132);
	programColours += wxColour(49, 89, 132);
	programColours += wxColour(49, 89, 132);
	programColours += wxColour(49, 89, 132);
	programColours += wxColour(49, 89, 132);
	programColours += wxColour(49, 89, 132);
	programColours += wxColour(49, 89, 132);

	// load stored settings
	lrt::Array<int> colNums = Wx::ReadArray(-1, "/GUI/ProgramColours");
	for(int i = 0; i < colNums.length(); i++)
	{
		int num = colNums[i];
		if(num < 0) 
			continue; 
		int r = (num & 0xFF0000) >> 16;
		int g = (num & 0x00FF00) >> 8;
		int b = (num & 0x0000FF);
		programColours[i] = wxColour(r,g,b);
	}
}

void BotPainter::StoreSettings()
{
	lrt::Array<int> colNums(programColours.length());
	for(int i = 0; i < programColours.length(); i++)
	{
		wxColour& col = GetProgramColour(i);
		colNums[i] = (col.Red() << 16) | (col.Green() << 8) | col.Blue();
	}
	Wx::StoreArray(colNums, "/GUI/ProgramColours");
}

void BotPainter::SetProgramColour(int progNum, const wxColour& col)
{
	if(progNum < 0 || progNum >= programColours.length())
		return;
	programColours[progNum] = col;
}

wxColour& BotPainter::GetProgramColour(int progNum)
{
	return programColours[progNum % programColours.length()];
}

int BotPainter::GetProgramColourCount()
{
	return programColours.length();
}

void BotPainter::DrawBot(BotInfo& bot, wxDC& dc, const wxPoint& pos, int size, bool bg)
{
	int px = pos.x, py = pos.y;

	// actual drawing comes...
	if(bg)
	{
		// draw background
		dc.SetBrush(*backBrush);
		dc.SetPen(*backPen);
		dc.DrawRectangle(px, py, size, size);
	}

	if(bot.program >= 0)
	{
		// draw bot
		wxColour& botCol = GetProgramColour(bot.program);
		botBrush->SetColour(botCol);
		dc.SetBrush(*botBrush);
		botPen->SetStyle(GetFrameStyle(bot.type));
		dc.SetPen(*botPen);
		dc.DrawEllipse(px, py, size, size);
		// draw heads
		botPen->SetStyle(wxSOLID);
		dc.SetPen(*botPen);
		for(int head = 0; head < 4; head++)
		{
			if(bot.head[head] < 0)
				continue;
			botBrush->SetColour(GetProgramColour(bot.head[head]));
			dc.SetBrush(*botBrush);
			wxPoint points[3];
			CalcPoints(head, size, points);
			dc.DrawPolygon(3, points, px, py);
		}

		if(bot.active <= 0)
		{
			// draw inactivity mark
			dc.SetPen(*activePen);
			dc.DrawLine(px+1, py+1, px+size-1, py+size-1);
			dc.DrawLine(px+size-1, py+1, px+1, py+size-1);
		}

	}

}

wxImageList* BotPainter::CreateImageList(int imgSize)
{
	static wxColour transparency(0,1,0);

	wxImageList* ilist = new wxImageList(imgSize, imgSize);

	for(int i = 0; i <= GetProgramColourCount(); i++)
	{
		wxBitmap bmp(imgSize, imgSize);
		wxMemoryDC dc;
		dc.SelectObject(bmp);
		dc.SetBrush(wxBrush(transparency, wxSOLID));
		dc.Clear();

		if(i != 0)
		{
			BotInfo info; 
			info.program = i-1;
			info.active = true;
			info.head[i%4] = i-1;
			info.type = 2;
			DrawBot(info, dc, wxPoint(0,0), imgSize, false);
		}

		dc.SelectObject(wxNullBitmap);

		ilist->Add(bmp, transparency);
	}
	return ilist; 
}


int BotPainter::GetFrameStyle(int botType)
{
	switch(botType)
	{
	  case 0:
		return wxTRANSPARENT;
	  case 1:
		return wxDOT;
      case 2:
		return wxSOLID;
	  default: // unreachable ?!
		wxLogError("invalid instruction set: %d", botType);
		return wxDOT_DASH;
	}
}

void BotPainter::CalcPoints(int dir, int fieldSize, wxPoint* points)
{
	int start = 0, up = fieldSize * 2 / 5, mid = fieldSize / 2, down = fieldSize * 3 / 5, end = fieldSize;

	switch(dir)
	{
	  case dirRight:
		points[0] = wxPoint(mid, down);
		points[1] = wxPoint(end, mid);
		points[2] = wxPoint(mid, up);
		break;
	  case dirUp:
		points[0] = wxPoint(down, mid);
		points[1] = wxPoint(mid, start);
		points[2] = wxPoint(up, mid);
		break;
	  case dirLeft:
		points[0] = wxPoint(mid, down);
		points[1] = wxPoint(mid, up);
		points[2] = wxPoint(start, mid);
		break;
	  case dirDown:
		points[0] = wxPoint(up, mid);
		points[1] = wxPoint(mid, end);
		points[2] = wxPoint(down, mid);
		break;
	  default: // should be unreachable...
		wxLogError("invalid robot head direction: %d", dir);
		break;
	}
}
