
// Copyright (c) 1996-2003 The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
// SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
// IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE,
// OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE FOR ANY DAMAGES SUFFERED BY
// LICENSEE AS A RESULT OF USING, RESULT OF USING, MODIFYING OR
// DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the U.S.,
// and the terms of this license.

// You may modify, distribute, and use the software contained in this
// package under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE"
// version 2, June 1991. A copy of this license agreement can be found in
// the file "LGPL", distributed with this archive.

// Authors: Philip A. Wilsey	philip.wilsey@ieee.org
//          Dale E. Martin	dmartin@cliftonlabs.com
//          Malolan Chetlur     
//          Timothy J. McBrayer 
//          Narayanan Thondugulam
//	    Magnus Danielson	cfmd@swipnet.se

#include "IIRScram.hh"
#include "IIRScram_ArchitectureStatementList.hh"
#include "IIR_ArchitectureStatement.hh"
#include "IIR_SimultaneousStatement.hh"
#include "IIR_ProcessStatement.hh"
#include "savant.hh"
#include "error_func.hh"
#include "published_cc_file.hh"
#include "language_processing_control.hh"
#include <iostream>

using std::cerr;

IIRScram_ArchitectureStatementList::~IIRScram_ArchitectureStatementList() {}

extern language_processing_control *lang_proc;

void 
IIRScram_ArchitectureStatementList::_publish_cc( published_file &_cc_out ) {
  IIR_ArchitectureStatement *node = NULL;
  IIR_ArchitectureStatement *transmutedNode = NULL;
  IIR *_old_publish_node = _get_current_publish_node();

  SCRAM_CC_REF( _cc_out, "IIRScram_ArchitectureStatementList::_publish_cc" );

  for (node = first(); node != NULL; node = successor(node)) {
    node->_set_enclosing_scope( _old_publish_node );
    transmutedNode = static_cast<IIR_ArchitectureStatement *>(node->_transmute());
    if(node != transmutedNode) {
      _replace(node, transmutedNode);
      node = transmutedNode;
    }
    node->_set_enclosing_scope( _old_publish_node );
    node->_publish_cc( _cc_out );
  }
}

IIR_Boolean
IIRScram_ArchitectureStatementList::_publish_cc_type_info( published_file &_cc_out,
							 IIR_Boolean firstDeclFlag ){
  IIR_ArchitectureStatement *node = NULL;
  IIR_ArchitectureStatement *transmutedNode = NULL;
  
  SCRAM_CC_REF( _cc_out, "IIRScram_ArchitectureStatementList::_publish_cc_type_info" );
  
  for (node = first(); node != NULL; node = successor(node)) {
    //4 lines added to do transmogrification once
    transmutedNode = (IIR_ArchitectureStatement*)node->_transmute();
    if(node != transmutedNode) {
      _replace(node, transmutedNode);
      node = transmutedNode;
    }
    if (node->get_kind() == IIR_PROCESS_STATEMENT ||
	node->get_kind() == IIR_BLOCK_STATEMENT){
      firstDeclFlag = firstDeclFlag && node->_publish_cc_type_info( _cc_out, firstDeclFlag );
    }
  }
  return firstDeclFlag;
}

void 
IIRScram_ArchitectureStatementList::_publish_cc_extern_type_info( published_file &_cc_out ) {
  IIR_ArchitectureStatement *node = NULL;
  IIR_ArchitectureStatement *transmutedNode = NULL;

  SCRAM_CC_REF( _cc_out, "IIRScram_ArchitectureStatementList::_publish_cc_extern_type_info" );
  
  for (node = first(); node != NULL; node = successor(node)) {
    //4 lines added to do transmogrification once
    transmutedNode = (IIR_ArchitectureStatement*)node->_transmute();
    if(node != transmutedNode) {
      _replace(node, transmutedNode);
      //      insert_after(node, transmutedNode);
      //      remove(node);
      node = transmutedNode;
    }
    if (node->get_kind() == IIR_PROCESS_STATEMENT ||
	node->get_kind() == IIR_BLOCK_STATEMENT){
      node->_publish_cc_extern_type_info( _cc_out );
    }
  }
}

IIR_Label *
IIRScram_ArchitectureStatementList::_find_instantiate_label( IIR_SimpleName *to_find ){
  IIR_Label *retval = NULL;

  IIR_ArchitectureStatement *current_statement = first();
  while( current_statement != NULL ){
    retval = current_statement->_find_instantiate_label( to_find );
    if( retval != NULL ){
      break;
    }

    current_statement = successor( current_statement );
  }

  return retval;
}

void 
IIRScram_ArchitectureStatementList::_type_check_instantiate_statements(){
  IIR_ArchitectureStatement *current = first();
  while( current != NULL ){
    current->_type_check_instantiate_statements();
    current = successor( current );
  }
}

void 
IIRScram_ArchitectureStatementList::_publish_cc_instantiate_calls( published_file &_cc_out ){
  IIR_ArchitectureStatement *current = first();
  while( current != NULL ){
    current->_publish_cc_instantiate_call( _cc_out );
    current = successor( current );
  }
}

int
IIRScram_ArchitectureStatementList::_get_number_of_component_instantiations() {
  int compInstCounter = 0;
  IIR_ArchitectureStatement *current = first();
  
  while (current != NULL) {
    if (current->get_kind() == IIR_COMPONENT_INSTANTIATION_STATEMENT) {
      compInstCounter++;
    }
    current = successor(current);
  }
  return compInstCounter;
}

visitor_return_type *IIRScram_ArchitectureStatementList::_accept_visitor(node_visitor *visitor, visitor_argument_type *arg) {
  ASSERT(visitor != NULL);
  return visitor->visit_IIR_ArchitectureStatementList(this, arg);
};

void
IIRScram_ArchitectureStatementList::_publish_cc_characteristic_expressions(published_file &_cc_out) {
  IIR_ArchitectureStatement *node;
  for( node = first(); node != NULL; node = successor(node) ) {
    if(node->get_kind() == IIR_SIMPLE_SIMULTANEOUS_STATEMENT || 
       node->get_kind() == IIR_SIMULTANEOUS_IF_STATEMENT ) {
      node->_publish_cc_characteristic_expressions(IIRScram_ArchitectureStatement::None,_cc_out);
    }
  }
}

void
IIRScram_ArchitectureStatementList::_publish_cc_ams_function_body(published_file & _cc_out) {
  IIR_ArchitectureStatement *node;
  for( node = first(); node != NULL; node = successor(node) ){
    switch( node->get_kind() ){
    case IIR_SIMPLE_SIMULTANEOUS_STATEMENT:
    case IIR_SIMULTANEOUS_IF_STATEMENT:
      ((IIR_SimultaneousStatement *)node)->_publish_cc_ams_function_body(_cc_out);
      break;
    default:
      // No action required
      break;
    }
  }
}

void
IIRScram_ArchitectureStatementList::_publish_cc_ams_function_prototype(published_file &_cc_out) {
  IIR_ArchitectureStatement *node;
  if( lang_proc->processing_vhdl_ams() ){
    _cc_out<<"#include \"Component.hh\"\n";
    _cc_out<<"#include <adouble.h>\n";
    _cc_out<<"#include <adutils.h>\n";
  } 
  for( node = first(); node != NULL; node = successor(node) ) {
    switch(node->get_kind()){
    case IIR_SIMPLE_SIMULTANEOUS_STATEMENT:
    case IIR_SIMULTANEOUS_IF_STATEMENT:
      static_cast<IIR_SimultaneousStatement *>(node)->_publish_cc_ams_function_prototype(_cc_out);
      break;
    default:
      // No action required
      break;
    }
  }
}

void
IIRScram_ArchitectureStatementList::_publish_cc_createNetInfo( published_file &_cc_out ) {
  // Publish createNetinfo from architecture statement part
  IIR_ArchitectureStatement *arch_stmt = first();
  while (arch_stmt!=NULL) {
    _set_current_publish_name( "" );
    _set_current_publish_node( NULL );

    switch( arch_stmt->get_kind() ){
    case IIR_PROCESS_STATEMENT:
      _set_current_elab_name( static_cast<IIR_ProcessStatement *>(arch_stmt)->_get_cc_binding_name() + "_elab_obj" );
      // No break here as we want the arch_stmt->_publish_createNetInfo() here.
    case IIR_BLOCK_STATEMENT:
    case IIR_COMPONENT_INSTANTIATION_STATEMENT:{
      arch_stmt->_publish_createNetInfo( _cc_out );
      break;    
    }
    case IIR_CONCURRENT_GENERATE_FOR_STATEMENT:
    case IIR_CONCURRENT_GENERATE_IF_STATEMENT:
      // No action is needed in this case.
      break;
    case IIR_SIMPLE_SIMULTANEOUS_STATEMENT:
    case IIR_SIMULTANEOUS_IF_STATEMENT:
      // No action is needed in this case.
      break;
    default:
      cerr << "ERROR! IIRScram_ArchitectureDeclaration::"
	   << "_publish_cc_createNetInfo( _cc_out ): unknown conc_statement "
	   << "type |" << arch_stmt->get_kind_text() << "| in arch\n";
      abort();
    }
    _set_current_elab_name( "" );
    arch_stmt = successor(arch_stmt);
  }
}
