
// Copyright (c) 1996-2003 The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
// SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
// IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE,
// OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE FOR ANY DAMAGES SUFFERED BY
// LICENSEE AS A RESULT OF USING, RESULT OF USING, MODIFYING OR
// DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the U.S.,
// and the terms of this license.

// You may modify, distribute, and use the software contained in this
// package under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE"
// version 2, June 1991. A copy of this license agreement can be found in
// the file "LGPL", distributed with this archive.

// Authors: Philip A. Wilsey	philip.wilsey@ieee.org
//          Dale E. Martin	dmartin@cliftonlabs.com
//          Timothy J. McBrayer 
//          Malolan Chetlur     
//          Krishnan Subramani  
//          Umesh Kumar V. Rajasekaran
//          Radharamanan Radhakrishnan
//          Narayanan Thondugulam
//          Swaminathan Subramanian
//	    Magnus Danielson	cfmd@swipnet.se

#include "IIRScram.hh"
#include "IIR_ArraySubtypeDefinition.hh"
#include "IIR_Attribute.hh"
#include "IIR_EnumerationSubtypeDefinition.hh"
#include "IIR_IntegerSubtypeDefinition.hh"
#include "IIR_RangeTypeDefinition.hh"
#include "IIR_TextLiteral.hh"
#include "error_func.hh"
#include "symbol_table.hh"
#include "resolution_func.hh"
#include "StandardPackage.hh"
#include "published_file.hh"
#include "sstream-wrap.hh"

IIRScram_ArrayTypeDefinition::IIRScram_ArrayTypeDefinition(){}

IIRScram_ArrayTypeDefinition::~IIRScram_ArrayTypeDefinition(){}

ostream&
IIRScram_ArrayTypeDefinition::_print(ostream &os) {
  os << "array ( " << *_get_index_subtype(); 
  
  IIR_TypeDefinition *element_type = get_element_subtype();
  
  while( element_type->_is_array_type() && element_type->_is_anonymous() ){
    os << ", " << *element_type->_get_index_subtype();
    element_type = element_type->_get_element_subtype();
  }
  
  os << " ) of " << *element_type->_get_declarator();

  return os;
}

IIR_TypeDefinition *
IIRScram_ArrayTypeDefinition::_get_element_subtype(){
  return get_element_subtype();
}

IIR_ScalarTypeDefinition *
IIRScram_ArrayTypeDefinition::_get_index_subtype(){
  return get_index_subtype();
}

void 
IIRScram_ArrayTypeDefinition::_set_element_subtype( IIR_TypeDefinition *new_element_type ){
  set_element_subtype( new_element_type );
}

void 
IIRScram_ArrayTypeDefinition::_set_index_subtype( IIR_ScalarTypeDefinition *new_index_type ){
  set_index_subtype( new_index_type );
}

void
IIRScram_ArrayTypeDefinition::_publish_vhdl_subtype_decl(ostream &_vhdl_out){
  if (_is_anonymous() == FALSE) {
    if (_get_type_mark() != NULL) {
      _get_type_mark()->_get_declaration()->get_declarator()->_publish_vhdl(_vhdl_out);
    }
    else {
      _get_declaration()->get_declarator()->_publish_vhdl(_vhdl_out);
    }
  }
  else {
    _publish_vhdl_decl(_vhdl_out);
  }
}

void 
IIRScram_ArrayTypeDefinition::_publish_vhdl_decl(ostream &_vhdl_out) {
  IIR_TypeDefinition *node = this;
  int max_index = _get_num_indexes();

  ASSERT(get_element_subtype() != NULL);
  ASSERT(_get_index_subtype() != NULL);

  _vhdl_out << "array (";
  
  node = this;
  while ((node->_is_array_type() == TRUE) && (max_index > 0)) {
    node->_get_index_subtype()->_publish_vhdl_index(_vhdl_out);
    max_index--;
    node =  node->_get_element_subtype();
    if (max_index > 0) {
      _vhdl_out << ", ";
      ASSERT ( node != NULL );
    }
  }

  ASSERT ( node != NULL );
  _vhdl_out << ") of ";
  node->_publish_vhdl(_vhdl_out);
}

void 
IIRScram_ArrayTypeDefinition::_publish_cc_lvalue( published_file &_cc_out ) {
  // SCRAM_CC_REF( _cc_out, "IIRScram_ArrayTypeDefinition::_publish_cc" );
  _cc_out.get_stream() << "Savant"
		      << *_get_declarator()
		      << "Type";
}

void 
IIRScram_ArrayTypeDefinition::_publish_cc_left( published_file &_cc_out ) {

  SCRAM_CC_REF( _cc_out, "IIRScram_ArrayTypeDefinition::_publish_cc_left" );

  ASSERT(get_element_subtype() != NULL);
  get_element_subtype()->_publish_cc_left( _cc_out );
}

IIR_Boolean
IIRScram_ArrayTypeDefinition::_is_discrete_type(){
  if(get_element_subtype()->_is_scalar_type() == TRUE){
    return get_element_subtype()->_is_discrete_type();
  }
  else {
    return FALSE;
  }
}

const string
IIRScram_ArrayTypeDefinition::_get_cc_kernel_type(){
  return "ArrayType";
}

void
IIRScram_ArrayTypeDefinition::_publish_cc_universal_type( published_file &_cc_out ) {
  // This is wrong. Just a fix as of now.
  _cc_out << "UniversalInteger";
  // Dummy function introduced to remove a undefined function report
}

void
IIRScram_ArrayTypeDefinition::_publish_cc_headers( published_file &_cc_out ) {
  SCRAM_CC_REF( _cc_out, "IIRScram_ArrayTypeDefinition::_publish_cc_headers" );
  IIRScram::_publish_cc_include( _cc_out, "tyvis/ArrayInfo.hh" );

  if( get_index_subtype() != 0 ){
    get_index_subtype()->_publish_cc_headers( _cc_out );
  }
  if( get_element_subtype() != 0 ){
    get_element_subtype()->_publish_cc_headers( _cc_out );
  }
  
  IIR_TypeDefinition::_publish_cc_headers( _cc_out );
}

void
IIRScram_ArrayTypeDefinition::_publish_cc_constructor_args( published_file &_cc_out ) {

  SCRAM_CC_REF( _cc_out, "IIRScram_ArrayTypeDefinition::_publish_cc_constructor_args" );

  //ObjectBase::Objecttype is published in object declaration
  if (_is_subtype() == TRUE) {
    if ((get_base_type()->_get_declaration() == NULL) ||
	(_is_anonymous() == TRUE)) {
      _cc_out << ", ";
      _publish_cc_range( _cc_out );
    }
  }
}

void
IIRScram_ArrayTypeDefinition::_publish_cc_range_args( published_file &_cc_out ) {
  int dimensions = _get_num_indexes();
  IIR_TypeDefinition *node = this;

  SCRAM_CC_REF( _cc_out, "IIRScram_ArrayTypeDefinition::_publish_cc_range_args" );
  int i;
  for( i = 1; i <= dimensions; i++) {
    _cc_out << ", int left" << i << ", ArrayInfo::ArrayDirn_t dirn" 
	    << i << ", int right" << i;
    node =  node->_get_element_subtype();
  }
}

void
IIRScram_ArrayTypeDefinition::_publish_cc_constructor_ranges( published_file &_cc_out ) {

  SCRAM_CC_REF( _cc_out, "IIRScram_ArrayTypeDefinition::_publish_cc_constructor_ranges" );
  _cc_out << _get_cc_type_name() <<
    "(ObjectBase::ObjectType objType, const TypeInfo &tInfo";

  if(_is_unconstrained_array_type() == TRUE) {
    _publish_cc_range_args( _cc_out );
  }
}

void
IIRScram_ArrayTypeDefinition::_publish_cc_constructor_prelude( published_file &_cc_out ){
  IIR_Int32 dimensions = _get_num_indexes();
  IIR_TypeDefinition *node = _get_element_subtype();
  ostringstream params;

  SCRAM_CC_REF( _cc_out, "IIRScram_ArrayTypeDefinition::_publish_cc_constructor_prelude" );

  _cc_out << "    ArrayInfo aInfo[] = {";
  _get_index_subtype()->_publish_cc_array_info( _cc_out );
  for(int i = 2; (i <= dimensions); i++) {
    _cc_out << ", ";
    node->_get_index_subtype()->_publish_cc_array_info( _cc_out );
    node = node->_get_element_subtype();
  }
  _cc_out << "};\n";
}


void
IIRScram_ArrayTypeDefinition::_publish_cc_set_element_range_info( published_file &_cc_out ) {

  SCRAM_CC_REF( _cc_out, "IIRScram_ArrayTypeDefinition::_publish_cc_set_element_range_info" );

  if(_is_element_unconstrained_subtype() == TRUE) {
    _cc_out << "    ArrayInfo* tempRangeInfo = ArrayType::rangeInfo;\n";
    _cc_out << "    ArrayInfo elementRangeInfo[] = { ";

    IIR_ArrayTypeDefinition* element_type_def = (IIR_ArrayTypeDefinition*)_get_element_type();
    IIR_Int32 dimensions = element_type_def->_get_num_indexes();
    IIR_TypeDefinition *node = element_type_def->_get_element_subtype();
    IIR_Int32 i;

    if(element_type_def->_get_index_subtype()->get_left() != NULL) {
      ASSERT(element_type_def->_get_index_subtype()->get_right() != NULL);
      ASSERT(element_type_def->_get_index_subtype()->get_direction() != NULL);
      element_type_def->_publish_cc_array_info( _cc_out );
    }

    for( i = 2; i <= dimensions; i++) {
      if(node->_get_index_subtype()->get_left() != NULL) {
	ASSERT(node->_get_index_subtype()->get_right() != NULL);
	ASSERT(node->_get_index_subtype()->get_direction() != NULL);
	node->_publish_cc_array_info( _cc_out );
      } 
      node = node->_get_element_subtype();
    }

    _cc_out << " };\n";
    _cc_out << "    ArrayType::rangeInfo = elementRangeInfo;\n";
  }
}

IIR_TypeDefinition*
IIRScram_ArrayTypeDefinition::_get_element_type() {
  IIR_TypeDefinition *current_subtype = get_element_subtype();
  ASSERT(current_subtype != NULL);
  while((current_subtype->_is_element() == FALSE ) &&
	(current_subtype->_is_array_type()== TRUE)) {
    current_subtype = current_subtype->_get_element_subtype();
    ASSERT(current_subtype != NULL);
  }
  return current_subtype;
}

IIR_Boolean
IIRScram_ArrayTypeDefinition::_is_element_unconstrained_subtype() {
  IIR_TypeDefinition* element_type = _get_element_type();
  if(element_type->_is_array_type() == FALSE) {
    return FALSE;
  }
  else {
    if((element_type->_is_anonymous() == TRUE) &&
       (element_type->get_base_type()->_is_unconstrained_array_type() == TRUE)) {
      return TRUE;
    }
  }
  return FALSE;
}

void
IIRScram_ArrayTypeDefinition::_publish_cc_restore_range_info( published_file &_cc_out ) {

  SCRAM_CC_REF( _cc_out, "IIRScram_ArrayTypeDefinition::_publish_cc_restore_range_info" );

  if(_is_element_unconstrained_subtype() == TRUE) {
    _cc_out << "    ArrayType::rangeInfo = tempRangeInfo;\n";
  }
}


void
IIRScram_ArrayTypeDefinition::_publish_cc_object_type( published_file &_cc_out,
						       IIR_Boolean typeNameFlag, 
						       const char *suffix) {
  if (typeNameFlag == TRUE) {
    _cc_out << _get_cc_type_name();
  }

  if (suffix != NULL) {
    _cc_out << suffix;
  }
  
  _cc_out << "(ObjectBase::ObjectType objType, const TypeInfo &tInfo";
}


IIR_Boolean 
IIRScram_ArrayTypeDefinition::_is_character_type(){
  IIR_Boolean retval = FALSE;

  if( get_base_type() != 0 ){
    retval = get_base_type()->_is_character_type();
  }
  else{
    if( get_element_subtype()->_is_character_type() == TRUE &&
	get_element_subtype()->_is_scalar_type() == TRUE ){
      retval = TRUE;
    }
  }
    
  return retval;
}

void
IIRScram_ArrayTypeDefinition::_publish_cc_define_array_type_attributes( published_file &_cc_out ) {

  SCRAM_CC_REF( _cc_out, "IIRScram_ArrayTypeDefinition::_publish_cc_define_array_type_attributes" );

  // These are defined only for subtypes.
  _report_undefined_scram_fn("_publish_cc_define_array_type_attributes( _cc_out )");
}

void
IIRScram_ArrayTypeDefinition::_publish_cc_init_signal( published_file &_cc_out ) {

  ostringstream tmpname;
  ostringstream tmpanothername;
  ostringstream tmpsuffixname;
  
  const string tmpPtr1 = _get_current_publish_name();
  const string tmpPtr2 = _get_current_another_name();
  const string tmpPtr4 = _get_current_suffix_name();
  IIR* tmpPtr3 = _get_current_publish_node();

  SCRAM_CC_REF( _cc_out, "IIRScram_ArrayTypeDefinition::_publish_cc_init_signal" );
  
  _set_index_level( _get_index_level() + 1 );
  _cc_out << "{\n";//begin dummy block for scoping
  _cc_out << "int index" << _get_index_level() << " =0;\n";
  _cc_out << "for(index" << _get_index_level() << "=0;";
  _cc_out << " index";
  _cc_out << _get_index_level() << " < (";
  _cc_out << _get_current_publish_name();
  _cc_out << ").get_number_of_elements(); index" << _get_index_level() << "++) {\n";
  tmpname << _get_current_publish_name() << ".get_element(index" << _get_index_level()
	  << ")";
  tmpanothername << _get_current_another_name() << ".get_element(index"
		 <<_get_index_level()<< ")";

  if( _get_current_suffix_name() != "" ){
    tmpsuffixname << _get_current_suffix_name();
  }
  
  tmpsuffixname << ".get_element(index"
		<< _get_index_level() << ")";
  
  _set_current_publish_name( tmpname.str() );
  _set_current_another_name( tmpanothername.str() );
  _set_current_suffix_name( tmpsuffixname.str() );
  
  get_element_subtype()->_publish_cc_init_signal( _cc_out );
  
  _cc_out << "}\n";
  _cc_out << "};\n";//end dummy block
  
  _set_current_publish_name( tmpPtr1 );
  _set_current_another_name( tmpPtr2 );
  _set_current_publish_node( tmpPtr3 );
  _set_current_suffix_name( tmpPtr4 );
}

void
IIRScram_ArrayTypeDefinition::_publish_cc_composite_init( published_file &_cc_out ){
  ostringstream tmpname, tmpRecName;
  const string tmpPtr1 = _get_current_publish_name();

  SCRAM_CC_REF( _cc_out, "IIRScram_ArrayTypeDefinition::_publish_cc_composite_init" );

  _set_index_level( _get_index_level() + 1 );
  _cc_out << "for( int index" << _get_index_level() << " = 0;" << OS("")
	  << "index" << _get_index_level() << " < "
	  << _get_current_publish_name() << ".get_number_of_elements();" << NL() 
	  << "index" << _get_index_level() << "++ )" << OS("{") << CS("");
  tmpname << _get_current_publish_name() << ".get_element(index" << _get_index_level() << ")";
  _set_current_publish_name( tmpname.str() );
  // Need to change this I guess. May not be the right way to solve the
  // problem.
  tmpRecName << "((" << get_element_subtype()->_get_cc_type_name() << " &) "
	     << _get_current_publish_name() << ")";

  _set_current_publish_name( tmpRecName.str() );
  
  get_element_subtype()->_publish_cc_composite_init( _cc_out );
  _cc_out << CS("}"); //end for block
  _set_current_publish_name( tmpPtr1 );
}

void
IIRScram_ArrayTypeDefinition::_publish_cc_range( published_file &_cc_out ) {
  IIR_Int32 num_indexes = _get_num_indexes();
  IIR_TypeDefinition *elem_subtype = this;

  SCRAM_CC_REF( _cc_out, "IIRScram_ArrayTypeDefinition::_publish_cc_range" );

  elem_subtype->_get_index_subtype()->_publish_cc_range( _cc_out );
  int i;
  for( i = 1; i < num_indexes; i++) {
    _cc_out << "," << NL();
    elem_subtype = elem_subtype->_get_element_subtype();
    ASSERT(elem_subtype != NULL);
    elem_subtype->_get_index_subtype()->_publish_cc_range( _cc_out );
  }
}

IIR_Int32
IIRScram_ArrayTypeDefinition::_get_num_indexes() {
  IIR_Int32 num_indexes = 1;

  IIR_TypeDefinition *current_subtype = get_element_subtype();
  ASSERT(current_subtype != NULL);
  while( current_subtype->_is_array_type() == TRUE
	 && current_subtype->_is_element() == FALSE ){
    num_indexes++;
    current_subtype = current_subtype->_get_element_subtype();
    ASSERT(current_subtype != NULL);
  }

  return num_indexes;
}

IIR_Boolean 
IIRScram_ArrayTypeDefinition::_has_access_type() {
  return _get_element_type()->_has_access_type();
}

IIR_Boolean 
IIRScram_ArrayTypeDefinition::_is_single_dimensional_array_type() {
  if(_get_num_indexes() == 1) {
    return TRUE;
  }
  else {
    return FALSE;
  }
}

IIR_Boolean 
IIRScram_ArrayTypeDefinition::_is_unconstrained_array_type(){
  IIR_ScalarTypeDefinition *index = _get_index_subtype();
  ASSERT(index != NULL);
  if( index->get_left() == NULL ){
    ASSERT( index->get_right() == NULL );
    ASSERT( index->get_direction() == NULL );

    return TRUE;
  }
  else{
    return FALSE;
  }
}


set<IIR_Declaration> *
IIRScram_ArrayTypeDefinition::_find_declarations( IIR_Name *to_find ){
  return get_element_subtype()->_find_declarations( to_find );
}


void
IIRScram_ArrayTypeDefinition::_publish_cc_event_vector_instantiation( published_file &_cc_out ) {

  IIR_Int32 dimensions = _get_num_indexes();
  IIR_TypeDefinition *node = NULL;
  int i;

  SCRAM_CC_REF( _cc_out, "IIRScram_ArrayTypeDefinition::_publish_cc_event_vector_instantiation" );

  _publish_cc_set_element_range_info( _cc_out );

  _cc_out << "    object = new ";
  node = this;
  for(i = 1; i <= dimensions; i++) {
    node = node->_get_element_subtype();
    _cc_out << "Vector<";
  }
  
  if(node->_is_scalar_type() == TRUE) {
    _cc_out << "EnumerationType";
  }
  else {
    node->_get_cc_type_name();
    _cc_out << "_event";
  }
  
  for(i = 1; i <= dimensions; i++) {
    _cc_out << "> ";
  }
  _cc_out << "(objType);\n";
  _publish_cc_restore_range_info( _cc_out );
}

void 
IIRScram_ArrayTypeDefinition::_publish_cc_array_info( published_file &_cc_out ){

  SCRAM_CC_REF( _cc_out, "IIRScram_ArrayTypeDefinition::_publish_cc_array_info" );

  if ((_is_unconstrained_array_type() == TRUE) || (_get_num_indexes() > 1)) {
    _cc_out << "nullInfo";
  } 
  else {
    _cc_out << "ArrayInfo(";
    _publish_cc_range( _cc_out );
    _cc_out << ")";
  }
}


void
IIRScram_ArrayTypeDefinition::_publish_cc_bounds( published_file &_cc_out ){

  SCRAM_CC_REF( _cc_out, "IIRScram_ArrayTypeDefinition::_publish_cc_bounds" );

  _publish_cc_array_info( _cc_out );
}

void
IIRScram_ArrayTypeDefinition::_publish_cc_necessary_decl_in_state( published_file &_cc_out )
{

  SCRAM_CC_REF( _cc_out, "IIRScram_ArrayTypeDefinition::_publish_cc_necessary_decl_in_state" );

  _get_cc_type_name();
  _cc_out << "_event ";
  _cc_out << _get_current_publish_name();
  _cc_out << "_event;\n";
  
  _get_cc_type_name();
  _cc_out << "_lastevent  ";
  _cc_out << _get_current_publish_name();
  _cc_out << "_last_event;\n";
}

void 
IIRScram_ArrayTypeDefinition::_publish_cc_extern_type_info( published_file &_cc_out ){
  string externFlag;
  
  SCRAM_CC_REF( _cc_out, "IIRScram_ArrayTypeDefinition::_publish_cc_extern_type_info" );
  
  if ((_get_currently_publishing_unit() == PACKAGE_PUB) ||
      (_get_currently_publishing_unit() == PACKAGE_BODY)) {
    externFlag = "extern";
  }

  _cc_out << externFlag << " ArrayTypeInfo ";
  _publish_cc_lvalue( _cc_out );
  _cc_out << "_info;" << NL();
  
  _cc_out << externFlag << " ArrayTypeInfo ";
  _publish_cc_lvalue( _cc_out );
  _cc_out << "_event_info;" << NL();
  
  _cc_out << externFlag << " ArrayTypeInfo ";
  _publish_cc_lvalue( _cc_out );
  _cc_out << "_lastevent_info;" << NL();
}

void
IIRScram_ArrayTypeDefinition::_publish_cc_temporary_type_info( published_file &_cc_out,
							       char *suffix, 
							       char *elementTypeInfo) {
  IIR_Int32 dimensions = _get_num_indexes();
  IIR_TypeDefinition *node = this; 
  int i = dimensions;

  _cc_out << "ArrayTypeInfo" << OS("(") << dimensions << "," << NL()
	  << "&";
  
  if (elementTypeInfo == NULL) {
    if (_get_final_subtype()->_is_access_type() == TRUE) {
      // These ypes do not have type_info structures yet...
      _cc_out << "TypeInfo::getNullTypeInfo()";
    }
    else {
      _get_final_subtype()->_publish_cc_object_type_info( _cc_out, FALSE );
    }
  }
  else {
    _cc_out << elementTypeInfo;
  }

  _cc_out << "," << NL();

  _cc_out << ((_is_unconstrained_array_type() == TRUE) ? "true" : "false")
	  << "," << NL();
  _cc_out << ((_is_resolved_type() == TRUE) ? "true" : "false")
	  << "," << NL();
  
  _publish_cc_resolution_function_id( _cc_out );
  _cc_out << "," << NL();
  
  do {
    if(node->_get_index_subtype()->get_left() != NULL) {
      ASSERT(node->_get_index_subtype()->get_right() != NULL);
      ASSERT(node->_get_index_subtype()->get_direction() != NULL);
      _cc_out << "new ";
      node->_get_index_subtype()->_publish_cc_temporary_type_info( _cc_out,
								   suffix, 
								   elementTypeInfo );
    } 
    else {
      if (node->_get_index_subtype()->_get_base_type() != NULL) {
	_cc_out << "new ";
	node->_get_index_subtype()->_get_base_type()->_publish_cc_temporary_type_info( _cc_out,
										       suffix, 
										       elementTypeInfo);
      }
      else {
	cerr << "Error: Unable to determine range for dimension " << i
	     << " of array Type " << *_get_declarator() << ".\n";
	ASSERT ( FALSE );	
      }
    }
    
    i--;
    if (i > 0) {
      _cc_out << "," << NL();
      node = node->_get_element_subtype();
    }
  } 
  while (i > 0);
  
  _cc_out << CS(")");
}

void
IIRScram_ArrayTypeDefinition::_publish_cc_type_info( published_file &_cc_out ) {
  _publish_cc_type_info( _cc_out, NULL, NULL );

  if( _get_currently_publishing_unit() == PROCEDURE ||
      _get_currently_publishing_unit() == FUNCTION  ||
      _get_currently_publishing_unit() == PACKAGE_BODY ||
      _get_currently_publishing_unit() == PACKAGE_PUB ){
    _cc_out << ";";
  }
  else {
    _cc_out << ",";
  }
  _cc_out << NL();

  _publish_cc_type_info( _cc_out, "_event", "SavantbooleanType_info" );

  if ( _get_currently_publishing_unit() == PROCEDURE ||
       _get_currently_publishing_unit() == FUNCTION  ||
       _get_currently_publishing_unit() == PACKAGE_BODY ||
       _get_currently_publishing_unit() == PACKAGE_PUB ){
    _cc_out << ";";
  }
  else {
    _cc_out << ",";
  }
  _cc_out << NL();

  _publish_cc_type_info( _cc_out, "_lastevent", "SavanttimeType_info" );  
}

void
IIRScram_ArrayTypeDefinition::_publish_cc_type_info( published_file &_cc_out,
						     char *suffix, 
						     char *elementTypeInfo) {
  IIR_Int32 dimensions = _get_num_indexes();
  IIR_TypeDefinition *node = this; 
  IIR_Int32 i = dimensions;

  SCRAM_CC_REF( _cc_out, "IIRScram_ArrayTypeDefinition::_publish_cc_type_info" );

  if( _get_currently_publishing_unit() == PROCEDURE ||
      _get_currently_publishing_unit() == FUNCTION ||
      _get_currently_publishing_unit() == PACKAGE_BODY ||
      _get_currently_publishing_unit() == PACKAGE_PUB ){
    _cc_out << "ArrayTypeInfo" << NL();
  }
  
  _publish_cc_lvalue( _cc_out );
  if (suffix != NULL) {
    _cc_out << suffix;
  }
  _cc_out << OS("_info(")
	  << dimensions << "," << NL();

  // _publish the type_info structure for the elements of this array...
  ASSERT ( _get_element_subtype() != NULL );

  if (elementTypeInfo == NULL) {
    if (_get_final_subtype()->_is_access_type() == TRUE) {
      // These ypes do not have type_info structures yet...
      _cc_out << "&TypeInfo::getNullTypeInfo()";
    }
    else {
      if ((_get_final_subtype()->_is_array_type() == TRUE) &&
	  (_get_final_subtype()->_is_anonymous() == TRUE)) {
	_cc_out << "new ";
      }
      else {
	_cc_out << "&";
      }
      _get_final_subtype()->_publish_cc_object_type_info( _cc_out, FALSE, suffix, TRUE );
    }
  }
  else {
    _cc_out << "&" << elementTypeInfo;
  }
  
  _cc_out << "," << NL();

  _cc_out << ((_is_unconstrained_array_type() == TRUE) ? "true" : "false")
	  << "," << NL();

  _cc_out << ((_is_resolved_type() == TRUE) ? "true" : "false")
	  << "," << NL();
  
  _publish_cc_resolution_function_id( _cc_out );
  _cc_out << "," << NL();

  do {
    if(node->_get_index_subtype()->get_left() != NULL) {
      ASSERT(node->_get_index_subtype()->get_right() != NULL);
      ASSERT(node->_get_index_subtype()->get_direction() != NULL);
      if (node->_get_index_subtype()->_is_anonymous() == FALSE) {
	_cc_out << "&";
	node->_get_index_subtype()->_publish_cc_object_type_info( _cc_out, FALSE );
      }
      else {
	_cc_out << "new ";
	node->_get_index_subtype()->_publish_cc_temporary_type_info( _cc_out, 
								     suffix, 
								     elementTypeInfo );
      }
    } 
    else {
      if (node->_get_index_subtype()->_get_base_type() != NULL) {
	_cc_out << "new ";
	node->_get_index_subtype()->_get_base_type()->_publish_cc_temporary_type_info( _cc_out,
										       suffix, 
										       elementTypeInfo);
      }
      else {
	cerr << "Error: Unable to determine range for dimension " << i
	     << " of array Type " << *_get_declarator() << ".\n";
	ASSERT ( TRUE );
      }
    }
    
    i--;
    if (i > 0) {
      _cc_out << "," << NL();
      node = node->_get_element_subtype();
    }
  } while (i > 0);
  
  _cc_out << CS(")");
  
  if ((_get_currently_publishing_unit() == PACKAGE_BODY) ||
      (_get_currently_publishing_unit() == PACKAGE_PUB)) {
    _cc_out << ";" << NL();
  }
}

const string
IIRScram_ArrayTypeDefinition::_get_cc_type_name(){
  return "ArrayType";
}

IIR_ScalarTypeDefinition *
IIRScram_ArrayTypeDefinition::_build_proper_index( IIR_RangeTypeDefinition *range ){
  IIR_ScalarTypeDefinition *range_type =
    IIRScram_ScalarTypeDefinition::_determine_discrete_type( range );
  
  IIR_TypeDefinition *new_subtype =  range_type->_construct_new_subtype( NULL, range );

  ASSERT( new_subtype->_is_scalar_type() == TRUE );

  return (IIR_ScalarTypeDefinition *)new_subtype;
}

IIR_TypeDefinition *
IIRScram_ArrayTypeDefinition::_index_constrain_array( IIR_ScalarTypeDefinition *constraint ){
#ifdef DEVELOPER_ASSERTIONS
  IIR_TypeDefinition *original_index_subtype = _get_index_subtype();
#endif

  IIR_ArraySubtypeDefinition *retval = NULL;

  IIR_TypeDefinition *current_index_type = _get_index_subtype(); 
  
  current_index_type = current_index_type->_construct_new_subtype( NULL, constraint);
  if( constraint->_is_subtype() == FALSE && constraint->_is_anonymous() == FALSE ){
    current_index_type->_set_type_mark( constraint );
  }
  else{
    current_index_type->_set_type_mark( constraint->_get_type_mark() );
  }

  current_index_type->_set_declaration( constraint->_get_declaration() );

  retval = new IIR_ArraySubtypeDefinition();
  copy_location( constraint, retval );

  ASSERT( current_index_type->_is_scalar_type() == TRUE );
  retval->_set_index_subtype( (IIR_ScalarTypeDefinition *)current_index_type );

  if( _is_subtype() == TRUE ){
    retval->set_base_type( get_base_type() );    
  }
  else{
    retval->set_base_type( this );
  }

  ASSERT( original_index_subtype == _get_index_subtype() );

  return retval;
}


IIR_TypeDefinition *
IIRScram_ArrayTypeDefinition::_construct_new_subtype( IIR_Name *resolution_function,
						      IIR_ScalarTypeDefinition *index_constraint ){

  IIR_TypeDefinition *retval = NULL;

  if( resolution_function != NULL ){
    retval = _construct_new_subtype_resolution_function_only( resolution_function );
  }

  if( index_constraint != NULL ){
    retval = _index_constrain_array( index_constraint );
    retval->_set_element_subtype( _get_element_subtype() );
  }
  
  if( retval == NULL ){
    // We'll just make a copy of ourseleves then.
    retval = _get_new_subtype();
    ASSERT( retval->_is_iir_array_type_definition() == TRUE );
    _clone( (IIR_ArrayTypeDefinition *)retval );
    if( _is_subtype() ){
      retval->set_base_type( get_base_type() );
    }
    else{
      retval->set_base_type( this );
    }
  }

  return retval;
}

IIR_ArrayTypeDefinition *
IIRScram_ArrayTypeDefinition::_construct_array_type( IIR_ScalarTypeDefinition *index_subtype,
						     IIR_TypeDefinition *element_subtype,
						     IIR_TypeDeclaration *declaration ){
  if( index_subtype->get_left() == NULL ){
    return _construct_unconstrained( index_subtype, element_subtype, declaration );
  }
  else{
    return _construct_constrained( index_subtype, element_subtype, declaration );
  }
}


IIR_ArraySubtypeDefinition *
IIRScram_ArrayTypeDefinition::_construct_unconstrained( IIR_ScalarTypeDefinition *index_subtype,
						        IIR_TypeDefinition *element_subtype,
							IIR_TypeDeclaration *declaration ){

  IIR_ArrayTypeDefinition *base_type = new IIR_ArrayTypeDefinition();
  copy_location( index_subtype, base_type );

  ASSERT( index_subtype->get_left() == NULL );
  ASSERT( index_subtype->get_direction() == NULL );
  ASSERT( index_subtype->get_right() == NULL );
  
  base_type->set_index_subtype( index_subtype );
  base_type->set_element_subtype( element_subtype );
  base_type->_set_declaration( declaration );

  IIR_TypeDefinition *temp =  base_type->_construct_new_subtype( 0, 0 );
  ASSERT( temp->get_kind() == IIR_ARRAY_SUBTYPE_DEFINITION );

  IIR_ArraySubtypeDefinition *retval = (IIR_ArraySubtypeDefinition *)temp;
  retval->_set_declaration( declaration );

  return retval;
}

IIR_ArraySubtypeDefinition *
IIRScram_ArrayTypeDefinition::_construct_constrained( IIR_ScalarTypeDefinition *index_subtype,
						      IIR_TypeDefinition *element_subtype,
						      IIR_TypeDeclaration *declaration ){

  // This is the unconstrained base type mandated by the LRM.
  IIR_ArrayTypeDefinition *base_type = new IIR_ArrayTypeDefinition();
  copy_location( index_subtype, base_type );

  // This is implementing the rules of pg. 41, line 314 of the LRM.
  IIR_TypeDefinition *base_index_type;
  base_index_type = index_subtype->_get_new_subtype();
  base_index_type->set_base_type( index_subtype->get_base_type() );
  base_index_type->_set_resolution_function( index_subtype->_get_resolution_function() );
  base_index_type->_set_type_mark( index_subtype );
  base_index_type->_set_declaration( index_subtype->_get_declaration() );
  ASSERT( base_index_type->_is_iir_scalar_type_definition() == TRUE );
  base_type->_set_index_subtype( (IIR_ScalarTypeDefinition *)base_index_type );
  base_type->_get_index_subtype()->_set_type_mark( index_subtype );
  IIR_TypeDefinition *temp =  base_type->_construct_new_subtype( 0, index_subtype );
  ASSERT( temp->get_kind() == IIR_ARRAY_SUBTYPE_DEFINITION );

  IIR_ArraySubtypeDefinition *retval = (IIR_ArraySubtypeDefinition *)temp;

  // This is a kludge for the code generator
  base_type->_set_type_mark( retval );
  base_type->_set_declaration( declaration );
  retval->_set_declaration( declaration );
  return retval; 
}

IIR_TypeDefinition *
IIRScram_ArrayTypeDefinition::_get_new_subtype(){
  IIR_TypeDefinition * retval = new IIR_ArraySubtypeDefinition();
  copy_location( this, retval );
  return retval;
}

void 
IIRScram_ArrayTypeDefinition::_add_decl_into_cgen_symbol_table() {
  if(get_base_type() != NULL && get_base_type()->_is_anonymous() == FALSE) {
    get_base_type()->_add_decl_into_cgen_symbol_table();
  }
  if(_get_index_subtype() != NULL) {
    _get_index_subtype()->_add_decl_into_cgen_symbol_table();
  }
  if(get_element_subtype() != NULL) {
    _get_element_subtype()->_add_decl_into_cgen_symbol_table();
  }
}

IIR *
IIRScram_ArrayTypeDefinition::_clone(){
  IIR_ArrayTypeDefinition *type_def = new IIR_ArrayTypeDefinition();
  _clone( type_def );

  return type_def;
}

void 
IIRScram_ArrayTypeDefinition::_clone( IIR *my_clone ){
  ASSERT( my_clone->_is_iir_array_type_definition() == TRUE );
  IIR_ArrayTypeDefinition *as_array_type = (IIR_ArrayTypeDefinition *)my_clone;
  
  as_array_type->set_index_subtype( get_index_subtype() );
  as_array_type->set_element_subtype( get_element_subtype() );

  IIR_TypeDefinition::_clone( my_clone );
}

void
IIRScram_ArrayTypeDefinition::_type_check(){
  // See lines 369-372 of 1076-1993 for this rule.  It seems to be OK to declare
  // an array of access types to unconstained arrays, hence the first part of the rule.
  if( _get_final_subtype()->_is_iir_array_type_definition() == TRUE &&
      _get_final_subtype()->_is_unconstrained_array_type() == TRUE ){
    ostringstream err;
    err << "The element subtype indication for an array type definition must define "
	<< "a constrained array subtype.";
    report_error( this, err.str() );
  }
}

void
IIRScram_ArrayTypeDefinition::_publish_vhdl(ostream &_vhdl_out){
  ASSERT( _get_declaration() != NULL );
  _get_declaration()->_publish_vhdl(_vhdl_out);
}


void 
IIRScram_ArrayTypeDefinition::_come_into_scope( symbol_table *sym_tab, IIR_TypeDeclaration *td ){
  IIR_TypeDefinition::_come_into_scope( sym_tab, td );

  if( _get_num_indexes() == 1 ){
    sym_tab->get_in_scope_one_d_array_types()->add( (IIR_ArrayTypeDefinition *)this );
  }

  sym_tab->get_in_scope_array_types()->add( (IIR_ArrayTypeDefinition *)this );
}

void 
IIRScram_ArrayTypeDefinition::_come_out_of_scope( symbol_table *sym_tab ){
  if( _get_num_indexes() == 1 ){
    sym_tab->get_in_scope_one_d_array_types()->remove( (IIR_ArrayTypeDefinition *)this );
  }

  sym_tab->get_in_scope_array_types()->remove( (IIR_ArrayTypeDefinition *)this );
}

void 
IIRScram_ArrayTypeDefinition::_build_implicit_operators( set<IIR_Declaration> *add_to ){
  char *shift_operators[] = { "\"sll\"", "\"srl\"", "\"sla\"", "\"sra\"",
			      "\"rol\"", "\"ror\"", NULL };

  IIR_TypeDefinition::_build_implicit_operators( add_to );

  if( _get_num_indexes() == 1 ){
    IIR_TypeDefinition *boolean_type = StandardPackage::get_boolean_type();
    IIR_TypeDefinition *bit_type =  StandardPackage::get_bit_type();
    IIR_TypeDefinition *integer_type = StandardPackage::get_integer_type();
    
    // The order of the operands is result type, left type, right type.

    if( get_element_subtype()->_is_discrete_type() == TRUE ){
      _build_ordering_operators( add_to );
    }
    
    if( get_element_subtype()->_is_compatible( boolean_type ) != NULL ||
	get_element_subtype()->_is_compatible( bit_type ) != NULL ){
      // return this type, left is this type, right is integer
      IIR_TypeDefinition::_build_implicit_operators( shift_operators, 
						     add_to,
						     this,
						     this,
						     integer_type );      
      IIR_TypeDefinition::_build_logical_operators( add_to );
    }

    IIR_TypeDefinition::_build_implicit_operator( "\"&\"", add_to, this, this, this );
    
    IIR_TypeDefinition::_build_implicit_operator( "\"&\"", 
						  add_to, this, get_element_subtype(),this);
    
    IIR_TypeDefinition::_build_implicit_operator( "\"&\"", 
						  add_to, this, this,get_element_subtype());
    
    IIR_TypeDefinition::_build_implicit_operator( "\"&\"", 
						  add_to, 
						  this, 
						  get_element_subtype(), 
						  get_element_subtype());
  }
}

IIR_TypeDefinition*
IIRScram_ArrayTypeDefinition::_get_final_subtype() {

  IIR_TypeDefinition* node = get_element_subtype();
  while ( (node->get_kind() == IIR_ARRAY_TYPE_DEFINITION ||
	   node->get_kind() == IIR_ARRAY_SUBTYPE_DEFINITION) &&
	  node->_is_element() == false ){
    node =  node->_get_element_subtype();
  }
  return node;
}

void
IIRScram_ArrayTypeDefinition::_publish_cc_constructor_args_type_info( published_file &_cc_out,
								      IIR *initializer ){
  _publish_cc_constructor_args_type_info_composite_type( _cc_out,
							 initializer );
}

void 
IIRScram_ArrayTypeDefinition::_set_resolution_function( IIR_FunctionDeclaration * ){
  ostringstream err;
  err << "Internal error - IIRScram_ArrayTypeDefinition::_set_resolution_function was "
      << "called.  Resolution functions can't be associated with an array type and this "
      << "should have been caught earlier.";
  report_error( this, err.str() );
}

visitor_return_type *
IIRScram_ArrayTypeDefinition::_accept_visitor(node_visitor *visitor, visitor_argument_type *arg) {
  ASSERT(visitor != NULL);
  return visitor->visit_IIR_ArrayTypeDefinition(this, arg);
}
