
// Copyright (c) 1996-2003 The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
// SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
// IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE,
// OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE FOR ANY DAMAGES SUFFERED BY
// LICENSEE AS A RESULT OF USING, RESULT OF USING, MODIFYING OR
// DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the U.S.,
// and the terms of this license.

// You may modify, distribute, and use the software contained in this package
// under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE" version 2,
// June 1991. A copy of this license agreement can be found in the file
// "LGPL", distributed with this archive.

// Authors: Philip A. Wilsey	philip.wilsey@ieee.org
//          Dale E. Martin	dmartin@cliftonlabs.com
//          Timothy J. McBrayer 
//          Malolan Chetlur     
//          Krishnan Subramani  
//          Umesh Kumar V. Rajasekaran
//          Narayanan Thondugulam 
//          Radharamanan Radhakrishnan
//          Swaminathan Subramanian
//	    Magnus Danielson	cfmd@swipnet.se

#include "IIRScram.hh"
#include "IIRScram_EnumerationLiteral.hh"
#include "IIR_EnumerationLiteral.hh"
#include "IIRScram_EnumerationLiteralList.hh"
#include "IIR_EnumerationTypeDefinition.hh"
#include "IIR_Identifier.hh"
#include "IIR_CharacterLiteral.hh"
#include "IIR_StringLiteral.hh"
#include "set.hh"
#include "symbol_table.hh"
#include "published_file.hh"

IIRScram_EnumerationLiteral::~IIRScram_EnumerationLiteral() {}

void 
IIRScram_EnumerationLiteral::_publish_vhdl(ostream &_vhdl_out) {
  get_declarator()->_publish_vhdl(_vhdl_out);
}

void 
IIRScram_EnumerationLiteral::_publish_vhdl_decl(ostream &_vhdl_out) {
  _publish_vhdl(_vhdl_out);
}

void 
IIRScram_EnumerationLiteral::_publish_cc_rvalue( published_file &_cc_out ) {
  SCRAM_CC_REF( _cc_out, "IIRScram_EnumerationLiteral::_publish_cc_rvalue" );
  _cc_out << "UniversalInteger(";
  _publish_cc_universal_value( _cc_out );
  _cc_out << ")";
}

void 
IIRScram_EnumerationLiteral::_publish_cc_lvalue( published_file &_cc_out ) {
  SCRAM_CC_REF( _cc_out, "IIRScram_EnumerationLiteral::_publish_cc_lvalue" );
  _cc_out << OS("EnumerationType(") 
	  << "ObjectBase::VARIABLE," << NL();
  _publish_cc_universal_value( _cc_out );
  _cc_out << CS(")");
}

void 
IIRScram_EnumerationLiteral::_publish_cc_universal_value( published_file &_cc_out ) {
  _publish_cc_value( _cc_out );
}

void 
IIRScram_EnumerationLiteral::_publish_cc_value( published_file &_cc_out ) {
  SCRAM_CC_REF( _cc_out, "IIRScram_EnumerationLiteral::_publish_cc_value" );

  ASSERT(get_subtype() != NULL);
  IIR_EnumerationTypeDefinition *enum_subtype 
    = dynamic_cast<IIR_EnumerationTypeDefinition *>(get_subtype());
  ASSERT( enum_subtype != 0 );
    
  _cc_out << enum_subtype->enumeration_literals._get_ordinal_position( static_cast<IIR_EnumerationLiteral *>(this) );
}

void 
IIRScram_EnumerationLiteral::_publish_cc_initialization_value( published_file &_cc_out ) {
  _publish_cc_rvalue( _cc_out );
}

void 
IIRScram_EnumerationLiteral::_publish_cc_state_object_init( published_file &_cc_out ) {
  _cc_out << "new ";
  _publish_cc_lvalue( _cc_out );
}

void 
IIRScram_EnumerationLiteral::_publish_cc_condition( published_file &_cc_out ) {
  _publish_cc_rvalue( _cc_out );
}

void 
IIRScram_EnumerationLiteral::_publish_cc_wait_data( published_file &_cc_out ) {
  _publish_cc_lvalue( _cc_out );
}

// Currently direction is implemented as an enumeration literal, where
// `to' is represented by IIR_EnumerationLiteral::declarator = "true", and
// `downto' is represented by IIR_EnumerationLiteral::declarator = "false"
// This method checks this condition and publishes the value of direction
// correspondingly. -- SK on Tue Mar 4.
void 
IIRScram_EnumerationLiteral::_publish_cc_direction( published_file &_cc_out ) {

  SCRAM_CC_REF( _cc_out, "IIRScram_EnumerationLiteral::_publish_cc_direction" );
  
  if(IIRScram_TextLiteral::_cmp(get_declarator(), "true") == 0) {
    _cc_out << "ArrayInfo::to";
  } else if(IIRScram_TextLiteral::_cmp(get_declarator(), "false") == 0) {
    _cc_out << "ArrayInfo::downto";
  } else {
    cerr << "Invalid direction: Neither`to' nor `downto'" << endl;
  }
}

void 
IIRScram_EnumerationLiteral::_publish_cc_headers( published_file &_cc_out ) {

  SCRAM_CC_REF( _cc_out, "IIRScram_EnumerationLiteral::_publish_cc_headers" );
  
  ASSERT(get_subtype() != NULL);
  //  get_subtype()->_publish_cc_headers( _cc_out );
}

void
IIRScram_EnumerationLiteral::_publish_cc_ams_function( published_file &_cc_out ) {
  if( _get_currently_publishing_unit() == SIMULTANEOUS_IF ) {
    _publish_cc_lvalue(_cc_out);
   }
}

IIRScram_Declaration::declaration_type 
IIRScram_EnumerationLiteral::_get_type() {
  return LITERAL;
}

set<IIR_TypeDefinition> *
IIRScram_EnumerationLiteral::_get_rval_set(IIR_Boolean (IIR::*constraint_function)()){
  
  set<IIR_TypeDefinition> *retval = NULL;

  if( _is_resolved() == true ){
    ASSERT( get_subtype() != NULL );
    retval = new set<IIR_TypeDefinition>( get_subtype() );
  }
  else{
    IIR_Declaration *current_decl;
    set<IIR_Declaration> *my_decls = _symbol_lookup();
    if( my_decls == NULL ){
      return NULL;
    }

    current_decl = my_decls->get_element();
    while( current_decl != NULL ){
      if( constraint_function == 0 || (current_decl->*constraint_function)() == TRUE ){
	if( retval == NULL ){
	  retval = new set<IIR_TypeDefinition>;
	}
      
	retval->add( current_decl->get_subtype() );
      } 
      current_decl = my_decls->get_next_element();
    }

    delete my_decls;
  }
  
  return retval;
}

set<IIR_Declaration> *
IIRScram_EnumerationLiteral::_symbol_lookup(){
  set<IIR_Declaration> *retval;
  retval = new set<IIR_Declaration>(*_get_symbol_table()->find_set( get_declarator() ));
  
  return retval;
}

ostream &
IIRScram_EnumerationLiteral::_print( ostream &os ){
  os << *get_declarator();

  return os;
}

IIR *
IIRScram_EnumerationLiteral::_rval_to_decl( IIR_TypeDefinition *my_rval ){
  ASSERT( my_rval != NULL );

  IIR_Declaration *retval = NULL;
  
  if( get_subtype() == NULL ){
    set<IIR_Declaration> *my_decls = _symbol_lookup();
    ASSERT( my_decls != NULL );

    IIR_Declaration *current_decl = my_decls->get_element();
    while( current_decl != NULL ){
      if( current_decl->get_subtype()->_is_compatible( my_rval ) != NULL ){
	retval = current_decl;
	break;
      }
      current_decl = my_decls->get_next_element();
    }

    ASSERT( retval != NULL );
  
    delete my_decls;
  }
  else{
    ASSERT( my_rval->_is_compatible( get_subtype() ) != NULL );
    retval = this;
  }

  return retval;
}

void 
IIRScram_EnumerationLiteral::_type_check( set<IIR_TypeDefinition> * ){
}

void
IIRScram_EnumerationLiteral::_publish_cc_bounds( published_file &_cc_out ) {

  SCRAM_CC_REF( _cc_out, "IIRScram_EnumerationLiteral::_publish_cc_bounds" );
  
  get_subtype()->_publish_cc_bounds( _cc_out );
}

void
IIRScram_EnumerationLiteral::_publish_cc_data( published_file &_cc_out ) {

  SCRAM_CC_REF( _cc_out, "IIRScram_EnumerationLiteral::_publish_cc_data" );
  
  //  _cc_out << *get_declarator();

  if(get_subtype() != NULL) {
  get_subtype()->_publish_cc_lvalue( _cc_out );
  _cc_out << "(ObjectBase::VARIABLE, ";
  }

  if(get_subtype() != NULL) {
    get_subtype()->_publish_cc_lvalue( _cc_out );
    _cc_out << "::enum";
  }

  ((IIR_StringLiteral *) _get_declarator())->_publish_cc_variable_name( _cc_out );

  if(get_subtype() != NULL) {
  _cc_out << ")";
  }
  //  _cc_out << "state.current->" << *get_declarator();
}

IIR_Boolean
IIRScram_EnumerationLiteral::_is_ascending_range() {
  if(IIRScram_TextLiteral::_cmp(get_declarator(), "true") == 0) {
    return TRUE;		// Temporary solution -- SK.
  } else if(IIRScram_TextLiteral::_cmp(_get_declarator(), "false") == 0) {
    return FALSE;
  } else {
    cerr << "Unknown enumeration literal in IIRScram_EnumerationLiteral::"
	 << "_is_ascending_range()" << endl;
    abort();
    return FALSE;
  }
}

IIR_Boolean 
IIRScram_EnumerationLiteral::_is_resolved(){
  if( get_subtype() == NULL ){
    return FALSE;
  }
  else{
    return TRUE;
  }
}

// According to the LRM Section 13.5, a character literal is
//   character_literal ::= 'graphic_character'
IIR_Boolean 
IIRScram_EnumerationLiteral::_is_character_literal() {
  IIR_TextLiteral *my_text_lit = get_declarator();

  if(my_text_lit->get_text_length() == 3 &&
     (*my_text_lit)[0] == '\'' && (*my_text_lit)[2] == '\'') {
    return TRUE;
  }
  return FALSE;
}

IIR*
IIRScram_EnumerationLiteral::_clone() {
  return this;
}

IIR_Boolean 
IIRScram_EnumerationLiteral::_is_homograph_of( IIR_Declaration *compare_to ){
  if( get_kind() != compare_to->get_kind() ){
    return FALSE;
  }
  
  if( get_subtype()->_is_compatible( compare_to->get_subtype() ) == NULL ){
    return FALSE;
  }

  return TRUE;
}

visitor_return_type *IIRScram_EnumerationLiteral::_accept_visitor(node_visitor *visitor, visitor_argument_type *arg) {
  ASSERT(visitor != NULL);
  return visitor->visit_IIR_EnumerationLiteral(this, arg);
};
