/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdlib.h>

#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>
#include "sg_edit_dialog.h"
#include "sg.h"
#include "sg_dataset.h"

#define DATASET_NAME_LENGTH 100

static gint default_nargs[] = {2, 2, 2, 2, 2, 3, 4, 3, 3, 3, 3, 3, 3, 3, 3};
static gint nargs[] = {2, 2, 2, 2, 2, 3, 4, 3, 3, 3, 3, 3, 3, 3, 3};

static gchar *column_labels[][10] = {
  {"X", "Y", "Z", "", "Xerr", "Yerr", "Zerr", "", "Lbls"},
  {"X", "Y", "", "", "", "", "", "", ""},
  {"X", "Y", "", "", "", "", "", "", ""},
  {"X", "Y", "Amp", "", "", "", "Aerr", "", ""},
  {"X", "Y", "Amp", "", "", "", "Aerr", "", ""},
  {"X", "Y", "Z", "Amp", "", "", "", "", ""},
  {"X", "Y", "Z", "", "DX", "DY", "DZ", "Amp", ""},
  {"X", "Y", "Z", "Size", "", "", "", "Amp", ""},
  {"X", "Y", "Z", "Amp", "", "", "", "", ""},
  {"X", "Y", "Z", "Amp", "", "", "", "", ""},
  {"X", "Y", "Z", "Amp", "", "", "", "", ""},
  {"X", "Y", "Z", "", "", "", "", "", ""},};

static gchar *dialog_names_2d[] = {"Worksheet",
                                   "Functions",
                                   "Python variable/expression",
                                   NULL};

static gchar *dialog_names_3d[] = {"Worksheet",
                                   "Matrix",
                                   "Python variable/expression",
                                    NULL};



static GtkWidget *column_entry[9];
static GtkWidget *left_button[9];
static GtkWidget *right_button[9];
static GtkWidget *entries_table = NULL;
static GtkWidget *entries_box = NULL;
static GtkWidget *exp_entry = NULL;
static GtkWidget *toggle_combo = NULL;
static gint toggle_selection = 0;
static SGdataStyle data_style;
static SGlayerType layer_type;
static gint num_entries;
static gint default_columns_mask[] = { 
                              SG_STYLE_LPOINTS_MASK,
                              SG_STYLE_VBARS_MASK,
                              SG_STYLE_HBARS_MASK,
                              SG_STYLE_VBOXES_MASK,
                              SG_STYLE_HBOXES_MASK,
                              SG_STYLE_BUBBLES_MASK,
                              SG_STYLE_FLUX_MASK,
                              SG_STYLE_COLORS_MASK,
                              SG_STYLE_CONTOUR_MASK,
                              SG_STYLE_CONTOUR_MASK,
                              SG_STYLE_CONTOUR_MASK,
                              SG_STYLE_CONTOUR_MASK };
static gint columns_mask[] = {
                              SG_STYLE_LPOINTS_MASK,
                              SG_STYLE_VBARS_MASK,
                              SG_STYLE_HBARS_MASK,
                              SG_STYLE_VBOXES_MASK,
                              SG_STYLE_HBOXES_MASK,
                              SG_STYLE_BUBBLES_MASK,
                              SG_STYLE_FLUX_MASK,
                              SG_STYLE_COLORS_MASK,
                              SG_STYLE_CONTOUR_MASK,
                              SG_STYLE_CONTOUR_MASK,
                              SG_STYLE_CONTOUR_MASK,
                              SG_STYLE_CONTOUR_MASK };

static SGdataset *the_dataset = NULL;

static GtkWidget *entries[9];

static void select_exp                          	(GtkWidget *widget,
                                                         gint row, gint col,
                                                         gpointer data);
static void init_exp					();
static gint focus_next					(GtkWidget *widget, 
							 GtkWidget *next);


SGdataset *
sg_edit_exp_update()
{
  gchar name[DATASET_NAME_LENGTH] = "";
  gint i, row = 0, narg = 0;
  gint dataset_columns[9];
  gchar column_text[9][100];
  gchar *column_name[9];
  gboolean old_dataset = FALSE;

  if(the_dataset) old_dataset = TRUE;

  narg=0;
  for(i = 0; i < 9; i++){
    dataset_columns[i] = -1;
    if(columns_mask[data_style] & (1 << i)){
#ifdef WITH_GNOME
       column_name[i] = gtk_entry_get_text(GTK_ENTRY(gnome_entry_gtk_entry(GNOME_ENTRY(entries[i]))));
#else
       column_name[i] = gtk_entry_get_text(GTK_ENTRY(entries[i]));
#endif
       narg++;
       if (strlen(column_name[i])>0)
       { 
         if (i>0) 
           g_snprintf(column_text[i], 100,",%s",column_name[i]);
         else 
           g_snprintf(column_text[i], 100,"%s",column_name[i]);
       }
       else
       { 
         column_name[i]=NULL;
         sprintf(column_text[i],"");
       }

    }
    else
    { 
        column_name[i]=NULL;
        sprintf(column_text[i],"");
    }
  }

  if(narg < nargs[data_style]) return NULL;

  if(old_dataset)
    sg_dataset_set_python(the_dataset, column_name);
  else
    the_dataset = sg_dataset_new_python(data_style, column_name);

  g_snprintf(name,100,"EXPR->(%s%s%s%s%s%s%s)", column_text[0],
                                     		column_text[1],
                                     		column_text[2],
                                     		column_text[3],
                                     		column_text[4],
                                     		column_text[5],
                                     		column_text[6]);

  sg_dataset_set_name(the_dataset, name);

  if(old_dataset) return the_dataset;

  switch(toggle_selection){
    case 3:
      the_dataset->real_data->symbol.symbol_type = GTK_PLOT_SYMBOL_IMPULSE;
      break;
    case 1:
      the_dataset->real_data->line.line_style = GTK_PLOT_LINE_SOLID;
      the_dataset->real_data->symbol.symbol_type = GTK_PLOT_SYMBOL_NONE;
      break;
    case 2:
      the_dataset->real_data->symbol.symbol_type = GTK_PLOT_SYMBOL_SQUARE;
      the_dataset->real_data->line.line_style = GTK_PLOT_LINE_SOLID;
      break;
    case 0:
    default:
      the_dataset->real_data->symbol.symbol_type = GTK_PLOT_SYMBOL_SQUARE;
      the_dataset->real_data->line.line_style = GTK_PLOT_LINE_NONE;
  }

  return the_dataset;
}

static gint
focus_next(GtkWidget *widget, GtkWidget *next)
{
    gtk_widget_grab_focus(next);
    return TRUE;
}

static gint
set_exp(GtkWidget *widget, gpointer data)
{
  GList *selection;
  gchar *text;
  gint row;

  text = gtk_entry_get_text(GTK_ENTRY(data));
  if(strlen(text) > 0) return TRUE;

  text = gtk_entry_get_text(GTK_ENTRY(exp_entry));
  gtk_signal_emit_by_name(GTK_OBJECT(exp_entry),"activate",NULL);
  if(!strlen(text)) return TRUE;
  gtk_entry_set_text(GTK_ENTRY(data), text);
  gtk_editable_delete_text(GTK_EDITABLE(exp_entry),0,-1);
  return TRUE;
}

static gint
restore_exp(GtkWidget *widget, gpointer data)
{
  GList *selection;
  gchar *text;
  gint row;

  text = gtk_entry_get_text(GTK_ENTRY(data));
  if(!strlen(text)) return TRUE;

  gtk_editable_delete_text(GTK_EDITABLE(data),0,-1);
  return TRUE;
}

GtkWidget *
sg_edit_exp_dialog (SGdataset *dataset, SGlayerType type)
{
  GtkWidget *frame, *vbox;
  GtkWidget *label;
  GtkWidget *table;
  GtkWidget *box;
  gint i;

  the_dataset = dataset;
  if(dataset)
    data_style = dataset->style;
  else
    data_style = (SGdataStyle)0;

  layer_type = type;
  
  for(i = 0; i < 12; i++){
     columns_mask[i] = default_columns_mask[i];
     nargs[i] = default_nargs[i];
  }

  if(layer_type == SG_LAYER_3D){
     columns_mask[SG_STYLE_LPOINTS] |= GTK_PLOT_DATA_Z | GTK_PLOT_DATA_DZ;
     columns_mask[SG_STYLE_LPOINTS] -= (GTK_PLOT_DATA_LABELS +
                                      GTK_PLOT_DATA_DX + GTK_PLOT_DATA_DY);
     columns_mask[SG_STYLE_BUBBLES] |= GTK_PLOT_DATA_Z;
     columns_mask[SG_STYLE_COLORS] |= GTK_PLOT_DATA_Z;
     columns_mask[SG_STYLE_FLUX] |= GTK_PLOT_DATA_Z | GTK_PLOT_DATA_DZ;
  
     nargs[SG_STYLE_LPOINTS]++;
     nargs[SG_STYLE_BUBBLES]++;
     nargs[SG_STYLE_COLORS]++;
     nargs[SG_STYLE_FLUX] += 2;
  }

  vbox = gtk_vbox_new(FALSE, 5);

/*********************************************************************/

  box = gtk_hbox_new(TRUE, 5);
  gtk_box_pack_start (GTK_BOX (vbox), box, FALSE, FALSE, 0);

  label = gtk_label_new("Dataset Style:");
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.);
  gtk_box_pack_start (GTK_BOX (box), label, FALSE, FALSE, 0);

  toggle_combo = sg_toggle_data_new(layer_type);
  gtk_box_pack_start (GTK_BOX (box), toggle_combo, FALSE, FALSE, 0);
  sg_toggle_data_init(toggle_combo, 0);

/*********************************************************************/

  frame = gtk_frame_new("Expressions");
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
  gtk_box_pack_start (GTK_BOX (vbox), frame, FALSE, FALSE, 0);

  entries_table = NULL;
  entries_box = gtk_hbox_new(TRUE, 5);
  gtk_container_set_border_width(GTK_CONTAINER(entries_box), 5);
  gtk_container_add (GTK_CONTAINER (frame), entries_box);

  if(dataset){
    init_exp();
    gtk_widget_set_sensitive(toggle_combo, FALSE);
  } else {
    select_exp(toggle_combo, 0, 0, NULL);
  }

  gtk_signal_connect(GTK_OBJECT(toggle_combo), "changed",
                     GTK_SIGNAL_FUNC(select_exp), NULL);

/********************************************************************/
  return vbox;
}

static void 
init_exp()
{
  GtkWidget *column_label;
  GtkWidget *button_box;
  gint i, nrows = 0, j, k;
  gchar label[9][100], ident[80];

  if(entries_table && GTK_IS_WIDGET(entries_table))
     gtk_container_remove(GTK_CONTAINER(entries_box), entries_table);

  toggle_selection = 0;

  switch(layer_type){ 
    case SG_LAYER_3D:
      switch(data_style){
        case SG_STYLE_LPOINTS:
          toggle_selection = 0;
          break;
        case SG_STYLE_BUBBLES:
          toggle_selection = 1;
          break;
        case SG_STYLE_COLORS:
          toggle_selection = 2;
          break;
        case SG_STYLE_FLUX:
          toggle_selection = 3;
          break;
        case SG_STYLE_SURFACE:
          toggle_selection = 4;
          break;
        case SG_STYLE_CSURFACE:
          toggle_selection = 5;
          break;
      }
      break;
    case SG_LAYER_POLAR:
      switch(data_style){
        case SG_STYLE_LPOINTS:
          toggle_selection = 2;
          break;
        case SG_STYLE_BUBBLES:
          toggle_selection = 4;
          break;
        case SG_STYLE_COLORS:
          toggle_selection = 5;
          break;
      }
      break;
    case SG_LAYER_2D:
      switch(data_style){
        case SG_STYLE_LPOINTS:
          toggle_selection = 2;
          break;
        case SG_STYLE_VBARS:
          toggle_selection = 4;
          break;
        case SG_STYLE_HBARS:
          toggle_selection = 5;
          break;
        case SG_STYLE_VBOXES:
          toggle_selection = 6;
          break;
        case SG_STYLE_HBOXES:
          toggle_selection = 7;
          break;
        case SG_STYLE_BUBBLES:
          toggle_selection = 8;
          break;
        case SG_STYLE_COLORS:
          toggle_selection = 9;
          break;
        case SG_STYLE_FLUX:
          toggle_selection = 10;
          break;
        case SG_STYLE_CONTOUR:
          toggle_selection = 11;
          break;
      }
    break;
  }

  sg_toggle_data_init(toggle_combo, toggle_selection);

  entries_table = gtk_table_new(num_entries, 2, FALSE);
  gtk_table_set_row_spacings(GTK_TABLE(entries_table), 5);
  gtk_table_set_col_spacings(GTK_TABLE(entries_table), 5);

  for(i = 0; i < 9; i++){
    sprintf(label[i], "%s:", column_labels[data_style][i]);
    if(columns_mask[data_style] & (1 << i)){

      column_label = gtk_label_new(label[i]);
      gtk_misc_set_alignment(GTK_MISC(column_label), .5, .5);
      gtk_table_attach_defaults(GTK_TABLE(entries_table), column_label,
                                0, 1, nrows, nrows + 1);
#ifdef WITH_GNOME
      g_snprintf(ident,80,"sg_dataset_exp_%s",label[i]);
      entries[i]=gnome_entry_new(ident);
#else
      entries[i] = gtk_entry_new();
#endif

      gtk_table_attach_defaults(GTK_TABLE (entries_table), entries[i], 
                                1, 2, nrows, 1+nrows);

      if(active_dataset){
        if(active_dataset->p_exp[i])
#ifdef WITH_GNOME
          gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(entries[i])->entry), active_dataset->p_exp[i]); 
#else
          gtk_entry_set_text(GTK_ENTRY(entries[i]), active_dataset->p_exp[i]); 
#endif
      } else {
#ifdef WITH_GNOME
          gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(entries[i])->entry), ""); 
#else
          gtk_entry_set_text(GTK_ENTRY(entries[i]), ""); 
#endif
      }

      nrows++;
    }
    else entries[i]=NULL;
  }

  for (j=0;j<nrows-1;j++)
  {   
    if (!entries[j]) continue;
    for (k=j+1;k<nrows;k++)
       if (entries[k]) break;
    if (k>=nrows) break;

#ifdef WITH_GNOME
      gtk_signal_connect(GTK_OBJECT(gnome_entry_gtk_entry(GNOME_ENTRY(entries[j]))),
                         "activate",(GtkSignalFunc) focus_next,
                         (gpointer)gnome_entry_gtk_entry(GNOME_ENTRY(entries[k])));
#else
      gtk_signal_connect(GTK_OBJECT(entries[j]), "activate",
                         (GtkSignalFunc) focus_next, (gpointer)entries[k]);
#endif
  }

  gtk_box_pack_start(GTK_BOX(entries_box), entries_table, FALSE, FALSE, 0);
  gtk_widget_show_all(entries_table);
}


static void 
select_exp(GtkWidget *widget, gint row, gint col, gpointer data)
{
  GtkWidget *column_label;
  GtkWidget *button_box;
  gint i, nrows = 0, j, k;
  gchar label[9][100], ident[80];

  toggle_selection = sg_toggle_data_get_selection(widget);

  if(entries_table && GTK_IS_WIDGET(entries_table))
     gtk_container_remove(GTK_CONTAINER(entries_box), entries_table);

  switch(toggle_selection){
    case 0:
      if(layer_type == SG_LAYER_3D){
         data_style = SG_STYLE_LPOINTS;
         break;
      }
    case 1:
      if(layer_type == SG_LAYER_3D){
         data_style = SG_STYLE_BUBBLES;
         break;
      }
    case 2:
      if(layer_type == SG_LAYER_3D){
         data_style = SG_STYLE_COLORS;
         break;
      }
    case 3:
      if(layer_type == SG_LAYER_3D){
         data_style = SG_STYLE_FLUX;
         break;
      }
      data_style = SG_STYLE_LPOINTS;
      break;
    case 4:
      if(layer_type == SG_LAYER_3D){
         data_style = SG_STYLE_SURFACE;
         break;
      }
      if(layer_type == SG_LAYER_POLAR){
         data_style = SG_STYLE_BUBBLES;
         break;
      }
      data_style = SG_STYLE_VBARS;
      break;
    case 5:
      if(layer_type == SG_LAYER_3D){
         data_style = SG_STYLE_CSURFACE;
         break;
      }
      if(layer_type == SG_LAYER_POLAR){
         data_style = SG_STYLE_COLORS;
         break;
      }
      data_style = SG_STYLE_HBARS;
      break;
    case 6:
      data_style = SG_STYLE_VBOXES;
      break;
    case 7:
      data_style = SG_STYLE_HBOXES;
      break;
    case 8:
      data_style = SG_STYLE_BUBBLES;
      break;
    case 9:
      data_style = SG_STYLE_COLORS;
      break;
    case 10:
      data_style = SG_STYLE_FLUX;
      break;
    case 11:
      data_style = SG_STYLE_CONTOUR;
      break;
  }

  entries_table = gtk_table_new(num_entries, 2, FALSE);
  gtk_table_set_row_spacings(GTK_TABLE(entries_table), 5);
  gtk_table_set_col_spacings(GTK_TABLE(entries_table), 5);

  for(i = 0; i < 9; i++){
    sprintf(label[i], "%s:", column_labels[data_style][i]);
    if(columns_mask[data_style] & (1 << i)){

      column_label = gtk_label_new(label[i]);
      gtk_misc_set_alignment(GTK_MISC(column_label), .5, .5);
      gtk_table_attach_defaults(GTK_TABLE(entries_table), column_label,
                                0, 1, nrows, nrows + 1);
#ifdef WITH_GNOME
      g_snprintf(ident,80,"sg_dataset_exp_%s",label[i]);
      entries[i] = gnome_entry_new(ident);
#else
      entries[i] = gtk_entry_new();
#endif
      gtk_table_attach_defaults(GTK_TABLE (entries_table), entries[i], 
                                1, 2, nrows, 1+nrows);

      if(active_dataset){
#ifdef WITH_GNOME
         gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(entries[i])->entry), active_dataset->p_exp[i]); 
#else
         gtk_entry_set_text(GTK_ENTRY(entries[i]), active_dataset->p_exp[i]); 
#endif
      } else {
#ifdef WITH_GNOME
          gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(entries[i])->entry), ""); 
#else
          gtk_entry_set_text(GTK_ENTRY(entries[i]), ""); 
#endif
      }

      nrows++;
    }
    else entries[i]=NULL;
  }

  for (j=0;j<nrows-1;j++)
  {   
    if (!entries[j]) continue;
    for (k=j+1;k<nrows;k++)
       if (entries[k]) break;
    if (k>=nrows) break;

#ifdef WITH_GNOME
      gtk_signal_connect(GTK_OBJECT(gnome_entry_gtk_entry(GNOME_ENTRY(entries[j]))),
                         "activate",(GtkSignalFunc) focus_next,
                         (gpointer)gnome_entry_gtk_entry(GNOME_ENTRY(entries[k])));
#else
      gtk_signal_connect(GTK_OBJECT(entries[j]), "activate",
                         (GtkSignalFunc) focus_next, (gpointer)entries[k]);
#endif
  }

  gtk_box_pack_start(GTK_BOX(entries_box), entries_table, FALSE, FALSE, 0);
  gtk_widget_show_all(entries_table);
}

