/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <math.h>
#include <stdio.h>
#include <gdk/gdk.h>
#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>

#include "sg_plot_menu.h"
#include "sg.h"  
#include "sg_dialogs.h"
#include "sg_layer.h"
#include "sg_text_dialog.h"   
#include "sg_clipboard.h"

#ifdef WITH_GNOME   
#include <gnome.h>
#include <libgnomeprint/gnome-print.h>
#include "gtkplotgnome.h"
#endif /* WITH GNOME */   

static gpointer sg_menu_fit_page		(SGplugin *plugin);
static gpointer sg_menu_fit_page_h		(SGplugin *plugin);
static gpointer sg_menu_fit_page_v		(SGplugin *plugin);
static gpointer sg_menu_zoom_in			(SGplugin *plugin);
static gpointer sg_menu_zoom_out		(SGplugin *plugin);
static gpointer sg_menu_export			(SGplugin *plugin);
static gpointer sg_menu_import			(SGplugin *plugin);
static gpointer sg_menu_new_layer		(SGplugin *plugin);
static gpointer sg_menu_new_layer_3d		(SGplugin *plugin);
static gpointer sg_menu_new_layer_polar		(SGplugin *plugin);
static gpointer sg_menu_clone_layer		(SGplugin *plugin);
static gpointer sg_menu_remove_layer		(SGplugin *plugin);
static gpointer sg_menu_arrange			(SGplugin *plugin);
static gpointer sg_menu_layer_control		(SGplugin *plugin);
static gpointer sg_menu_copy_page		(SGplugin *plugin);
static gpointer sg_menu_paste			(SGplugin *plugin);
static gpointer sg_menu_image			(SGplugin *plugin);
static gpointer sg_menu_view_normal		(SGplugin *plugin);
static gpointer sg_menu_view_antialias		(SGplugin *plugin);
static gpointer sg_menu_add_plot		(SGplugin *plugin);
static gpointer sg_menu_page			(SGplugin *plugin);

#ifdef WITH_GNOME_PRINT
#ifdef WITH_GNOME
static gpointer sg_menu_preview			(SGplugin *plugin);
#endif /* WITH GNOME */
#endif /* WITH GNOME_PRINT */

void
sg_plot_menu_show(GtkWidget *widget, GdkEventButton *event)
{
  GList *w;
  SGplot *plot = NULL;
  GtkWidget *data;
  GdkModifierType mods;

  w = plots;
  while(w){
    plot = (SGplot *)w->data;
    data = plot->real_canvas;
    if(data && data == widget) break;
    w = w->next;
  }

  active_plot = plot;

  gdk_window_get_pointer(widget->window, NULL, NULL, &mods);
  if(!(mods & GDK_BUTTON3_MASK)) return;

  gtk_menu_popup(GTK_MENU(plot_menu), NULL, NULL, NULL, NULL,
                 event->button, event->time);
}

void
sg_plot_build_menu()
{
  sg_plugin_new_c("Plot dataset", "Plot:", sg_menu_add_plot);

  /* FILE */

  sg_plugin_group_new("Plot:File:");
  sg_plugin_new_c("Export", "Plot:File:", sg_menu_export);
  sg_plugin_new_c("Import", "Plot:File:", sg_menu_import);
#ifdef WITH_GNOME_PRINT
#ifdef WITH_GNOME
  sg_plugin_new_c("Print preview", "Plot:File:", sg_menu_preview);
#endif
#endif

  /* EDIT */

  sg_plugin_group_new("Plot:Edit:");
  sg_plugin_new_c("Copy page", "Plot:Edit:", sg_menu_copy_page);
  sg_plugin_new_c("Paste", "Plot:Edit:", sg_menu_paste);

  /* VIEW */

  sg_plugin_group_new("Plot:View:");
  sg_plugin_new_c("Window view", "Plot:View:", sg_menu_view_normal);
#ifdef WITH_LIBART
  sg_plugin_new_c("Antialiased view", "Plot:View:", sg_menu_view_antialias);
  sg_plugin_new_c("Show whole page", "Plot:View:", sg_menu_fit_page);
  sg_plugin_new_c("Fit page to window width", "Plot:View:", sg_menu_fit_page_h);
  sg_plugin_new_c("Fit page to window height", "Plot:View:", sg_menu_fit_page_v);
  sg_plugin_new_c("Zoom in", "Plot:View:", sg_menu_zoom_in);
  sg_plugin_new_c("Zoom out", "Plot:View:", sg_menu_zoom_out);

#endif /* WITH_LIBART */

  /* LAYER */

  sg_plugin_group_new("Plot:Layer:");
  sg_plugin_group_new("Plot:Layer:New layer:");
  sg_plugin_new_c("2D", "Plot:Layer:New layer:", sg_menu_new_layer);
  sg_plugin_new_c("3D", "Plot:Layer:New layer:", sg_menu_new_layer_3d);
  sg_plugin_new_c("Polar", "Plot:Layer:New layer:", sg_menu_new_layer_polar);
  sg_plugin_new_c("Clone Layer", "Plot:Layer:", sg_menu_clone_layer);
  sg_plugin_new_c("Remove layer", "Plot:Layer:", sg_menu_remove_layer);
  sg_plugin_new_c("Layer control", "Plot:Layer:", sg_menu_layer_control);

  /* PAGE */

  sg_plugin_group_new("Plot:Page:");
  sg_plugin_new_c("Arrange all layers", "Plot:Page:", sg_menu_arrange);
  sg_plugin_new_c("Properties", "Plot:Page:", sg_menu_page);

  /* INSERT */

  sg_plugin_group_new("Plot:Insert:");
  sg_plugin_new_c("Image from file", "Plot:Insert:", sg_menu_image);

  /* PLUG-INS */

/*
  sg_plugin_group_new("Plot:Plugins:");
*/
}

static gpointer
sg_menu_fit_page(SGplugin *plugin)
{
  sg_plot_fit_page(active_plot);
}

static gpointer
sg_menu_fit_page_h(SGplugin *plugin)
{
  sg_plot_fit_page_h(active_plot);
}

static gpointer
sg_menu_fit_page_v(SGplugin *plugin)
{
  sg_plot_fit_page_v(active_plot);
}


static gpointer
sg_menu_arrange(SGplugin *plugin)
{
  sg_arrange_dialog(GTK_PLOT_CANVAS(active_plot->real_canvas));
}

static gpointer
sg_menu_clone_layer(SGplugin *plugin)
{
  SGlayer *layer;

  layer = sg_layer_clone(active_plot->active_layer);
  sg_plot_add_layer(active_plot, layer, .175, .15);

  gtk_plot_canvas_paint(GTK_PLOT_CANVAS(active_plot->real_canvas));
  gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(active_plot->real_canvas));
}

static gpointer
sg_menu_new_layer(SGplugin *plugin)
{
  SGlayer *layer;

  layer = sg_layer_new(SG_LAYER_2D, .65, .45);
  sg_plot_add_layer(active_plot, layer, .175, .15);

  gtk_plot_canvas_paint(GTK_PLOT_CANVAS(active_plot->real_canvas));
  gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(active_plot->real_canvas));
}

static gpointer
sg_menu_new_layer_3d(SGplugin *plugin)
{
  SGlayer *layer;

  layer = sg_layer_new(SG_LAYER_3D, .65, .45);
  sg_plot_add_layer(active_plot, layer, .175, .15);

  gtk_plot_canvas_paint(GTK_PLOT_CANVAS(active_plot->real_canvas));
  gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(active_plot->real_canvas));
}

static gpointer
sg_menu_new_layer_polar(SGplugin *plugin)
{
  SGlayer *layer;

  layer = sg_layer_new(SG_LAYER_POLAR, .75, .45);
  sg_plot_add_layer(active_plot, layer, .175, .15);

  gtk_plot_canvas_paint(GTK_PLOT_CANVAS(active_plot->real_canvas));
  gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(active_plot->real_canvas));
}

static gpointer
sg_menu_remove_layer(SGplugin *plugin)
{
  SGlayer *layer;
  GList *layers = NULL;
  gint selected_layer_nr = 0;
  gchar message[250];

  layer = active_plot->active_layer;
  layers = active_plot->layers;

  while(layers) {
     selected_layer_nr++;
     if(layers->data == layer)
       break;
     else
       layers = layers->next;
  }

  g_snprintf(message, 250, "Remove selected layer nr. %d?", selected_layer_nr);
  if(sg_accept_dialog(message, 1) == YES_CLICKED)
       sg_plot_remove_layer(active_plot, layer);

  if(active_plot->nlayers > 0 && active_plot->layers)
    sg_plot_set_active_layer(active_plot, (SGlayer *)active_plot->layers->data); 
}

static gpointer
sg_menu_layer_control(SGplugin *plugin)
{
  SGlayer *layer;
  GList *llink;
  gchar path[1000];

  layer = active_plot->active_layer;
  llink = g_list_find(active_plot->layers, layer);
  sprintf(path, "%s:%d", active_plot->name,
          g_list_position(active_plot->layers, llink)+1);

  sg_layer_control(path);
}

static gpointer
sg_menu_import(SGplugin *plugin)
{
/*
  sg_message_dialog("Sorry, this feature is not implemented", 2);
*/
  sg_plot_import();
}

static gpointer
sg_menu_export(SGplugin *plugin)
{
  sg_plot_export();
}

static gpointer
sg_menu_copy_page(SGplugin *plugin)
{
  sg_clipboard_copy_pixmap(active_plot);
}

static gpointer
sg_menu_paste(SGplugin *plugin)
{
  sg_clipboard_paste_pixmap(active_plot, 0., 0.);
}

static gpointer
sg_menu_image(SGplugin *plugin)
{
  GdkPixmap *pixmap;

  pixmap = sg_image_import();
  if(pixmap) sg_plot_put_pixmap(active_plot, pixmap);
}

#ifdef WITH_GNOME_PRINT
#ifdef WITH_GNOME
static gpointer
sg_menu_preview(SGplugin *plugin)
{
  GtkWidget *toplevel, *gcanvas, *sw;
  GtkPlotCanvas *canvas;
  GtkPlotPC *pc;
  GList *plots;
  gint old_width, old_height;
  GtkAllocation allocation[100];
  gdouble m;
  gint n;

  canvas = GTK_PLOT_CANVAS(active_plot->real_canvas);
  m = canvas->magnification;
  old_width = canvas->pixmap_width;
  old_height = canvas->pixmap_height;

  canvas->magnification = 1.0;
  canvas->pixmap_width = canvas->width;
  canvas->pixmap_height = canvas->height;

  n = 0;
  plots = canvas->plots;
  while(plots)
     {
       GtkPlot *plot;

       plot = (GtkPlot *)plots->data;

       plot->magnification =  1.0;

       allocation[n].width = GTK_WIDGET(plot)->allocation.width;
       allocation[n].height = GTK_WIDGET(plot)->allocation.height;
       n++;

       GTK_WIDGET(plot)->allocation.width = canvas->width;
       GTK_WIDGET(plot)->allocation.height = canvas->height;

       plots = plots->next;
     }

  gtk_widget_set_default_colormap (gdk_rgb_get_cmap ());
  gtk_widget_set_default_visual (gdk_rgb_get_visual ());

  toplevel = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_widget_set_usize (toplevel, canvas->width, canvas->height);

  sw = gtk_scrolled_window_new (NULL, NULL);
  gcanvas = gnome_canvas_new_aa ();
  gtk_container_add (GTK_CONTAINER (toplevel), sw);
  gtk_container_add (GTK_CONTAINER (sw), gcanvas);

  gtk_widget_show_all (toplevel);
  pc = canvas->pc; 
  canvas->pc = GTK_PLOT_PC(gtk_plot_gnome_preview_new(GNOME_CANVAS(gcanvas), 0, active_plot->page_size));


  gtk_plot_canvas_paint(canvas);

  gtk_object_destroy(GTK_OBJECT(canvas->pc));
  canvas->pc = pc;

  canvas->magnification = m;
  canvas->pixmap_width = old_width;
  canvas->pixmap_height = old_height;

  n = 0;
  plots = canvas->plots;
  while(plots)
     {
       GtkPlot *plot;

       plot = (GtkPlot *)plots->data;
       plot->magnification =  m;

       GTK_WIDGET(plot)->allocation.width = allocation[n].width;
       GTK_WIDGET(plot)->allocation.height = allocation[n].height;
       n++;

       plots = plots->next;
     }
}
#endif /* WITH_GNOME */
#endif /* WITH_GNOME_PRINT */


static gpointer
sg_menu_view_normal(SGplugin *plugin)
{
  sg_plot_antialias(active_plot, FALSE);
  gtk_plot_canvas_paint(GTK_PLOT_CANVAS(active_plot->real_canvas));
  gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(active_plot->real_canvas));
}

static gpointer
sg_menu_view_antialias(SGplugin *plugin)
{
  sg_plot_antialias(active_plot, TRUE);
  gtk_plot_canvas_paint(GTK_PLOT_CANVAS(active_plot->real_canvas));
  gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(active_plot->real_canvas));
}

static gpointer
sg_menu_zoom_in(SGplugin *plugin)
{
  sg_plot_rescale(active_plot, active_plot->scale + .1);
}

static gpointer
sg_menu_zoom_out(SGplugin *plugin)
{
  sg_plot_rescale(active_plot, active_plot->scale - .1);
}

static gpointer
sg_menu_add_plot(SGplugin *plugin)
{
  SGlayer *layer;
  GList *llink;
  gchar path[1000];

  layer = active_plot->active_layer;
  llink = g_list_find(active_plot->layers, layer);
  sprintf(path, "%s:%d:dataset", active_plot->name,
          g_list_position(active_plot->layers, llink)+1);
  sg_layer_control(path);
}

static gpointer
sg_menu_page(SGplugin *plugin)
{
  gchar *path;

  path = active_plot->name;
  sg_layer_control(path);
}


